import java.io.*;
import java.net.*;
import java.util.*;

/*
 *------------------------------------------------------------------
 *
 * LESS Group 
 * Computer Sciences Department 
 * University of Texas at Austin
 *
 * PerstConnection ---
 *          Simple persistent connection.
 *
 * For liveness, you must dedicate a thread to pulling stuff
 * out. It should not block anywhere outside of here.
 *
 * $Date: 2003/01/06 06:16:30 $ $Id: PerstConnection.java,v 1.1.1.1 2003/01/06 06:16:30 ypraveen Exp $
 *
 * Copyright (c) 2002 by Mike Dahlin.
 * 
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for noncommercial use without fee and
 * without written agreement is hereby granted, provided that 
 * the above copyright notice appear in all copies of this software.
 * 
 *------------------------------------------------------------------
 */
public class PerstConnection{

private Socket socket = null;
private OutputStream os = null;
private LineNumberReader lnr = null;
private int maxOutstanding;   // Put blocks if more than this
private String server;
private int port;
private LinkedList list;
private int mode;
private static final int MODE_PIPELINE = 1000;
private static final int MODE_RECOVERY = 1001;

private static final boolean verbose = true;

public PerstConnection(String server_, int port_, int maxOutstanding_)
  {
    server = server_;
    port = port_;
    maxOutstanding = maxOutstanding_;
    list = new LinkedList();
    mode = MODE_PIPELINE;
    connect();
  }


/*
 *------------------------------------------------------------------
 *
 * connect --
 *
 *          description.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public void connect()
  {
    try{
      Assert.myAssert(socket == null);
      socket = new Socket(server, port);
      os = socket.getOutputStream();
      InputStream is = socket.getInputStream();
      InputStreamReader isr = new InputStreamReader(is);
      lnr = new LineNumberReader(new BufferedReader(isr));
    }
    catch(IOException e){
      System.out.println("FailedConnect:" +  e);
      e.printStackTrace();
      System.exit(-1);
     }
  }


/*
 *------------------------------------------------------------------
 *
 * put --
 *
 *          Enqueue the request.
 *          If pipelining is active (e.g., we are not in recovery
 *          mode after a dropped connection), send the request.
 *
 *    ***************
 *    NOT THREAD SAFE -- since we need to ensure that replies
 *    ***************    come back in the same order as requests
 *                       go out, need to make sure that we
 *                       send requests in the same order
 *                       that we enqueue them. (Notice that
 *                       enqueue and send cannot be atomic
 *                       since send could then block while holding 
 *                       the lock)
 *
 *            The socket returned may be null (if we are recovery
 *            mode) or it may be stale (e.g., it may get closed
 *            out from under us before or while we send). 
 *
 *            Ignore errors -- on any error, the receiving
 *            thread will notice and switch over to
 *            recovery mode and retry our enqueued request.
 *
 *            May block in doPut to limit pipelining.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
/***** NOT SYNCHRONIZED *****/
public void put(PendingRequest pr)
  {
    OutputStream oldOS = doPut(pr);
    pr.launched();
    try{
      if(oldOS != null){
	sendRequestWire(oldOS, pr.getObjId());
      }
    }
    catch(IOException e){
      // Ignore -- receive thread will go into recovery
      // mode and retry our request on a new socket
      // in non-pipelined mode
      try{
	oldOS.close();
      }
      catch(Exception f){
      }
    }
  }


/*
 *------------------------------------------------------------------
 *
 * doPut --
 *
 *      Put the request on the queue.
 *
 *      Return the output stream on which to send the
 *      request *** that was in effect when the request
 *      was enqueued ***
 *
 *      The caller must try to send the request on the returned
 *      output stream.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized OutputStream doPut(PendingRequest pr)
  {
    while(list.size() >= maxOutstanding){
      try{
	wait();
      }
      catch(InterruptedException z){
      }
    }
    list.addLast(pr);
    if(mode == MODE_PIPELINE){
      return os;
    }
    else{
      return null;
    }
  }



/*
 *------------------------------------------------------------------
 *
 * get --
 *
 *      If we are in recovery mode
 *         get work if any // note: won't wait
 *         send it
 *      wait for next reply
 *      remove successful data form list
 *
 *     Notice: we do not wait while holding lock
 *     The only place we wait is when reading from socket
 *     We return if there is an error or if the data comes back
 *        
 *     If  there is an error, the receiver thread (this thread)
 *     is responsible for retrying pending requests
 * 
 *     NOTE: for liveness, there must always be an active
 *     receiving thread
 *
 *     NOTE: not thread safe. We assume exactly one receiving thread.
 *
 *     **********************
 *       NOT THREAD SAFE
 *     **********************
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
  /******** NOT SYNCHRONIZED ********/
public PendingRequest get()
  {
    boolean success = false;
    while(!success){
      try{
	PendingRequest req = getUnsentIfAny();
	if(req != null){
	  Assert.myAssert(mode == MODE_RECOVERY);
	  //
	  // Assume send does not block since we are not pipelining
	  //
	  OutputStream theOS = getOSMayBecomeStale();
	  sendRequestWire(theOS, req.getObjId()); 
	}
	int length = getReplyWire(lnr);
	// We must have succeeded
	PendingRequest done = removeHead();
	Assert.myAssert(done != null, "We only return normally if there was something on the list");
	done.succeeded(length);
	success = true;
	return done;
      }
      catch(Exception e){
       if(verbose){
	  System.out.println("Recovering persistent conection because: " + e.toString());
       }
	recover();
	success = false;
	// Assume the head object caused the problem 
	// Remove it so we don't loop forever.
	PendingRequest done = removeHead();
	if(done != null){
	  done.failed();
	  return done;
	}
	Assert.myAssert(done == null, "Connection was torn down due to no work. Keep going.");
	success = false;
      }
    }
    Assert.myAssert(false, "NOTREACHED");
    return null;
  }

/*
 *------------------------------------------------------------------
 *
 * getOSMayBecomeStale --
 *
 *       Return the OS. We do not guarantee that it will remain
 *       live in the future. Make sure you understand the
 *       recovery model.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */

private synchronized OutputStream getOSMayBecomeStale(){
  return os;
}
/*
 *------------------------------------------------------------------
 *
 * getUnsentIfAny --
 *
 *          If we are in recovery mode, return the top (unsent)
 *          object. Assumes at most one get() thread.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized PendingRequest getUnsentIfAny()
  {
    if(mode == MODE_PIPELINE){
      return null;
    }
    if(list.size() == 0){
      return null;
    }
    return (PendingRequest)list.getFirst();
  }

/*
 *------------------------------------------------------------------
 *
 * removeHead --
 *
 *          Remove the head of the list now that it is done.
 *
 *          Assumes at most one get() thread.
 *
 *          Wake the put() thread if we now have room
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized PendingRequest removeHead()
  {
    // Not an assertion: the server may kill our connection
    // due to idleness
    //  Assert.myAssert(list.size() != 0);
    if(list.size() == 0){
      mode = MODE_PIPELINE;
      return null;
    }
    if((mode == MODE_RECOVERY) && (list.size() == 1)){
      // Recovery is complete -- finishing the last one
      mode = MODE_PIPELINE;
    }
    notify();
    return (PendingRequest)list.removeFirst();
  }

/*
 *------------------------------------------------------------------
 *
 * recover
 *
 *        Throw away the old connections and make new ones.
 *
 *        Called by receiving thread.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized void recover()
  {
    mode = MODE_RECOVERY;
    try{
      os.close();
    }
    catch(Exception e){
      System.out.println("PerstConnection recovery error cleaning up: " + e);
      e.printStackTrace();
      // Continue
    }
    finally{
      os = null;
    }
    try{
      lnr.close();
    }
    catch(Exception e){
      System.out.println("PerstConnection recovery error cleaning up: " + e);
      e.printStackTrace();
      // Continue
    }
    finally{
      lnr = null;
    }
    try{
      socket.close();      
    }
    catch(Exception e){
      System.out.println("PerstConnection recovery error cleaning up: " + e);
      e.printStackTrace();
      // Continue
    }
    finally{
      socket = null;
    }
    connect();
  }


    // added by yala to allow disconnect

public void disconnect()
  {
    mode = MODE_RECOVERY;
    try{
      os.close();
    }
    catch(Exception e){
      System.out.println("PerstConnection recovery error cleaning up: " + e);
      e.printStackTrace();
      // Continue
    }
    finally{
      os = null;
    }
    try{
      lnr.close();
    }
    catch(Exception e){
      System.out.println("PerstConnection recovery error cleaning up: " + e);
      e.printStackTrace();
      // Continue
    }
    finally{
      lnr = null;
    }
    try{
      socket.close();      
    }
    catch(Exception e){
      System.out.println("PerstConnection recovery error cleaning up: " + e);
      e.printStackTrace();
      // Continue
    }
    finally{
      socket = null;
    }
  }


/*
 *------------------------------------------------------------------
 *
 * sendRequestWire --
 *
 *       Send a request for objId on the specified output stream
 *
 *       DO NOT USE THE MEMBER VARIABLE os
 *
 *       Instead, we are given an output stream on which 
 *       to send. If things get torn down, the data 
 *       structures will assume the os they passed out
 *       will go away.
 *
 *       NOT SYNCHRONIZED -- may block on send.
 *
 *       
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
  /**** NOT SYNCHRONIZED ***/
private void sendRequestWire(OutputStream oldOS, String obj)
  throws IOException
  {
    String req = "GET http://" + server + ":" + port + "/" + obj + " HTTP/1.1\r\n"
      + "host: " + server + ":" + port + "\r\n\r\n";
    byte buf[] = req.getBytes();
    oldOS.write(buf);
    oldOS.flush();
  }

/*
 *------------------------------------------------------------------
 *
 * getReplyWire --
 *
 *      Get the next reply off the wire. Return its length.
 *      Throw an exception if there is a problem. The caller
 *      will tear down the connection, go into recovery mode.
 *
 *  *********************
 *   NOT SYNCHRONIZED -- could block
 *  **********************
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Lenght of the reply
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
  /**** NOT SYNCHRONIZED ***/
private int getReplyWire(LineNumberReader lnr)
throws IOException 
  {
    int length = -999;
    length = readHeader(lnr);
    if(length > 0){
      lnr.skip(length);  // Skip the body
    }
    if(length < 0){
      // This typically happens on 404 not found, which apache
      // returns as chunked encoding rather than encoding lenght
      // This might also happen on .cgi programs.
      // We should probably add code to understand chunked encoding
      //
      throw new IOException("Got confused by header -- no length found");
    }
    return length;    
  }

/*
 *------------------------------------------------------------------
 *
 * readHeader --
 *
 *          Read the header and return the length.
 *
 * TBD: we don't handle "chunked encoding" e.g. 404, cgi?
 *
 *  *********************
 *   NOT SYNCHRONIZED -- could block
 *  **********************
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      LIne length or throws exception on error
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
  /**** NOT SYNCHRONIZED ***/
private static int readHeader(LineNumberReader lnr)
throws IOException
  {
    int length = -1;

    while(true){
      String line = lnr.readLine();
      //	System.out.println("READ HEADER: " + line);
      int ndx = line.indexOf("Content-Length:");
      if(ndx == 0){
	String lenS = line.substring(16);
	length = (new Integer(lenS)).intValue();
      }
      if(line.length() == 0){ // Empty line signals end of header
	return length;
      }
    }
  }


private int testCount;
private int testSoFar;
private synchronized void testStart(int count_)
  {
    testCount = count_;
    testSoFar = 0;
  }
private synchronized void testWait()
  {
    while(testSoFar < testCount){
      try{
	wait();
      }
      catch(InterruptedException e){
      }
    }
  }

private synchronized void testDone()
  {
    testSoFar++;
    notifyAll();
  }

public static void main(String argv[])
  {
    //PerstConnection c = new PerstConnection("orchid.cs.utexas.edu", 9000, 10);
    PerstConnection c = new PerstConnection("balsam.cs.utexas.edu", 8083, 10);
    c.doTest();
  }

private void doTest()
  {
    testStart(2);
    String names[] = {"preload-check/imagecheck.html"};
    //String names[] = {"index.html", "index2.html"};
    //long listLen[] = {3891, 81608};
    long listLen[] = {141};
    TestPutThread t = new TestPutThread(this, names, 1000, this);
    t.start();
    TestGetThread g = new TestGetThread(this, listLen, names, 1000, this, false);
    g.start();
    testWait();
    System.out.println("Test 1 done.");


    testStart(2);
    String names2[] = {"index.html", "index2.html", "does-not-exist.404"};
    long listLen2[] = {3891, 81608, -1};
    t = new TestPutThread(this, names2, 100, this);
    t.start();
    g = new TestGetThread(this, listLen2, names2, 100, this, false);
    g.start();
    testWait();
    System.out.println("Test 2 done.");


    testStart(2);
    String names3[] = {"index.html", "index2.html"};
    long listLen3[] = {3891, 81608};
    t = new TestPutThread(this, names3, 10, this);
    t.start();
    g = new TestGetThread(this, listLen3, names3, 10, this, false);
    g.start();
    testWait();
    System.out.println("Test 3 done.");



    testStart(4);
    String names4a[] = {"index.html", "index2.html"};
    long listLen4a[] = {3891, 81608};
    String names4b[] =  {"index.html", "index2.html", "does-not-exist.404"};
    long listLen4b[] = {3891, 81608, -1};
    PerstConnection c = new PerstConnection(server, port, 1);
    t = new TestPutThread(this, names4a, 200, this);
    t.start();
    t = new TestPutThread(c, names4b, 40, this);
    t.start();
    g = new TestGetThread(this, listLen4a, names4a, 200, this, false);
    g.start();
    g = new TestGetThread(c, listLen4b, names4b, 40, this, false);
    g.start();
    testWait();
    System.out.println("Test 4 done.");



    testStart(2);
    String names5[] = {"index.html", "index2.html"};
    long listLen5[] = {3891, 81608};
    t = new TestPutThread(this, names5, 1000, this);
    t.start();
    g = new TestGetThread(this, listLen5, names5, 1000, this, true);
    g.start();
    int ii;
    for(ii = 0; ii < 100; ii++){
      try{
	Thread.sleep(50);
	this.socket.close();
      }
      catch(Exception e){
	System.out.println("Exception: " + e.toString());
	e.printStackTrace();
      }
    }
    testWait();
    System.out.println("Test 5 done.");


    System.out.println("All done.");

  }


/*
 *------------------------------------------------------------------
 *
 * TestPutThread --
 *
 *          Inner class for testing
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected class TestGetThread extends Thread{
private long list[];
private PerstConnection c;
private int count;
private String names[];
private PerstConnection testMaster;
private boolean expectTheUnexpected;

protected TestGetThread(PerstConnection c_, long list_[], String names_[], int count_,
			PerstConnection testMaster_,
			boolean expectErrors)
  {
    c = c_;
    list = list_;
    count = count_;
    names = names_;
    testMaster = testMaster_;
    expectTheUnexpected = expectErrors;
  }

public void run()
  {
    int ii;
    for(ii = 0; ii < count; ii++){
      int ij;
      for(ij = 0; ij < list.length; ij++){
	if(verbose){
	  System.out.println("Getting req " + (ii*list.length + ij));
	}
	PendingRequest pr = c.get();
	if(verbose){
	  System.out.println("Got req " + (ii*list.length + ij));
	  System.out.println("  got length: " + pr.getGotSize() 
			     + " expected: " + list[ij] );
	  System.out.println("  got name: " + pr.getObjId()
			     + " expected: " + names[ij]);
	}
	if(expectTheUnexpected){
	  Assert.myAssert(list[ij] == -1 || pr.getGotSize() == list[ij] || pr.getGotSize() == -1);
	}
	else{
	  Assert.myAssert(list[ij] == -1 || pr.getGotSize() == list[ij]);
	}
	Assert.myAssert(pr.getObjId().equals(names[ij]));
      }
    }
    testMaster.testDone();
  }
};


/*
 *------------------------------------------------------------------
 *
 * TestPutThread --
 *
 *          Inner class for testing
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected class TestPutThread extends Thread{
private String list[];
private PerstConnection c;
private int count;
private PerstConnection testMaster;

protected TestPutThread(PerstConnection c_, String list_[], int count_,
			PerstConnection testMaster_)
  {
    c = c_;
    list = list_;
    count = count_;
    testMaster = testMaster_;
  }

public void run()
  {
    int ii;
    for(ii = 0; ii < count; ii++){
      int ij;
      for(ij = 0; ij < list.length; ij++){
	if(verbose){
	  System.out.println("Putting req " + list[ij] + " as req " + (ii*list.length + ij));
	}
	c.put(new PendingRequest(list[ij], null));
	if(verbose){
	  System.out.println("Put req " + list[ij] + " as req " + (ii*list.length + ij));
	}
      }
    }
    testMaster.testDone();
  }
};
};


