import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.util.Hashtable;
/*
 *------------------------------------------------------------------
 *
 * LESS Group 
 * Computer Sciences Department 
 * University of Texas at Austin
 *
 * PrefetchClient ---
 *          Client for tcp-vegas-nice experiment.
 *
 * $Date: 2003/01/06 06:16:30 $ $Id: PrefetchClient.java,v 1.1.1.1 2003/01/06 06:16:30 ypraveen Exp $
 *
 * Copyright (c) 2002 by Mike Dahlin.
 * 
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for noncommercial use without fee and
 * without written agreement is hereby granted, provided that 
 * the above copyright notice appear in all copies of this software.
 * 
 *------------------------------------------------------------------
 */


public class PrefetchClient{

public static boolean verbose = true;

/*
 * Globals set by arguments
 */
private static String client = "UNINITIALIZED";
private static String server = "UNINITIALIZED";
private static float timeFactor = (float)-1.0;
private static float prefThresh = (float)-1.0;
private static long embargoTimeMS = (long) 1;
private static String traceFormat = "UNINITIALIZED";
private static String prefetchStateFile = "UNINITIALIZED";
private static String mangleDemandPrefix = "UNINITIALIZED";
private static String manglePrefetchPrefix = "UNINITIALIZED";
private static String prefPredAlg = "UNINITIALIZED";
  // Randomly distribute the clients we simulate
  // across nSimulatedClientCaches, each with 
  // a separate prefetch queue and fetch queue
  // Note: we make several native threads per
  // simulated cache, so this cannot be too large
  // or we run out of native threads.
private static int nSimulatedClientCaches = 1;

/*
 *------------------------------------------------------------------
 *
 * printArgs --
 *
 *          Print out the arguments we were called with.
 *
 * Arguments:
 *      String argv[]
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public static void printArgs(String argv[])
  {
    int ii;
    System.out.print("PrefetchClient ");
    for(ii = 0; ii < argv.length; ii++){
      System.out.print(argv[ii] + " ");
    }
    System.out.println("");
  }

/*
 *------------------------------------------------------------------
 *
 * main --
 *
 *          Read the specified trace and send demand and
 *          prefetch requests across the network.
 *
 * Arguments:
 *      see usage();
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public static void main(String argv[])
  {
    int c;

    final int MAX_OPT = 11;
    final int REQUIRED_COUNT = 9; 
    int gotCount = 0;
    LongOpt[] longopts = new LongOpt[MAX_OPT];
    longopts[0] = new LongOpt("client", LongOpt.REQUIRED_ARGUMENT, 
			      null, 0);
    longopts[1] = new LongOpt("server", LongOpt.REQUIRED_ARGUMENT, 
			      null, 1);
    longopts[2] = new LongOpt("timeFactor", LongOpt.REQUIRED_ARGUMENT, 
			      null, 2);
    longopts[3] = new LongOpt("prefPredAlg", LongOpt.REQUIRED_ARGUMENT, 
			      null, 3);
    longopts[4] = new LongOpt("prefThresh", LongOpt.REQUIRED_ARGUMENT, 
			      null, 4);
    longopts[5] = new LongOpt("traceFormat", LongOpt.REQUIRED_ARGUMENT, 
			      null, 5);
    longopts[6] = new LongOpt("prefetchStateFile", LongOpt.REQUIRED_ARGUMENT, 
			      null, 6);
    longopts[7] = new LongOpt("mangleDemandPrefix", LongOpt.REQUIRED_ARGUMENT, 
			      null, 7);
    longopts[8] = new LongOpt("manglePrefetchPrefix",
			      LongOpt.REQUIRED_ARGUMENT, 
			      null, 8);
    longopts[9] = new LongOpt("nSimulatedClientCaches", 
			      LongOpt.REQUIRED_ARGUMENT,
			      null, 9);
    longopts[10] = new LongOpt("embargoTimeMS", LongOpt.REQUIRED_ARGUMENT, 
			      null, 10);

    Getopt g = new Getopt("PrefetchClient", argv, "", longopts, true);
    while((c = g.getopt()) != -1){
      switch(c){
      case 0:
	client = g.getOptarg();
	gotCount++;
	break;
      case 1:
	server = g.getOptarg();
	gotCount++;
	break;
      case 2:
	timeFactor = (new Float(g.getOptarg())).floatValue();
	gotCount++;
	break;
      case 3:
	prefPredAlg = g.getOptarg();
	gotCount++;
	Assert.myAssert(prefPredAlg.equals("useless") 
			|| prefPredAlg.equals("markov")
			|| prefPredAlg.equals("off"), 
			"-prefPredAlg must be 'useless' or 'markov' or 'off'");
	break;
      case 4:
	prefThresh = (new Float(g.getOptarg())).floatValue();
	gotCount++;
	break;
      case 5:
	traceFormat = g.getOptarg();
	Assert.myAssert(traceFormat.equals("bu")
			|| traceFormat.equals("squid"),
		      "-traceFormat must be 'bu' or 'squid'");
	gotCount++;
	break;
      case 6:
	prefetchStateFile = g.getOptarg();
	gotCount++;
	break;
      case 7:
	mangleDemandPrefix = g.getOptarg();
	gotCount++;
	break;
      case 8:
	manglePrefetchPrefix = g.getOptarg();
	gotCount++;
	break;
      case 9:
	nSimulatedClientCaches = (new Integer(g.getOptarg())).intValue();
	break;
      case 10:
	embargoTimeMS = (new Long(g.getOptarg())).longValue();
	break;
     case '?':
	System.out.println("The option '" + (char)g.getOptopt() 
			   + "' is not valid.");
	System.exit(-1);
	break;
      default:
	System.out.println("getopt() returned " + c);
	System.exit(-1);
	break;
      }
    }

    printArgs(argv);

    if(gotCount < REQUIRED_COUNT){
      System.out.println("Missing a required argument. Exiting.");
      System.exit(-1);
    }

    doIt();
  }

/*
 *------------------------------------------------------------------
 *
 * doIt --
 *
 *          Process the trace.
 *
 * Arguments:
 *      Uses the global params.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void doIt()
  {
    Cache cache;
    Prefetcher prefetcher;
    PrefetchPredictor prefetchPredictor;
     Fetcher fetcher;
     RawReq.setMangle(mangleDemandPrefix, manglePrefetchPrefix);
     RawReq req;
     Hashtable caches = new Hashtable();
     Hashtable prefetchers = new Hashtable();
     Hashtable fetchers = new Hashtable();
     Stats stats = new Stats();
     InputTrace inputTrace = new InputTrace(System.in, traceFormat, 
					    client, server, timeFactor);
     Integer cacheSet;

     //
     // Note: prefetch predictor already keeps per-client
     // histories.
     //
     if(prefPredAlg.equals("useless")){
       prefetchPredictor = new UselessPrefetchPredictor();
     }
     else if(prefPredAlg.equals("markov")){
       prefetchPredictor = new MarkovPrefetchPredictor(prefetchStateFile);
     }
     else if(prefPredAlg.equals("off")){
       prefetchPredictor = new NullPrefetchPredictor();
     }
     else{
       Assert.myAssert(false, "Bad prediction algorithm " + prefPredAlg
		       + " selected.");
       prefetchPredictor = null;
     }


     initializeCacheSets(caches, prefetchers, fetchers, 
			 stats, embargoTimeMS, prefetchPredictor);

     try{
       int count = 0;
       while(true){
	 count++;
	 if(count % 100 == 0){
	   System.out.print(".");
	 }

	 req = inputTrace.next();

	 String cacheID = req.getClient();
	 cacheSet = getCacheSet(cacheID);

	 cache = (Cache)caches.get(cacheSet);
	 Assert.myAssert(cache != null,
			 "All cache sets now created in initializeCacheSet");
	 prefetcher = (Prefetcher)prefetchers.get(cacheSet);
	 Assert.myAssert(prefetcher != null);
	 fetcher = (Fetcher)fetchers.get(cacheSet);
	 Assert.myAssert(fetcher != null);


	 if(cache.checkMissUpdateSize(req)){
	   stats.incrementMiss(req);
	   //
	   // Don't want multiple outstanding demand requests
	   // for same data.
	   //
	   cache.insertIfCachable(req); 
	   prefetcher.notifyDemand(req); // Prefetcher MAY stop prefetching
	   fetcher.asyncFetch(req);
	 }
	 else{
	   stats.incrementHit(req);
	 }
       }
     }
     catch(EndOfInputException e){
       try{
	 Thread.sleep(4000); // Wait for prefetches to clear.
       }
       catch(InterruptedException iez){
	 System.out.println("InterruptedException in outer loop?");
       }
       stats.done();
       System.out.println("Done.");
     }
     catch(MalformedRecordException m){
       System.out.println("Bad record in input: " + m.toString());
       System.exit(-1);
     }
     catch(RuntimeException z){
       System.out.println("Runtime exception: " + z.toString());
       z.printStackTrace();
       System.exit(-1);
     }

     System.exit(0);    
   }



/*
 *------------------------------------------------------------------
 *
 * getCacheSet --
 *
 *          Randomly split the collection of clients
 *          in this trace across sets of simulated caches.
 * 
 *          We need to multiplex clients across caches since
 *          each cache/prefetcher ends up spawning a separate
 *          collection of prefetcher threads.
 *
 *          We use round robin assignment since that way if there 
 *          are fewer than nSimulatedClientCaches clients
 *          in the trace, then everyone gets their own cache.
 *
 * Arguments:
 *      String cacheID -- the id of the client in the trace
 *
 * Results:
 *      Integer -- the set between 0..nSimulatedClients-1
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static int nextSet = 0;
private static Hashtable clientSetMap = new Hashtable();
private static Integer getCacheSet(String cacheID)
  {
    Integer set = (Integer)clientSetMap.get(cacheID);
    if(set == null){
      set = new Integer(nextSet);
      nextSet = (nextSet+1) % nSimulatedClientCaches;
      Assert.myAssert(nextSet >= 0);
      Assert.myAssert(nextSet < nSimulatedClientCaches);
      clientSetMap.put(cacheID, set);
    }
    return set;
  }

private static void initializeCacheSets(Hashtable caches, 
					Hashtable prefetchers,
					Hashtable fetchers, 
					Stats stats, long embargoTimeMS,
					PrefetchPredictor prefetchPredictor)
  {
    int ii;
    for(ii = 0; ii < nSimulatedClientCaches; ii++){
      Cache cache;
      Fetcher fetcher;
      Prefetcher prefetcher;
      Integer cacheSet;

      cacheSet = new Integer(ii);
      cache = new Cache();
      caches.put(cacheSet, cache);
      prefetcher = new Prefetcher(cache, stats, prefThresh, embargoTimeMS);
      prefetchers.put(cacheSet, prefetcher);
      fetcher = new Fetcher(cache, stats, prefetcher,
			    prefetchPredictor);
      fetchers.put(cacheSet, fetcher);
    }
    
  }

};

