import se.sics.tasim.props.*;
import se.sics.isl.transport.*;
import java.util.*;

/**Holds all information pertaining to components.  The
 *  information is updated by using the incoming messages at the beginning
 *  of each day and the outgoing messages at the end of each day.  This means
 *  that there is no need to modify any values during the agent's decision
 *  making process, and so this object should essentially be read-only during
 *  the day. (But fields have been left public to allow easy access - too much
 *  work to add get-methods for everything!)  Note that this object is not
 *  responsible for verifying messages.  For example, if a production schedule
 *  is sent that is not feasible given the components available, the problem
 *  will not be detected.
 */

public class ComponentInfo {
	//IMPORTANT: arrays are indexed by [component][day] or by [component]
	
	/**completed deliveries*/
	public int deliveries[][]; 
	/**expected future deliveries*/
	public int expectedDeliveries[][]; 
	/** inventory each day*/
	public int inventory[][]; 
	/**tomorrow's available inventory (including today's deliveries, which aren't included in today's inventory)*/
	public int tomorrowsInventory[];  
	/**used each day*/
	public int used[][]; 
	/**number ordered each day*/
	public int orderedToday[][]; 
	/**total number currently ordered*/
	public int orderedTotal[]; 
	/** component base prices*/
	public int basePrices[]; 
	
	/**ID for each index*/
	public int[] productIDs; 
	/**stores RFQs by rfqID*/
	public Hashtable rfqTable = new Hashtable(); 
	/**a list of orders that have not been delivered for each component*/
	public Hashtable[] currentOrders; 
	/**stores offers by offerID*/
	public Hashtable offerTable = new Hashtable(); 
	/**list of all offers*/
	public Vector todaysOffers; 
	/**list of orders for each component*/
	public Vector[] todaysOrders;
	/**a list of all orders for each component, in order of date*/
	public Vector[] ordersByDate; 
	/**offers sorted by date, divided into [component][supplier]*/
	public Vector[][] todaysSortedOffers; 
	/**information on each supplier line, as [component][supplier]*/
	public SupplierModel[][] supplierModel; 
	
	private GameSettings settings;
	/**width of summary table columns*/
	private final static int columnWidth = 11; 
	private int numComponents; //total number of components
	private int numDays; //total number of days
	
	/**
	 * Returns a copy of tomorrow's available inventory (including components
	 * delivered today).
	 * 
	 * @return  array with quantity of each component
	 */
	public int[] getTomorrowsInventoryCopy(){
		int[] i = new int[numComponents];
		System.arraycopy(tomorrowsInventory, 0, i, 0, numComponents);
		return i;
	}

	/** 
	 * 
	 * @param settings  the game settings
	 */
	public ComponentInfo(GameSettings settings){
		this.settings = settings;
		numDays = settings.numberOfDays;
		numComponents = settings.catalog.size();
		deliveries = new int[numComponents][numDays+1];
		expectedDeliveries = new int[numComponents][numDays+1];
		inventory = new int[numComponents][numDays+1];
		tomorrowsInventory = new int[numComponents];
		used = new int[numComponents][numDays+1];
		orderedToday = new int[numComponents][numDays+1];
		orderedTotal = new int[numComponents];
		productIDs = new int[numComponents];
		currentOrders = new Hashtable[numComponents];
		todaysOffers = new Vector();
		ordersByDate = new Vector[numComponents];
		todaysOrders = new Vector[numComponents];
		
		todaysSortedOffers = new Vector[numComponents][2];
		supplierModel = new SupplierModel[numComponents][2];
		for (int i = 0; i < numComponents; i++){
			productIDs[i] = settings.catalog.getProductID(i);
			currentOrders[i] = new Hashtable();
			ordersByDate[i] = new Vector();
			todaysOrders[i] = new Vector();
			todaysSortedOffers[i][0] = new Vector();
			todaysSortedOffers[i][1] = new Vector();
			String[] suppliers = settings.catalog.getSuppliers(i);
			supplierModel[i][0] = new SupplierModel(i, productIDs[i], suppliers[0], settings.catalog.getProductBasePrice(i));
			if (suppliers.length == 2)
				supplierModel[i][1] = new SupplierModel(i, productIDs[i], suppliers[1], settings.catalog.getProductBasePrice(i));
		}
	
		basePrices = new int[numComponents];
		for (int j = 0; j < numComponents; j++){
			basePrices[j] = settings.catalog.getProductBasePrice(j);
		}
	}

	
	/**
	 * Generates a compact table summarizing today's component information.
	 * 
	 * @return  summary table
	 */
	public String getDailySummary(int date){
		StringBuffer buf = new StringBuffer();
		buf.append("COMPONENTS:\n");
		buf.append(pad("component"));
		buf.append(pad("delivered"));
		buf.append(pad("inventory"));
		buf.append(pad("used"));
		buf.append(pad("orderToday"));
		buf.append(pad("orderTotal"));
		buf.append("\n");

		for(int i = 0; i < numComponents; i++){
			buf.append(pad(productIDs[i]));
			buf.append(pad(deliveries[i][date]));
			buf.append(pad(inventory[i][date]));
			buf.append(pad(used[i][date]));
			buf.append(pad(orderedToday[i][date]));
			buf.append(pad(orderedTotal[i]));
			buf.append("\n");
		}
		return buf.toString();
	}
	
	/**
	 * Updates the component information using today's incoming messages.
	 * 
	 * @param dayInfo  information received today
	 */
	public void morningUpdate(DailyInfo dayInfo){
		int date = dayInfo.date;
		todaysOffers.clear();
		for (int i = 0; i < numComponents; i++)
			todaysOrders[i].clear();
		
		//store inventory
		InventoryStatus dayInventory = dayInfo.inventory;
		if (dayInventory != null){
			for(int i = 0, n = dayInventory.getProductCount(); i < n; i++){
				int index = getIndex(dayInventory.getProductID(i));
				if (index >= 0){ //unknown IDs (for computers) will return -1
					inventory[index][date] = dayInventory.getQuantity(i);
				}
			}
			
			//copy this inventory into tommorowsInventory (deliveries will be added later)
			for (int i = 0; i < numComponents; i++)
				tomorrowsInventory[i] = inventory[i][date];
				//System.arraycopy(inventory[date], 0, tomorrowsInventory, 0, numComponents);	
		}
		
		//store deliveries
		Vector dayDeliveries = dayInfo.deliveries;
		DeliveryNotice delivery;
		if (dayDeliveries != null && dayDeliveries.size() > 0){
			for (int i = 0, ni = dayDeliveries.size(); i < ni; i++) {
				delivery = (DeliveryNotice) dayDeliveries.get(i);
				for (int j = 0, nj = delivery.size(); j < nj; j++){
					int index = getIndex(delivery.getProductID(j));
					deliveries[index][date] += delivery.getQuantity(j);
					//add delivery to inventory available for tomorrow
					tomorrowsInventory[index] += delivery.getQuantity(j);
					orderedTotal[index] -= delivery.getQuantity(j);
					OrderInfo order = (OrderInfo) currentOrders[index].get(new Integer(delivery.getOrderID(j)));
									
					if (order != null){ //this should never be null
						//update supplier model
						if (order.name.equals(supplierModel[order.index][0].getName()))
							supplierModel[order.index][0].addDelivery(order, date);
						else
							supplierModel[order.index][1].addDelivery(order, date);

						//remove order from the list of expected orders
						currentOrders[index].remove(new Integer(order.orderID));
					}
				}
			}
		}

		//offers from suppliers
		Vector supplierOffers = dayInfo.supplierOffers;
		OfferBundle bundle;
		OfferInfo lastOffer = null;
		RFQInfo rfq;
		//alternative offers appear in sequence, so check for consecutive IDs that match
		int lastRfqID = -1; 
		if (supplierOffers != null && supplierOffers.size() > 0){
			for(int i = 0, n = supplierOffers.size(); i < n; i++){
				bundle = (OfferBundle) supplierOffers.get(i);
				for (int j = 0, nj = bundle.size(); j < nj; j++){
					if (bundle.getRFQID(j) == lastRfqID) //this is an alternative offer
						rfq = lastOffer.rfq;
					else
						rfq = (RFQInfo) rfqTable.remove(new Integer(bundle.getRFQID(j)));
					
					if (rfq != null){ //this should never be null
						OfferInfo offer = new OfferInfo(bundle.getOfferID(j), bundle.getQuantity(j),
							bundle.getDueDate(j), (int)(bundle.getUnitPrice(j)), rfq, this.getIndex(bundle.getOfferID(j)));
						offer.index = getIndex(offer.productID);
						//add to table of existing offers
						offerTable.put(new Integer(offer.offerID), offer);						
						
						//if the rfqID is the same as for the last offer, this an alternative offer
						if (lastOffer != null && lastOffer.rfq.rfqID == offer.rfq.rfqID)
							lastOffer.alternative = offer; //store this offer with the last offer
						else
							todaysOffers.add(offer);
						lastRfqID = rfq.rfqID;
						lastOffer = offer;
					}
				}
			}
		}
		
		
		//divide offers by supplier
		for (int c = 0; c < 10; c++){	
			todaysSortedOffers[c][0].clear();
			todaysSortedOffers[c][1].clear();
			for (int i = 0, n = todaysOffers.size(); i < n; i++){
				OfferInfo offer = (OfferInfo) todaysOffers.get(i);
				if (offer.index == c)
					if (offer.name.equals(supplierModel[c][0].getName()))
						todaysSortedOffers[c][0].add(offer);
					else
						todaysSortedOffers[c][1].add(offer);
			}
		}
		
		//now update supplier models	
		for (int c = 0; c < 10; c++){
			supplierModel[c][0].update(date, todaysSortedOffers[c][0]);
			if (supplierModel[c][1] != null)
				supplierModel[c][1].update(date, todaysSortedOffers[c][1]);
		}
		
		//give market report to suppliers
		if (dayInfo.marketReport != null){
			for (int c = 0; c < 10; c++){
				supplierModel[c][0].processMarketReport(dayInfo.marketReport);
				if (supplierModel[c][1] != null)
					supplierModel[c][1].processMarketReport(dayInfo.marketReport);
			}
		}
		
		//calculate expected deliveries
		for (int c = 0; c < 10; c++){
			int[] dels = supplierModel[c][0].getExpectedDeliveries();
			for (int d = date; d < 220; d++)
				expectedDeliveries[c][d] = dels[d];
			if (supplierModel[c][1] != null){
				dels = supplierModel[c][1].getExpectedDeliveries();
				for (int d = date; d < 220; d++)
					expectedDeliveries[c][d] += dels[d];
			}
		}
	}

	/**
	 * Updates the component information using today's outgoing messages
	 * (actions taken by the agent).
	 * 
	 * @param date  the date
	 * @param actions  the agent's actions for the day
	 */
	public void eveningUpdate(int date, DailyActions actions){
		Vector messages = actions.messages;
		
		//go through all messages in the actions
		for(int m = 0; m < messages.size(); m++){
			Transportable content = (Transportable) messages.get(m);
			
			//production schedule
			if (content instanceof ProductionSchedule){ 
				ProductionSchedule production = (ProductionSchedule) content;
				for (int i = 0, n = production.size(); i < n; i++){
					int quantity = production.getQuantity(i);
					int comps[] = settings.bom.getComponentsForProductID(production.getProductID(i));
					for (int j = 0; j < comps.length; j++){
						int index = getIndex(comps[j]);
						used[index][date] += quantity;
					}		
				}
			}
			
			//order bundle
			else if (content instanceof OrderBundle){
				OrderBundle orders = (OrderBundle) content;
				for (int i = 0, n = orders.size(); i < n; i++){
					OfferInfo offer = (OfferInfo) offerTable.get(new Integer(orders.getOfferID(i)));
					if (offer != null){ //this should never be null
			 			OrderInfo order = new OrderInfo(offer, orders.getOrderID(i));
			 			int index = getIndex(offer.productID);
			 			currentOrders[index].put(new Integer(order.orderID), order);
			 			orderedTotal[index] += offer.quantity;
			 			orderedToday[index][date] += offer.quantity;
						ordersByDate[index].add(order);
						todaysOrders[index].add(order);
						
						//update supplier model
						if (order.name.equals(supplierModel[order.index][0].getName()))
							supplierModel[order.index][0].addOrder(order, date);
						else
							supplierModel[order.index][1].addOrder(order, date);
			 		}
			 	}
			}
		}
		
		//go through all RFQs to suppliers
		Enumeration enumm = actions.supplyTable.keys();
		while (enumm.hasMoreElements()) {
			String receiver = (String) enumm.nextElement();
			RFQBundle bundle = (RFQBundle) actions.supplyTable.get(receiver);
			for (int i = 0, n = bundle.size(); i < n; i++){
				//add RFQ to table of sent RFQs
				RFQInfo rfq = new RFQInfo(bundle.getRFQID(i), bundle.getProductID(i),
					bundle.getQuantity(i), bundle.getDueDate(i), 0, i, receiver);
				rfqTable.put(new Integer(rfq.rfqID), rfq);
			}	
		}
		
		//update reputation with suppliers
		for (int c = 0; c < 10; c++){
			supplierModel[c][0].updateReputation(todaysOrders[c], todaysOffers);
			if (supplierModel[c][1] != null)
				supplierModel[c][1].updateReputation(todaysOrders[c], todaysOffers);
		}
	}
	
	
	
	/**
	 * Returns the index for the productID
	 * 
	 * @param productID (e.g., 101)
	 * @return  index (0-9)
	 */
	public int getIndex(int productID) {
		for (int i = 0; i < numComponents; i++) {
			if (productIDs[i] == productID)
				return i;
		}
		return -1;
	}

	//return a fixed-width string representing the integer - used to make table
	private String pad(int i){
		return pad(String.valueOf(i));
	}

	//return a fixed-width string - used to make table
	private String pad(String s){
		s += "                        ";
		return s.substring(0, columnWidth);
	}		

	
}
