import se.sics.isl.transport.*;
import se.sics.tasim.props.*;
import java.util.*;

/** Holds all information pertaining to computers.  The
 *  information is updated by using the incoming messages at the beginning
 *  of each day and the outgoing messages at the end of each day.  This means
 *  that there is no need to modify any values during the agent's decision
 *  making process, and so this object should essentially be read-only during
 *  the day. (But fields have been left public to allow easy access - too much
 *  work to add get methods for everything!)  Note that this object is not
 *  responsible for verifying messages.  For example, if a delivery schedule
 *  is sent that is not feasible given the computers available, the problem
 *  will not be detected.
 */

public class ComputerInfo {
	//IMPORTANT: arrays are indexed by [computer][day] or by [computer]
	
	/**total number of RFQs per day*/
	public int[] dailyRFQCount; 
	/**number of RFQs per day in high segment*/
	public int[] highSegmentRFQCount; 
	/**number of RFQs per day in mid segment*/
	public int[] midSegmentRFQCount; 
	/**number of RFQs per day in low segment*/
	public int[] lowSegmentRFQCount; 
	/**number of RFQs received for each type of computer*/
	public int wholeRfqNumber[][]; 
	/**total computers in all rfqs each day*/
	public int rfqNumber[][]; 
	/**total number of computers offered each day*/
	public int offerNumber[][]; 
	/**total number of offers made each day*/ 
	public int wholeOfferNumber[][]; 
	/**sum of all offers each day*/
	public int offerPrice[][]; 
	/**number of computers ordered each day*/
	public int orderNumber[][]; 
	/**total number of orders each day*/
	public int wholeOrderNumber[][]; 
	/**avg bid per computer type each day*/
	public int averageOfferPrice[][];
	/**total price for all computers ordered each day*/
	public int orderPrice[][]; 
	/**number of computers in inventory each day*/
	public int inventory[][]; 
	/**number of computers available for delivery tomorrow*/
	public int tomorrowsInventory[]; 
	/**computers produced each day*/
	public int produced[][]; 
	/**number delivered each day*/
	public int delivered[][]; 
	/**total number of each computer currently ordered*/
	public int totalOrders[];
	/**high price from each day's product report*/
	public int highPrice[][];
	/**low price from each day's product report*/
	public int lowPrice[][]; 
	/**product IDs for each index*/
	public int[] productIDs; 
	
	/**stores current RFQs*/
	public  Hashtable rfqTable = new Hashtable(); 
	/**stores all offers*/
	public Hashtable offerTable = new Hashtable();
	/** stores current orders*/
	public Hashtable orderTable = new Hashtable();
	/** offers made yesterday for each computer type*/
	public Vector[] yesterdaysOffers;
	/**list of components for each computer, stored by index instead of ID*/
	public int[][] componentIndexList; 
	/**base prices for each computer: sum of component base prices*/
	public int[] basePrices; 
	
		 	
	private GameSettings settings;
	/**width of columns in summary table*/
	private final static int columnWidth = 8; 
	private int numComputers;
	private int numDays;
	private int date;
	
	/**
	 * 
	 * @param settings the game settings
	 */
	public ComputerInfo(GameSettings settings){
		this.settings = settings;
		numDays = settings.numberOfDays;
		numComputers = settings.bom.size();
		dailyRFQCount = new int[numDays+1];
		highSegmentRFQCount = new int[numDays+1];
		midSegmentRFQCount = new int[numDays+1];
		lowSegmentRFQCount = new int[numDays+1];
		wholeRfqNumber = new int[numComputers][numDays+1];
		rfqNumber = new int[numComputers][numDays+1];
		offerNumber = new int[numComputers][numDays+1];
		wholeOfferNumber = new int[numComputers][numDays+1];
		offerPrice = new int[numComputers][numDays+1];
		orderNumber = new int[numComputers][numDays+1];
		wholeOrderNumber = new int[numComputers][numDays+1];
		orderPrice = new int[numComputers][numDays+1];
		averageOfferPrice = new int[numComputers][numDays+1];
		inventory = new int[numComputers][numDays+1];
		tomorrowsInventory = new int[numComputers];
		produced = new int[numComputers][numDays+1];
		delivered = new int[numComputers][numDays+1];
		totalOrders = new int[numComputers];
		highPrice = new int[numComputers][numDays+1];
		lowPrice = new int[numComputers][numDays+1];
		productIDs = new int[numComputers];
		yesterdaysOffers = new Vector[numComputers];
		for (int i = 0; i < numComputers; i++){
			productIDs[i] = settings.bom.getProductID(i);
			yesterdaysOffers[i] = new Vector();
		}
		
		componentIndexList = new int[numComputers][4]; //4 components per computer
		basePrices = new int[numComputers];
		int priceSum;
		//store the components needed for each computer, along with the total base price of these components
		for (int i = 0; i < numComputers; i++){
			priceSum = 0;
			int[] comps = settings.bom.getComponents(i);
			for (int j = 0; j < 4; j++){
				componentIndexList[i][j] = settings.catalog.getIndexFor(comps[j]);
				priceSum += settings.catalog.getProductBasePrice(componentIndexList[i][j]);
			}
			basePrices[i] = priceSum;
		}

	}
	
	/**
	 * Returns a copy of tomorrow's available computers (including computers being
	 * produced on the current day)
	 * 
	 * @return  array with quantity of each computer
	 */
	public int[] getTomorrowsInventoryCopy(){
		int[] i = new int[numComputers];
		System.arraycopy(tomorrowsInventory, 0, i, 0, numComputers);
		return i;
	}
		
	/**
	 * Generates a compact table summarizing today's computer information.
	 * 
	 * @return  summary table
	 */
	public String getDailySummary(int date){
		StringBuffer buf = new StringBuffer();
		buf.append("COMPUTERS:\n");
		buf.append(pad("computr"));
		buf.append(pad("Yrfqs"));
		buf.append(pad("Yoffers"));
		buf.append(pad("YavgOf$"));
		buf.append(pad("orders"));
		buf.append(pad("avgOrd$"));
		buf.append(pad("hiRpt$"));
		buf.append(pad("loRpt$"));
		buf.append(pad("invent"));
		buf.append(pad("producd"));
		buf.append(pad("delivd"));
		buf.append(pad("totlOrd"));
		buf.append("\n");
		 
		int totalRfq = 0, totalOrder = 0, totalOffer = 0;
		for(int i = 0; i < numComputers; i++){
			int yesterday = (date == 0) ? 0 : date-1; //don't want -1
			buf.append(pad(productIDs[i]));
			buf.append(pad(rfqNumber[i][yesterday]));
			buf.append(pad(offerNumber[i][yesterday]));	
			if (offerNumber[i][yesterday] == 0)
				buf.append(pad("0"));
			else
				buf.append(pad(offerPrice[i][yesterday] / offerNumber[i][yesterday]));	
			buf.append(pad(orderNumber[i][date]));	
			if (orderNumber[i][date] == 0)
				buf.append(pad("0"));
			else
				buf.append(pad(orderPrice[i][date] / orderNumber[i][date]));	
			buf.append(pad(highPrice[i][date]));	
			buf.append(pad(lowPrice[i][date]));				
			buf.append(pad(inventory[i][date]));	
			buf.append(pad(produced[i][date]));	
			buf.append(pad(delivered[i][date]));	
			buf.append(pad(totalOrders[i]));	
			buf.append("\n");

			totalRfq += rfqNumber[i][date];
			totalOffer += offerNumber[i][date];
			totalOrder += orderNumber[i][date];
		}
		
		buf.append("Total requested: " + totalRfq + " Total offered: " + totalOffer + " Total ordered: " + totalOrder + "\n");
		
		int yesterdaysOffers = 0;
		if (date == 0)
			yesterdaysOffers = 0;
		else for(int i = 0; i < numComputers; i++)
			yesterdaysOffers += offerNumber[i][date-1];
		if (yesterdaysOffers == 0)
			yesterdaysOffers = 1; //fraction will be 0
		buf.append("% of yesterday's offers accepted: " + getDecimal(100.0 * totalOrder/yesterdaysOffers, 1) + "\n");
		
		return buf.toString();
	}

	/**
	 * Returns the index for the productID
	 * 
	 * @param productID 
	 * @return  index (0-15)
	 */
	public int getIndex(int productID) {
		for (int i = 0; i < numComputers; i++)
			if (productIDs[i] == productID)
				return i;
		return -1;
	}
	
	
	/**
	 * Updates the computer information using today's incoming messages.
	 * 
	 * @param dayInfo  information received today
	 */
	public void morningUpdate(DailyInfo dayInfo){
		date = dayInfo.date;
	
		//remove cancelled orders
		Enumeration enumm = orderTable.keys();
		while (enumm.hasMoreElements()) {
			OrderInfo order = (OrderInfo) orderTable.get(enumm.nextElement());
			if ( (date - order.dueDate + 1) >= settings.daysBeforeVoid ){ //cancel order
				orderTable.remove(new Integer(order.orderID));
				totalOrders[getIndex(order.productID)] -= order.quantity;
			}	
		}	
		
		//store inventory
		InventoryStatus dayInventory = dayInfo.inventory;
		
		//add yesterday's production schedule (actually being produced today) to the inventory available tomorrow
		if (date > 0)
			for (int c = 0; c < numComputers; c++)
				tomorrowsInventory[c] = produced[c][date-1]; 

		if (dayInventory != null){
			for(int i = 0, n = dayInventory.getProductCount(); i < n; i++){
				int index = getIndex(dayInventory.getProductID(i));
				if (index >= 0){ //unknown IDs (for components) will return -1
					inventory[index][date] = dayInventory.getQuantity(i);
					tomorrowsInventory[index] += inventory[index][date]; 
				}
			}
		}
	
		//RFQs
		RFQBundle rfqs = dayInfo.customerRFQs;
		rfqTable.clear();
		if (rfqs != null)
			dailyRFQCount[date] = rfqs.size();
		if (rfqs != null){
			for (int i = 0; i < rfqs.size(); i++){
				RFQInfo rfq = new RFQInfo(rfqs.getRFQID(i), rfqs.getProductID(i), rfqs.getQuantity(i),
					rfqs.getDueDate(i), rfqs.getReservePricePerUnit(i), rfqs.getPenalty(i), dayInfo.customerName);
				rfq.receivedDate = date;
				rfqTable.put(new Integer(rfq.rfqID), rfq);
				rfqNumber[getIndex(rfq.productID)][date] += rfq.quantity;	
				wholeRfqNumber[getIndex(rfq.productID)][date]++;
				
				//get segment demand
				int pid = rfqs.getProductID(i);
				if (pid == 1 || pid == 2 || pid == 9 || pid == 10 || pid == 12)
					lowSegmentRFQCount[date]++;
				else if (pid == 6 || pid == 7 || pid == 8 || pid == 15 || pid == 16)
					highSegmentRFQCount[date]++;
				else
					midSegmentRFQCount[date]++;
			}
		}
				
		//Orders
		OrderBundle orders = dayInfo.customerOrders;
	 	if (orders != null){
		 	for (int i = 0, n = orders.size(); i < n; i++){
				OfferInfo offer = (OfferInfo) offerTable.remove(new Integer(orders.getOfferID(i)));
				if (offer != null){ //this should never be null
					offer.ordered = true;
					OrderInfo order = new OrderInfo(offer, orders.getOrderID(i));
					int index = getIndex(offer.productID);
					orderTable.put(new Integer(order.orderID), order);
					orderNumber[index][date] += offer.quantity;
					wholeOrderNumber[index][date]++;
					orderPrice[index][date] += offer.price * offer.quantity;
					totalOrders[index] += offer.quantity;
				}
		 	}
	 	}
	
	 	//price report
	 	PriceReport p = dayInfo.priceReport;
	 	if (p != null){
	 		for (int i = 0; i < p.size(); i++){
	 			int product = getIndex(p.getProductID(i));
	 			if (product != -1){ //there may be garbage values in the price report
	 				//a price of 0 probably indicates that none were sold, so just use the previous price
	 				int hi = p.getHighestPrice(i);
		 			if (hi < .0000001 && date > 1) 
		 				hi = highPrice[product][date-1];
					int lo = p.getLowestPrice(i);
		 			if (lo < .0000001 && date > 1)
						lo = lowPrice[product][date-1];
		 			
		 			lowPrice[product][date] = lo;
		 			highPrice[product][date] = hi;
		 		}
	 		}
	 	}
	}
	
	/**
	 * Updates the component information using today's outgoing messages
	 * (actions taken by the agent).
	 * 
	 * @param date  the date
	 * @param actions  the agent's actions for the day
	 */
	public void eveningUpdate(int date, DailyActions actions){
		Vector messages = actions.messages;
		
		//clear yesterday's offers
		for (int i = 0; i < numComputers; i++)
			yesterdaysOffers[i].clear();
		
		//go through all messages in the actions
		for(int m = 0; m < messages.size(); m++){
			Transportable content = (Transportable) messages.get(m);
			
			//ProductionSchedule
			if (content instanceof ProductionSchedule){ 
				ProductionSchedule production = (ProductionSchedule) content;
				for (int i = 0, n = production.size(); i < n; i++)
					produced[getIndex(production.getProductID(i))][date] += production.getQuantity(i);
			}
			
			//offers
			else if (content instanceof OfferBundle){
				OfferBundle offers = (OfferBundle) content;
				int[] offerSum = new int[numComputers];
				
				for (int i = 0; i < offers.size(); i++){
					RFQInfo rfq = (RFQInfo) rfqTable.get(new Integer(offers.getRFQID(i)));
					if (rfq != null){ //this should never be null
						OfferInfo offer = new OfferInfo(offers.getOfferID(i), offers.getQuantity(i),
							offers.getDueDate(i), offers.getUnitPrice(i), rfq, this.getIndex(rfq.productID));
						int index = getIndex(offer.productID);
						offerNumber[index][date] += offer.quantity;
						wholeOfferNumber[index][date]++;
						offerPrice[index][date] += offer.quantity * offer.price;
						offerSum[index] += offer.price;		
						offerTable.put(new Integer(offer.offerID), offer);
						yesterdaysOffers[index].add(offer);
					}
				}
				for (int i = 0; i < numComputers; i++)
					if (wholeOfferNumber[i][date] != 0)
						averageOfferPrice[i][date] = offerSum[i]/wholeOfferNumber[i][date]; 
			}

			//deliveries
			else if (content instanceof DeliverySchedule){
				DeliverySchedule deliveries = (DeliverySchedule) content;
				for (int i = 0; i < deliveries.size(); i++){
					delivered[getIndex(deliveries.getProductID(i))][date] += deliveries.getQuantity(i);
					int oid = deliveries.getOrderID(i);
					OrderInfo order = (OrderInfo) orderTable.remove(new Integer(oid));
					if (order != null)
						totalOrders[order.index] -= order.quantity;
				}
			}
		}
	}

	//return a fixed-width string representing the integer - used to make table
	private String pad(int i){
		return pad(String.valueOf(i));
	}

	//return a fixed-width string - used to make table	
	private String pad(String s){
		s += "               ";
		return s.substring(0, columnWidth);
	}		
	
	//used to make table
	private String getDecimal(double d, int length){
		String s = String.valueOf(d);
		if (s.indexOf('.') > -1){
			int pos = s.indexOf('.')+length+1;
			if (pos > s.length())
				pos = s.length();
			s = s.substring(0, pos);
		}
		return s;
	}
}
