import se.sics.tasim.props.*;
import se.sics.isl.transport.*;
import java.util.*;
import java.io.*;

import javax.xml.parsers.*; 
import javax.xml.transform.*;
import javax.xml.transform.stream.*;
import javax.xml.transform.dom.*;
import org.w3c.dom.*;

/**Stores the actions taken by the agent during a single day.
 * The actions are simply the messages to be sent at the end of the day.
 */

public class DailyActions {
	public Vector messages = new Vector(); //stores Messages
	public Vector messageRecipients = new Vector(); //stores Strings representing the intended recipients of the corresponding messages
	public Hashtable supplyTable = new Hashtable();
	
	/**
	 * Add a message to be sent at the end of the day.
	 * 
	 * @param recipient (e.g., "Customer")
	 * @param content  (e.g., a DeliverySchedule)
	 */
	public void addMessage(String recipient, Transportable content){
		messages.add(content);
		messageRecipients.add(recipient);
	}	

	/**
	 * Summarize the messages sent today
	 * 
	 * @return list of messages
	 */
	public String toString(){
		StringBuffer buf = new StringBuffer();
		buf.append("\n---------------------\n");
		buf.append("Messages sent today:\n");
		buf.append("---------------------\n");
		
		//go through messages
		for(int m = 0; m < messages.size(); m++){
			Transportable content = (Transportable) messages.get(m);
			//DeliverySchedule
			if (content instanceof DeliverySchedule){
				DeliverySchedule delivery = (DeliverySchedule) content;
				buf.append("Sending delivery schedule to factory: \n");
					for (int i = 0, n = delivery.size(); i < n; i++) {
						buf.append("\tdeliver " + delivery.getQuantity(i) + " units of "
							+ delivery.getProductID(i) + " to " + delivery.getCustomer(i) +
							" for order number " + delivery.getOrderID(i) + "\n"); 
					}
					buf.append("\n");
			}
			//ProductionSchedule
			else if (content instanceof ProductionSchedule){ 
				ProductionSchedule production = (ProductionSchedule) content;
				buf.append("Sending production schedule to factory: \n");
				for (int i = 0, n = production.size(); i < n; i++) {
					buf.append("\tproduce " + production.getQuantity(i) + " units of "
						+ production.getProductID(i) + "\n"); 
				}
				buf.append("\n");
			}
			//OrderBundle
			else if (content instanceof OrderBundle){
				OrderBundle orders = (OrderBundle) content;
			 	for (int i = 0, n = orders.size(); i < n; i++)
				 	buf.append("Accepting offer " + orders.getOfferID(i) + " from " +
				 			messageRecipients.get(m) + " as order number " + orders.getOrderID(i) + "\n");
			}
			//OfferBundle
			else if (content instanceof OfferBundle){
		 		OfferBundle offer = (OfferBundle) content;
		 		if (offer.size() > 0) {
					buf.append("Making offers on all RFQs for a price below the reserve:\n");
					for (int i = 0, n = offer.size(); i < n; i++) {
						buf.append("\tOffer " + offer.getOfferID(i) + " for RFQ " + offer.getRFQID(i) +
							" is for " + offer.getQuantity(i) + " units on day " + offer.getDueDate(i) + 
							" for $" + offer.getUnitPrice(i) + " each\n");
					}	
					buf.append("\n");
		 		}
			}
		}
		
		//go through all RFQs to suppliers
		Enumeration enumm = supplyTable.keys();
		if (enumm.hasMoreElements())
			buf.append("Sending RFQs:\n");
		while (enumm.hasMoreElements()) {
			String receiver = (String) enumm.nextElement();
			RFQBundle rfq = (RFQBundle) supplyTable.get(receiver);
			for (int i = 0, n = rfq.size(); i < n; i++)
				buf.append("\tRFQ " + rfq.getRFQID(i) + " for " + rfq.getQuantity(i) + 
					" units of product " + rfq.getProductID(i) + " from " + receiver +
					" by day " + rfq.getDueDate(i) + "\n");
		}
		
		return buf.toString();		
	}

	public void writeToXMLFile(String fileName){
		Document document;
		try {
			//create a new document
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();

			Element root = document.createElement("dailyActions"); 
			document.appendChild(root);
			
			//go through messages
			for(int m = 0; m < messages.size(); m++){
				Transportable content = (Transportable) messages.get(m);
				
				//DeliverySchedule
				if (content instanceof DeliverySchedule){
					DeliverySchedule delivery = (DeliverySchedule) content;
					Element eltDeliverySchedule = document.createElement("deliverySchedule");
					root.appendChild(eltDeliverySchedule);
					eltDeliverySchedule.setAttribute("recipient", (String) messageRecipients.get(m));
					for (int i = 0, n = delivery.size(); i < n; i++) {
						Element eltDelivery = document.createElement("delivery");
						eltDeliverySchedule.appendChild(eltDelivery);
						eltDelivery.setAttribute("quantity", "" + delivery.getQuantity(i));
						eltDelivery.setAttribute("productID", "" + delivery.getProductID(i));
						eltDelivery.setAttribute("customer", "" + delivery.getCustomer(i));
						eltDelivery.setAttribute("orderID", "" + delivery.getOrderID(i));
					}
				}
				
				//ProductionSchedule
				else if (content instanceof ProductionSchedule){ 
					ProductionSchedule production = (ProductionSchedule) content;
					Element eltProductionSchedule = document.createElement("productionSchedule");
					root.appendChild(eltProductionSchedule);
					eltProductionSchedule.setAttribute("recipient", (String) messageRecipients.get(m));
					for (int i = 0, n = production.size(); i < n; i++) {
						Element eltProduction = document.createElement("production");
						eltProductionSchedule.appendChild(eltProduction);
						eltProduction.setAttribute("quantity", "" + production.getQuantity(i));
						eltProduction.setAttribute("productID", "" + production.getProductID(i));
					}
				}

				//OrderBundle
				else if (content instanceof OrderBundle){
					OrderBundle orders = (OrderBundle) content;
					Element eltOrderBundle = document.createElement("orderBundle");
					root.appendChild(eltOrderBundle);
					eltOrderBundle.setAttribute("recipient", (String) messageRecipients.get(m));
				 	for (int i = 0, n = orders.size(); i < n; i++){
						Element eltOrder = document.createElement("order");
						eltOrderBundle.appendChild(eltOrder);
						eltOrder.setAttribute("orderID", "" + orders.getOrderID(i));
						eltOrder.setAttribute("offerID", "" + orders.getOfferID(i));
				 	}
				}
				
				//OfferBundle
				else if (content instanceof OfferBundle){
			 		OfferBundle offer = (OfferBundle) content;
			 		Element eltOfferBundle = document.createElement("offerBundle");
					root.appendChild(eltOfferBundle);
					eltOfferBundle.setAttribute("recipient", (String) messageRecipients.get(m));
					for (int i = 0, n = offer.size(); i < n; i++) {
						Element eltOffer = document.createElement("offer");
						eltOfferBundle.appendChild(eltOffer);
						eltOffer.setAttribute("offerID", "" + offer.getOfferID(i));
						eltOffer.setAttribute("rfqID", "" + offer.getRFQID(i));
						eltOffer.setAttribute("quantity", "" + offer.getQuantity(i));
						eltOffer.setAttribute("dueDate", "" + offer.getDueDate(i));
						eltOffer.setAttribute("unitPrice", "" + offer.getUnitPrice(i));
					}
				}
			}
			
			//go through all RFQs to suppliers
			Enumeration enumm = supplyTable.keys();
	 		Element eltRFQList = document.createElement("rfqList");
			root.appendChild(eltRFQList);
	
			while (enumm.hasMoreElements()) {
				String receiver = (String) enumm.nextElement();
				RFQBundle rfq = (RFQBundle) supplyTable.get(receiver);
				Element eltRFQBundle = document.createElement("rfqBundle");
				eltRFQList.appendChild(eltRFQBundle);
				eltRFQBundle.setAttribute("recipient", receiver);
				for (int i = 0, n = rfq.size(); i < n; i++){
					Element eltRFQ = document.createElement("rfq");
					eltRFQBundle.appendChild(eltRFQ);
					eltRFQ.setAttribute("rfqID", "" + rfq.getRFQID(i));
					eltRFQ.setAttribute("quantity", "" + rfq.getQuantity(i));
					eltRFQ.setAttribute("productID", "" + rfq.getProductID(i));
					eltRFQ.setAttribute("dueDate", "" + rfq.getDueDate(i));
					eltRFQ.setAttribute("reservePricePerUnit", "" + rfq.getReservePricePerUnit(i));
					eltRFQ.setAttribute("penalty", "" + rfq.getPenalty(i));
				}	
			}
			
			// Write the DOM document to the file						
			File file = new File(fileName);
			Result result = new StreamResult(file);
			DOMSource source = new DOMSource(document.getDocumentElement());
			Transformer xformer = TransformerFactory.newInstance().newTransformer();
			xformer.transform(source, result);	
		} catch (Exception ex) { ex.printStackTrace(); }
	}	

	public void readFromXMLFile(String fileName){
		try{
			Document document;
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.parse(new File(fileName));
			
			Element root = document.getDocumentElement();
			
			NodeList nodes = root.getChildNodes();
			for (int n = 0; n < nodes.getLength(); n++) {
				Element e = (Element) nodes.item(n);
				
				//DeliverySchedule
				if (e.getTagName() == "deliverySchedule"){
					DeliverySchedule delivery = new DeliverySchedule();
					messageRecipients.add(e.getAttribute("recipient"));
					NodeList deliveryNodes = e.getChildNodes();
					for(int i = 0; i < deliveryNodes.getLength(); i++){
						Element eltDelivery = (Element) deliveryNodes.item(i);
						int quantity = Integer.parseInt(eltDelivery.getAttribute("quantity"));
						int productID = Integer.parseInt(eltDelivery.getAttribute("productID"));
						String customer = eltDelivery.getAttribute("customer");
						int orderID = Integer.parseInt(eltDelivery.getAttribute("orderID"));
						delivery.addDelivery(productID, quantity, orderID, customer);
					}
					messages.add(delivery);
				}
				
				
				//ProductionSchedule
				else if (e.getTagName() == "productionSchedule"){
					ProductionSchedule production = new ProductionSchedule();
					messageRecipients.add(e.getAttribute("recipient"));
					NodeList productionNodes = e.getChildNodes();
					for(int i = 0; i < productionNodes.getLength(); i++){
						Element eltProduction = (Element) productionNodes.item(i);
						int quantity = Integer.parseInt(eltProduction.getAttribute("quantity"));
						int productID = Integer.parseInt(eltProduction.getAttribute("productID"));
						production.addProduction(productID, quantity);
					}
					messages.add(production);
				}
				
				
				//OrderBundle
				else if (e.getTagName() == "orderBundle"){
				OrderBundle orderBundle = new OrderBundle();
				messageRecipients.add(e.getAttribute("recipient"));
				NodeList orderNodes = e.getChildNodes();
				for(int i = 0; i < orderNodes.getLength(); i++){
					Element eltOrder = (Element) orderNodes.item(i);
					int orderID = Integer.parseInt(eltOrder.getAttribute("orderID"));
					int offerID = Integer.parseInt(eltOrder.getAttribute("offerID"));
					orderBundle.addOrder(orderID, offerID);	
				}
				messages.add(orderBundle);
			}
				
				//OfferBundle
				else if (e.getTagName() == "offerBundle"){
				OfferBundle offers = new OfferBundle();
				messageRecipients.add(e.getAttribute("recipient"));
				NodeList offerNodes = e.getChildNodes();
				for(int i = 0; i < offerNodes.getLength(); i++){
					Element eltOffer = (Element) offerNodes.item(i);
					int quantity = Integer.parseInt(eltOffer.getAttribute("quantity"));
					int offerID = Integer.parseInt(eltOffer.getAttribute("offerID"));
					int rfqID = Integer.parseInt(eltOffer.getAttribute("rfqID"));
					int dueDate = Integer.parseInt(eltOffer.getAttribute("dueDate"));
					int unitPrice = Integer.parseInt(eltOffer.getAttribute("unitPrice"));
					offers.addOffer(offerID, rfqID, unitPrice, dueDate, quantity);	
				}
				messages.add(offers);
			}
						
			//go through all RFQs to suppliers
				else if (e.getTagName() == "rfqList"){
					NodeList rfqListNodes = e.getChildNodes();
					for(int b = 0; b < rfqListNodes.getLength(); b++){
						Element eltRFQBundle = (Element) rfqListNodes.item(b);
						String recipient = eltRFQBundle.getAttribute("recipient");
						NodeList rfqBundleNodes = eltRFQBundle.getChildNodes();
						RFQBundle rfqBundle = new RFQBundle();
						for(int i = 0; i < rfqBundleNodes.getLength(); i++){
							Element eltRFQ = (Element) rfqBundleNodes.item(i);
							int rfqID = Integer.parseInt(eltRFQ.getAttribute("rfqID"));
							int quantity = Integer.parseInt(eltRFQ.getAttribute("quantity"));
							int productID = Integer.parseInt(eltRFQ.getAttribute("productID"));
							int dueDate = Integer.parseInt(eltRFQ.getAttribute("dueDate"));
							int reservePricePerUnit = Integer.parseInt(eltRFQ.getAttribute("reservePricePerUnit"));
							int penalty = Integer.parseInt(eltRFQ.getAttribute("penalty"));																	
							rfqBundle.addRFQ(rfqID, productID, quantity, reservePricePerUnit, dueDate, penalty);
						}
						supplyTable.put(recipient, rfqBundle);
					}
				}
			}
		} catch (Exception ex) { ex.printStackTrace(); }
	}
}
