import se.sics.tasim.props.*;

import java.util.*;
import java.io.*;

import javax.xml.parsers.*; 
import javax.xml.transform.*;
import javax.xml.transform.stream.*;
import javax.xml.transform.dom.*;
import org.w3c.dom.*;


/**Stores all of the information sent to the agent at the 
 * beginning of the day.
 */


public class DailyInfo{

	public InventoryStatus inventory;
	public RFQBundle customerRFQs;
	public OrderBundle customerOrders;
	public Vector deliveries = new Vector();	//stores the DeliveryNotices
	public Vector supplierOffers = new Vector(); //stores the OfferBundles from suppliers
	public Vector supplierOfferSenders = new Vector(); //stores the name of the supplier for each offer
	public int date;
	public BankStatus bankStatus;
	public String customerName; //currently only one customer sends rfqs; this is its name
	public PriceReport priceReport;
	public MarketReport marketReport;
	
	public DailyInfo(){};

	/**
	 * Stores an offer from a supplier.
	 * 
	 * @param o  the offer
	 * @param sender  who sent it
	 */
	public void addSupplierOffer(OfferBundle o, String sender){
		supplierOffers.add(o);
		supplierOfferSenders.add(sender);
	}
	
	/**
	 * Stores a delivery notice for components.
	 * 
	 * @param dn  the notice
	 */
	public void addDeliveryNotice(DeliveryNotice dn){
		deliveries.add(dn);
	}

	/**
	 * Summarizes the bank status only
	 * 
	 * @return  summary string
	 */
	public String toString(){
		StringBuffer buf = new StringBuffer();
		buf.append("\n-------------------------\n");
		buf.append("DAY " + date + ":\n");
		buf.append("-------------------------\n");
		if (bankStatus != null){
			buf.append("Bank status: $" + bankStatus.getAccountStatus() + 
					" with " + bankStatus.getPenaltyCount() + " penalties today for $" +
					 bankStatus.getTotalPenaltyAmount() + "\n");
			buf.append("\n");
		}
		return buf.toString();
	}

	/**
	 * Summarizes all information received
	 * @return  summary string
	 */
	public String toVerboseString(){
		StringBuffer buf = new StringBuffer();
		buf.append("\n---------------------\n");
		buf.append("New information for day " + date + ":\n");
		buf.append("---------------------\n");
		//bank status
		if (bankStatus != null){
			buf.append("Bank status: $" + bankStatus.getAccountStatus() + 
					" with " + bankStatus.getPenaltyCount() + " penalties today\n");
			if (bankStatus.getPenaltyCount() > 0){
				for(int i = 0, n = bankStatus.getPenaltyCount(); i < n; i++){
					buf.append("\tpenalty of $" + bankStatus.getPenaltyAmount(i) +
						" for order " + bankStatus.getPenaltyOrderID(i) + "\n");
				}
			}
			buf.append("\n");
		}
		
		//inventory
		if (inventory != null){
			buf.append("Current inventory:");
			for(int i = 0, n = inventory.getProductCount(); i < n; i++){
				buf.append("\tProduct: " + inventory.getProductID(i) + "\tQuantity: "
					+ inventory.getQuantity(i) + "\n");
				}
		 	buf.append("\n");		
		}
		
		//deliveries
		if (deliveries != null && deliveries.size() > 0){
			buf.append("Today's deliveries:\n");
			for (int i = 0, ni = deliveries.size(); i < ni; i++) {
				DeliveryNotice d = (DeliveryNotice) deliveries.get(i);
				for (int j = 0, nj = d.size(); j < nj; j++) {
					buf.append("\t" + d.getSupplier() + " delivered order " + d.getOrderID(j)
						+ ": " + d.getQuantity(j) + " units of " + d.getProductID(j) + "\n");					
				}
			}
			buf.append("\n");
		}
			
		//orders from customers
		if (customerOrders != null && customerOrders.size() > 0){
		 	buf.append("Today's orders from customers:\n");
		 	for (int i = 0, n = customerOrders.size(); i < n; i++)
		 		buf.append("\torder " + customerOrders.getOrderID(i) + " accepting offer "
		 		+ customerOrders.getOfferID(i) + "\n");
		 	buf.append("\n");
		}
		
		//offers from suppliers
		if (supplierOffers != null && supplierOffers.size() > 0){
			buf.append("Today's offers received from suppliers:\n");
			for(int i = 0, n = supplierOffers.size(); i < n; i++){
				OfferBundle o = (OfferBundle) supplierOffers.get(i);
				for (int j = 0, nj = o.size(); j < nj; j++){
					buf.append("\tOffer " + o.getOfferID(j) + " from " +
						supplierOfferSenders.get(i) + " concerning RFQ " + o.getRFQID(j) +
					" is for " + o.getQuantity(j) + " units on day " + o.getDueDate(j) + 
					" for $" + o.getUnitPrice(j) + "\n");
				}
			}
			buf.append("\n");
		}
					
		//rfqs from customers
		if (customerRFQs != null){
			buf.append("Today's RFQs from customers:\n");
			for (int i = 0, n = customerRFQs.size(); i < n; i++) {
				buf.append("\tRFQ " + customerRFQs.getRFQID(i) + " for " +
					customerRFQs.getQuantity(i) + " units of computer " +
					customerRFQs.getProductID(i) + " due by day " + customerRFQs.getDueDate(i)
					+ " with a reserve price of $" + customerRFQs.getReservePricePerUnit(i)
					+ " and a penalty of $" + customerRFQs.getPenalty(i) + " per late day\n");
			}
			buf.append("\n");
		}	
	
		if (marketReport != null){
			buf.append("Market report received:\n");
			buf.append(marketReport.toString());
			buf.append("\n\n");
		}
		
		if (priceReport != null){
			buf.append("Price report received:\n");
			buf.append(priceReport.toString());
		}
			
		return buf.toString();
	}

	public void writeToXMLFile(String fileName){
		Document document;
		try {
			//create a new document
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();

			Element root = document.createElement("dailyInfo"); 
			document.appendChild(root);
			root.setAttribute("date", "" + date);
			
			//bankStatus
			if (bankStatus != null){
				Element eltBank = document.createElement("bankStatus");
				root.appendChild(eltBank);
				eltBank.setAttribute("acountStatus", "" + bankStatus.getAccountStatus());
				for(int i = 0, n = bankStatus.getPenaltyCount(); i < n; i++){
					Element eltPenalty = document.createElement("penalty");
					eltBank.appendChild(eltPenalty);
					eltPenalty.setAttribute("amount", "" + bankStatus.getPenaltyAmount(i));
					eltPenalty.setAttribute("orderId", "" + bankStatus.getPenaltyOrderID(i));
				}
				for(int i = 0, n = bankStatus.getDeniedCount(); i < n; i++){
					Element eltDenied = document.createElement("denied");
					eltBank.appendChild(eltDenied);
					eltDenied.setAttribute("orderId", "" + bankStatus.getDeniedOrderID(i));
				}
			}

			//inventory
			if (inventory != null){
				Element eltInventory = document.createElement("inventory");
				root.appendChild(eltInventory);
				for(int i = 0, n = inventory.getProductCount(); i < n; i++){
					Element eltProduct = document.createElement("product");
					eltInventory.appendChild(eltProduct);
					eltProduct.setAttribute("id", "" + inventory.getProductID(i));
					eltProduct.setAttribute("quantity", "" + inventory.getQuantity(i));
				}
			}

			//deliveries
			if (deliveries != null && deliveries.size() > 0){
				Element eltDeliveries = document.createElement("deliveries");
				root.appendChild(eltDeliveries);
				for (int i = 0, ni = deliveries.size(); i < ni; i++) {
					DeliveryNotice d = (DeliveryNotice) deliveries.get(i);
					for (int j = 0, nj = d.size(); j < nj; j++) {
						Element eltDelivery = document.createElement("delivery");
						eltDeliveries.appendChild(eltDelivery);
						eltDelivery.setAttribute("supplier", d.getSupplier());
						eltDelivery.setAttribute("order", "" + d.getOrderID(j));
						eltDelivery.setAttribute("quantity", "" + d.getQuantity(j));						
						eltDelivery.setAttribute("productId", "" + d.getProductID(j));						
					}
				}
			}
			
			//orders from customers
			if (customerOrders != null && customerOrders.size() > 0){
				Element eltOrders = document.createElement("customerOrders");
				root.appendChild(eltOrders);
			 	for (int i = 0, n = customerOrders.size(); i < n; i++){
		 			Element eltOrder = document.createElement("order");
					eltOrders.appendChild(eltOrder);
					eltOrder.setAttribute("orderId", "" + customerOrders.getOrderID(i));
					eltOrder.setAttribute("offerId", "" + customerOrders.getOfferID(i));
				}
			}
			
			//offers from suppliers
			if (supplierOffers != null && supplierOffers.size() > 0){
				Element eltOffers = document.createElement("supplierOffers");
				root.appendChild(eltOffers);
				for(int i = 0, n = supplierOffers.size(); i < n; i++){
					OfferBundle o = (OfferBundle) supplierOffers.get(i);
					Element eltOfferBundle = document.createElement("offerBundle");
					eltOffers.appendChild(eltOfferBundle);
					eltOfferBundle.setAttribute("supplier", "" + supplierOfferSenders.get(i));
					for (int j = 0, nj = o.size(); j < nj; j++){
						Element eltOffer = document.createElement("offer");
						eltOfferBundle.appendChild(eltOffer);
						eltOffer.setAttribute("offerId", "" + o.getOfferID(j));
						eltOffer.setAttribute("rfq", "" + o.getRFQID(j));
						eltOffer.setAttribute("quantity", "" + o.getQuantity(j));
						eltOffer.setAttribute("dueDate", "" + o.getDueDate(j));
						eltOffer.setAttribute("unitPrice", "" + o.getUnitPrice(j));
					}
				}
			}
			
			//rfqs from customers
			if (customerRFQs != null){
				Element eltRFQs = document.createElement("rfqs");
				root.appendChild(eltRFQs);
				eltRFQs.setAttribute("customerName", customerName);
				for (int i = 0, n = customerRFQs.size(); i < n; i++) {
					Element eltRFQ = document.createElement("rfq");
					eltRFQs.appendChild(eltRFQ);
					eltRFQ.setAttribute("rfqId", "" + customerRFQs.getRFQID(i));
					eltRFQ.setAttribute("productId", "" + customerRFQs.getProductID(i));
					eltRFQ.setAttribute("quantity", "" + customerRFQs.getQuantity(i));
					eltRFQ.setAttribute("dueDate", "" + customerRFQs.getDueDate(i));
					eltRFQ.setAttribute("reserve", "" + customerRFQs.getReservePricePerUnit(i));
					eltRFQ.setAttribute("penalty", "" + customerRFQs.getPenalty(i));
				}
			}	

			//price report
			if (priceReport != null){
				Element eltPrices = document.createElement("priceReport");
				root.appendChild(eltPrices);
				for (int i = 0, n = priceReport.size(); i < n; i++) {
					Element eltProduct = document.createElement("product");
					eltPrices.appendChild(eltProduct);
					eltProduct.setAttribute("productId", "" + priceReport.getProductID(i));
					eltProduct.setAttribute("highestPrice", "" + priceReport.getHighestPrice(i));
					eltProduct.setAttribute("lowestPrice", "" + priceReport.getLowestPrice(i));										
				}
			}
			
			//market report
			if (marketReport != null){
				Element eltMarket = document.createElement("marketReport");
				root.appendChild(eltMarket);
				eltMarket.setAttribute("startDate", "" + marketReport.getStartDate());
				eltMarket.setAttribute("endDate", "" + marketReport.getEndDate());				
				Element eltSupply = document.createElement("supply");
				eltMarket.appendChild(eltSupply);
				Element eltDemand = document.createElement("demand");
				eltMarket.appendChild(eltDemand);
				Element eltSuppliers = document.createElement("suppliers");
				eltMarket.appendChild(eltSuppliers);
				
				for (int i = 0, n = marketReport.getDemandCount(); i < n; i++) {
					Element eltProduct = document.createElement("product");
					eltDemand.appendChild(eltProduct);
					eltProduct.setAttribute("productId", "" + marketReport.getDemandProductID(i));
					eltProduct.setAttribute("productsOrdered", "" + marketReport.getProductsOrdered(i));
					eltProduct.setAttribute("averagePrice", "" + marketReport.getAverageProductPrice(i));
					eltProduct.setAttribute("productsRequested", "" + marketReport.getProductsRequested(i));
				}
				for (int i = 0, n = marketReport.getSupplyCount(); i < n; i++) {
					Element eltProduct = document.createElement("product");
					eltSupply.appendChild(eltProduct);
					eltProduct.setAttribute("productId", "" + marketReport.getSupplyProductID(i));
					eltProduct.setAttribute("delivered", "" + marketReport.getSupplyDelivered(i));
					eltProduct.setAttribute("ordered", "" + marketReport.getSupplyOrdered(i));
					eltProduct.setAttribute("averagePrice", "" + marketReport.getAverageSupplyPrice(i));					
				}
				for (int i = 0, n = marketReport.getSupplierCount(); i < n; i++) {
					Element eltProduct = document.createElement("supplier");
					eltSuppliers.appendChild(eltProduct);
					eltProduct.setAttribute("productId", "" + marketReport.getSupplierProductID(i));
					eltProduct.setAttribute("supplierAddress", "" + marketReport.getSupplierAddress(i));
					eltProduct.setAttribute("averageCapacity", "" + marketReport.getAverageSupplierCapacity(i));
				}
			}
			
			// Write the DOM document to the file						
			File file = new File(fileName);
			Result result = new StreamResult(file);
			DOMSource source = new DOMSource(document.getDocumentElement());
			Transformer xformer = TransformerFactory.newInstance().newTransformer();
			xformer.transform(source, result);
		} catch (Exception ex) { ex.printStackTrace(); }
	}

	public void readFromXMLFile(String fileName){
		inventory = new InventoryStatus();
		customerRFQs = new RFQBundle();
		customerOrders = new OrderBundle();
		bankStatus = new BankStatus();
	
		try {
			Document document;
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.parse(new File(fileName));
			
			Element root = document.getDocumentElement();
			date = Integer.parseInt(root.getAttribute("date"));
		
			NodeList nodes = root.getChildNodes();
			for (int n = 0; n < nodes.getLength(); n++) {
				Element e = (Element) nodes.item(n);
			
				//bankStatus
				if (e.getTagName() == "bankStatus"){
					bankStatus.setAccountStatus(Integer.parseInt(e.getAttribute("acountStatus")));
					NodeList penalties = e.getChildNodes();
					for(int i = 0; i < penalties.getLength(); i++){
						Element eltPenalty = (Element) penalties.item(i);
						int amount = Integer.parseInt(eltPenalty.getAttribute("amount"));
						int orderId = Integer.parseInt(eltPenalty.getAttribute("orderId"));
						bankStatus.addPenalty("", orderId, amount);
					}
				}

				//inventory
				else if (e.getTagName() == "inventory"){
					NodeList products = e.getChildNodes();
					for(int i = 0; i < products.getLength(); i++){
						Element eltProduct = (Element) products.item(i);
						int productId = Integer.parseInt(eltProduct.getAttribute("id"));
						int quantity = Integer.parseInt(eltProduct.getAttribute("quantity"));
						inventory.addInventory(productId, quantity);
					}
				}

				//deliveries
				else if (e.getTagName() == "deliveries"){
					NodeList dlist = e.getChildNodes();
					for (int i = 0; i < dlist.getLength(); i++) {
						Element eltDelivery = (Element) dlist.item(i);
						String supplier = eltDelivery.getAttribute("supplier");
						int order = Integer.parseInt(eltDelivery.getAttribute("order"));
						int quantity = Integer.parseInt(eltDelivery.getAttribute("quantity"));						
						int productId = Integer.parseInt(eltDelivery.getAttribute("productId"));
						DeliveryNotice d = new DeliveryNotice(supplier, "");
						d.addDelivery(order, productId, quantity);
						deliveries.add(d);
					}
				}
			
				//orders from customers
				else if (e.getTagName() == "customerOrders"){
			 		NodeList orderList = e.getChildNodes();
					for (int i = 0; i < orderList.getLength(); i++){
			 			Element eltOrder = (Element) orderList.item(i);
						int orderId = Integer.parseInt(eltOrder.getAttribute("orderId"));
						int offerId = Integer.parseInt(eltOrder.getAttribute("offerId"));
						customerOrders.addOrder(orderId, offerId);
					}
				}
			
				//offers from suppliers
				else if (e.getTagName() == "supplierOffers"){
			 		NodeList offerList = e.getChildNodes();
					for(int i = 0; i < offerList.getLength(); i++){
						Element eltOfferBundle = (Element) offerList.item(i);
						supplierOfferSenders.add(eltOfferBundle.getAttribute("supplier"));
						OfferBundle o = new OfferBundle();
						NodeList itemList = eltOfferBundle.getChildNodes();
						for (int j = 0; j < itemList.getLength(); j++){
							Element eltOffer = (Element) itemList.item(j);
							int offerId = Integer.parseInt(eltOffer.getAttribute("offerId"));
							int rfq = Integer.parseInt(eltOffer.getAttribute("rfq"));
							int quantity = Integer.parseInt(eltOffer.getAttribute("quantity"));
							int dueDate = Integer.parseInt(eltOffer.getAttribute("dueDate"));
							int unitPrice = Integer.parseInt(eltOffer.getAttribute("unitPrice"));
							o.addOffer(offerId, rfq, unitPrice, dueDate, quantity);
						}
						supplierOffers.add(o);
					}
				}
			
				//rfqs from customers
				else if (e.getTagName() == "rfqs"){
					customerName = e.getAttribute("customerName");
			 		NodeList rfqList = e.getChildNodes();
					for (int i = 0; i < rfqList.getLength(); i++) {
						Element eltRFQ = (Element) rfqList.item(i);
						int rfqId = Integer.parseInt(eltRFQ.getAttribute("rfqId"));
						int productId = Integer.parseInt(eltRFQ.getAttribute("productId"));
						int quantity = Integer.parseInt(eltRFQ.getAttribute("quantity"));
						int dueDate = Integer.parseInt(eltRFQ.getAttribute("dueDate"));
						int reserve = Integer.parseInt(eltRFQ.getAttribute("reserve"));
						int penalty = Integer.parseInt(eltRFQ.getAttribute("penalty"));
						customerRFQs.addRFQ(rfqId, productId, quantity, reserve, dueDate, penalty);
					}
				}
				
				//price report
				else if (e.getTagName() == "priceReport"){
			 		priceReport = new PriceReport();
			 		NodeList productList = e.getChildNodes();
					for (int i = 0; i < productList.getLength(); i++) {
						Element eltProduct = (Element) productList.item(i);
						int productID = Integer.parseInt(eltProduct.getAttribute("productId"));
						int highestPrice = Integer.parseInt(eltProduct.getAttribute("highestPrice"));
						int lowestPrice = Integer.parseInt(eltProduct.getAttribute("lowestPrice"));
						priceReport.addPriceInfo(productID, lowestPrice, highestPrice);
					}
				}
			
				//market report
				else if (e.getTagName() == "marketReport"){
			 		int startDate = Integer.parseInt(e.getAttribute("startDate"));
			 		int endDate = Integer.parseInt(e.getAttribute("endDate"));
			 		marketReport = new MarketReport(startDate, endDate);
			 		NodeList sdList = e.getChildNodes();
					for (int i = 0; i < sdList.getLength(); i++) {
						Element sd = (Element) sdList.item(i);
						NodeList productList = sd.getChildNodes();
						if (sd.getTagName() == "demand"){
							for (int j = 0; j < productList.getLength(); j++) {
								Element eltProduct = (Element) productList.item(j);
								int productId = Integer.parseInt(eltProduct.getAttribute("productId"));
								int productsOrdered = Integer.parseInt(eltProduct.getAttribute("productsOrdered"));
								int averagePrice = Integer.parseInt(eltProduct.getAttribute("averagePrice"));
								int productsRequested = Integer.parseInt(eltProduct.getAttribute("productsRequested"));
								marketReport.addDemandForProduct(productId, productsRequested, productsOrdered, averagePrice);
							}
						}
						else if (sd.getTagName() == "supply"){
							for (int j = 0; j < productList.getLength(); j++) {
								Element eltProduct = (Element) productList.item(j);
								int productId = Integer.parseInt(eltProduct.getAttribute("productId"));
								int delivered = Integer.parseInt(eltProduct.getAttribute("delivered"));
								int ordered = Integer.parseInt(eltProduct.getAttribute("ordered"));
								int averagePrice = Integer.parseInt(eltProduct.getAttribute("averagePrice"));
								marketReport.addSupplyDelivered(productId, delivered);
								marketReport.addSupplyOrdered(productId, ordered, averagePrice);
								//marketReport.addSupplyProduced(productId, produced);
							}
						}
						else if (sd.getTagName() == "suppliers"){
							for (int j = 0; j < productList.getLength(); j++) {
								Element eltProduct = (Element) productList.item(j);
								int productId = Integer.parseInt(eltProduct.getAttribute("productId"));
								String supplierAddress = eltProduct.getAttribute("supplierAddress");
								int averageCapacity = Integer.parseInt(eltProduct.getAttribute("averageCapacity"));
								marketReport.addSupplierCapacity(supplierAddress, productId, averageCapacity);
							}
						}
					}
				}
			}
		} catch (Exception ex) { ex.printStackTrace(); }
	}
}
