import java.util.*;

import se.sics.tasim.props.*;

/**
 * A very simple demand manager.
 */
public class SimpleDemandManager implements InterfaceDemandManager{

	
	/**
	 * Used for sorting orders and/or RFQs
	 */
	class OrderHandler implements Comparable{
		public OrderInfo order = null;
		public RFQInfo rfq = null;
		public double profit;
				
		OrderHandler(OrderInfo order, double profit){
			this.order = order;
			this.profit = profit;
		}
		
		OrderHandler(RFQInfo rfq, double profit){
			this.rfq = rfq;
			this.profit = profit;
		}
		
		public int compareTo(Object other){ //want to sort in DESCENDING order
			double otherProfit = ((OrderHandler) other).profit;
			if (profit == otherProfit)
				return 0;
			else 
				return (profit > otherProfit) ? -1 : 1;
		}
	}
	
	
	final static int numComponents = 10;
	final static int numComputers = 16;
	
	private GameSettings settings;
	private DailyInfo dayInfo;
	private AgentInfo agentInfo;
	private DailyActions actions;	
	private int date;
	/**how many days of production to plan for*/
	private int numProductionDays; 
	private OfferBundle offers = new OfferBundle();
	
	private double[] componentCost = new double[numComponents];//replacement cost per component used
	private double[] computerCost = new double[numComputers];//replacement cost of all components in computer
	
	/**number of components remaining*/
	private int[][] components; 
	/**number of computers remaining in inventory*/
	private int[] completedComputers; 
	/**cycles remaining each day*/
	private int[] cycles; 
	/**how many of each computer are produced per day*/
	private int[][] producedPerDay; 
	
	private Random random = new Random();
	
	/**
	 * Predicts replacement costs for each type of component and computer
	 */
	
	private void calculateCosts(){
		//first determine component replacement costs
		for (int i = 0; i < numComponents; i++){
			//find the last price paid for each component
			int size = agentInfo.components.ordersByDate[i].size();
			if (size == 0) // no orders yet
				componentCost[i] = 0.75 * agentInfo.components.basePrices[i];
			else 
				componentCost[i] = ((OrderInfo)agentInfo.components.ordersByDate[i].get(size-1)).price;
		}
		
		//now get the cost of each computer type
		for (int i = 0; i < numComputers; i++){
			computerCost[i] = 0;
			for (int c = 0; c < agentInfo.computers.componentIndexList[i].length; c++)
				computerCost[i] += componentCost[agentInfo.computers.componentIndexList[i][c]];
		}	
	}
	
	
	
	/**
	 * Determines how many days of production should be planned for.
	 */
	private void getProductionDays(){
		//do 10 days, unless near the end of the game
		//there is no reason to produce past day 219
		numProductionDays = 10;
		if (date + numProductionDays >= settings.numberOfDays - 2)
			numProductionDays = settings.numberOfDays - date - 2;
	}

	/**
	 * Initializes the resources - cycles, components, and completed computers.
	 *
	 */
	private void setupResources(){
		int size = numProductionDays;
		if (size < 0)
			size = 0;
		cycles = new int[size];
		//set all cycles to factory capacity
		for (int i = 0; i < size; i++)
			cycles[i] = settings.factoryCapacity;
		
		components = new int[numComponents][size];
		completedComputers = new int[numComputers];
		System.arraycopy(agentInfo.tomorrowsRemainingComputers, 0, completedComputers, 0, numComputers);

		if (numProductionDays >= 1){
			for (int i = 0; i < numComponents; i++){
				components[i][0] = agentInfo.tomorrowsRemainingComponents[i];
				for (int d = 1; d < numProductionDays; d++)
					components[i][d] = agentInfo.components.expectedDeliveries[i][dayInfo.date + d] + agentInfo.newComponentDeliveries[i][dayInfo.date + d];
			}
		}
		
		producedPerDay = new int[numComputers][numProductionDays];
	}
	
	
	/**
	 * Tries to deliver or produce existing orders.  Orders are sorted by
	 * profit, then considered one at a time.  Orders that are due or late
	 * will be delivered if possible.  Other orders will not be delivered yet.
	 * An attempt will be made to produce each order up to its cancellation
	 * date.
	 */
	private void produceExistingOrders(){
		
		DeliverySchedule dSchedule = new DeliverySchedule();
		Vector allOrders = new Vector();
		Iterator it = agentInfo.computers.orderTable.values().iterator();
		while (it.hasNext()){
			OrderInfo o = (OrderInfo) it.next();
			OrderHandler oHandler = new OrderHandler(o, o.price - computerCost[o.index]);
			allOrders.add(oHandler);
		}
		
		//sort orders by profit
		Collections.sort(allOrders);
		
		//now try to produce in order
		for (int i = 0; i < allOrders.size(); i++){
			OrderInfo o = ((OrderHandler)allOrders.get(i)).order;
			//try to deliver if due or late
			if (o.dueDate <= date + 1 && completedComputers[o.index] >= o.quantity){
				dSchedule.addDelivery(o.productID, o.quantity, o.orderID, o.name);
				completedComputers[o.index] -= o.quantity;
				continue;
			}
			
			//now try to produce, up to 4 days late
			for (int late = 0; late < settings.daysBeforeVoid - 1; late++){
				int daysUntilDue = o.dueDate - date + late;
				if (daysUntilDue < 1) //can't deliver before tomorrow
					continue;
				if (tryToProduce(o.index, o.quantity, daysUntilDue)) //success
					break;
			}
		}

		actions.addMessage(settings.factoryName, dSchedule);
				
	}

	/**
	 * Makes offers on RFQs.  RFQs are considered in random order, and a
	 * random price is chosen for each RFQ.
	 * An offer will be made for any RFQ that can be produced in time.  
	 */
	private void handleRFQs(){
		
		//get list of RFQs - order should be effectively random 
		Vector rfqsToConsider = new Vector();
		Iterator it = agentInfo.computers.rfqTable.values().iterator();
		while (it.hasNext()){
			RFQInfo rfq = (RFQInfo)it.next();
			rfqsToConsider.add(rfq);
		}
			
		offers = new OfferBundle();
		
		//now try to produce in order
		for (int i = 0; i < rfqsToConsider.size(); i++){
			RFQInfo rfq = (RFQInfo) rfqsToConsider.get(i);
			int index = agentInfo.computers.getIndex(rfq.productID);
			int price;
			
			if (date == 0 || agentInfo.computers.lowPrice[index][date-1] == 0){
				//no previous high/low, so offer at reserve price
				price = rfq.reservePrice;
			} 
			else { //offer between low and 1.5*(high-low)
				double x = random.nextDouble();
				int p = 2;
				x = 1.5 * (1-Math.pow(x, p)); //skew so higher values are likely
				if (agentInfo.computers.highPrice[index][date-1] < computerCost[index])//will lose money, so shift higher 
					x += .75; 

				int range = agentInfo.computers.highPrice[index][date-1] - agentInfo.computers.lowPrice[index][date-1];
				if (range < 50) 
					range = 50;
				price = (int) (agentInfo.computers.lowPrice[index][date-1] + x * range); 
			}

			if (price > rfq.reservePrice) //skip
				continue;
			 
			//now try to produce, but don't consider late delivery
			int daysUntilDue = rfq.dueDate - date;
			if (tryToProduce(index, rfq.quantity, daysUntilDue)){ //success, so issue offer
				offers.addOffer(agentInfo.getNextID(), rfq.rfqID, price, rfq.dueDate, rfq.quantity);	
			}
		}
		
		actions.addMessage(dayInfo.customerName, offers);
	}
	
	/**
	 * Tries to produce the needed computers.  The computers will be produced 
	 * using cyles and components from the latest possible days.  If not
	 * all computers can be produced, the remainder will be taken from
	 * already completed computers if possible.
	 * 
	 * @param index  computer index
	 * @param quantity  how many are needed
	 * @param daysUntilDue  how far in the future the due date is
	 * @return  whether production was possible
	 */
	private boolean tryToProduce(int index, int quantity, int daysUntilDue){
		//the day with index 0 in the production schedule is actually tomorrow,
		//so an order due in two days would need to be produced on day 0
		int currentDay = daysUntilDue - 2;
		if (currentDay >= numProductionDays) //this date is after the last day of production
			return false;
		
		int[] componentList = agentInfo.computers.componentIndexList[index]; //components in this computer
		int numberLeft = quantity; //how many more are needed
		int cyclesPerComputer = settings.bom.getAssemblyCyclesRequired(index);
		
		//make arrays for recording changes so they can be undone
		int[] usedCycles = new int[cycles.length];
		int[][] usedComponents = new int[componentList.length][numProductionDays];
		
		
		while (currentDay >= 0 && numberLeft > 0){//make as much as possible on current date
			
			int tempCycles = (int)cycles[currentDay];
			//check if there are enough cycles
			int maxProduction = (int)Math.floor(tempCycles / cyclesPerComputer);
			if (maxProduction > numberLeft)
				maxProduction = numberLeft;
			int component;
			int totalComps;
			
			//find out how many components are available, reduce production if needed
			//The number of components in the components array is the number delivered
			//each day, so all components at an index on or before currentDay are available.
			for (component = 0; component < componentList.length; component++){
				totalComps = 0; //how many are available to use
				int componentIndex = componentList[component];
				for (int d = currentDay; d >= 0; d--)
					totalComps += components[componentIndex][d];
				if (totalComps < maxProduction){
					maxProduction = totalComps;
				}
			}
				
			//remove used components - start with the latest to be delivered
			for (component = 0; component < componentList.length; component++){
				int componentIndex = componentList[component];
				int needed = maxProduction;
				for (int d = currentDay; d >= 0; d--){
					if (components[componentIndex][d] >= needed){
						components[componentIndex][d] -= needed;
						usedComponents[component][d] += needed;
						needed = 0;
						break;
					}
					else{
						int available = components[componentIndex][d];
						needed -= available;
						components[componentIndex][d] = 0;
						usedComponents[component][d] += available;
					}
				}
			}
			numberLeft -= maxProduction;
			cycles[currentDay] -= maxProduction * cyclesPerComputer;
			producedPerDay[index][currentDay] += maxProduction;
			usedCycles[currentDay] += maxProduction * cyclesPerComputer;
			currentDay--;
		}
		
		//now if any more computers are needed, check inventory for completed computers
		if (completedComputers[index] >= numberLeft){
			completedComputers[index] -= numberLeft;
			return true;
		}
		else { //can't complete order, so put back used cycles and components
			for (int d = 0; d < usedCycles.length; d++){
				cycles[d] += usedCycles[d];
				producedPerDay[index][d] -= usedCycles[d]/cyclesPerComputer;
				for (int c = 0; c < usedComponents.length; c++)
					components[componentList[c]][d] += usedComponents[c][d];
			}
			return false;
		}
	}
		
	/**
	 * Determines the production schedule for tomorrow, and sends it to the
	 * factory.  If there are free cycles, projected production from future
	 * days is moved forwards.
	 */
	private void sendProductionSchedule(){
		if (date >= 218) //no production
			return;
		int cycles = settings.factoryCapacity;
		int[] comps = new int[numComponents];
		for (int i = 0; i < numComponents; i++)
			comps[i] = agentInfo.tomorrowsRemainingComponents[i];
	
		//do all production scheduled for tomorrow
		for (int computer = 0; computer < numComputers; computer++){
			int[] componentList = agentInfo.computers.componentIndexList[computer];
			int num = (int) producedPerDay[computer][0];
			int cyclesPerComputer = settings.bom.getAssemblyCyclesRequired(computer); 	
			cycles -= cyclesPerComputer * num;
			for (int c = 0; c < 4; c++)
				comps[componentList[c]] -= num;
		}
	
		ProductionSchedule production = new ProductionSchedule();
		for (int computer = 0; computer < numComputers; computer++)
			production.addProduction(agentInfo.computers.productIDs[computer], producedPerDay[computer][0]);
		actions.addMessage(settings.factoryName, production);
	}
	
	/**
	 * Project component use for some number of days in the future, so the
	 * supply manager will know what to order.
	 */
	public void projectFutureComponentUse(){
		//for now, this just sets projected use to the planned production
		agentInfo.projectedComponentUse = new double[numComponents][settings.numberOfDays + 1];
		for (int d = 0; d < numProductionDays; d++){
			for (int computer = 0; computer < numComputers; computer++){
				int[] componentList = agentInfo.computers.componentIndexList[computer];
				for (int c = 0; c < componentList.length; c++)
					agentInfo.projectedComponentUse[componentList[c]][date + d] += producedPerDay[computer][d];
			}
		}
		
	}
	
	/**
	 * Performs all of the demand manager's actions: produce and deliver 
	 * computers, and bid on RFQs from customers.
	 */
	public void produceDeliverBid(GameSettings settings, DailyInfo dayInfo, AgentInfo agentInfo, DailyActions actions){
		date = dayInfo.date;
		if (date == 0 || date == 219) //nothing to do
			return;
		
		this.settings = settings;
		this.agentInfo = agentInfo;
		this.dayInfo = dayInfo;
		this.actions = actions;
		
		calculateCosts();
		getProductionDays();
		setupResources();
		produceExistingOrders();
		handleRFQs();
		sendProductionSchedule();
		projectFutureComponentUse();
	}
}
