import java.util.*;
import se.sics.tasim.props.*;

/**
 * A very simple supply manager.   
 */
public class SimpleSupplyManager implements InterfaceSupplyManager{
	
	final static int numComponents = 10;
	final static int inventoryGoal = 800;
	//final static int maxDays = 10;
	Random random = new Random();
	
	/**
	 * Sends RFQs to suppliers for needed components.  
	 */
	public void sendRFQsToSuppliers(GameSettings settings, DailyInfo dayInfo, AgentInfo agentInfo, DailyActions actions){
		int date = dayInfo.date;
		if (date > 210)//don't want any more
			return;
		if (date == 0){ //use a fixed set of orders
			for (int c = 0; c < numComponents; c++){ //loop through components
				for (int s = 0; s < 2; s++){ //loop through both suppliers
					SupplierModel supplier = agentInfo.components.supplierModel[c][s];
					if (supplier != null){//there may only be one supplier
						//use existing bundle, if there is one
						RFQBundle b = (RFQBundle) actions.supplyTable.get(supplier.getName());
						if (b == null){ //no bundle yet
							b = new RFQBundle();
							actions.supplyTable.put(supplier.getName(), b);
						}
						actions.supplyTable.put(supplier.getName(), b);
						for (int r = 0; r < 5; r++){ //5 RFQs, request 500 every 5 days
							//addRFQ has arguments (RFQ id, product id, quantity, reserve, due date, penalty (always 0) )
							b.addRFQ(agentInfo.getNextID(), supplier.getProductID(), 500, 9999, 5 + 5 * r, 0);
						}
					}
				}
			}
			return;
		}
		
		//normal case
		for (int c = 0; c < numComponents; c++){
			int previousDay = 0;
			int inv = agentInfo.tomorrowsRemainingComponents[c];
			int[] requestDays = new int[3];
			int[] needed = new int[3];
			int dailyUse = 0;
						
			//assume daily use equals average over past 10 days
			int sum = 0;
			int daysBack = 10;
			if (date < 10)
				daysBack = date;
			for (int i = 1; i <= daysBack; i++)
				sum += agentInfo.components.used[c][date-i];
			dailyUse = sum / daysBack;
			if (dailyUse > 200) //don't order a huge amount
				dailyUse = 200;
				
			
			//get one day in 5-10, another in 11-20, and another in 21-30
			requestDays[0] = 5 + random.nextInt(6);
			requestDays[1] = 11 + random.nextInt(10);
			requestDays[2] = 21 + random.nextInt(10);
			
			for (int r = 0; r < 3; r++){//send three rfqs with nonzero quantities, if needed
				if (date + requestDays[r] > settings.numberOfDays - 1)//past end of game
					continue;
				
				for (int d = previousDay + 1; d < requestDays[r]; d++)//subtract planned component usage, and add deliveries
					inv += agentInfo.newComponentDeliveries[c][date+d] + agentInfo.components.expectedDeliveries[c][date+d] - dailyUse;
				int goal = inventoryGoal;
				if (date + requestDays[r] > settings.numberOfDays - 10) //drop threshold at end
					goal = 0;
				if (c < 4) //CPU, only need half as many
					needed[r] = goal/2 - inv;	
				else
					needed[r] = goal - inv;
				
				//reduce needed by 3% each day after 5, so that a mix of long and short term orders result 
				needed[r] = (int) (needed[r] * (1 - .03 * (requestDays[r] - 5)));
				if (needed[r] < 0)
					needed[r] = 0;
				
				if (c >= 4) //request half from each supplier
					needed[r] /= 2;
				
				previousDay = requestDays[r];	
			}
			
			//now send rfqs
			for (int s = 0; s < 2; s++){
				SupplierModel supplier = agentInfo.components.supplierModel[c][s];
				if (supplier == null)
					continue;
				//use existing bundle, if there is one
				RFQBundle b = (RFQBundle) actions.supplyTable.get(supplier.getName());
				if (b == null){ //no bundle yet
					b = new RFQBundle();
					actions.supplyTable.put(supplier.getName(), b);
				}
				for (int i = 0; i < 3; i++){
					int dueDate = date + requestDays[i];
					if (dueDate > settings.numberOfDays - 1)//past end of game
						continue;
					int quantity = needed[i];
					int reserve = agentInfo.components.basePrices[c];
					b.addRFQ(agentInfo.getNextID(), supplier.getProductID(), quantity, reserve, dueDate, 0);
				}
				
				//add zero-quantity RFQ due between 5 and 30 days
				//use negative ID so that it will be recognized and used for a needed prediction
				int range = 26;
				if (date + range > settings.numberOfDays - 1)
					range = settings.numberOfDays - 1 - date;
				int dueDate = date + 5 + random.nextInt(range);
				b.addRFQ(-agentInfo.getNextID(), supplier.getProductID(), 0, 0, dueDate, 0);
				
				//add zero-quantity RFQ due between 31 and 60 days
				dueDate = date + 31 + random.nextInt(30);
				if (dueDate <= settings.numberOfDays - 1)//past end of game
					b.addRFQ(agentInfo.getNextID(), supplier.getProductID(), 0, 0, dueDate, 0);
			}
		}
	}
	
	/**
	 * Responds to offers from suppliers by accepting all of them.  In the case
	 * of alternative offers, the later offer is accepted.
	 */
	public void respondToSupplierOffers(GameSettings settings, DailyInfo dayInfo, AgentInfo agentInfo, DailyActions actions){
		
		//reset new deliveries to zero
		agentInfo.newComponentDeliveries = new int[10][221];
		
		for (int c = 0; c < numComponents; c++){ //loop through components	
			for (int s = 0; s < 2; s++){ //loop through both suppliers (may only be one)
				Vector offers = agentInfo.components.todaysSortedOffers[c][s];
				for (int o = 0; o < offers.size(); o++){ //loop through all orders
					OfferInfo offer = (OfferInfo)offers.get(o);

					if (offer.quantity == 0) //0 quantity probe
						continue;
					
					if (offer.alternative != null) 
					{
					 	//there is an alternative offer, probably with relaxed due date
						//take the offer with later due date, to ensure perfect reputation
						if (offer.alternative.dueDate > offer.dueDate)
							offer = offer.alternative;
					}
					
					//always accept the offer
					agentInfo.newComponentDeliveries[offer.index][offer.dueDate] += offer.quantity;
					OrderBundle order = new OrderBundle();
					order.addOrder(agentInfo.getNextID(), offer.offerID);
					actions.addMessage(offer.name, order);
				}
			}
		}
	}
}
