import java.util.*;
import se.sics.tasim.props.MarketReport;

/** Keeps track of a single supplier and component.  There will be two objects
 *  per supplier (e.g., pintel will have one both CPU types).  Reputation is 
 *  tracked, predictions can be made about prices the supplier will offer in
 *  response to RFQs, and deliveries can be predicted.  The ComponentInfo
 *  object will update this object at the start of each day with information
 *  about offers made and components delivered, and at the end of each day 
 *  with the offers accepted.  
 *  
 */


public class SupplierModel {
	final static int initialAverageCapacity = 550; //average starting capacity
	final static int initialReputationQuantity = 2000; //used to initialize reputation
	final static double cpuAPR = 0.75; //APR for cpus
	final static double otherAPR = 0.45;  //APR for all other components
	final static int reputationRecovery = 100;  //added to offers, orders daily
	private int componentIndex;
	private int productId;
	private String name;
	private int basePrice;
	private double capacity;
	//static hashtables used because reputation depends on both products for each supplier
	private static Hashtable quantityOrdered = new Hashtable(8); //used for calculating reputation
	private static Hashtable quantityOffered = new Hashtable(8); //used for calculating reputation
	private double reputation;
	private double[] fractionCommitted = null; //estimated fraction committed on a number of days
	private int[] fractionCommittedDates = null; //days corresponding to estimated fraction committed
		
	private Hashtable myOrders = new Hashtable(); //record all orders	
	private int date;
	
	/**
	 * @param componentIndex  index (0-9)
	 * @param productId  (e.g., 301)
	 * @param name  supplier name
	 * @param basePrice  the component's base price
	 */
	public SupplierModel(int componentIndex, int productId, String name, int basePrice){
		this.basePrice = basePrice;
		this.name = name;
		this.componentIndex = componentIndex; 	
		this.productId = productId;
	
		capacity = initialAverageCapacity; //this is the average starting capacity, real capacity is unknown
		quantityOrdered.put(name,new Integer(initialReputationQuantity));
		quantityOffered.put(name,new Integer(initialReputationQuantity));
		reputation = 1;
		date = 0;
		
	}

	/**
	 * Returns the supplier's name.
	 * @return  name
	 */
	public String getName(){
		return name;
	}

	/**
	 * Returns the product ID
	 * @return  ID
	 */
	public int getProductID(){
		return productId;
	}
	
	/**
	 * Updates estimates of the supplier's fraction of capacity used.
	 * @param date  today's date
	 * @param offers  a vector of offers
	 */
	public void update(int date, Vector offers){
		//note: the ComponentInfo object currently sorts offers in order of
		//ascending due date
		
		this.date = date;
		//now find amount committed each day
		fractionCommitted = new double[offers.size()];
		fractionCommittedDates = new int[offers.size()];
		for (int o = 0; o < offers.size(); o++){
			OfferInfo offer = (OfferInfo)offers.get(o);
			fractionCommittedDates[o] = offer.dueDate;	
			//For a due date d, we can directly calculate the fraction committed
			//between now and d using equation (10) from the specs
			fractionCommitted[o] = ((double) offer.price / basePrice / 0.5 - 1);
		}
	}
	
	/**
	 * Processes a periodic (every 20 days) market report.
	 * Currently, this only records the supplier's reported capacity,
	 * which is actually the average capacity over the period.
	 * 
	 * @param mr  the market report
	 */ 
	public void processMarketReport(MarketReport mr){
		int index = mr.getSupplierIndexFor(name, productId);
		capacity = mr.getAverageSupplierCapacity(index);
	}	
	
	/**
	 * Record an order from this supplier, and update reputation.
	 * 
	 * @param order  a single order
	 * @param todaysDate  the date
	 */
	public void addOrder(OrderInfo order, int todaysDate){
		myOrders.put(new Integer(order.orderID), order);
		//update reputation
		quantityOrdered.put(name,new Integer(((Integer)quantityOrdered.get(name)).intValue() + order.quantity));
	}

	/**
	 * Updates the reputation with this supplier.
	 * 
	 * @param orders  a vector with the day's orders
	 * @param offers  a vector with the day's offers
	 */
	public void updateReputation(Vector orders, Vector offers){
		//update quantityOffered for reputation
		int qOrdered = ((Integer)quantityOrdered.get(name)).intValue(); //total ever ordered
		int qOffered = ((Integer)quantityOffered.get(name)).intValue(); //total ever offered
		for (int o = 0; o < offers.size(); o++){
			OfferInfo offer = (OfferInfo)offers.get(o);
			if(offer.index != componentIndex || offer.rfq.quantity == 0 || !offer.name.equals(name)) //nothing to do
				continue;
			if(offer.alternative != null){ 
				//there was a partial offer and an earliest-complete offer
				boolean ordered = false;
				for (int p = 0; p < orders.size(); p++){
					OfferInfo order = (OfferInfo)orders.get(p);
					if (order.offerID == offer.offerID || order.offerID == offer.alternative.offerID){
						//quantity offered gets updated by max of ordered and 20% RFQ
						qOffered += Math.max(order.quantity, order.rfq.quantity * 0.2);
						ordered = true;
						break;
					}
				}
				if (!ordered){
					//quantity offered gets updated with max of partial offer and 20% RFQ
					qOffered += Math.max(Math.min(offer.quantity,offer.alternative.quantity), offer.rfq.quantity*0.2);
				}
			}
			else{
				//quantity offered gets updated with max of offer and 20% RFQ
				qOffered += Math.max(offer.quantity, offer.rfq.quantity * 0.2);
			}
		}
		
		double apr;
		if (componentIndex < 4) //cpu
			apr = cpuAPR;
		else
			apr = otherAPR;			
		
		//suppliers are forgiving, but only update each supplier once
		//each supplier produces two components with consecutive indices,
		//so just do this for the odd one
		if ((componentIndex % 2) == 0)
		{
			qOrdered += reputationRecovery;
			qOffered += reputationRecovery;
		}
		quantityOrdered.put(name, new Integer(qOrdered));
		quantityOffered.put(name, new Integer(qOffered));
		reputation = Math.min(apr, (double)qOrdered / (double)qOffered) / apr;
	}	

	/**
	 * Returns the amount by which offers could be increased without the
	 * reputation dropping below 1.
	 *
	 * @return  offer quantity 
	 */
	public double getSpareOffers(){
		if (reputation < 1)
			return 0;
		int qpur = ((Integer)quantityOrdered.get(name)).intValue();
		int qoffers = ((Integer)quantityOffered.get(name)).intValue();

		double apr;
		if (componentIndex < 4) //cpu
			apr = 0.75;
		else
			apr = 0.45;

		double spare = qpur / apr - qoffers;
		if (spare < 0) //shouldn't if reputation is 1
			spare = 0;
		return spare; 		
	}

	/**
	 * Records a delivery from the supplier.
	 * 
	 * @param order  the order that was delivered
	 * @param todaysDate
	 */
	public void addDelivery(OrderInfo order, int todaysDate){
		myOrders.remove(new Integer(order.orderID));
	}
	
	/**
	 * Predicts the price (per component) that will be offered for an RFQ with
	 * the given due date and quantity.  The prediction is based on past
	 * offers from the supplier, and takes into account the quantity requested
	 * in the RFQ will raise the price.
	 * 
	 * @param dueDate  when delivery is requested
	 * @param quantity  the quantity requested
	 * @return predicted cost per component
	 */
	public double calculatePrice(int dueDate, double quantity){ 
		if (dueDate - date < 2)
			return Double.POSITIVE_INFINITY; //can't request that
		
		//of the days for which capacity estimates are available, 
		//choose the one that is either the same as or the first to
		//come after the dueDate, or the last one if none come after
		int estimateDate = 0;
		int numEstimates = fractionCommittedDates.length;
		
		if (numEstimates == 0){ //no information, so guess the base price
			return basePrice;
		}
		
		if (dueDate > fractionCommittedDates[numEstimates - 1])
			estimateDate = numEstimates - 1;
		else{
			while (fractionCommittedDates[estimateDate] < dueDate)
				estimateDate++;
		}
		
		double fraction = fractionCommitted[estimateDate];
		double freeCapacity = capacity * (1 - fraction) * (dueDate - date) - quantity;
		//calculate price using Eq. (10) from the specs
		double price = basePrice * (1 - .5 * freeCapacity / ((dueDate - date) * capacity));
		return price;
	}
	
	/**
	 * Returns an array indicating the expected future deliveries from the
	 * supplier.  The array has a length of 220, and a delivery on day d is
	 * indicated at index d.
	 *  
	 * @return array of deliveries
	 */
	public int[] getExpectedDeliveries(){
		//Note: it is assumed that all deliveries will be on time.
		//It may be possible to estimate how late deliveries will be.
		
		int[] del = new int[221];
					
		Iterator it = myOrders.values().iterator();
		while (it.hasNext()){
			OrderInfo order = (OrderInfo) it.next();
			int dueDate = order.dueDate;
			if (dueDate <= date) //assume late orders will come tomorrow
				dueDate = date + 1;
			del[dueDate] += order.quantity;
		}	 
		
		return del;
	}

}
