import se.sics.isl.transport.Transportable;
import se.sics.tasim.aw.Agent;
import se.sics.tasim.aw.TimeListener;
import se.sics.tasim.aw.Message;
import se.sics.tasim.props.*;
import java.io.*;
import java.util.*;

/**
 * The main agent class.  Each day, messages from the server are received and
 * processed.  The agent's information is updated, the managers perform their
 * tasks, the information is updated again with the results, and then messages
 * are sent to the server.
 */
public class TACAgent extends Agent implements TimeListener {
	private boolean debugMode = false; 
	/**the information received at the beginning of the day*/
	private DailyInfo todaysInfo = new DailyInfo();  
	/**the game parameters sent at the beginning of the game*/
	private GameSettings settings = new GameSettings(); 
	/**stores the agent's state*/
	private AgentInfo agentInfo = new AgentInfo(); 
	/**the messages to be sent by the agent at the end of the day*/
	private DailyActions actions = new DailyActions(); 
	/** where to write any output*/
	private PrintStream textLog = System.out; 
	/** the agent's name*/
	private String agentName = null;
	/** stores manager classes by name to allow reuse*/
	private Hashtable managers; 
	/** the demand manager object*/
	private InterfaceDemandManager demandManager;
	/** the supply manager object*/
	private InterfaceSupplyManager supplyManager;
	
	/**
	 * The constructor for debug mode.
	 * 
	 * @param name  the name the agent ran as (e.g., Dummy3)
	 */
	public TACAgent(String name){ 
		debugMode = true; //no communication with server
		this.agentName = name;

	}
	
	public TACAgent() throws Exception {
	}

	//this method is required to implement TimeListener
	public void nextTimeUnit(int date) {
	}

	//read next line from the file, ignoring lines starting with #
	private String getNextLine(BufferedReader f) throws Exception{
		String s;
		do {s = f.readLine();} while (s != null && s.charAt(0) == '#');
		return s;
	}
	
	
	/**
	 * Return a manager object based on the class name.  If a manager of that 
	 * class already exists, the existing object is returned (this way, a
	 * single object may implement multiple managers. 
	 * @param name  the class name
	 * @return a manager object
	 * @throws Exception
	 */
	private Object getManager(String name) throws Exception{
		Object manager;
		if (managers.get(name) == null){ //create new object of the specified type
			Class managerClass = Class.forName(name);
			manager = managerClass.newInstance();
			managers.put(name, manager);
		}
		else //reuse existing object
			manager = managers.get(name);	
		
		return manager;
	}

	/**
	 * A method of the Agent superclass, called when the agent starts.
	 */
	protected void simulationSetup() {
		addTimeListener(this);
		if (agentName == null)
			agentName = this.getName();
		supplyManager = new SimpleSupplyManager(); 
		demandManager = new SimpleDemandManager();
		
	}

	
 	/**
 	 * A method of the Agent superclass, called when the game ends.
 	 */
	protected void simulationFinished() {
		removeTimeListener(this);
		//for some reason, the agent doesn't seem to exit cleanly without this
		//but this prevents the use of autojoin
		System.exit(0);
	}

 	/**
 	 * A method of the Agent superclass, called with each message received
 	 * from the server, either at the start of the game or the beginning of
 	 * each day.
 	 */
	protected void messageReceived(Message message) {
		Transportable content = message.getContent();

		//BOMBundle, ComponentCatalog, and StartInfo are only received at the beginning of the game
		if (content instanceof BOMBundle) {
				settings.bom = (BOMBundle) content;
		} else if (content instanceof ComponentCatalog) {
			settings.catalog = (ComponentCatalog) content;
		} else if (content instanceof StartInfo) {
			processStartInfo((StartInfo) content);
		}
		
		//MarketReports and PriceReports are received periodically
		else if (content instanceof MarketReport) {
			todaysInfo.marketReport = (MarketReport) content;
		} else if (content instanceof PriceReport) {
			todaysInfo.priceReport = (PriceReport) content;
		}
		
	 	// The rest of these should be received daily
	 	else if (content instanceof InventoryStatus) {
			todaysInfo.inventory = (InventoryStatus) content;
		} else if (content instanceof OfferBundle) {
			todaysInfo.addSupplierOffer((OfferBundle) content, message.getSender());
		} else if (content instanceof RFQBundle) {
			todaysInfo.customerRFQs = (RFQBundle) content;
			todaysInfo.customerName = message.getSender();			 
		} else if (content instanceof OrderBundle) {
			todaysInfo.customerOrders = (OrderBundle) content;
		} else if (content instanceof DeliveryNotice) {
			todaysInfo.addDeliveryNotice((DeliveryNotice) content);
		} else if (content instanceof BankStatus) {
			todaysInfo.bankStatus = (BankStatus) content;
		}	else if (content instanceof SimulationStatus) {
			todaysInfo.date = ((SimulationStatus) content).getCurrentDate();
			//all messages have been received, so start working
			if (todaysInfo.date != settings.numberOfDays)//nothing to do on last day
				try{
					//update agentInfo with today's incoming messages
					agentInfo.morningUpdate(todaysInfo, settings);
					
					startDay();
				} catch (Exception e) {e.printStackTrace(textLog);}
		}		
	}

	/**
	 * The method that performs each day's tasks.
	 * @throws Exception
	 */
	public void startDay() throws Exception{
		long startTime = System.currentTimeMillis();
		
		//textLog.print(todaysInfo.toString());
		
		//clear actions
		actions = new DailyActions();
					
		//perform the basic activities
		//this order seems reasonable, but can be changed
		supplyManager.respondToSupplierOffers(settings, todaysInfo, agentInfo, actions);
		demandManager.produceDeliverBid(settings, todaysInfo, agentInfo, actions);
		supplyManager.sendRFQsToSuppliers(settings, todaysInfo, agentInfo, actions);
		
		long totalTime = System.currentTimeMillis() - startTime;
		if (totalTime > settings.secondsPerDay * 1000) //took too long 
			textLog.println("EXCEEDED DAILY TIME LIMIT");
		else{ //only do this if there is time, if we go over 15s our messages won't be received
			//update agentInfo with today's outgoing messages
			agentInfo.eveningUpdate(actions);
	
			//send all of the messages from today's actions
			if (!debugMode)
				sendMessages();
		}
				
		//this is very verbose
		//textLog.print(actions.toString());
		
		//print message about the agentInfo state
		//textLog.print(agentInfo.getDailySummary());

		//textLog.println("Time: " + totalTime + "ms");

		//when done for the day, clear the day's info
		todaysInfo = new DailyInfo();
		
	}

	/**
	 * Sends all of the messages generated today.
	 */
	private void sendMessages(){
		for(int i = 0; i < actions.messages.size(); i++)
			sendMessage((String) actions.messageRecipients.get(i), (Transportable) actions.messages.get(i));	
		sendMessages(actions.supplyTable);
	}
		
	/**
	 * Handles the information we are given at the start of the game.
	 * @param info starting info (BOM, catalog, etc.)
	 */
	private void processStartInfo(StartInfo info){
		settings.factoryName = info.getAttribute("factory.address");
		double ir = Float.valueOf(info.getAttribute("bank.interestRate")).floatValue();
		settings.debtInterestRate = ir / 100.0 / 220;//this is how the bank actually calculates it
		ir = Float.valueOf(info.getAttribute("bank.depositInterestRate")).floatValue();
		settings.depositInterestRate = ir / 100.0 / 220;//this is how the bank actually calculates it
		settings.factoryCapacity = Integer.valueOf(info.getAttribute("factory.capacity")).intValue();		
		double sc = Float.valueOf(info.getAttribute("factory.storageCost")).floatValue();
		settings.storageCost = Math.pow(1 + sc/100.0, 1/220.0) - 1; //calculated differently from interest
		settings.daysBeforeVoid = Integer.valueOf(info.getAttribute("customer.daysBeforeVoid")).intValue();		
		settings.secondsPerDay = info.getSecondsPerDay();
		settings.numberOfDays = info.getNumberOfDays();
		settings.simulationID = info.getSimulationID();
		settings.agentName = this.getName();
		
		textLog = System.out; //write text logs out to the screen
		
		//textLog.print("Using the following task manager classes:\n");
		//textLog.print("supplyManager: " + supplyManager.getClass().getName() + "\n");
		//textLog.print("demandManager: " + demandManager.getClass().getName() + "\n");
	}
	
 	/**
 	 * A method of the Agent superclass, called when the game ends.
 	 */
	public void simulationStopped(){
	}
	
	/**
	 * Update agent with information received at the start of the day.
	 * @param todaysInfo  all messages received
	 */
	public void morningUpdate(DailyInfo todaysInfo){
		this.todaysInfo = todaysInfo;
		agentInfo.morningUpdate(todaysInfo, settings);
	}

	/**
	 * Update agent with actions performed that day.
	 * @param actions  all outgoing messages
	 */
	public void eveningUpdate(DailyActions actions){
		agentInfo.eveningUpdate(actions);
	}
	
	/**
	 * Sets the game settings.
	 * @param settings
	 */
	public void setGameSettings(GameSettings settings){
		this.settings = settings;
	}
	
}
