;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;;    array1.lisp
;;;
;;;    Characterization of 1-dimensional arrays.
;;;
;;;    This book requires nothing more than the Acl2 boot-strap theory for
;;;    its certification.
;;;    
;;;    Bishop Brock
;;;    Computational Logic, Inc.
;;;    1717 West 6th St., Suite 290
;;;    Austin, Texas 78703
;;;    brock@cli.com
;;;
;;;    Copyright 1994, Computational Logic, Inc.  All Rights Reserved
;;;   
;;;   Modified by Jun Sawada for ACL2 V1-9.  May 1997. 
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(in-package "ACL2")

(deflabel array1
  :doc ":doc-section array1
  
  A book of lemmas that characterize 1-dimensional arrays.~/

  Because many of the functions characterized by this book are non-recursive,
  one should always (DISABLE ARRAY1-FUNCTIONS) after including this book, or
  the lemmas will not be applicable.
  ~/

  The lemmas exported by this book should completely characterize
  1-dimensional arrays for most purposes.  Given the lemmas exported by this
  book, it should not be necessary to ENABLE any of the 1-dimensional array
  functions except under special circumstances.

  This book also includes a macro, DEFARRAY1TYPE, which defines recognizers
  and supporting lemmas for 1-dimensional arrays whose elements are all of a
  fixed type.  See :DOC DEFARRAY1TYPE for complete information.~/")

;;;  Rich and Warren added this because it's used below but not defined.
;;;  We presume that this existed in an earlier version of ACL2.
;;;  Wed Apr 30 18:33:40 CDT 1997.

;(defun keyword-listp (lst)
;  (if (atom lst)
;      t
;    (and (keywordp (car lst))
;         (keyword-listp (cdr lst)))))

;;;****************************************************************************
;;;
;;;    These are general lemmas about ALISTs and ALIST functions.  None of
;;;    these lemmas are exported by this book.  Perhaps someday they will
;;;    appear in an ALIST book.  
;;;
;;;****************************************************************************

;;;  EQLABLE-ALISTP

(local
 (defthm eqlable-alistp-implies-alistp
  (implies
   (eqlable-alistp l)
   (alistp l))
  :rule-classes (:rewrite :forward-chaining)))

;;;  ASSOC

(local
 (defthm assoc-properties
  (implies
   (and (eqlable-alistp l)
	(assoc x l))
   (and (consp (assoc x l))
	(equal (car (assoc x l)) x)))))

(local
 (defthm eqlablep-car-assoc
   (implies
    (and (eqlable-alistp l)
	 (assoc x l))
    (eqlablep (car (assoc x l))))))

;;;  ASSOC-EQ

(local
 (defthm assoc-eq-properties
   (implies
    (and (alistp l)
	 (symbolp x)
	 (assoc-eq x l))
    (and (consp (assoc-eq x l))
	 (equal (car (assoc-eq x l)) x)))))

;;;  BOUNDED-INTEGER-ALISTP

(local
 (defthm bounded-integer-alistp-eqlable-alistp
  (implies
   (bounded-integer-alistp l n)
   (eqlable-alistp l))
  :rule-classes (:rewrite :forward-chaining)))

(local
 (defthm bounded-integer-alistp-car-assoc-properties
   (implies
    (and (bounded-integer-alistp l n)
	 (assoc i l)
	 (not (equal (car (assoc i l)) :header)))
    (and (integerp (car (assoc i l)))
	 (>= (car (assoc i l)) 0)
	 (< (car (assoc i l)) n)))))


;;;****************************************************************************
;;;
;;;    Local array1 events.
;;;
;;;****************************************************************************

;;;  We prove a :FORWARD-CHAINING lemma for ARRAY1P and a couple of other
;;;  LOCAL lemmas, then disable ARRAY1P.  Note that for external consumption
;;;  we provide a :FORWARD-CHAINING lemma written in terms of HEADER,
;;;  DIMENSIONS, MAXIMUM-VALUE, etc.  DON'T MESS WITH THIS ARRANGEMENT, or
;;;  you'll be very frustrated and very sorry!

(local
 (defthm array1p-forward-local
   (implies
    (array1p name l)
    (and
     (symbolp name)
     (alistp l)
     (keyword-value-listp (cdr (assoc-eq :header l)))
     (true-listp
      (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
     (equal
      (length (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
      1)
     (integerp
      (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
     (integerp
      (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
     (< 0
	(car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
     (<= (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
	 (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
     (<= (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l))))
	 *maximum-positive-32-bit-integer*)
     (bounded-integer-alistp
      l
      (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))))
   :rule-classes :forward-chaining))

(local
 (defthm array1p-header-exists
   (implies
    (array1p name l)
    (assoc-eq :header l))))

; ARRAY1P-CONS (in a slightly different format) is part of the
; BOOT-STRAP-THEORY of Acl2 1.5.

(local
 (defthm our-array1p-cons
   (implies
    (and (array1p name l)
	 (integerp n)
	 (>= n 0)
	 (< n (car (dimensions name l))))
    (array1p name (cons (cons n x) l)))))

(local (in-theory (disable array1p)))

;;;  Now, we prove everthing we need to know about COMPRESS11, and then use
;;;  these lemmas to characterize COMPRESS1.

(local
 (defthm eqlable-alistp-compress11
   (implies
    (and (array1p name l)
	 (integerp i)
	 (integerp n)
	 (<= i n))
    (eqlable-alistp (compress11 name l i n default)))))

(local
 (defthm bounded-integer-alistp-compress11
   (implies
    (and (array1p name l)
	 (integerp i)
	 (integerp n)
	 (>= i 0)
	 (<= i n))
    (bounded-integer-alistp (compress11 name l i n default) n))))

(local
 (defthm compress11-assoc-property-0
   (implies
    (and (array1p name l)
	 (integerp i)
	 (integerp n)
	 (<= i n)
	 (integerp j)
	 (assoc j l)
	 (assoc j (compress11 name l i n default)))
    (equal (assoc j (compress11 name l i n default))
	   (assoc j l)))))

(local
 (defthm compress11-assoc-property-1
   (implies
    (and (array1p name l)
	 (not (assoc j (compress11 name l i n default))) ;Free vars!
	 (integerp i)
	 (integerp n)
	 (<= i n)
	 (integerp j)
	 (<= i j)
	 (< j n)
	 (assoc j l))
    (equal (cdr (assoc j l))
	   default))))

(local
 (defthm compress11-assoc-property-2
   (implies
    (and (array1p name l)
	 (integerp i)
	 (integerp n)
	 (<= i n)
	 (integerp j)
	 (not (assoc j l)))
    (not (assoc j (compress11 name l i n default))))))

(local
 (defthm compress1-assoc-property-0
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))) 
	(assoc n l)
	(assoc n (compress1 name l)))
   (equal (cdr (assoc n (compress1 name l)))
	  (cdr (assoc n l))))))

(local
 (defthm compress1-assoc-property-1
   (implies
    (and (array1p name l)
	 (integerp n)
	 (>= n 0)
	 (< n (car (dimensions name l))) 
	 (assoc n l)
	 (not (assoc n (compress1 name l))))
    (equal (cdr (assoc n l))
	   (cadr (assoc-keyword :default (cdr (assoc-eq :header l))))))))

(local
 (defthm compress1-assoc-property-2
   (implies
    (and (array1p name l)
	 (integerp n)
	 (>= n 0)
	 (< n (car (dimensions name l))) 
	 (not (assoc n l)))
    (not (assoc n (compress1 name l))))))

(local
 (defthm header-compress1-crock
   (implies
    (array1p name l)
    (equal (assoc-eq :header (compress1 name l))
	   (assoc-eq :header l)))))


;;;****************************************************************************
;;;
;;;    Exported Events.
;;;
;;;****************************************************************************

(defthm array1p-compress1
  (implies
   (array1p name l)
   (array1p name (compress1 name l)))
  :hints
  (("Goal"
    :in-theory (enable array1p)
    :use array1p-header-exists))
  :doc ":doc-section array1p-compress1
  Rewrite: (ARRAY1P name (COMPRESS1 name l)).
  ~/~/~/")

(defthm header-compress1
  (implies
   (array1p name l)
   (equal (header name (compress1 name l))
	  (header name l)))
  :hints
  (("Goal"
    :in-theory (disable array1p-compress1)
    :use ((:instance array1p-compress1))))
  :doc ":doc-section header-compress1
  Rewrite: (HEADER name (COMPRESS1 name l)) = (HEADER name l).
  ~/~/~/")

(defthm dimensions-compress1
  (implies
   (array1p name l)
   (equal (dimensions name (compress1 name l))
	  (dimensions name l)))
  :hints
  (("Goal"
    :in-theory (disable array1p-compress1)
    :use ((:instance array1p-compress1))))
  :doc ":doc-section dimensions-compress1
  Rewrite: (DIMENSIONS name (COMPRESS1 name l)) = (DIMENSIONS name l).
  ~/~/~/")
  
(defthm maximum-length-compress1
  (implies
   (array1p name l)
   (equal (maximum-length name (compress1 name l))
	  (maximum-length name l)))
  :hints
  (("Goal"
    :in-theory (disable array1p-compress1)
    :use ((:instance array1p-compress1))))
  :doc ":doc-section maximum-length-compress1
  Rewrite: (MAXIMUM-LENGTH name (COMPRESS1 name l)) = 
           (MAXIMUM-LENGTH name l).
  ~/~/~/")

;;;  COMPRESS1 is now fully characterized, so we DISABLE it and start proving
;;;  the interesting theorems.

(local (in-theory (disable compress1)))

(defthm array1p-aset1
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))))
   (array1p name (aset1 name l n val)))
  :doc ":doc-section array1p-aset1
  Rewrite: (ARRAY1P name (ASET1 name l n val)).
  ~/~/~/")

(defthm header-aset1
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))))
   (equal (header name (aset1 name l n val))
	  (header name l)))
  :doc ":doc-section header-aset1
  Rewrite: (HEADER name (ASET1 name l n val)) = (HEADER name l).
  ~/~/~/")

(defthm dimensions-aset1
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))))
   (equal (dimensions name (aset1 name l n val))
	  (dimensions name l)))
  :doc ":doc-section dimensions-aset1
  Rewrite: (DIMENSIONS name (ASET1 name l n val)) = (DIMENSIONS name l).
  ~/~/~/")

(defthm maximum-length-aset1
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))))
   (equal (maximum-length name (aset1 name l n val))
	  (maximum-length name l)))
  :doc ":doc-section maximum-length-aset1
  Rewrite: (MAXIMUM-LENGTH name (ASET1 name l n val)) =
           (MAXIMUM-LENGTH name l).
  ~/~/~/")

(defthm aref1-compress1
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))))
   (equal (aref1 name (compress1 name l) n)
	  (aref1 name l n)))
  :doc ":doc-section aref1-compress1
  Rewrite: (AREF1 name (COMPRESS1 name l) n) = (AREF name l n).
  ~/~/~/")

;;;  These two theorems for the AREF1-ASET1 cases are used to prove a
;;;  combined result, and then exported DISABLEd.

(defthm aref1-aset1-equal
  (implies
   (and (array1p name l)
	(integerp n)
	(>= n 0)
	(< n (car (dimensions name l))))
   (equal (aref1 name (aset1 name l n val) n)
	  val))
  :doc ":doc-section aref1-aset1-equal
  Rewrite: (AREF1 name (ASET1 name l n val) n) = val.~/
  Note that this rule is exported DISABLEd by default in favor of
  AREF1-ASET1.
  ~/~/")

(defthm aref1-aset1-not-equal
  (implies
   (and (array1p name l)
	(integerp n1)
	(>= n1 0)
	(< n1 (car (dimensions name l)))
	(integerp n2)
	(>= n2 0)
	(< n2 (car (dimensions name l)))
	(not (equal n1 n2)))
   (equal (aref1 name (aset1 name l n1 val) n2)
	  (aref1 name l n2)))
  :doc ":doc-section aref1-aset1-not-equal
  Rewrite: (AREF1 name (ASET1 name l n1 val) n2) = (AREF1 name l n2),
           when n1 /= n2.~/
  Note that this rule is exported DISABLEd by default in favor of
  AREF1-ASET1.
  ~/~/")

(defthm aref1-aset1
  (implies
   (and (array1p name l)
	(integerp n1)
	(>= n1 0)
	(< n1 (car (dimensions name l)))
	(integerp n2)
	(>= n2 0)
	(< n2 (car (dimensions name l))))
   (equal (aref1 name (aset1 name l n1 val) n2)
	  (if (equal n1 n2)
	      val
	    (aref1 name l n2))))
  :hints
  (("Goal"
    :in-theory (disable aref1 aset1)))
  :doc ":doc-section aref1-aset1
  Rewrite: (AREF1 name (ASET1 name l n1 val) n2) = 
              (IF (EQUAL n1 n2) val (AREF1 name l n2)).
  ~/
  Note that this lemma forces the decision of the equality of n1 and n2.  If
  this causes problems, then DISABLE this lemma and
  (ENABLE AREF1-ASET1-EQUAL AREF1-ASET1-NOT-EQUAL).~/~/")

(in-theory (disable aref1-aset1-equal aref1-aset1-not-equal))

;;;  The final form of the :FORWARD-CHAINING lemma for ARRAY1P.

(defthm array1p-forward-modular
  (implies
   (array1p name l)
   (and (symbolp name)
	(alistp l)
	(keyword-value-listp (cdr (header name l)))
	(true-listp (dimensions name l))
	(equal (length (dimensions name l)) 1)
	(integerp (car (dimensions name l)))
	(integerp (maximum-length name l))
	(< 0 (car (dimensions name l)))
	(<= (car (dimensions name l)) (maximum-length name l))
	(<= (maximum-length name l) *maximum-positive-32-bit-integer*)
	(bounded-integer-alistp l (car (dimensions name l)))))
  :rule-classes :forward-chaining
  :hints
  (("Goal"
    :in-theory (disable length)))
  :doc ":doc-section array1p-forward-modular
  Forward Chaining: A forward definition of (ARRAY1P name l), in terms of
  HEADER, DIMENSIONS, and MAXIMUM-LENGTH.
  ~/
  Note that Acl2 1.5 also defines a lemma ARRAY1P-FORWARD, but that lemma
  is in terms of the expansions of HEADER, DIMENSIONS, and MAXIMUM-LENGTH.
  ~/
  One should normaly DISABLE ARRAY1P in favor of this :FORWARD-CHAINING rule.
  If allowed to open, ARRAY1P can cause severe performance degradation due to
  its large size and many recursive functions.  This lemma is designed to be
  used with the ARRAY1-FUNCTIONS theory DISABLEd.~/")

(defconst *array1-functions*
  '(array1p aset1 aref1 compress1 header dimensions maximum-length)
  ":doc-section *array1-functions*
  A list of function symbols used to define the ARRAY1-FUNCTIONS theory.
  ~/~/~/")

(deftheory array1-functions *array1-functions*
  :doc ":doc-section array1
  A theory of all functions specific to 1-dimensional arrays.
  ~/
  This theory must be DISABLEd in order for the lemmas exported by the
  \"array1\" book to be applicable.~/~/ :cite *array1-functions*")

(defconst *array1-lemmas*
  '(array1p-forward-modular
    array1p-compress1 header-compress1 dimensions-compress1
    maximum-length-compress1 array1p-aset1 header-aset1 dimensions-aset1
    maximum-length-aset1 aref1-compress1 aref1-aset1)
  ":doc-section *array1-lemmas*
    A list of lemma names used to define the ARRAY1-LEMMAS theory.
    ~/~/~/")
    
(deftheory array1-lemmas *array1-lemmas*
  :doc ":doc-section array1
  A theory of all ENABLEd rules exported by the \"array1\" book.
  ~/
  Note that in order for these rules to be applicable you will first need to
  (DISABLE ARRAY1-FUNCTIONS).  The following rules are found in this theory:
  ~/~/

  :cite *array1-lemmas*
  :cite array1p-forward-modular
  :cite array1p-compress1
  :cite header-compress1
  :cite dimensions-compress1
  :cite maximum-length-compress1
  :cite array1p-aset1
  :cite header-aset1
  :cite dimensions-aset1
  :cite maximum-length-aset1
  :cite aref1-compress1
  :cite aref1-aset1")

(deftheory array1-disabled-lemmas '(aref1-aset1-equal aref1-aset1-not-equal)
    :doc ":doc-section array1

  A theory of all rules exported DISABLEd by the \"array1\" book.~/ 

  Note that in order for these rules to be applicable you will first need to
  (DISABLE ARRAY1-FUNCTIONS).  Look at the :DOC for each lemma for an
  explanation of why the lemma is exported DISABLEd. The following rules are
  found in this theory:~/~/ 

  :cite aref1-aset1-equal
  :cite aref1-aset1-not-equal")


;;;****************************************************************************
;;;
;;;    DEFARRAY1TYPE
;;;
;;;****************************************************************************
	
(defmacro defarray1type
  (recognizer predicate &key
	      size doc
	      (guard-check 'nil)
              (aref1-lemma-rule-classes ':REWRITE)
              (aset1-lemma-rule-classes ':REWRITE))

  ":doc-section array1
  
  Characterize 1-dimensional arrays with a fixed element type.~/

  Example form:

  (DEFARRAY1TYPE INTEGERP-ARRAY1P INTEGERP)

  The above example defines a recognizer, INTEGERP-ARRAYP, for 1-dimensional
  arrays whose elements are all INTEGERP.~/

  General form:

  (DEF1ARRAYTYPE recognizer predicate 
                 &key size doc
                      (guard-check nil)
                      (aref1-lemma-rule-classes ':REWRITE)
                      (aset1-lemma-rule-classes ':REWRITE))

  DEFARRAY1TYPE defines a recognizer for 1-dimensional arrays whose elements
  are all of a single type.  The recognizer argument is a symbol that is used
  as the name of the recognizer.  The predicate argument should be a
  1-argument, unguarded Boolean function that recognizes objects of the
  desired type.  The predicate may either be a symbol (the name of the
  predicate), or a LAMBDA expression.  

  If :SIZE is specified it should be a variable-free term that will evaluate
  to a positive integer.  If specified, then the recognizer will only
  recognize 1-dimensional arrays of the given type and of a fixed size.

  If :DOC is specified it should be a string, and it will be inserted as the
  documentation string in the recognizer.

  If :GUARD-CHECK is specified to be a non-nil value, guards are inserted 
  into the array type predicate definition.  The default value fo GUARD-CHECK 
  is nil.

  DEFARRAY1TYPE defines a recognizer:

  (recognizer NAME L),

  and proves several useful theorems about it including

  (IMPLIES
    (recognizer NAME L)
    (ARRAY1P NAME L)),

  (IMPLIES
    (AND (recognizer NAME L)
         (< N (CAR (DIMENSIONS NAME L)))
         (INTEGERP N)
         (>= N 0))
    (predicate (AREF1 NAME L N))),

  and
  
  (IMPLIES
   (AND (recognizer NAME L)
        (< N (CAR (DIMENSIONS NAME L)))
        (INTEGERP N)
        (>= N 0)
        (predicate VAL))
   (recognizer NAME (ASET1 NAME L N VAL))).

  The first theorem is stored as both :REWRITE and :FORWARD-CHAINING rules.
  The :RULE-CLASSES of the second and third lemmas default to :REWRITE, but
  are selectable by the user by means of the :AREF1-LEMMA-RULE-CLASSES and
  :ASET1-LEMMA-RULE-CLASSSES arguments to DEFARRAY1TYPE (respectively).  If
  using :RULE-CLASSES other than :REWRITE the user should bear in mind the
  documented restrictions on the applicability of :TYPE-PRESCRIPTION and
  :FORWARD-CHAINING rules.

  For fixed-size arrays the above theorems are modified so that the first
  includes the constraint on the array size, and in the second and third the
  term (CAR (DIMENSIONS NAME L)) is replaced by the size.  To see the exact
  forms of the created lemmas one can :TRANS the DEFARRAY1TYPE form (see :DOC
  trans).

  WARNING: The recognizer is just a shell for a call to a recursive
  recoignizer, named <recognizer>-FN.  THE RECURSIVE RECOGNIZER MUST BE
  COMPILED BEFORE YOU TRY TO EXECUTE IT, OR IT WILL PROBABLY CAUSE A STACK
  OVERFLOW.  Also note that the recognizer will be DISABLEd after execution
  of this macro.  The user must insure that the recognizer remains DISABLEd,
  otherwise the above lemmas will never be applied.

  DEFARRAY1TYPE proves the generated lemmas in a minimal, ENCAPSULATEd theory
  that should guarantee that the proofs always succeed.  If one should
  encounter a case where a proof fails (as opposed to a translation or other
  syntax failure), please notify the author.~/"

  (declare (xargs :guard (and (symbolp recognizer)
                              (pseudo-termp predicate)
			      (implies doc (stringp doc)))))

  ;;  A form for the size, and function and lemma names.

  (let
    ((size-form (if size size '(CAR (DIMENSIONS NAME L))))
     (recognizer-fn
      (intern-in-package-of-symbol
       (coerce (packn1 (list recognizer '-fn)) 'string)
       recognizer))
     (recognizer-lemma
      (intern-in-package-of-symbol
       (coerce (packn1 (list recognizer '-array1p)) 'string) 
       recognizer))
     (aref1-lemma
      (intern-in-package-of-symbol
       (coerce (packn1 (list recognizer '-aref1)) 'string) 
       recognizer))
     (aset1-lemma
      (intern-in-package-of-symbol
       (coerce (packn1 (list recognizer '-aset1)) 'string) 
       recognizer)))
    `(ENCAPSULATE ()

       ;;  Set up a theory guaranteed to admit the functions and prove the
       ;;  lemmas.  We assume that the "array1" book has been loaded!
       
       ;; Acl2 1.4 (LOCAL (IN-THEORY (CURRENT-THEORY 'EXIT-BOOT-STRAP-MODE)))

       (LOCAL (IN-THEORY (CURRENT-THEORY 'INVISIBLE-FNS-ALIST)))
       (LOCAL (IN-THEORY (DISABLE ARRAY1-FUNCTIONS)))
       (LOCAL (IN-THEORY (ENABLE ARRAY1-LEMMAS)))
       (LOCAL (IN-THEORY (DISABLE (FORCE)))) ;FORCE causes problems in Acl2
					     ;1.5

       ;;  The recursive recognizer.

       (DEFUN ,recognizer-fn (NAME L I N)
         (DECLARE (XARGS ,@(if guard-check
			       `(:GUARD (AND (ARRAY1P NAME L)
					 (INTEGERP I)
					 (>= I 0)
					 (INTEGERP N)
					 (<= I N)
					 (<= N (CAR (DIMENSIONS NAME L)))))
			       nil)
                         :MEASURE (nfix (- N I))))
         (COND
	   ((not (and (integerp I) (<= 0 I) (integerp n) (<= 0 n))) nil)
	   ((zp (- N I)) T)
	   (T (AND (,predicate (AREF1 NAME L I))
                     (,recognizer-fn NAME L (+ I 1) N)))))

       ;; The recognizer.

       (DEFUN ,recognizer (NAME L)
	 ,@(if guard-check `((DECLARE (XARGS :GUARD T))) NIL)
	 ,@(if doc (list doc) NIL)
         (AND (ARRAY1P NAME L)
	      ,@(if size (list `(EQUAL (CAR (DIMENSIONS NAME L)) ,size)) NIL)
              (,recognizer-fn NAME L 0 ,size-form)))

       ;; The recognizer lemma.

       (DEFTHM ,recognizer-lemma
         (IMPLIES
          (,recognizer NAME L)
          (AND (ARRAY1P NAME L)
	       ,@(if size (list `(EQUAL (CAR (DIMENSIONS NAME L)) ,size))
		   NIL)))
         :RULE-CLASSES (:REWRITE :FORWARD-CHAINING))

       ;; We prove that AREF1 returns objects of the proper type.

       (LOCAL
        (DEFTHM CROCK0
	  (IMPLIES
	   (AND (ARRAY1P NAME L)
		(,recognizer-fn NAME L I N)
		(INTEGERP I)
		(>= I 0)
		(INTEGERP N)
		(<= I N)
		(<= N (CAR (DIMENSIONS NAME L)))
		(INTEGERP J)
		(<= I J)
		(< J N))
	   (,predicate (AREF1 NAME L J)))
	  :RULE-CLASSES NIL))

       (DEFTHM ,aref1-lemma
         (IMPLIES
          (AND (,recognizer NAME L)
               (< N ,size-form)
               (INTEGERP N)
               (>= N 0))
          (,predicate (AREF1 NAME L N)))
         :RULE-CLASSES ,aref1-lemma-rule-classes
         :HINTS
         (("Goal"
           :USE
           ((:INSTANCE CROCK0
                       (NAME NAME) (L L) (I 0) (N (CAR (DIMENSIONS NAME L)))
                       (J N))))))

       ;;  We prove that ASET1 returns arrays of the proper type.

       (LOCAL
        (DEFTHM CROCK1
	  (IMPLIES
	   (AND (ARRAY1P NAME L)
		(,recognizer-fn NAME L I N)
		(INTEGERP I)
		(>= I 0)
		(INTEGERP N)
		(<= I N)
		(<= N (CAR (DIMENSIONS NAME L)))
		(INTEGERP J)
		(>= J 0)
		(< J I))
	   (,recognizer-fn NAME (ASET1 NAME L J VAL) I N))))

       (LOCAL
        (DEFTHM CROCK2
	  (IMPLIES
	   (AND (ARRAY1P NAME L)
		(,recognizer-fn NAME L I N)
		(INTEGERP I)
		(>= I 0)
		(INTEGERP N)
		(<= I N)
		(<= N (CAR (DIMENSIONS NAME L)))
		(INTEGERP J)
		(<= I J)
		(< J N)
		(,predicate VAL))
	   (,recognizer-fn NAME (ASET1 NAME L J VAL) I N))
	  :RULE-CLASSES NIL))

       (DEFTHM ,aset1-lemma
         (IMPLIES
          (AND (,recognizer NAME L)
               (< N ,size-form)
               (INTEGERP N)
               (>= N 0)
               (,predicate VAL))
          (,recognizer NAME (ASET1 NAME L N VAL)))
         :RULE-CLASSES ,aset1-lemma-rule-classes
         :HINTS
         (("Goal"
           :USE
           ((:INSTANCE CROCK2
		       (NAME NAME) (L L) (I 0) (N (CAR (DIMENSIONS NAME L)))
		       (J N))))))

       ;;  We DISABLE the recognizer.

       (IN-THEORY (DISABLE ,recognizer)))))
     
	
       
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;     
; Disable non-recursive definitions of one-dimension array functions ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(in-theory (disable array1p compress1 default dimensions header
		    aref1 aset1 maximum-length))

