;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;;    fm9001.lisp
;;;
;;;    An Acl2 book that defines the FM9001 top-level specification using the
;;;    IHS library.  This book is for illustrative purposes only, and
;;;    contains a few axioms.
;;;
;;;    Bishop Brock
;;;    Computational Logic, Inc.
;;;    1717 West 6th Street, Suite 290
;;;    Austin, Texas 78703
;;;    (512) 322-9951
;;;    brock@cli.com
;;;
;;;    Copyright 1994, Computational Logic, Inc.  All Rights Reserved.
;;;
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(in-package "ACL2")

(include-book "ihs")
(maximum-ihs-theory)

(include-book "array1")

(include-book "ihs-absolute-paths")

; In order to use a include-book command directly, comment out the following 
; two s-expressions, and uncomment the include-book commands following '#',
; and set the path to the ACL2 public/deflist and public/structures books.
; 
;(include-book "/p/src/formal-methods/acl2-v1-9/books/public/deflist")
;(include-book "/p/src/formal-methods/acl2-v1-9/books/public/structures")

(include-book "ihs-absolute-paths")
(include-acl2-book "public/deflist")
(include-acl2-book "public/structures")

(in-theory (disable mv-nth))

;;;****************************************************************************
;;;
;;;  INTRODUCTION
;;;
;;;****************************************************************************

(deflabel fm9001
  :doc ":doc-section fm9001
An Acl2 specification of the FM9001 using the IHS libraries.
~/~/

The FM9001:

   Instruction format:
                                                          { N/A  mode-a rn-a
                                                          {  3     2      4
unused op-code store-cc set-flags mode-b rn-b a-immediate |
  4       4       4        4        2      4        1     {     immediate
                                                          {         9


The A operand is a 10 bit field.  If the high order bit is set, the low order
9 bits are treated as a signed immediate.  Otherwise, the low order six bits
of the A operand are a mode/register pair identical to the B operand
specifier.

Addressing Modes for `a' and `b'.

 00  Register Direct
 01  Register Indirect
 10  Register Indirect with Pre-decrement
 11  Register Indirect with Post-increment

Interpretation of the OP-CODE.

 0000  b <- a             Move
 0001  b <- a + 1         Increment
 0010  b <- a + b + c     Add with carry
 0011  b <- b + a         Add
 0100  b <- 0 - a         Negation
 0101  b <- a - 1         Decrement
 0110  b <- b - a - c     Subtract with borrow
 0111  b <- b - a         Subtract
 1000  b <- a >> 1        Rotate right, shifted through carry
 1001  b <- a >> 1        Arithmetic shift right, top bit copied 
 1010  b <- a >> 1        Logical shift right, top bit zero
 1011  b <- b XOR a       Exclusive or
 1100  b <- b | a         Or
 1101  b <- b & a         And
 1110  b <- ~a            Not
 1111  b <- a             Move

Flags are set conditionally based on the SET-FLAGS field.

 0000  ----
 0001  ---Z
 0010  --N-
 0011  --NZ
 0100  -V--
 0101  -V-Z
 0110  -VN-
 0111  -VNZ
 1000  C---
 1001  C--Z
 1010  C-N-
 1011  C-NZ
 1100  CV--
 1101  CV-Z
 1110  CVN-
 1111  CVNZ

If the condition signified by the STORE-CC field is true, then the ALU output
will be stored in the destination.  Otherwise no storage takes place, but
flags may be updated based on the SET-FLAGS field.

 Condition                           Mnemonic
 ---------                           --------

 0000  (~c)                          Carry clear
 0001  (c)                           Carry set
 0010  (~v)                          Overflow clear
 0011  (v)                           Overflow set
 0100  (~n)                          Plus
 0101  (n)                           Negative
 0110  (~z)                          Not equal
 0111  (z)                           Equal
 1000  (~c & ~z)                     Higher
 1001  (c | z)                       Lower or same
 1010  (n & v | ~n & ~v)             Greater or equal
 1011  (n & ~v | ~n & v)             Less than
 1100  (n & v & ~z | ~n & ~v & ~z)   Greater than
 1101  (z | n & ~v | ~n & v)         Less or equal
 1110  (t)                           True
 1111  (nil)                         False

~/")


;;;****************************************************************************
;;;
;;;  Types
;;;
;;;****************************************************************************

(deflabel fm9001-types
  :doc ":doc-section fm9001
Types used in the FM9001 specification.
~/~/~/")

;  The DEFBYTETYPE macro defines and characterizes subranges of the integers.
;  E.g., (WORD-P i) recognizes 32-bit unsigned integers, and (WORD i) coerces
;  any integer into a 32-bit unsigned integer by clipping the high-order bits.

(defbytetype word 32 :unsigned
  :doc ":doc-section fm9001-types
The standard FM9001 data type: An unsigned, 32-bit word.
~/~/

Although the FM9001 can operate on signed and unsigned data, for simplicity
in the specification all data will be represented as 32-bit, unsigned values.
To convert any integer to a signed 32-bit integer use (SIGNED-WORD i).~/")

(defbytetype signed-word 32 :signed
  :doc ":doc-section fm9001-types
A signed, 32-bit word.
~/~/

The standard data type in the FM9001 specification is an unsigned, 32-bit
word.  Occasionaly we coerce unsigned integers to 32-bit signed integers for
special purposes.  See for example the specification of the alu (:DOC ALU).~/")

(defbytetype immediate-word 9 :signed
  :doc ":doc-section fm9001-types
  The nominal type of FM9001 immediate values.
  ~/~/

  We define this type in order to define the coercion operation
  IMMEDIATE-WORD. For an example of the use of this operation, see
  the definition of FM9001-OPERAND-A.~/")

;  The DEFTUPLE macro defines and characterizes n-tuples of identically typed
;  objects.  This invocation defines a 16-tuple of WORD-P integers which will
;  serve as the FM9001 register file.  The accessor call (READ-REG n regfile)
;  returns the value of register n.  The updater call 
;  (WRITE-REG n regfile value) updates register n with value.
;
;  Note that due to the large size of the register file, the proofs
;  associated with this macro require more than 10 minutes on the SPARC
;  10.  The :UNSAFE T parameter below enters the theorems as axioms.  Remove
;  the :UNSAFE T if you wish to be convinced that they really are
;  theorems.

#|
(deftuple regfile 16
  :guard word-p
  :accessor read-reg
  :updater write-reg
  :type-rule-classes (:rewrite :forward-chaining)
  :unsafe t
  :doc ":doc-section fm9001-types
The FM9001 register file is modeled as a 16-tuple of WORD-P integers.
~/~/~/")
|#

(deflist word-listp (l)
  (declare (xargs :guard t))
  word-p)

(defthm word-listp-update-nth
    (implies (and (word-listp l)
		  (< (nfix n) (len l))
		  (word-p val))
	     (word-listp (update-nth n val l)))
  :hints (("goal" :in-theory (enable len update-nth))))

(defun regfile-p (regs)
  (declare (xargs :guard t))
  (and (word-listp regs) (equal (len regs) 16)))

(defun regfile-index-p (rn)
    (declare (xargs :guard t))
    (and (integerp rn) (<= 0 rn) (< rn 16)))

(defthm regfile-p-true-listp
    (implies (regfile-p regs)
	     (true-listp regs))
  :rule-classes :forward-chaining)

(defthm regfile-index-p-4bit-byte
    (regfile-index-p (rdb (bsp 4 p) word))
  :hints (("goal" :in-theory (enable rdb bsp-size))))

(defun read-reg (rn regs)
  (declare (xargs :guard (and (regfile-p regs)
			      (regfile-index-p rn))))
  (nth rn regs))

(defun write-reg (rn regs result)
  (declare (xargs :guard (and (regfile-p regs)
			      (regfile-index-p rn)
			      (word-p result))))
  (update-nth rn result regs))

(defthm word-p-read-reg
    (implies (and (regfile-p regs)
		  (regfile-index-p rn))
	     (word-p (read-reg rn regs)))
  :rule-classes 
  ((:rewrite)
   (:type-prescription :corollary 
		       (implies (and (regfile-p regs)
				     (regfile-index-p rn))
				(and (integerp (read-reg rn regs))
				     (<= 0 (read-reg rn regs))))
		       :hints (("Goal" :in-theory (e/d (word-p)
						       (WORD-LISTP-NTH)))))
      (:rewrite :corollary 
		(implies (and (regfile-p regs)
			      (regfile-index-p rn))
			 (integerp (read-reg rn regs)))
		:hints (("Goal" :in-theory (e/d (word-p)
						(WORD-LISTP-NTH)))))
       (:rewrite :corollary 
		(implies (and (regfile-p regs)
			      (regfile-index-p rn))
			 (acl2-numberp (read-reg rn regs)))
		:hints (("Goal" :in-theory (e/d (word-p)
						(WORD-LISTP-NTH)))))
     (:linear :corollary 
	(implies (and (regfile-p regs)
		      (regfile-index-p rn))
		 (<= 0 (read-reg rn regs)))
		 :hints (("Goal" :in-theory (e/d (word-p)
						 (WORD-LISTP-NTH)))))))

(defthm word-p-write-reg
    (implies (and (regfile-p regs)
		  (regfile-index-p rn)
		  (word-p result))
	     (regfile-p (write-reg rn regs result)))
  :hints (("Goal" :in-theory (enable len-update-nth))))

(defthm read/write-reg
    (implies (and (regfile-p regs)
		  (regfile-index-p rn1)
		  (regfile-index-p rn2)
		  (word-p value))
	     (equal (read-reg rn2 (write-reg rn1 regs value))
		    (if (equal rn1 rn2) value (read-reg rn2 regs))))
  :hints (("goal" :in-theory (enable NTH-UPDATE-NTH))))

(in-theory (disable read-reg write-reg regfile-p regfile-index-p))


;  The DEFSTRUCTURE macro defines a characterizes structures in a way similar
;  to Common Lisp's DEFSTRUCT, and NQTHM's ADD-SHELL.  This invocation
;  creates a number of functions and lemmas.  See the specification function
;  NEW-FLAGS for examples of creating and accessing this kind of structure.

(defstructure flags
 ":doc-section fm9001-types
   The FM9001 flags register is modeled as 4 independent bits.
   ~/~/~/"
  (c (:assert (bitp c)))
  (v (:assert (bitp v)))
  (n (:assert (bitp n)))
  (z (:assert (bitp z)))
  (:options :guards))


;  The DEFARRAY1TYPE macro characterizes 1-dimensional arrays whose elements
;  are all of a known type.  This invocation creates a recognizer
;  WORD-P-ARRAY1P which recognizes arrays of WORD-P integers.  The macro also
;  generates a number of lemmas concerning these types of arrays.  This array
;  type will be used to define the FM9001 memory.

(defarray1type word-p-array1p word-p
  :guard-check t :doc ":doc-section fm9001-types
Arrays of WORD-P integers, used to define the FM9001 memory.
~/~/~/")

(defconst *mem-size* 1024
  ":doc-section fm9001-types
The maximum size of the FM9001 memory.
~/~/

This constant defines the amount of low-address RAM contained in the
FM9001 memory system.  The remainder of the memory is unimplemented.~/")

(defun mem-p (pair)
  ":doc-section fm9001-types
A recognizer for the FM9001 memory structure.
~/

This simple structure associates a fixed-size, WORD-P-ARRAY1P array with
the name of the array.  This structure is used to represent the FM9001
memory.
~/

The name of the array is an artifact of the implementation of arrays in Acl2.
Although the name has no formal meaning, the efficiency of the implementation
of arrays depends on the uniqueness of all names associated with real arrays.
See :DOC ARRAYS for more information.~/"
  (declare (xargs :guard t))
  (and (consp pair)
       (symbolp (car pair))
       (word-p-array1p (car pair) (cdr pair))
       (>= (car (dimensions (car pair) (cdr pair))) *mem-size*)))

;  The definition of the FM9001 state.  The accesors are FM9001-REGFILE etc.,
;  and the updaters are SET-FM9001-REGFILE etc.

(defstructure fm9001-state
":doc-section fm9001-types The state of the FM9001.
~/~/
The state of the FM9001 consists of the register file REGFILE, the
condition codes FLAGS, and the memory system MEM."
  (regfile (:assert (regfile-p regfile)))
  (flags (:assert (flags-p flags)))
  (mem (:assert (mem-p mem)))
  (:options :guards
	    (:conc-name fm9001-)
	    (:set-conc-name set-fm9001-)))



;;;****************************************************************************
;;;
;;;  Memory System
;;;
;;;****************************************************************************

;  We have previosuly defined the memory types.  Now we define the accessor
;  READ-MEM and updater WRITE-MEM.

(defun read-mem (address mem)
  ":doc-section read-mem
  Read a value from memory.
  ~/~/

  Note that reading an address >= *MEM-SIZE* returns 0.~/
  :cited-by mem-p"
  
  (declare (xargs :guard (and (word-p address)
			      (mem-p mem))))
  (cond
   ((< address *mem-size*) (aref1 (car mem) (cdr mem) address))
   (t 0)))

(defun write-mem (value address mem)
  ":doc-section write-mem
  Write a new value into memory, returning the new memory state.
  ~/~/

  Note that writing an address >= *MEM-SIZE* returns an unaltered memory.~/
  :cited-by mem-p"

  (declare (xargs :guard (and (word-p value)
			      (word-p address)
			      (mem-p mem))))
  (cond
   ((< address *mem-size*) 
    (cons (car mem) (aset1 (car mem) (cdr mem) address value)))
   (t mem)))

;  Guard proofs and type lemmas.

(encapsulate ()

  (verify-guards read-mem)
  (verify-guards write-mem)

  (defthm type-of-read-mem
    (implies
     (and (word-p address)
	  (mem-p mem))
     (word-p (read-mem address mem)))
    :doc ":doc-section read-mem
    Rewrite: (WORD-P (READ-MEM address mem)).
    ~/~/~/")

  (defthm type-of-write-mem
    (implies
     (and (word-p value)
          (word-p address)
          (mem-p mem))
     (mem-p (write-mem value address mem)))
    :doc ":doc-section write-mem
    Rewrite: (MEM-P (WRITE-MEM value address mem))
    ~/~/~/")

  (defthm read/write-mem
    (implies
     (and (word-p value)
          (word-p address1)
	  (< address1 *mem-size*)
          (word-p address2)
	  (< address1 *mem-size*)
          (mem-p mem))
     (equal (read-mem address1 (write-mem value address2 mem))
            (if (equal address1 address2)
                value
              (read-mem address1 mem))))
    :doc ":doc-section read-mem
    Rewrite: (READ-MEM address1 (WRITE-MEM value address2 mem)) =
             (IF (EQUAL address1 address2)
                 value
               (READ-MEM address1 mem)),
             when address1,address2 < *MEM-SIZE*.
    ~/~/~/
    :cited-by write-mem"))

(in-theory (disable mem-p read-mem write-mem))


;;;****************************************************************************
;;;
;;;  Instruction Word and addressing modes.
;;;
;;;****************************************************************************

;  The DEFWORD macro describes the layout of word.  The numbers associated
;  with each field are the byte specification of the location of the field
;  (see :DOC LOGOPS-BYTE-FUNCTIONS). This one is set up so that the accessors
;  for the field are the field names.

(defword inst-word
  ((rn-a 4 0)
   (mode-a 2 4)
   (immediate 9 0)
   (a-immediate 1 9)
   (rn-b 4 10)
   (mode-b 2 14)
   (set-flags 4 16)
   (store-cc 4 20)
   (op-code 4 24))
  :conc-name ||)

(defmacro reg-direct-p   (mode) `(EQUAL ,mode 0))
(defmacro reg-indirect-p (mode) `(EQUAL ,mode 1))
(defmacro pre-dec-p      (mode) `(EQUAL ,mode 2))
(defmacro post-inc-p     (mode) `(EQUAL ,mode 3))


;;;****************************************************************************
;;;
;;;  Interpretation of the SET-FLAGS field.
;;;
;;;****************************************************************************

(defword set-flag-word
  ((z 1 0)
   (n 1 1)
   (v 1 2)
   (c 1 3))
  :conc-name set-)

(defun new-flags (flags set-flags c v word)
  ":doc-section new-flags
Specification of the SET-FLAGS field of the instruction word.
~/~/~/"

  (declare (xargs :guard (and (flags-p flags)
			      (unsigned-byte-p 4 set-flags)
			      (bitp c)
			      (bitp v)
			      (word-p word))))

  (make-flags
   :c (if (= (set-c set-flags) 1) c (flags-c flags))
   :v (if (= (set-v set-flags) 1) v (flags-v flags))
   :n (if (= (set-n set-flags) 1)
	  (if (< (signed-word word) 0) 1 0)
	(flags-c flags))
   :z (if (= (set-c set-flags) 1)
	  (if (= word 0) 1 0)
	(flags-z flags))))

(defthm type-of-new-flags
  (implies
   (and (flags-p flags)
	(unsigned-byte-p 4 set-flags)
	(bitp c)
	(bitp v)
	(word-p word))
   (flags-p (new-flags flags set-flags c v word)))
  :doc ":doc-section new-flags
  Rewrite: (FLAGS-P (NEW-FLAGS flags set-flags c v word)).
  ~/~/~/")

(in-theory (disable new-flags))
 

;;;****************************************************************************
;;;
;;;  Interpretation of the STORE-CC field.
;;;
;;;****************************************************************************

(defun store-result-p (store-cc flags)
  ":doc-section store-result-p
Interpretation of the STORE-CC field of the instruction word.
~/~/~/"

  (declare (xargs :guard (and (unsigned-byte-p 4 store-cc)
			      (flags-p flags))))

  ;;  Since this is a predicate we will reason using Boolean values instead
  ;;  of bits.

  (let*
    ((c (= (flags-c flags) 1))
     (v (= (flags-v flags) 1))
     (n (= (flags-n flags) 1))
     (z (= (flags-z flags) 1))
     (~c (not c))
     (~v (not v))
     (~n (not n))
     (~z (not z)))

    (case store-cc
      (0  ~c)
      (1  c)
      (2  ~v)
      (3  v)
      (4  ~n)
      (5  n)
      (6  ~z)
      (7  z)
      (8  (and ~c ~z))
      (9  (or c z))
      (10 (or (and n v) (and ~n ~v)))
      (11 (or (and n ~v) (and ~n v)))
      (12 (or (and n v ~z) (and ~n ~v ~z)))
      (13 (or z (and n ~v) (and ~n v)))
      (14 t)
      (otherwise nil))))

;  NB!  The system has inferred that STORE-RESULT-P is a predicate.

(in-theory (disable store-result-p))


;;;****************************************************************************
;;;
;;;  The ALU
;;;
;;;****************************************************************************

(defun alu+ (c a b)
  ":doc-section alu+
  Compute a sum (+ c a b), carry out, and overflow bit.
  ~/~/

This function accepts a carry bit c, and two WORD-P integers a and b, and
produces 3 results:  The sum (+ a b c) coerced to a WORD-P integer, a carry
bit and an overflow bit.  Note that the carry out and overflow are determined
by a simple comparison of the exact unsigned or signed sum with the
fixed-size unsigned or signed result.~/"

  (declare (xargs :guard (and (bitp c)
			      (word-p a)
			      (word-p b))))

  (let*
    ((signed-a (signed-word a))
     (signed-b (signed-word b))
     (sum (+ a b c))
     (signed-sum (+ signed-a signed-b c))
     (result (word sum))
     (signed-result (signed-word signed-sum)))

    (mv result				;word
	(if (= result sum) 0 1)		;c
	(if (= signed-result signed-sum) 0 1))))

(defun alu- (c a b)
  ":doc-section alu+
  Compute a difference (- (- b a) c), carry out bit, and overflow bit.
  ~/~/

This function accepts a carry (borrow) bit c, and two WORD-P integers a and
b, and produces 3 results: The difference (- (- b a) c) coerced to a WORD-P
integer, a carry (borrow) bit and an overflow bit.  Note that the carry out and
overflow are determined by a simple comparison of the exact unsigned or
signed difference with the fixed-size unsigned or signed result.~/"

  (declare (xargs :guard (and (bitp c)
			      (word-p a)
			      (word-p b))))

  (let*
    ((signed-a (signed-word a))
     (signed-b (signed-word b))
     (diff (- (- b a) c))
     (signed-diff (- (- signed-b signed-a) c))
     (result (word diff))
     (signed-result (signed-word signed-diff)))

    (mv result				;word
	(if (= result diff) 0 1)		;c
	(if (= signed-result signed-diff) 0 1)))) ;v

(encapsulate ()

  (local (in-theory (enable mv-nth)))

  ;; This crock is necessitated by the equality rewriting heuristics.

  (local
   (defthm crock0
     (implies
      (and (integerp i)
	   (equal (word i) i))
      (word-p i))
     :hints
     (("Goal"
       :use word-p-word))))

  (defthm type-of-alu+
    (implies
     (and (bitp c)
	  (word-p a)
	  (word-p b))
     (and (word-p (mv-nth 0 (alu+ c a b)))
	  (bitp   (mv-nth 1 (alu+ c a b)))
	  (bitp   (mv-nth 2 (alu+ c a b)))))
    :doc ":doc-section alu+
  Rewrite:
  (AND (WORD-P (MV-NTH 0 (ALU+ c a b)))
       (BITP   (MV-NTH 1 (ALU+ c a b)))
       (BITP   (MV-NTH 2 (ALU+ c a b)))).
    ~/~/~/")

  (defthm type-of-alu-
    (implies
     (and (bitp c)
	  (word-p a)
	  (word-p b))
     (and (word-p (mv-nth 0 (alu- c a b)))
	  (bitp   (mv-nth 1 (alu- c a b)))
	  (bitp   (mv-nth 2 (alu- c a b)))))
    :doc ":doc-section alu-
  Rewrite:
  (AND (WORD-P (MV-NTH 0 (ALU- c a b)))
       (BITP   (MV-NTH 1 (ALU- c a b)))
       (BITP   (MV-NTH 2 (ALU- c a b)))).
    ~/~/~/"))

(in-theory (disable alu+ alu-))


(defun alu (c a b op-code)
  ":doc-section alu
The specification of the FM9001 ALU.
~/~/

The ALU computes a new result based on the carry in C, the A and B operands,
and a 4-bit OP-CODE.  The ALU returns 3 values: (word c v), which are the
32-bit result, the carry out bit, and the overflow bit respectively.~/"

  (declare (xargs :guard (and (bitp c)
			      (word-p a)
			      (word-p b)
			      (unsigned-byte-p 4 op-code))))

    (case op-code
      (0 (mv a 0 0))			;MOV
      (1 (alu+ 0 a 1))			;INC
      (2 (alu+ c a b))			;ADDC
      (3 (alu+ 0 a b))			;ADD
      (4 (alu- 0 a 0))			;NEG
      (5 (alu- 0 1 a))			;DEC
      (6 (alu- c a b))			;SUBB
      (7 (alu- 0 a b))			;SUB
      (8 (mv (logior (ash a -1) (ash c 31)) ;ROR
	     (logcar a)
	     0))
      (9 (mv (word (ash (signed-word a) -1)) ;ASR
	     (logcar a)
	     0))
      (10 (mv (ash a -1) (logcar a) 0))	;LSR
      (11 (mv (logxor a b) 0 0))	;XOR
      (12 (mv (logior a b) 0 0))	;OR
      (13 (mv (logand a b) 0 0))	;AND
      (14 (mv (word (lognot a)) 0 0))	;NOT
      (otherwise (mv a 0 0))))		;MOV

;  To prove this we would need several lemmas about logical operations that I
;  don't have at the moment.

(defaxiom type-of-alu
  (implies
   (and (bitp c)
	(word-p a)
	(word-p b)
	(unsigned-byte-p 4 op-code))
   (and (word-p (mv-nth 0 (alu c a b op-code)))
	(bitp   (mv-nth 1 (alu c a b op-code)))
	(bitp   (mv-nth 2 (alu c a b op-code)))))
  :doc ":doc-section alu
  Rewrite:
  (AND (WORD-P (MV-NTH 0 (ALU c a b op-code)))
       (BITP   (MV-NTH 1 (ALU c a b op-code)))
       (BITP   (MV-NTH 2 (ALU c a b op-code))))
  ~/~/~/")

(in-theory (disable alu))
      

;;;****************************************************************************
;;;
;;;  The FM9001
;;;
;;;****************************************************************************

;  We first define all of the parts of the specification, then at the end do
;  the guard proofs and type lemmas.

(defun fm9001-alu-operation
  (regfile flags mem instr operand-a operand-b b-address)
  ":doc-section fm9001-alu-operation
   Compute, and conditionaly store the result.  Return a new state.  
  ~/~/~/"

  (declare (xargs :guard (and (regfile-p regfile)
			      (flags-p flags)
			      (mem-p mem)
			      (word-p instr)
			      (word-p operand-a)
			      (word-p operand-b)
			      (word-p b-address))))
  (let*
    ((mode-b (mode-b instr))
     (rn-b (rn-b instr))
     (op-code (op-code instr))
     (store-cc (store-cc instr))
     (set-flags (set-flags instr))
     (storep (store-result-p store-cc flags)))
	    
    (mv-let
      (result c v)
      (alu (flags-c flags) operand-a operand-b op-code)

      (make-fm9001-state
       :regfile (if (and storep (reg-direct-p mode-b))
		    (write-reg rn-b regfile result)
		  regfile)
       :flags (new-flags flags set-flags c v result)
       :mem (if (and storep (not (reg-direct-p mode-b)))
		(write-mem result b-address mem)
	      mem)))))

(defun fm9001-operand-b (regfile flags mem instr operand-a)
  ":doc-section fm9001-operand-b
  Fetch the B operand, and side-effect the B operand register.  The
  B address is retained for a write to memory.
  ~/~/~/"

  (declare (xargs :guard (and (regfile-p regfile)
			      (flags-p flags)
			      (mem-p mem)
			      (word-p instr)
			      (word-p operand-a))))

  (let*
    ((mode-b (mode-b instr))
     (rn-b (rn-b instr))
     (reg (read-reg rn-b regfile))
     (reg+ (word (+ reg 1)))
     (reg- (word (- reg 1)))
     (b-address (if (pre-dec-p mode-b) reg- reg))

     (operand-b (if (reg-direct-p mode-b)
		    reg
		  (read-mem b-address mem)))

     (regfile (cond
	       ((pre-dec-p mode-b) (write-reg rn-b regfile reg-))
	       ((post-inc-p mode-b) (write-reg rn-b regfile reg+))
	       (t regfile))))

    (fm9001-alu-operation regfile flags mem instr
			  operand-a operand-b b-address)))

(defun fm9001-operand-a (regfile flags mem instr)
  ":doc-section fm9001-operand-a
  Fetch the A operand, and side-effect the A operand register.
  ~/~/~/"

  (declare (xargs :guard (and (regfile-p regfile)
			      (flags-p flags)
			      (mem-p mem)
			      (word-p instr))))
  (let*
    ((a-immediate-p (= (a-immediate instr) 1))
     (mode-a (mode-a instr))
     (rn-a (rn-a instr))
     (reg (read-reg rn-a regfile))
     (reg+ (word (+ reg 1)))
     (reg- (word (- reg 1)))
	       
     (operand-a (cond
		 (a-immediate-p (word
				 (immediate-word (immediate instr))))
		 ((reg-direct-p mode-a) reg)
		 ((pre-dec-p mode-a) (read-mem reg- mem))
		 (t (read-mem reg mem))))

     (regfile (cond
	       (a-immediate-p regfile)
	       ((pre-dec-p mode-a) (write-reg rn-a regfile reg-))
	       ((post-inc-p mode-a) (write-reg rn-a regfile reg+))
	       (t regfile))))

    (fm9001-operand-b regfile flags mem instr operand-a)))

(defun fm9001-step (fm9001-state pc-reg)
  ":doc-section fm9001
  Simulate a single step of the FM9001 with a given PC register.
  ~/~/~/
  :cite fm9001-operand-a
  :cite fm9001-operand-b
  :cite fm9001-alu-operation"

  (declare (xargs :guard (and (fm9001-state-p fm9001-state)
			      (regfile-index-p pc-reg))))

  ;;  Unpack the state

  (let*
    ((regfile (fm9001-regfile fm9001-state))
     (flags   (fm9001-flags   fm9001-state))
     (mem     (fm9001-mem     fm9001-state)))

    ;;  Fetch the instruction and update the PC.

    (let*
      ((pc (read-reg pc-reg regfile))
       (instr (read-mem pc mem))
       (regfile (write-reg pc-reg regfile (word (+ pc 1)))))

      (fm9001-operand-a regfile flags mem instr))))


(defun fm9001-sim (fm9001-state n)
  ":doc-section fm9001
  Simulates N instructions, using register 15 as the PC.
  ~/~/~/
  :cite fm9001-step"

  (declare (xargs :guard (and (fm9001-state-p fm9001-state)
			      (integerp n)
			      (>= n 0))
		  :verify-guards nil))
  (if (zp n)
      fm9001-state
    (fm9001-sim (fm9001-step fm9001-state 15) (- n 1))))



; Guard proofs and type lemmas.

(encapsulate ()

  (local
   (defthm crock0
     (implies
      (integerp i)
      (and (regfile-index-p (rn-a i))
	   (regfile-index-p (rn-b i))))
     :hints
     (("Goal"
       :in-theory (enable regfile-index-p)))))

  (local
   (defthm crock1
     (implies
      (word-p i)
      (integerp i))
     :rule-classes :rewrite))

  (defthm type-of-fm9001-alu-operation
    (implies
     (and (regfile-p regfile)
	  (flags-p flags)
	  (mem-p mem)
	  (word-p instr)
	  (word-p operand-a)
	  (word-p operand-b)
	  (word-p b-address))
     (fm9001-state-p (fm9001-alu-operation regfile flags mem instr
					   operand-a operand-b b-address)))
    :doc ":doc-section fm9001-alu-operation
  Rewrite:
  (FM9001-STATE-P (FM9001-ALU-OPERATION regfile flags mem instr
                                        operand-a operand-b b-address))).
  ~/~/~/")

  (in-theory (disable fm9001-alu-operation))

  (defthm type-of-fm9001-operand-b
    (implies
     (and (regfile-p regfile)
	  (flags-p flags)
	  (mem-p mem)
	  (word-p instr)
	  (word-p operand-a))
     (fm9001-state-p (fm9001-operand-b regfile flags mem instr operand-a)))
    :doc ":doc-section fm9001-operand-b
  Rewrite:
  (FM9001-STATE-P (FM9001-OPERAND-B regfile flags mem instr operand-a)).
  ~/~/~/")

  (in-theory (disable fm9001-operand-b))

  (defthm type-of-fm9001-operand-a
    (implies
     (and (regfile-p regfile)
	  (flags-p flags)
	  (mem-p mem)
	  (word-p instr))
     (fm9001-state-p (fm9001-operand-a regfile flags mem instr)))
    :doc ":doc-section fm9001-operand-a
  Rewrite: (FM9001-STATE-P (FM9001-OPERAND-A regfile flags mem instr)).
  ~/~/~/")

  (in-theory (disable fm9001-operand-a))

  (defthm type-of-fm9001-step
    (implies
     (and (fm9001-state-p fm9001-state)
	  (regfile-index-p pc-reg))
     (fm9001-state-p (fm9001-step fm9001-state pc-reg)))
    :doc ":doc-section fm9001-step
  Rewrite: (FM9001-STATE-P (FM9001-STEP fm9001-state pc-reg)).
  ~/~/~/")

  (in-theory (disable fm9001-step))

  (verify-guards fm9001-sim)

  (defthm type-of-fm9001-sim
    (implies
     (and (fm9001-state-p fm9001-state)
	  (integerp n)
	  (>= n 0))
     (fm9001-state-p (fm9001-sim fm9001-state n)))
    :doc ":doc-section fm9001-sim
  Rewrite: (FM9001-STATE-P (FM9001-SIM fm9001-state n)).
  ~/~/~/")

  (in-theory (disable fm9001-sim)))


;;;****************************************************************************
;;;
;;;  Testing
;;;
;;;****************************************************************************
#|

The book needs to be compiled before you begin testing.  The compliation
bombs, however.

(include-book "fm9001")
(comp 'word-p-array1p-fn)		;;?????????????

Initial flags:

(assign flags (make-flags :c 0 :v 0 :n 0 :z 1))

A regfile: Set R0 and R1 to what you want.

(assign regs '(1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign regs '(2 2 0 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign regs (list (word -1) (word -1) 0 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign regs (list (word -1) 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0))

ADD R0,R1

(assign add01-mem (cons 'mem (compress1
			'mem
			(list '(:header :name mem
					:dimensions (1024)
					:default 0
					:maximum-length 2048)
			      '(0 . #x03ef0400)))))

(regfile-p (@ regs))
(flags-p (@ flags))
(mem-p (@ add01-mem))

(assign fm9001-state (make-fm9001-state
		      :regfile (@ regs)
		      :flags (@ flags)
		      :mem (@ add01-mem)))

(assign new-state (fm9001-step (@ fm9001-state) 15))

The ADD Rn,Rm instruction seems to work.

ADD (R0),(R1)

(assign regs (list 1 2 0 0 0 0 0 0 0 0 0 0 0 0 0 0))	;Add 1,2

(assign add12-mem (cons 'mem (compress1
			'mem
			(list '(:header :name mem
					:dimensions (1024)
					:default 0
					:maximum-length 2048)
			      '(0 . #x03ef4410)
			      '(1 . 2)
			      '(2 . 2)))))

(assign add12-mem (cons 'mem (compress1
			'mem
			(list '(:header :name mem
					:dimensions (1024)
					:default 0
					:maximum-length 2048)
			      '(0 . #x03ef4410)
			      (cons 1 (word -2))
			      '(2 . 2))))) ;-2 + 2
			      
(assign fm9001-state (make-fm9001-state
		      :regfile (@ regs)
		      :flags (@ flags)
		      :mem (@ add12-mem)))

(assign new-state (fm9001-step (@ fm9001-state) 15))

So far so good.			      

|#

