;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;;    ihs.lisp
;;;
;;;    The top-level book in the IHS (Integer Hardware Specification)
;;;    library. 
;;;
;;;    Bishop Brock
;;;    Jun Sawada
;;;    Computational Logic, Inc.
;;;    1717 West 6th Street, Suite 290
;;;    Austin, Texas 78703
;;;    brock@cli.com
;;;
;;;    Copyright 1994,1997 Computational Logic, Inc.  All Rights Reserved
;;;    
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(in-package "ACL2")

(include-book "ihs-absolute-paths")
(include-acl2-book "public/utilities")

(deflabel before-loading-ihs
    :doc "The point before loading the IHS library.")

(INCLUDE-BOOK "ihs-init")
(include-book "ihs-init")
(include-book "ihs-theories")
(include-book "logops-definitions")
(include-book "math-lemmas")
(include-book "quotient-remainder-lemmas")
(include-book "logops-lemmas")

(deflabel ihs
  :doc ":doc-section ihs
  The IHS (Integer Hardware Specification) library.
  ~/~/

  Including the \"ihs\" book includes all of the books of definitions in the
  IHS library. To set up the proper theory, see ihs-theory. ~/")


(defmacro minimal-ihs-theory ()
  ":doc-section ihs
  Set up a minimal theory based on the IHS library.
  ~/~/

  Executing the macro call

  (MINIMAL-IHS-THEORY)

  will set up a minimal theory based on the theories exported by the books in
  the IHS libraries.  This disables all the previously defined definitions
  and lemmas.  We consider the original boot-strap theory of ACL2 contains
  too many recursive definitions and rewriting lemmas, slowing down the 
  performance of the prover.  By presenting only the minimal definitions and
  rules to the prover, we expect we can improve the performance of the prover.

  This is a possibly HAZARDOUS macro, as it disables most of the previously 
  defined recursive definitions and rewriting lemmas, regardless whether
  they are built-in or user defined.  One suggestion is that you should
  include the IHS library and run minimal-ihs-theory as earlier as possible
  in your ACL2 book.  By doing this, we disable all recursive definitions
  of built-in functions, but we don't disable user-defined recursive
  definitions of functions.  

  If you don't want to disable previously introduced definitions and lemmas,
  use DEFAULT-IHS-THEORY. ~/"

  `(PROGN
    (IN-THEORY NIL)
    (IN-THEORY
     (ENABLE
      BASIC-BOOT-STRAP			; From "ihs-theories"
      IHS-MATH				; From "math-lemmas"
      QUOTIENT-REMAINDER-RULES		; From "quotient-remainder-lemmas"
      LOGOPS-LEMMAS-THEORY))))		; From "logops-lemmas"


(defmacro default-ihs-theory ()
  ":doc-section ihs
  Set up the default theory based on the IHS library.
  ~/~/

  Executing the macro call

  (DEFAULT-IHS-THEORY)

  will set up the default theory based on the theories exported by the
  books in the IHS libraries.  This macro disables the definitions and
  rewriting rules in the ground-zero theory. This is because the
  definitions of built-in functions often hinder effective proofs of
  theorems.

  This may cause certain problems. Since all the definitions of built-in
  functions, such as append and len, are disabled, the ACL2 prover
  probably becomes unable to prove almost anything about those
  functions unless their definitions and related lemmas are explicitly
  enabled.  We consider this is a trade-off, and we think it is user's
  benefit to disable them in the default theory.  However, this will
  not indicate that any other theory is worse than the default IHS
  theory.  One possible improvement of the default theory is enabling
  all the definitions of recursive functions in ground-zero, since
  recursive functions are opened up only when the prover's heuristics
  suggest that it is a good idea.

  Generally speaking, there is no perfect set of definitions and
  lemmas for all purposes.  We will present, for now, only the minimal
  theory and the default theory as a good start for the users.  The
  users are recommended to come up with their own set of enabled
  definitions and lemmas that works better for the user's purpose.  We
  believe the definition of default-IHS-theory and minimal-IHS-theory
  gives some hints.

  Warning! This theory includes ACL2 utilities and arithmetic
  books.  The set of the activated rules from the arithmetic books
  may vary if the user specificly loads arithmetic library independently.~/"

  `(PROGN
    (IN-THEORY
     (SET-DIFFERENCE-THEORIES
      (CURRENT-THEORY :HERE)
      (UNION-THEORIES
       (UNIVERSAL-THEORY 'GROUND-ZERO)
       (SET-DIFFERENCE-THEORIES
	(UNIVERSAL-THEORY 'IHS)
	(UNIVERSAL-THEORY 'BEFORE-LOADING-IHS)))))
    (IN-THEORY
     (ENABLE
      BASIC-BOOT-STRAP
      IHS-MATH				; From "math-lemmas"
      QUOTIENT-REMAINDER-RULES		; From "quotient-remainder-lemmas"
      LOGOPS-LEMMAS-THEORY))))		; From "logops-lemmas"


(defmacro conservative-extension-ihs-theory ()
  ":doc-section ihs
  Set up a conservative theory based on the IHS library.
  ~/~/

  Executing the macro call

  (CONSERVATIVE-EXTENSION-IHS-THEORY)

  will conservatively extend the original theory before the IHS
  library is loaded.  This macro does not disable previously
  introduced definitions and lemmas, but adds all the IHS definitions
  and rules to the theory.

  However, conservatively introducing the IHS theory will probably
  make certain proofs more difficult, unless the user knows exactly
  what he or she is doing.  One possible problem is its bad
  performance.  Having too many definitions and rules will slow down
  the ACL2 theorem prover drastically.  If the user thinks the prover
  becomes too slow after loading IHS-theory, he or she should consider
  using the minimal theory or the default theory instead.  The other
  problem is the incompatibility with the previously introduced
  lemmas.  Most likely, the definitions and rules contained in the
  ground-zero theory do not cooperate with IHS definitions and rules
  in many cases.  For example, infinite rewriting loops may well be
  introduced by extending the original theory with IHS
  conservatively. Using the minimal or default theory will probably
  cause less incompatibility problems.

  Generally speaking, there is no perfect set of definitions and
  lemmas for all purposes.  We will present, for now, only a few
  theories as a good start for the users.  The users are recommended
  to come up with their own set of enabled definitions and lemmas that
  works better for the user's purpose.  We believe the definitions of
  default-IHS-theory, minimal-IHS-theory and
  conservative-extend-ihs-theory give some hints.~/"

  `(PROGN
    (IN-THEORY
     (SET-DIFFERENCE-THEORIES (CURRENT-THEORY :HERE)
      (SET-DIFFERENCE-THEORIES  (CURRENT-THEORY 'IHS)
				(CURRENT-THEORY 'BEFORE-LOADING-IHS))))
    (IN-THEORY
     (ENABLE
      IHS-MATH				; From "math-lemmas"
      QUOTIENT-REMAINDER-RULES		; From "quotient-remainder-lemmas"
      LOGOPS-LEMMAS-THEORY))))		; From "logops-lemmas"

(deftheory default-IHS-incompatibles
    '(oddp evenp floor mod rem truncate
      loghead logtail logior lognot logand logand
      logeqv logorc1 logorc2 logandc1 logandc2 logcount
      lognand lognor logxor))

(defmacro maximum-ihs-theory ()
  ":doc-section ihs
  Highly experimental new IHS theory.
  ~/~/

  Executing the macro call

  (MAXIMUM-IHS-THEORY)
  
  THIS IS A HIGHLY EXPERIMENTAL THEORY, AND USER HAS TO BE RESPONSIBLE FOR
  ANY RESULT OF ITS USAGE.

  After struggling many problems in the IHS theories, I decided to make a yet
  another IHS theory.  This one will not disable bootstrap theory by default.
  However, this will disable any built-in rules which may cause problems in
  using IHS.~/"

  `(PROGN
    (IN-THEORY
     (SET-DIFFERENCE-THEORIES (CURRENT-THEORY :HERE)
      (SET-DIFFERENCE-THEORIES  (CURRENT-THEORY 'IHS)
				(CURRENT-THEORY 'BEFORE-LOADING-IHS))))
    (IN-THEORY
     (DISABLE DEFAULT-IHS-INCOMPATIBLES))
    (IN-THEORY
     (ENABLE
      IHS-MATH				; From "math-lemmas"
      QUOTIENT-REMAINDER-RULES		; From "quotient-remainder-lemmas"
      LOGOPS-LEMMAS-THEORY))))		; From "logops-lemmas"


(defmacro ihs-theory (&key theory)
  ":doc-section ihs
  Set up a theory for integer hardware specification.
  ~/

  Example:

  (IHS-THEORY :theory 'MINIMAL)

  Sets up a minimal theory for reasoning about integer specifications of
  hardware. 

  (IHS-THEORY :theory 'CONSERVATIVE)
  
  Adds IHS definitions and rules and extend the original theory
  conservatively.

  (IHS-THEORY :theory 'DEFAULT)

  Sets up the default IHS theory.

  ~/

  General Form:

  (IHS-THEORY &key theory)

  This functions sets up the theory presented to the user.

  If :THEORY 'MINIMAL is specified, the resulting theory will be
  minimal, i.e., the working theory will be BASIC-BOOT-STRAP (see
  :DOC) extended with the theories exported by the included books.
  MINIMAL theory disables definitions and lemmas defined before
  including this book, and may result in an very weak theory setting .
  If :THEORY 'CONSERVATIVE is specified, the original theory is
  extended conservatively.  However the conservative theory will be
  slower than the minimal theory and may result in an undesirable
  behavior of the prover.  If no :THEORY is specified, or :THEORY
  'DEFAULT is specified, the default theory is introduced.  The
  default theory disables definitions and rules of the ground zero
  theory, but will not affect the rules and definitions introduced by
  the user.~/"

  (cond ((equal theory ''minimal)
	 '(minimal-ihs-theory))
	((equal theory ''maximum)
	 '(maximum-ihs-theory))
	((equal theory ''conservative)
	 '(conservative-extension-ihs-theory))
         (t '(default-ihs-theory))))

