;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ISA-def.lisp:
;  This file includes the definitions of our ISA. 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; This ISA implements following exceptions.
;  External Interrupt
;  Memory Protection Violation (For instruction and data accesses)
;  Illegal instruction(Software Trap)
;  
; When an exception happens, following actions are taken by the machine.
;   su <--- 0  (Set Supervisor mode)
;   sr0 <-- The restarting address after exception handling
;   sr1 <-- su
;   pc  <-- Exception Vectors 
;   regs <-- All write to the registers take place before the exception
;            handling takes place.
;   mem <-- All write to the memory by the preceding instructions must
;           complete.  If another error happens during the completion, 
;           the first instruction in the program order is considered to
;           have caused the memory error.
;
; External Interrupt
;  The machine will complete all issued instructions, but does not issue 
;  any new instructions.  Sr0 will point to the next instruction to be 
;  executed after the exception handling.  Exception vector is 0x30
; 
; Fetch Error
;  If the instruction fetch address is not readable, a fetch error will be
;  caused.  Sr0 will point to the instruction address which the processor 
;  failed to access.  Exception Vector is 0x10.  Note that speculative fetch
;  may cause a fake fetch error, but the machine should not go into the
;  exception handling cycle.
; 
; Data Access Error
;  If the processor fails to load or store data on the memory, an data access
;  error occurs and the exception is caused. Sr0 points to the instruction
;  that caused the data access error.  Exception vector is 0x20. 
; 
; Illegal instruction(Software Trap)
;  If the processor tries to execute an undefined instruction, or it tries to
;  execute an privileged instruction from the user mode, Illegal 
;  exception is taken and the processor goes into the exception handling.
;  Sr0 points to the next instruction after the Illegal instruction.
;  Software trap is implemented by an undefined instruction in our machine.
;  The exception vector is 0x0
; 
; Memory protection issue on Supervisor and User mode.
;  Memory protection is effective only in the user mode.  In the supervisor
;  mode, the processor can access to any address.  Memory address translation
;  does not exist in our model.
; 
; Added instructions 
;  In order to control the exception handling, we add a few more instructions.
; RFEH (Return From Exception Handling) privileged instruction
;     su <- 1
;     pc <- Sr0
; MFSR rc (Move From Special Register) Privileged instruction
;     rc <- SR0  
; MTSR rc (Move To Special Register) Privileged instruction
;     SR0 <- rc
; SPM rc,ra+rb (Set page mode) (proposed)
;     The access mode of the page containing ra+rb address is 
;     set to rc.  If rc does not contain 0, 1 or 2, the page mode is 
;     set to no-access.
; 
(in-package "ACL2")

(include-book "utils")
(include-book "basic-def")
(include-book "b-ops-aux")


;; Beginning of the definition of the Instruction-Set Architecture.
(deflabel begin-ISA-def)	     

(deflabel begin-ISA-state-def)

; An ISA state consists of a program counter, register file
; special register file and a memory.
(defstructure ISA-state 
  (pc (:assert (addr-p pc) :rewrite (:rewrite (Integerp pc))))
  (regs (:assert (regs-p regs) :rewrite))
  (sregs (:assert (sregs-p sregs) :rewrite))
  (mem (:assert (mem-p mem) :rewrite))
  (:options :guards  (:conc-name ISA-))) 

; An ISA oracle contains a flag for external interrupt.
(defstructure ISA-oracle
  (exint (:assert (bitp exint) :rewrite))
  (:options :guards))


(deflist ISA-oracle-listp (l)
  (declare (xargs :guard t))
  ISA-oracle-p)

(deflabel end-ISA-state-def)

(defun read-error? (addr mem su)
  (declare (xargs :guard (and (addr-p addr) (mem-p mem) (bitp su))))
  (b-nor su (readable-addr? addr mem)))

(defthm bitp-read-error (bitp (read-error? addr mem su)))

(defun write-error? (addr mem su)
  (declare (xargs :guard (and (addr-p addr) (mem-p mem) (bitp su))))
  (b-nor su (writable-addr? addr mem)))

(defthm bitp-write-error (bitp (write-error? addr mem su)))

(deflabel begin-ISA-step-functions)

(defun superviser-mode? (s)
  (declare (xargs :guard (ISA-state-p s)))
  (sregs-su (ISA-sregs s)))

(defthm bitp-superviser-mode 
    (implies (ISA-state-p s) (bitp (superviser-mode? s))))

;; Definitions of states after jumping to exception handling states.
(defun ISA-fetch-error (s)
  (declare (xargs :guard (ISA-state-p s)))
  (ISA-state #x10
	     (ISA-regs s)
	     (sregs 1 (word (ISA-pc s)) (word (sregs-su (ISA-sregs s))))
	     (ISA-mem s)))

(defun ISA-data-accs-error (s)
  (declare (xargs :guard (ISA-state-p s)))
  (ISA-state #x20
	     (ISA-regs s)
	     (sregs 1 (word (ISA-pc s)) (word (sregs-su (ISA-sregs s))))
	     (ISA-mem s)))

(defun ISA-illegal-inst (s)
  (declare (xargs :guard (ISA-state-p s)))
  (ISA-state #x0
	     (ISA-regs s)
	     (sregs 1 (word (1+ (ISA-pc s))) (word (sregs-su (ISA-sregs s))))
	     (ISA-mem s)))

(defun ISA-external-intr (s)
  (declare (xargs :guard (ISA-state-p s)))
  (ISA-state #x30
	     (ISA-regs s)
	     (sregs 1 (word (ISA-pc s)) (word (sregs-su (ISA-sregs s))))
	     (ISA-mem s)))

;; In the following definitions, rc ra and rb represent the field value of
;; the current instruction, and s represents the current state of the machine.
(defun ISA-add (rc ra rb s)
  (declare (xargs :guard (and (rname-p rc) (rname-p ra) (rname-p rb)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s)))
    (ISA-state (addr (1+ (ISA-pc s)))
	       (write-reg (word (+ (read-reg ra regs) (read-reg rb regs)))
			  rc regs)
	       (ISA-sregs s)
	       (ISA-mem s))))

(defun ISA-mult (rc ra rb s)		  
  (declare (xargs :guard (and (rname-p rc) (rname-p ra) (rname-p rb)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s)))
    (ISA-state (addr (1+ (ISA-pc s)))
	       (write-reg (word (* (read-reg ra regs) (read-reg rb regs)))
			  rc regs)
	       (ISA-sregs s)
	       (ISA-mem s))))

(defun ISA-branch (rc im s)
  (declare (xargs :guard (and (rname-p rc) (immediate-p im)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s))
	(pc (ISA-pc s)))
    (ISA-state (if (equal 0 (read-reg rc regs))
		   (addr (+ (logextu *addr-size* *immediate-size* im) pc))
		   (addr (1+ (ISA-pc s))))
	       (ISA-regs s)
	       (ISA-sregs s)
	       (ISA-mem s))))

(defun ISA-load (rc ra rb s)
  (declare (xargs :guard (and (rname-p rc) (rname-p ra) (rname-p rb)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s)))
    (let ((data-addr (addr (+ (read-reg ra regs) (read-reg rb regs)))))
      (b-if (read-error? data-addr (ISA-mem s) (sregs-su (ISA-sregs s)))
	    (ISA-data-accs-error s)
	    (ISA-state (addr (1+ (ISA-pc s)))
		       (write-reg (read-mem data-addr (ISA-mem s)) rc regs)
		       (ISA-sregs s)
		       (ISA-mem s))))))

(defun ISA-load-im (rc im s)
  (declare (xargs :guard (and (rname-p rc) (immediate-p im)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s)))
    (let ((data-addr (addr im)))
      (b-if (read-error? data-addr (ISA-mem s) (sregs-su (ISA-sregs s)))
	    (ISA-data-accs-error s)
	    (ISA-state (addr (1+ (ISA-pc s)))
		       (write-reg (read-mem data-addr (ISA-mem s)) rc regs)
		       (ISA-sregs s)
		       (ISA-mem s))))))

(defun ISA-store (rc ra rb s)
  (declare (xargs :guard (and (rname-p rc) (rname-p ra) (rname-p rb)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s)))
    (let ((data-addr (addr (+ (read-reg ra regs) (read-reg rb regs)))))
      (b-if (write-error? data-addr (ISA-mem s) (sregs-su (ISA-sregs s)))
	    (ISA-data-accs-error s)
	    (let ((new-pc (addr (1+ (ISA-pc s))))
		  (new-regs (ISA-regs s))
		  (new-sregs (ISA-sregs s)))
	      (let ((new-mem (write-mem (read-reg rc regs)
					data-addr (ISA-mem s))))
		(ISA-state new-pc new-regs new-sregs new-mem)))))))


(defun ISA-store-im (rc im s)
  (declare (xargs :guard (and (rname-p rc) (immediate-p im)
			      (ISA-state-p s))))
  (let ((regs (ISA-regs s)))
    (let ((data-addr (addr im)))
      (b-if (write-error? data-addr (ISA-mem s) (sregs-su (ISA-sregs s)))
	    (ISA-data-accs-error s)
	    (let ((new-pc (addr (1+ (ISA-pc s))))
		  (new-regs (ISA-regs s))
		  (new-sregs (ISA-sregs s)))
	      (let ((new-mem (write-mem (read-reg rc regs)
					data-addr
					(ISA-mem s))))
		(ISA-state new-pc new-regs new-sregs new-mem)))))))

(defun ISA-rfeh (s)
  (declare (xargs :guard (ISA-state-p s)))
  (b-if (superviser-mode? s)
	(ISA-state (addr (sregs-sr0 (ISA-sregs s)))
		   (ISA-regs s)
		   (sregs (logcar (sregs-sr1 (ISA-sregs s)))
			  (sregs-sr0 (ISA-sregs s))
			  (sregs-sr1 (ISA-sregs s)))
		   (ISA-mem s))
	(ISA-illegal-inst s)))
		    
(defun ISA-mfsr (rc ra s)
  (declare (xargs :guard (and (rname-p rc) (rname-p ra) (ISA-state-p s))))
  (cond ((zbp (superviser-mode? s))
	 (ISA-illegal-inst s))
	((or (equal ra 0) (equal ra 1))
	 (ISA-state (addr (1+ (ISA-pc s)))
		    (write-reg (read-sreg ra (ISA-sregs s))
			       rc (ISA-regs s))
		    (ISA-sregs s)
		    (ISA-mem s)))
	(t (ISA-illegal-inst s))))

(defun ISA-mtsr (rc ra s)
  (declare (xargs :guard (and (rname-p rc) (rname-p ra) (ISA-state-p s))))
  (let ((srs (ISA-sregs s))
	(regs (ISA-regs s)))
    (cond ((zbp (superviser-mode? s))
	   (ISA-illegal-inst s))
	  ((or (equal ra 0) (equal ra 1))
	   (ISA-state (addr (1+ (ISA-pc s)))
		      regs
		      (write-sreg (read-reg rc regs) ra srs)
		      (ISA-mem s)))
	  (t (ISA-illegal-inst s)))))

(defun ISA-sync (s)
  (declare (xargs :guard (ISA-state-p s)))
  (ISA-state (addr (1+ (ISA-pc s)))
	     (ISA-regs s)
	     (ISA-sregs s)
	     (ISA-mem s)))

(deflabel end-ISA-step-functions)

(defun ISA-step (s orcl)
  "s represents the current state."
  (declare (xargs :guard (and (ISA-state-p s) (ISA-oracle-p orcl))))
  (b-if (ISA-oracle-exint orcl)
	(ISA-external-intr s)
  ; otherwise      
  (b-if (read-error? (ISA-pc s) (ISA-mem s) (sregs-su (ISA-sregs s)))
	(ISA-fetch-error s)
	(let ((inst (read-mem (ISA-pc s) (ISA-mem s))))
	  (let ((op (opcode inst))
		(rc (rc-field inst))
		(ra (ra-field inst))
		(rb (rb-field inst))
		(im (im-field inst)))
	    (cond ((equal op 0)		; add
		   (ISA-add rc ra rb s))
		  ((equal op 1)		; mult
		   (ISA-mult rc ra rb s))
		  ((equal op 2)		; branch
		   (ISA-branch rc im s))
		  ((equal op 3)		; load
		   (ISA-load rc ra rb s))
		  ((equal op 6)		; load from an immediate address
		   (ISA-load-im rc im s))
		  ((equal op 4)		; store
		   (ISA-store rc ra rb s))
		  ((equal op 7)		; store at an immediate address
		   (ISA-store-im rc im s))
		  ((equal op 5)		; sync
		   (ISA-sync s))
		  ((equal op 8) ; RFEH
		   (ISA-rfeh s))
		  ((equal op 9) ; MFSR
		   (ISA-mfsr rc ra s))
		  ((equal op 10) ; MTSR
		   (ISA-mtsr rc ra s))
		  (t (ISA-illegal-inst s))))))))


(defun ISA-stepn (s oracles n)
  (declare (xargs :guard (and (ISA-state-p s) (integerp n) (>= n 0)
			      (ISA-oracle-listp oracles)
			      (<= n (len oracles)))
		  :verify-guards nil))
  (if (zp n)
      s
      (ISA-stepn (ISA-step s (car oracles)) (cdr oracles) (1- n))))

(verify-guards ISA-stepn)

(defthm ISA-state-p-ISA-fetch-error
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-fetch-error s))))

(defthm ISA-state-p-ISA-data-accs-error
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-data-accs-error s))))

(defthm ISA-state-p-ISA-illegal-inst
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-illegal-inst s))))

(defthm ISA-state-p-ISA-external-intr 
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-external-intr s))))

(defthm ISA-state-p-ISA-add
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-add rc ra rb s))))

(defthm ISA-state-p-ISA-mult
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-mult rc ra rb s))))

(defthm ISA-state-p-ISA-branch
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-branch rc im s))))

(defthm ISA-state-p-ISA-load
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-load rc ra rb s))))

(defthm ISA-state-p-ISA-load-im
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-load-im rc im s))))

(defthm ISA-state-p-ISA-store
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-store rc ra rb s))))

(defthm ISA-state-p-ISA-store-im
    (implies (and (ISA-state-p s) (rname-p rc))
	     (ISA-state-p (ISA-store-im rc im s))))

(defthm ISA-state-p-ISA-rfeh
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-rfeh s))))

(defthm ISA-state-p-ISA-mfsr
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-mfsr rc ra s))))

(defthm ISA-state-p-ISA-mtsr
    (implies (and (ISA-state-p s)
		  (rname-p rc))
	     (ISA-state-p (ISA-mtsr rc ra s))))

(defthm ISA-state-p-ISA-sync
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-sync s))))

(defthm ISA-state-p-ISA-step
    (implies (ISA-state-p s)
	     (Isa-state-p (ISA-step s orcl))))

(in-theory (disable ISA-step))


(defthm ISA-state-p-ISA-stepn
    (implies (ISA-state-p s)
	     (ISA-state-p (ISA-stepn s orcls n))))

(deflabel end-ISA-def)	     

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Definition of No-Self-Modifying Code 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(deflabel begin-ISA-functions)

(defun store-inst-p (inst)
  (declare (xargs :guard (word-p inst)))
  (or (equal (opcode inst) 7) (equal (opcode inst) 4)))

(defun ISA-store-inst-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (store-inst-p (read-mem (ISA-pc s) (ISA-mem s))))
    
(defun ISA-store-addr (s)
  (declare (xargs :guard (ISA-state-p s)))
  (let ((inst (read-mem (ISA-pc s) (ISA-mem s)))
	(regs (ISA-regs s)))
    (cond ((equal (opcode inst) 7)
	   (addr (im-field inst)))
	  ((equal (opcode inst) 4)
	   (addr (+ (read-reg (ra-field inst) regs)
		    (read-reg (rb-field inst) regs))))
	(t 0))))
  
(defun ISA-fetch-error-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (b1p (read-error? (ISA-pc s) (ISA-mem s) (sregs-su (ISA-sregs s)))))
  
(defun ISA-decode-error-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (let ((opcd (opcode (read-mem (ISA-pc s) (ISA-mem s))))
	(ra (ra-field (read-mem (ISA-pc s) (ISA-mem s))))
	(su (sregs-su (ISA-sregs s))))
    (not (or (equal opcd 0)
	     (equal opcd 1)
	     (equal opcd 2)
	     (equal opcd 3)
	     (equal opcd 4)
	     (equal opcd 5)
	     (equal opcd 6)
	     (equal opcd 7)
	     (and (equal opcd 8) (b1p su))
	     (and (equal opcd 9) (b1p su) (or (equal ra 0) (equal ra 1)))
	     (and (equal opcd 10) (b1p su) (or (equal ra 0) (equal ra 1)))))))

(defun ISA-load-access-error-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (let ((inst (read-mem (ISA-pc s) (ISA-mem s)))
	(su (sregs-su (ISA-sregs s)))
	(mem (ISA-mem s))
	(regs (ISA-regs s)))
    (if (equal (opcode inst) 6)
	(b1p (read-error? (addr (im-field inst)) mem su))
	(if (equal (opcode inst) 3)
	    (b1p (read-error? (addr (+ (read-reg (ra-field inst) regs)
				       (read-reg (rb-field inst) regs)))
			      mem su))
	    nil))))

(defun ISA-store-access-error-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (let ((inst (read-mem (ISA-pc s) (ISA-mem s)))
	(su (sregs-su (ISA-sregs s)))
	(mem (ISA-mem s))
	(regs (ISA-regs s)))
    (if (equal (opcode inst) 7)
	(b1p (write-error? (addr (im-field inst)) mem su))
	(if (equal (opcode inst) 4)
	    (b1p (write-error? (addr (+ (read-reg (ra-field inst) regs)
					(read-reg (rb-field inst) regs)))
			       mem su))
	    nil))))

(defun ISA-data-access-error-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (or (ISA-load-access-error-p s) (ISA-store-access-error-p s)))

(defun ISA-excpt-p (s)
  (declare (xargs :guard (ISA-state-p s)))
  (or (ISA-fetch-error-p s) (ISA-decode-error-p s) (ISA-data-access-error-p s)))

(defun ISA-stepn-fetches-from (addr s orcls n)
  (declare (xargs :guard (and (addr-p addr) (ISA-state-p s)
			      (integerp n) (<= 0 n)
			      (ISA-oracle-listp orcls)
			      (<= n (len orcls)))
		  :measure (nfix n)))
  (if (zp n)
      nil
      (or (and (equal addr (ISA-pc s))
	       (not (b1p (ISA-oracle-exint (car orcls)))))
	  (ISA-stepn-fetches-from addr (ISA-step s (car orcls))
				  (cdr orcls) (1- n)))))

(defun ISA-stepn-self-modifies-p (s orcls n)
  (declare (xargs :guard (and (ISA-state-p s) (integerp n) (<= 0 n)
			      (ISA-oracle-listp orcls)
			      (<= n (len orcls)))
		  :measure (nfix n)))
  (if (zp n)
      nil
      (if (and (ISA-store-inst-p s)
	       (not (ISA-excpt-p s))
	       (not (b1p (ISA-oracle-exint (car orcls)))))
	  (or (ISA-stepn-fetches-from (ISA-store-addr s)
				      (ISA-step s (car orcls))
				      (cdr orcls)
				      (1- n))
	      (ISA-stepn-self-modifies-p (ISA-step s (car orcls))
					 (cdr orcls)
					 (1- n)))
	  (ISA-stepn-self-modifies-p (ISA-step s (car orcls))
				     (cdr orcls)
				     (1- n)))))


(defun ISA-stepn-writes-at (addr s orcls n)
  (declare (xargs :guard (and (addr-p addr) (ISA-state-p s)
			      (integerp n) (<= 0 n)
			      (ISA-oracle-listp orcls)
			      (<= n (len orcls)))
		  :measure (nfix n)))
  (if (zp n)
      nil
      (or (and (ISA-store-inst-p s)
	       (not (ISA-excpt-p s))
	       (not (b1p (ISA-oracle-exint (car orcls))))
	       (equal (ISA-store-addr s) addr))
	  (ISA-stepn-writes-at addr (ISA-step s (car orcls))
			       (cdr orcls) (1- n)))))

(deflabel end-ISA-functions)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ISA theories
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(deftheory ISA-def
    (set-difference-theories (function-theory 'end-ISA-def)
			     (function-theory 'begin-ISA-def)))

(deftheory ISA-state-def
    (set-difference-theories (function-theory 'end-ISA-state-def)
			     (function-theory 'begin-ISA-state-def)))
    
(deftheory ISA-step-functions
    (definition-theory
	(set-difference-theories (universal-theory 'end-ISA-step-functions)
				 (universal-theory 'begin-ISA-step-functions))))
(in-theory (disable ISA-step-functions))

(deftheory ISA-functions
    (set-difference-theories (function-theory 'end-ISA-functions)
			     (function-theory 'begin-ISA-functions)))

(deftheory ISA-non-rec-functions
    (non-rec-functions (theory 'ISA-functions) world))

(in-theory (disable ISA-def))
(in-theory (disable ISA-non-rec-functions))

#|
Here is a simple example program for our sequential machine.

Our program calculates the factorial of the number at address #x800 and stores
it at address #x801.

Initial memory setting:

#x0: ST R0,(#x50)
#x1: LD R0,(#x3)
#x2: BZ R0, 0
#x3: 0

#x10: ST R0,(#x50)
#x11: LD R0,(#x13)
#x12: BZ R0, 0
#x13: 0

#x20: ST R0,(#x50)
#x21: LD R0,(#x23)
#x22: BZ R0, 0
#x23: 0

#x30: ST R0,(#x50)
#x31: LD R0,(#x33)
#x32: BZ R0, 0
#x33: 0

#x60: 0
#x61: 1
#x62: 2
#x63: -1

#x70: #x400
#x71: #x800

#x100: LD R15,(#x70) ; program base
#x101: LD R14,(#x71) ; data base
#x102: LD R0, (#x60) ; 0      
#x103: LD R1, (#x61) ; 1
#x104: LD R2, (#x62) ; 2
#x105: LD R3, (#x63) ; -1
#x106: MTSR SR0,R15
#x107: MTSR SR1,R0
#x108: RFEH

Initial memory image:
#x400     LD R5,(R14+R0) ; R5 holds counter
#x401     ADD R6, R0, R1     ; R6 holds factorial. Initially 1.
Loop:
#x402:    Mul R6, R6, R5  ; coutner * fact -> fact
#x403:    ADD R5, R5, R3  ; decrement fact
#x404:    BZ  R5, Exit; if counter is zero, exit
#x405:    BZ  R0, Loop ; always jump to loop
EXIT:
#x406: ST  R6, (R14+R1)
#x407: SYNC
#x408: Trap

#x800: 5
#x801: 0
#x802: 5 ; Offset to Loop
#x803: 9 ; Offset ot Exit

(assign mem-alist '(
; Exception Handler
(#x0 . #x7050) ; ST R0,(#x50)
(#x1 . #x6003) ; LD R0,(#x3)
(#x2 . #x2000) ; BZ R0, 0
(#x3 . 0)
; Exception Handler
(#x10 . #x7050) ; ST R0,(#x50)
(#x11 . #x6013) ; LD R0,(#x13)
(#x12 . #x2000) ; BZ R0, 0
(#x13 . 0)
; Exception Handler
(#x20 . #x7050) ; ST R0,(#x50)
(#x21 . #x6023) ; LD R0,(#x23)
(#x22 . #x2000) ; BZ R0, 0
(#x23 . 0)

; Exception Handler
(#x30 . #x7050) ; ST R0,(#x50)
(#x31 . #x6033) ; LD R0,(#x33)
(#x32 . #x2000) ; BZ R0, 0
(#x33 . 0)

; Kernel Data Section
(#x60 .  0)
(#x61 . 1)
(#x62 . 2)
(#x63 . #xFFFF) ; -1
(#x70 . #x400)
(#x71 . #x800)
; Kernel Dispatching code
(#x100 . #x6F70) ; LD R15,(#x70) ; program base
(#x101 . #x6E71) ;  LD R14,(#x71) ; data base
(#x102 . #x6060) ;  LD R0, (#x60) ; 0      
(#x103 . #x6161) ;  LD R1, (#x61) ; 1
(#x104 . #x6262) ; LD R2, (#x62) ; 2
(#x105 . #x6363) ; LD R3, (#x63) ; -1
(#x106 . #xAF00) ;  MTSR SR0,R15
(#x107 . #xA010) ;  MTSR SR1,R0
(#x108 . #x8000) ; #x103: RFEH
; Program
(#x400 . #x35E0) ;  LD R5,(R14+R0) ; R5 holds counter
(#x401 . #x0601) ;  ADD R6, R0, R1     ; R6 holds factorial. Initially 1.
; Loop:
(#x402 . #x1665) ;  Mul R6, R6, R5  ; coutner * fact -> fact
(#x403 . #x0553) ;  ADD R5, R5, R3  ; decrement fact
(#x404 . #x2502) ;  BZ  R5, Exit; if counter is zero, exit
(#x405 . #x20FD) ;  BZ  R0, Loop ; always jump to loop
; EXIT:
(#x406 . #x46E1) ; ST  R6, (R14+R1)
(#x407 . #x5000) ; SYNC
(#x408 . #xB000) ; Trap

; Data Section
(#x800 . 5)
(#x801 . 0)
(#x802 . 5) ; Offset to Loop
(#x803 . 9) ; Offset ot Exit
))

(assign mem (set-page-mode *read-only* 1 (compress1 'mem *init-mem*)))
(assign mem (set-page-mode *read-write* 2 (@ mem)))
(assign mem (compress1 'mem (load-mem-alist (@ mem-alist) (@ mem))))




How to run the program:
1. Certify and compile all the proof scripts.
   (You may skip this, but the execution will be slow.)
2. Run Acl2.
3. Type command '(ld "ISA-def.lisp")'.
4. Run following assign commands, which defines initial state s.
 (assign regs '(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0)) 
 (assign mem  See Above) 
 (assign sregs (sregs 1 0 0))
 (assign s (ISA-state #x100 (@ regs) (@ sregs) (@ mem)))

5. You can run the ISA machine for one cycle by
   (ISA-step (@ s) (ISA-oracle 1)).
   You can also run the machine for multiple cycles with ISA-stepn.
   For instance, if you want to run the machine 15 cycles, type:
     (assign oracle-list (make-list 15 :initial-element (ISA-oracle 0)))
     (ISA-stepn (@ s) 15).  

6. Following macro may be useful to evaluate and assign an ISA state
and print out only pc and register file, not memory.  The memory is expressed 
as an ACL2 array, and it is tedious to print out.

(defmacro eval-set-print-ISA-regs (s expr)
  `(pprogn (f-put-global ',s ,expr state)
           (mv nil
	     (list (ISA-pc (f-get-global ',s state))
 	           (ISA-regs (f-get-global ',s state))
	           (ISA-sregs (f-get-global ',s state)))
	      state)))

|#
