;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; MA2-def.lisp
; This file contains the micro architectural definition of our
; pipelined machine example.  This version deploys the Tomasulo Algorithm
; with a re-order buffer.  This is a out-of-order multi-issue machine.
; The specification is written using the IHS library.  It also requires
; the book basic-def, which defines the register file and the memory.
;
; The next-state function of the pipelined machine is given as
; MA-step. In order to execute, see below.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(in-package "ACL2")

(include-book "b-ops-aux-def")
(include-book "basic-def")

(deflabel begin-MA2-def)
(defconst *abstract-impl-flag* 'executable)

(defun wrap-local (defs)
  (if (endp defs)
      nil
      (cons `(local ,(car defs)) (wrap-local (cdr defs)))))

(defmacro encapsulate-impl (name stubs defs thms)
  (let ((begin-def-label (pack-intern name "BEGIN-DEF-" name))
	(end-def-label (pack-intern name "END-DEF-" name))
	(def-label (pack-intern name "DEF-" name))
	(begin-theory-label (pack-intern name "BEGIN-THEORY-" name))
	(end-theory-label (pack-intern name "END-THEORY-" name))
	(theory-label (pack-intern name "THEORY-" name)))
    (cond ((equal *abstract-impl-flag* 'abstract)
	   `(encapsulate ,stubs
	    ,@(wrap-local defs)
	    (deflabel ,begin-theory-label)
	    ,@thms
	    (deflabel ,end-theory-label)
	    (deftheory ,theory-label
		(set-difference-theories
		 (universal-theory ',end-theory-label)
		 (universal-theory ',begin-theory-label)))))
	 ((equal *abstract-impl-flag* 'executable)
	  `(encapsulate nil
	    (deflabel ,begin-def-label)
	    ,@defs
	    (deflabel ,end-def-label)
	    (deflabel ,begin-theory-label)
	    ,@thms
	    (deflabel ,end-theory-label)
	    (deftheory ,def-label
		(set-difference-theories
		 (universal-theory ',end-def-label)
		 (universal-theory ',begin-def-label)))
	    (deftheory ,theory-label
		(set-difference-theories
		 (universal-theory ',end-theory-label)
		 (universal-theory ',begin-theory-label)))
	    (in-theory (disable ,def-label))))
	 (t
	  (er hard 'encapsulate-impl "*abstract-impl-flag* must be 'abstract or 'executable")))))



(deflabel begin-MA-state)

(defconst *rob-index-size* 3)
(defconst *rob-size* (expt 2 *rob-index-size*))
(defbytetype rob-index *rob-index-size* :unsigned)

(defthm rob-index-p-forward-unsigned-byte
    (implies (rob-index-p idx) (unsigned-byte-p *rob-index-size* idx))
    :hints (("goal" :in-theory (enable rob-index-p)))
    :rule-classes :forward-chaining)

(defthm rob-index-p-bv-eqv-iff-equal
    (implies (and (rob-index-p idx1) (rob-index-p idx2))
	     (equal (b1p (bv-eqv *rob-index-size* idx1 idx2))
		    (equal idx1 idx2)))
    :hints (("goal" :in-theory (enable rob-index-p))))

#|
THE DEFINITION OF CONTROL VECTOR:

The control bits are defined as follows:
 exunit: indicate the selected execution unit.
          exunit bit 0: integer unit
                 bit 1: multiplier
                 bit 2: load-store unit
                 bit 3: branch?
                 bit 4: no execution unit

 operand: indicates the operand format.
          bit 0: dispatcher outputs RA to port val1 and RB to port val2
          bit 1: dispatcher outputs immediate to port val1
          bit 2: dispatcher outputs RC to port val1
          bit 3: dispatcher outputs special register value to port val1
 br-predict?: branch is taken speculatively?
 ld-st?:  load or store when exunit=3.
                 0: load
                 1: store
 wb?:     Whether instruction needs to write back its result.
 wb-sreg?:    whether results should be written back to a special register.
 sync?:   whether the instruction forces a synchronization.
 rfeh?:   This bit is on for instruction RFEH

|#

;; The definition of the control vector. An instruction is decoded into this
;; vector at the decode stage, and will be passed to the following latches.
;;
;; Access functions to the fields are give by cntlv-<name of field>.
;; For instance, cntlv-exunit is the access function to field exunit.
(defword cntlv-word ((exunit 5 10)
		     (operand 4 6)
		     (br-predict? 1 5)
		     (ld-st? 1 4)
		     (wb? 1 3)
		     (wb-sreg? 1 2)
		     (sync? 1 1)
		     (rfeh? 1 0))
  :conc-name cntlv-)

(defbytetype cntlv 15 :unsigned)

#|
Exception Flags
 raised?:     Exception is raised
 ex-type: Exception type
                 0: Illegal Instruction
                 1: Fetch Error
                 2: Data Access Error
                 3: External Interrupt
|#
(defword excpt-word ((raised? 1 2)
		    (type 2 0))
  :conc-name excpt-)

(defbytetype excpt-flags 3 :unsigned)

;; Micro-architecture specification uses three oracles corrensponding
;; to an external interrupt, a fetch unit response and a data unit
;; response.  1 means the corresponding event takes place.  For
;; instance, an external interrupt is requested when exintr
;; is 1.  An instruction fetch can take place when fetch is 1.  We try to
;; model the asynchronous behavior of memory with the oracle.  If we
;; keep MA-oracle-fetch to be 0, we can flush the pipeline.
(defstructure MA-oracle
  (exintr (:assert (bitp exintr) :rewrite))
  (br-predict (:assert (bitp br-predict) :rewrite))
  (fetch (:assert (bitp fetch) :rewrite))
  (data  (:assert (bitp data) :rewrite))
  (:options :guards))

(deflist MA-oracle-listp (l)
  (declare (xargs :guard t))
  MA-oracle-p)

; The instruction fetch unit.  The program counter is considered to
; belong to a different unit.  The instruction fetch unit stores a
; instruction temporarily, and send it to dispatching queue if
; there is an empty queue entry.
(defstructure IFU
  (valid? (:assert (bitp valid?) :rewrite))
  (excpt (:assert (excpt-flags-p excpt) :rewrite (:rewrite (integerp excpt))))
  (pc (:assert (addr-p pc) :rewrite (:rewrite (integerp pc))
	       (:rewrite (acl2-numberp pc))))
  (inst (:assert (word-p inst) :rewrite (:rewrite (integerp inst))))
  (:options :guards))

(defstructure dispatch-entry
  (valid? (:assert (bitp valid?) :rewrite))
  (excpt (:assert (excpt-flags-p excpt) :rewrite (:rewrite (integerp excpt))))
  (pc (:assert (addr-p pc) :rewrite (:rewrite (integerp pc))
	       (:rewrite (acl2-numberp pc))))
  (cntlv (:assert (cntlv-p cntlv) :rewrite (:rewrite (integerp cntlv))))
  (rcn (:assert (rname-p rcn) :rewrite))
  (ran (:assert (rname-p ran) :rewrite))
  (rbn (:assert (rname-p rbn) :rewrite))
  (im (:assert (immediate-p im) :rewrite))
  (br-target (:assert (addr-p br-target) :rewrite
		      (:rewrite (integerp br-target))))
  (:options :guards (:conc-name DE-)))

; Register Reference Table
(defstructure reg-ref
  (wait? (:assert (bitp wait?) :rewrite))
  (robe (:assert (rob-index-p robe) :rewrite (:rewrite (integerp robe))))
  (:options :guards))

(deflist reg-ref-listp (l)
    (declare (xargs :guard t))
    reg-ref-p)

(defun reg-tbl-p (regs)
  (declare (xargs :guard t))
  (and (reg-ref-listp regs) (equal (len regs) *num-regs*)))

(defun reg-tbl-nth (n tbl)
  (declare (xargs :guard (and (rname-p n) (reg-tbl-p tbl))))
  (nth n tbl))

(defthm reg-ref-p-reg-tbl-nth
    (implies (and (reg-tbl-p tbl) (rname-p n))
	     (reg-ref-p (reg-tbl-nth n tbl))))

(in-theory (disable reg-tbl-p reg-tbl-nth))

(defstructure sreg-tbl
  (sr0 (:assert (reg-ref-p sr0) :rewrite))
  (sr1 (:assert (reg-ref-p sr1) :rewrite))
  (:options :guards))

(defun sreg-tbl-nth (n tbl)
  (declare (xargs :guard (and (rname-p n) (sreg-tbl-p tbl))))
  (b-if (bv-eqv *rname-size* n 0) (sreg-tbl-sr0 tbl)
  (b-if (bv-eqv *rname-size* n 1) (sreg-tbl-sr1 tbl)
	(reg-ref 0 0))))

(defthm sreg-ref-p-sreg-tbl-nth
    (implies (and (sreg-tbl-p tbl) (rname-p n))
	     (reg-ref-p (sreg-tbl-nth n tbl))))
(in-theory (disable sreg-tbl sreg-tbl-nth))

(defstructure DQ
  (DE0 (:assert (dispatch-entry-p DE0) :rewrite))
  (DE1 (:assert (dispatch-entry-p DE1) :rewrite))
  (DE2 (:assert (dispatch-entry-p DE2) :rewrite))
  (DE3 (:assert (dispatch-entry-p DE3) :rewrite))
  (reg-tbl (:assert (reg-tbl-p reg-tbl) :rewrite))
  (sreg-tbl (:assert (sreg-tbl-p sreg-tbl) :rewrite))
  (:options :guards))


(defstructure ROB-entry
  (valid? (:assert (bitp valid?) :rewrite))
  (complete? (:assert (bitp complete?) :rewrite))
  (excpt (:assert (excpt-flags-p excpt) :rewrite (:rewrite (integerp excpt))))
  (wb? (:assert (bitp wb?) :rewrite))
  (wb-sreg? (:assert (bitp wb-sreg?) :rewrite))
  (sync? (:assert (bitp sync?) :rewrite))
  (branch? (:assert (bitp branch?) :rewrite))
  (rfeh? (:assert (bitp rfeh?) :rewrite))
  (br-predict? (:assert (bitp br-predict?) :rewrite))
  (br-actual? (:assert (bitp br-actual?) :rewrite))
  (pc (:assert (addr-p pc) :rewrite (:rewrite (integerp pc))
	       (:rewrite (acl2-numberp pc))))
  (val (:assert (word-p val) :rewrite (:rewrite (integerp val))))
  (dest (:assert (rname-p dest) :rewrite (:rewrite (integerp dest))))
  (:options :guards (:conc-name ROBE-)))

(deflist ROBE-listp (l)
  (declare (xargs :guard t))
  ROB-entry-p)

(defun ROB-entries-p (l)
  (declare (xargs :guard t))
  (and (ROBE-listp l) (equal (len l) *rob-size*)))

(defthm true-listp-rob-entries
    (implies (ROB-entries-p l) (true-listp l)))

(in-theory (disable ROB-entries-p))

; Reorder buffer contains re-order buffer entries and indecies to their head
; and tail. Re-order buffer is a FIFO queue.  Head points to the oldest
; instruction which will be committed next.  Tail points to the newest
; instruction which is dispatched most recently.
; It also contains the external interrupt flag exintr? and a flag
; to indicate whether head is larger  than equal to tail.   This flag is
; set when the tail is wrapped-around to 0, and it is cleared when the head
; is wrapped-around to 0.  If head and tail are equal and flag is off,
; the reorder buffer is empty.  If head and tail are equal and flag is on,
; the reorder buffer is full.
;
;
;        -------	        -------
;      0|       | flg=0       0|  I2   |  flg=1
;      1|       |             1|  I3   |
;      2|-------|	      2|-------|
;      3|   I1  |<- HEAD      3|       |<- TAIL
;      4|   I2  |	      4|       |
;      5|   I3  |             5|       |
;      6|-------|	      6|-------|
;      7|       |<- TAIL      7|  I1   |<- HEAD
;        -------                -------
(defstructure ROB
  (flg (:assert (bitp flg) :rewrite))
  (exintr? (:assert (bitp exintr?) :rewrite))
  (head (:assert (rob-index-p head) :rewrite
		 (:type-prescription (integerp head))))
  (tail (:assert (rob-index-p tail) :rewrite
		 (:type-prescription (integerp tail))))
  (entries (:assert (rob-entries-p entries) :rewrite))
  (:options :guards))

(defthm rob-listp-rob-entries
    (implies (rob-p rob) (robe-listp (rob-entries rob)))
  :hints (("Goal" :in-theory (enable rob-entries-p)
		  :use (defs-rob-assertions))))

(defun nth-ROBE (n ROB)
  (declare (xargs :guard (and (rob-index-p n) (ROB-p ROB))))
  (nth n (ROB-entries ROB)))

(defthm ROB-entry-p-nth-rob-entries
    (implies (and (rob-index-p n) (ROB-entries-p rob-entries))
	     (ROB-entry-p (nth n rob-entries)))
  :hints (("goal" :in-theory (enable rob-index-p rob-entries-p))))

(defthm ROB-entry-p-nth-robe
    (implies (and (rob-index-p n) (ROB-p ROB))
	     (ROB-entry-p (nth-ROBE n ROB))))

(in-theory (disable nth-ROBE))

; The structure of a reservation station.
; Op indicates what type of integer operation should be done with adder.
; When Op=0, IU returns the sum of two operands. When Op=1, IU returns the
; the first value itself.
(defstructure RS
  (valid? (:assert (bitp valid?) :rewrite))
  (op (:assert (bitp op) :rewrite))
  (dest (:assert (rob-index-p dest) :rewrite))
  (ready1? (:assert (bitp ready1?) :rewrite))
  (ready2? (:assert (bitp ready2?) :rewrite))
  (val1 (:assert (word-p val1) :rewrite (:rewrite (integerp val1))
		 (:rewrite (acl2-numberp val1))))
  (val2 (:assert (word-p val2) :rewrite (:rewrite (integerp val2))
		 (:rewrite (acl2-numberp val2))))
  (src1 (:assert (rob-index-p src1) :rewrite))
  (src2 (:assert (rob-index-p src2) :rewrite))
  (:options :guards))

(defstructure integer-unit
  (rs0 (:assert (RS-p rs0) :rewrite))
  (rs1 (:assert (RS-p rs1) :rewrite))
  (:options :guards (:conc-name IU-)))

(defstructure MU-latch
  (valid? (:assert (bitp valid?) :rewrite))
  (dest (:assert (rob-index-p dest) :rewrite))
  (data)
  (:options :guards))

(defstructure mult-unit
  (rs0 (:assert (RS-p rs0) :rewrite))
  (rs1 (:assert (RS-p rs1) :rewrite))
  (lch1 (:assert (MU-latch-p lch1) :rewrite))
  (lch2 (:assert (MU-latch-p lch2) :rewrite))
  (:options :guards (:conc-name MU-)))

; The structure of a reservation station for the memory unit.
; Op indicates how to calculate the access address. 0 means the sum of
; RA and RB registers is the access address.  1 means that the immediate
; value is the access address.
(defstructure LSU-RS
  (valid? (:assert (bitp valid?) :rewrite))
  (op (:assert (bitp op) :rewrite))
  (ld-st? (:assert (bitp ld-st?) :rewrite))
  (dest (:assert (rob-index-p dest) :rewrite))
  (rdy3? (:assert (bitp rdy3?) :rewrite))
  (val3 (:assert (word-p val3) :rewrite (:rewrite (integerp val3))))
  (src3 (:assert (rob-index-p src3) :rewrite))
  (rdy1? (:assert (bitp rdy1?) :rewrite))
  (val1 (:assert (word-p val1) :rewrite (:rewrite (integerp val1))
		 (:rewrite (acl2-numberp val1))))
  (src1 (:assert (rob-index-p src1) :rewrite))
  (rdy2? (:assert (bitp rdy2?) :rewrite))
  (val2 (:assert (word-p val2) :rewrite (:rewrite (integerp val2))
		 (:rewrite (acl2-numberp val2))))
  (src2 (:assert (rob-index-p src2) :rewrite))
  (:options :guards))

(defstructure read-buffer
  (valid? (:assert (bitp valid?) :rewrite))
  (dest (:assert (rob-index-p dest) :rewrite))
  (addr (:assert (addr-p addr) :rewrite (:rewrite (integerp addr))))
  (:options :guards (:conc-name rbuf-)))

(defstructure write-buffer
  (valid? (:assert (bitp valid?) :rewrite))
  (complete? (:assert (bitp complete?) :rewrite))
  (commit? (:assert (bitp commit?) :rewrite))
  (dest (:assert (rob-index-p dest) :rewrite))
  (addr (:assert (addr-p addr) :rewrite (:rewrite (integerp addr))))
  (val (:assert (word-p val) :rewrite (:rewrite (integerp val))))
  (:options :guards (:conc-name wbuf-)))

(defstructure LSU-latch
  (valid? (:assert (bitp valid?) :rewrite))
  (excpt (:assert (excpt-flags-p excpt) :rewrite (:rewrite (integerp excpt))))
  (dest (:assert (rob-index-p dest) :rewrite))
  (val (:assert (word-p val) :rewrite (:rewrite (integerp val))))
  (:options :guards))


; The reservation stataions in memory forms a queue, because the
; memory access order is critical in executing programs.  The head of the
; queue is indicated by the flag rs1-head?.  When rs1-head? is on, RS1 is
; the head of the queue.  Otherwise, RS0 is.
(defstructure load-store-unit
  (rs1-head? (:assert (bitp rs1-head?) :rewrite))
  (rs0 (:assert (LSU-RS-p rs0) :rewrite))
  (rs1 (:assert (LSU-RS-p rs1) :rewrite))
  (rbuf (:assert (read-buffer-p rbuf) :rewrite))
  (wbuf0 (:assert (write-buffer-p wbuf0) :rewrite))
  (wbuf1 (:assert (write-buffer-p wbuf1) :rewrite))
  (lch (:assert (LSU-latch-p lch) :rewrite))
  (:options :guards (:conc-name LSU-)))

(defstructure BU-RS
  (valid? (:assert (bitp valid?) :rewrite))
  (dest (:assert (rob-index-p dest) :rewrite))
  (ready? (:assert (bitp ready?) :rewrite))
  (val (:assert (word-p val) :rewrite (:rewrite (integerp val))))
  (src (:assert (rob-index-p src) :rewrite))
  (:options :guards))


(defstructure branch-unit
  (rs0 (:assert (BU-RS-p rs0) :rewrite))
  (rs1 (:assert (BU-RS-p rs1) :rewrite))
  (:options :guards (:conc-name BU-)))


;; Definition of the pipelined machine states. A machine state contains
;; a program counter, a register file, special register file, instruction
;; fetch unit, dipatch unit, re-order buffer, integer unit, multiplier unit
;; branch unit memory unit and memory.
(defstructure MA-state
  (pc (:assert (addr-p pc) :rewrite (:rewrite (integerp pc))
	       (:rewrite (acl2-numberp pc))))
  (regs (:assert (regs-p regs) :rewrite))
  (sregs (:assert (sregs-p sregs) :rewrite))
  (IFU  (:assert (IFU-p IFU) :rewrite))
  (DQ   (:assert (DQ-p DQ) :rewrite))
  (ROB (:assert (rob-p ROB) :rewrite :type-prescription))
  (IU  (:assert (integer-unit-p IU) :rewrite))
  (MU  (:assert (mult-unit-p MU) :rewrite))
  (BU (:assert (branch-unit-p BU) :rewrite))
  (LSU (:assert (load-store-unit-p LSU) :rewrite))
  (mem (:assert (mem-p mem) :rewrite))
  (:options :guards (:conc-name MA-)))

(deflabel end-MA-state)

(deflabel begin-MA-def)

; This is the branch prediction function.  We can replace the
; implementation with other realistic ones.
(encapsulate-impl branch-predict
   ((branch-predict? (IFU s orcl) t))
((defun branch-predict? (IFU s orcl)
   (declare (xargs :guard (and (IFU-p IFU) (MA-state-p s) (MA-oracle-p orcl))))
   (MA-oracle-br-predict orcl)))

((defthm bitp-branch-predict?
     (implies (MA-oracle-p orcl) (bitp (branch-predict? IFU s orcl))))))

; IFU-branch-predict? is set if the decode unit finds a branch instruction
; in IFU and it predicts the branch is taken.  IFU-branch-target should
; post the target address when IFU-branch-predict? is set.
(defun IFU-branch-predict? (IFU s orcl)
  (declare (xargs :guard (and (IFU-p IFU) (MA-state-p s) (MA-oracle-p orcl))))
  (bs-and (IFU-valid? IFU)
	  (b-not (excpt-raised? (IFU-excpt IFU)))
	  (bv-eqv *opcode-size* (opcode (IFU-inst IFU)) 2)
	  (branch-predict? IFU s orcl)))

(defthm bitp-IFU-branch-predict (bitp (IFU-branch-predict? IFU s orcl)))

(defun IFU-branch-target (IFU)
  (declare (xargs :guard (IFU-p IFU)))
  (addr (+ (IFU-pc IFU)
	   (logextu *addr-size* *immediate-size* (im-field (IFU-inst IFU))))))

(defthm addr-p-IFU-branch-target
    (implies (IFU-p IFU) (addr-p (IFU-branch-target IFU))))
(in-theory (disable IFU-branch-target))


(defun IFU-fetch-prohibited? (pc mem su)
  (declare (xargs :guard (and (addr-p pc) (mem-p mem) (bitp su))))
  (b-nor (readable-addr? pc mem) su))

#|

THE DEFINITION OF CONTROL VECTOR:

The control bits are defined as follows:
 exunit: indicate the selected execution unit.
          exunit bit 0: adder
                 bit 1: multiplier
                 bit 2: load-store unit
                 bit 3: branch?
                 bit 4: no execution unit
 operand: indicates the operand format.
          bit 0: dispatcher outputs RA to port val1 and RB to port val2
          bit 1: dispatcher outputs immediate to port val1
          bit 2: dispatcher outputs RC to port val1
          bit 3: dispatcher outputs special register value to port val1
 branch-predict?: branch is taken speculatively?
 ld-st?:  load or store when exunit=3.
                 0: load
                 1: store
 wb?:     Whether instruction needs to write back its result.
 wb-sreg?:    whether results should be written back to a special register.
 sync?:   whether the instruction forces a synchronization.
 rfeh?:   This bit is on for instruction RFEH


We define six opcodes:

 ADD 0		Addition
 MUL 1		Multiplication
 BR  2		Conditional Branch
 LD  3		Load Memory
 ST  4		Store Memory
 SYNC 5		Synchronize
 LD-IM 6        Load from an immediate address
 ST-IM 7        Store at an immediate address
 RFEH  8        Return from Exception Handling (privileged)
 MFSR  9        Move From a Special Register (privileged)
 MTSR  10       Move To a Special Register (privileged)
|#


(defun decode (opcd branch-predict?)
  (declare (xargs :guard (and (opcd-p opcd) (bitp branch-predict?))
		  :guard-hints (("Goal" :in-theory (enable opcd-p)))))
  (logcons (bv-eqv *opcode-size* opcd 8) ; rfeh?
  (logcons (b-ior (bv-eqv *opcode-size* opcd 5)
		  (bv-eqv *opcode-size* opcd 8)) ; sync?
  (logcons (bv-eqv *opcode-size* opcd 10) ; wb-sreg?
  (logcons (bs-ior (bv-eqv *opcode-size* opcd 0)
		   (bv-eqv *opcode-size* opcd 1)
		   (bv-eqv *opcode-size* opcd 3)
		   (bv-eqv *opcode-size* opcd 6)
		   (bv-eqv *opcode-size* opcd 9)
		   (bv-eqv *opcode-size* opcd 10)) ; wb?
  (logcons (b-ior (bv-eqv *opcode-size* opcd 4)
		  (bv-eqv *opcode-size* opcd 7)) ; ld-st?
  (logcons branch-predict? ; branch-predict?
  (logcons (bs-ior (bv-eqv *opcode-size* opcd 0)
		   (bv-eqv *opcode-size* opcd 1)
		   (bv-eqv *opcode-size* opcd 3)
		   (bv-eqv *opcode-size* opcd 4)) ; operand:0
  (logcons (b-ior (bv-eqv *opcode-size* opcd 6)
		  (bv-eqv *opcode-size* opcd 7)) ; operand:1
  (logcons (b-ior (bv-eqv *opcode-size* opcd 2)
		  (bv-eqv *opcode-size* opcd 10)) ; operand:2
  (logcons (bv-eqv *opcode-size* opcd 9) ;operand:3
  (logcons (bs-ior (bv-eqv *opcode-size* opcd 0)
		   (bv-eqv *opcode-size* opcd 9)
		   (bv-eqv *opcode-size* opcd 10)) ; exunit:0
  (logcons (bv-eqv *opcode-size* opcd 1) ; exunit:1
  (logcons (bs-ior (bv-eqv *opcode-size* opcd 3)
		   (bv-eqv *opcode-size* opcd 4)
		   (bv-eqv *opcode-size* opcd 6)
		   (bv-eqv *opcode-size* opcd 7))
  (logcons (bv-eqv *opcode-size* opcd 2)
	   (b-ior (bv-eqv *opcode-size* opcd 5)
		  (bv-eqv *opcode-size* opcd 8))))))))))))))))) ; exunit:2

(defthm cntlv-p-decode
    (implies (and (opcd-p opcd) (bitp flg)) (cntlv-p (decode opcd flg)))
  :hints (("Goal" :in-theory (enable cntlv-p unsigned-byte-p*))))

(in-theory (disable decode))

(defun decode-illegal-inst? (opcd su ra)
  (declare (xargs :guard (and (opcd-p opcd) (bitp su) (rname-p ra))
		  :guard-hints (("Goal" :in-theory (enable opcd-p)))))
  (bs-and (b-not (bv-eqv *opcode-size* opcd 0))
	  (b-not (bv-eqv *opcode-size* opcd 1))
	  (b-not (bv-eqv *opcode-size* opcd 2))
	  (b-not (bv-eqv *opcode-size* opcd 3))
	  (b-not (bv-eqv *opcode-size* opcd 4))
	  (b-not (bv-eqv *opcode-size* opcd 5))
	  (b-not (bv-eqv *opcode-size* opcd 6))
	  (b-not (bv-eqv *opcode-size* opcd 7))
	  (b-not (b-and (bv-eqv *opcode-size* opcd 8) su))
	  (b-not (bs-and (bv-eqv *opcode-size* opcd 9)
			 su
			 (b-ior (bv-eqv *rname-size* ra 0)
				(bv-eqv *rname-size* ra 1))))
	  (b-not (bs-and (bv-eqv *opcode-size* opcd 10)
			 su
			 (b-ior (bv-eqv *rname-size* ra 0)
				(bv-eqv *rname-size* ra 1))))))

(defthm bitp-decode-illegal-inst (bitp (decode-illegal-inst? opcd su ra)))

(defun decode-output (IFU s orcl)
  (declare (xargs :guard (and (IFU-p IFU) (MA-state-p s) (MA-oracle-p orcl))
		  :verify-guards nil))
  (dispatch-entry (IFU-valid? IFU)
		  (b-if (excpt-raised? (IFU-excpt IFU))
			(IFU-excpt IFU)
			(b-if (decode-illegal-inst?
			       (opcode (IFU-inst IFU)) (sregs-su (MA-sregs s))
			       (ra-field (IFU-inst IFU)))
			      #b100 0))
		  (IFU-pc IFU)
		  (decode (opcode (IFU-inst IFU))
			  (IFU-branch-predict? IFU s orcl))
		  (rc-field (IFU-inst IFU))
		  (ra-field (IFU-inst IFU))
		  (rb-field (IFU-inst IFU))
		  (im-field (IFU-inst IFU))
		  (IFU-branch-target IFU)))

(verify-guards decode-output
    :hints (("Goal" :in-theory (enable opcd-p))))

(defthm dispatch-entry-p-decode-output
    (implies (and (IFU-p IFU) (MA-state-p s) (MA-oracle-p orcl))
	     (dispatch-entry-p (decode-output IFU s orcl))))

(in-theory (disable decode-output))


(defun DQ-out-dest-reg (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (let ((DE0 (DQ-DE0 DQ)))
    (b-if (cntlv-wb-sreg? (DE-cntlv DE0))
	  (DE-ran DE0)
	  (DE-rcn DE0))))

(defun DQ-out-ready1? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (let ((DE0 (DQ-DE0 DQ))
	(cntlv (DE-cntlv (DQ-DE0 DQ))))
    (b-if (logbit 0 (cntlv-operand cntlv))
	  (b-not (reg-ref-wait? (reg-tbl-nth (DE-ran DE0) (DQ-reg-tbl DQ))))
    (b-if (logbit 1 (cntlv-operand cntlv)) 1
    (b-if (logbit 2 (cntlv-operand cntlv))
	  (b-not (reg-ref-wait? (reg-tbl-nth (DE-rcn DE0) (DQ-reg-tbl DQ))))
    (b-if (logbit 3 (cntlv-operand cntlv))
	  (b-not (reg-ref-wait?
		  (sreg-tbl-nth (DE-ran DE0) (DQ-sreg-tbl DQ))))
	  0))))))

(defthm bitp-DQ-out-ready1
    (implies (DQ-p DQ) (bitp (DQ-out-ready1? DQ))))

(in-theory (disable DQ-out-ready1?))

(defun DQ-out-robe1 (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (let ((DE0 (DQ-DE0 DQ)) (cntlv (DE-cntlv (DQ-DE0 DQ))))
    (b-if (logbit 0 (cntlv-operand cntlv))
	  (reg-ref-robe (reg-tbl-nth (DE-ran DE0) (DQ-reg-tbl DQ)))
    (b-if (logbit 2 (cntlv-operand cntlv))
	  (reg-ref-robe (reg-tbl-nth (DE-rcn DE0) (DQ-reg-tbl DQ)))
    (b-if (logbit 3 (cntlv-operand cntlv))
	  (reg-ref-robe (sreg-tbl-nth (DE-ran DE0) (DQ-sreg-tbl DQ)))
	  0)))))

(defthm rob-index-p-DQ-out-robe1
    (implies (DQ-p DQ) (rob-index-p (DQ-out-robe1 DQ)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (DQ-p DQ) (integerp (DQ-out-robe1 DQ))))))

(in-theory (disable DQ-out-robe1))

; DQ-read-val1 reads the value of the first operand from the corresponding
; dispatch queue entry or register file.
(defun DQ-read-val1 (DQ s)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s))))
  (let ((regs (MA-regs s))
	(sregs (MA-sregs s))
	(DE0 (DQ-DE0 DQ))
	(cntlv (DE-cntlv (DQ-DE0 DQ))))
    (b-if (logbit 0 (cntlv-operand cntlv)) (read-reg (DE-ran DE0) regs)
    (b-if (logbit 1 (cntlv-operand cntlv)) (word (DE-im DE0))
    (b-if (logbit 2 (cntlv-operand cntlv)) (read-reg (DE-rcn DE0) regs)
    (b-if (logbit 3 (cntlv-operand cntlv)) (read-sreg (DE-ran DE0) sregs)
	  0))))))

(defthm word-p-DQ-read-val1
    (implies (and (DQ-p DQ) (MA-state-p s))
	     (word-p (DQ-read-val1 DQ s))))
(in-theory (disable DQ-read-val1))

(defun DQ-out-ready2? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (b-not (reg-ref-wait? (reg-tbl-nth (DE-rbn (DQ-DE0 DQ))
					(DQ-reg-tbl DQ)))))

(defthm bitp-DQ-out-ready2
    (implies (DQ-p DQ) (bitp (DQ-out-ready2? DQ))))

(in-theory (disable DQ-out-ready2?))


(defun DQ-out-reg2 (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (DE-rbn (DQ-DE0 DQ)))

(defthm rname-p-DQ-out-reg2
    (implies (DQ-p DQ) (rname-p (DQ-out-reg2 DQ))))

(in-theory (disable DQ-out-reg2))

(defun DQ-out-robe2 (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (reg-ref-robe (reg-tbl-nth (DE-rbn (DQ-DE0 DQ)) (DQ-reg-tbl DQ))))

(defthm rob-index-p-DQ-out-robe2
    (implies (DQ-p DQ) (rob-index-p (DQ-out-robe2 DQ)))
    :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (DQ-p DQ) (integerp (DQ-out-robe2 DQ))))))

(in-theory (disable DQ-out-robe2))


(defun DQ-out-ready3? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (b-not (reg-ref-wait? (reg-tbl-nth (DE-rcn (DQ-DE0 DQ))
					(DQ-reg-tbl DQ)))))

(defthm bitp-DQ-out-ready3
    (implies (DQ-p DQ) (bitp (DQ-out-ready3? DQ))))

(in-theory (disable DQ-out-ready3?))


(defun DQ-out-reg3 (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (DE-rcn (DQ-DE0 DQ)))

(defthm rname-p-DQ-out-reg3
    (implies (DQ-p DQ) (rname-p (DQ-out-reg3 DQ))))

(in-theory (disable DQ-out-reg3))


(defun DQ-out-robe3 (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (reg-ref-robe (reg-tbl-nth (DE-rcn (DQ-DE0 DQ)) (DQ-reg-tbl DQ))))

(defthm rob-index-p-DQ-out-robe3
    (implies (DQ-p DQ) (rob-index-p (DQ-out-robe3 DQ)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (DQ-p DQ) (integerp (DQ-out-robe3 DQ))))))

(in-theory (disable DQ-out-robe3))


; DQ-full? is set when dispatch queue is full.  The fetching is stalled
; until DQ has an avaiable slot.
(defun DQ-full? (DQ)
  (declare (xargs :guard (and (DQ-p DQ))))
  (DE-valid? (DQ-DE3 DQ)))

(defthm bitp-DQ-full? (implies (DQ-p DQ) (bitp (DQ-full? DQ))))
(in-theory (disable DQ-full?))

(defun DQ-ready-no-unit? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (b-and (DE-valid? (DQ-DE0 DQ))
	 (b-ior (logbit 4 (cntlv-exunit (DE-cntlv (DQ-DE0 DQ))))
		(excpt-raised? (DE-excpt (DQ-DE0 DQ))))))

(defthm bitp-DQ-ready-no-unit (bitp (DQ-ready-no-unit? DQ)))

(defun DQ-ready-to-IU? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (bs-and (DE-valid? (DQ-DE0 DQ))
	  (logbit 0 (cntlv-exunit (DE-cntlv (DQ-DE0 DQ))))
	  (b-not (excpt-raised? (DE-excpt (DQ-DE0 DQ))))))

(defthm bitp-DQ-ready-to-IU (bitp (DQ-ready-to-IU? DQ)))

(defun DQ-ready-to-MU? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (bs-and (DE-valid? (DQ-DE0 DQ))
	  (logbit 1 (cntlv-exunit (DE-cntlv (DQ-DE0 DQ))))
	  (b-not (excpt-raised? (DE-excpt (DQ-DE0 DQ))))))

(defthm bitp-DQ-ready-to-MU (bitp (DQ-ready-to-MU? DQ)))

(defun DQ-ready-to-LSU? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (bs-and (DE-valid? (DQ-DE0 DQ))
	  (logbit 2 (cntlv-exunit (DE-cntlv (DQ-DE0 DQ))))
	  (b-not (excpt-raised? (DE-excpt (DQ-DE0 DQ))))))

(defthm bitp-DQ-ready-to-LSU (bitp (DQ-ready-to-LSU? DQ)))

(defun DQ-ready-to-BU? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (bs-and (DE-valid? (DQ-DE0 DQ))
	  (logbit 3 (cntlv-exunit (DE-cntlv (DQ-DE0 DQ))))
	  (b-not (excpt-raised? (DE-excpt (DQ-DE0 DQ))))))

(defthm bitp-DQ-ready-to-BU (bitp (DQ-ready-to-BU? DQ)))

; If the ROB-flg is set, and ROB-head and ROB-tail are equal to each other,
; then the reorder buffer is full.
(defun ROB-full? (ROB)
  (declare (xargs :guard (ROB-p ROB)))
  (b-and (ROB-flg ROB)
	 (bv-eqv *rob-index-size* (ROB-head ROB) (ROB-tail ROB))))

(defthm bitp-ROB-full (bitp (ROB-full? ROB)))

; If the ROB-flg is unset, and ROB-head and ROB-tail are equal to each other,
; then the reorder buffer is empty.
(defun ROB-empty? (ROB)
  (declare (xargs :guard (ROB-p ROB)))
  (b-andc1 (ROB-flg ROB)
	   (bv-eqv *rob-index-size* (ROB-head ROB) (ROB-tail ROB))))

(defthm bitp-ROB-empty (bitp (ROB-empty? ROB)))

;
(defun LSU-pending-writes? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (let ((wbuf0 (LSU-wbuf0 LSU)) (wbuf1 (LSU-wbuf1 LSU)))
    (b-ior (b-and (wbuf-valid? wbuf0) (wbuf-commit? wbuf0))
	   (b-and (wbuf-valid? wbuf1) (wbuf-commit? wbuf1)))))

(defthm bitp-LSU-pending-writes (bitp (LSU-pending-writes? LSU)))

; This line coming out of ROB is raised if the ROB detects a branch
; misprediction and syncronization.  When entering an exception or leaving
; one, this line may not be raised.
(defun commit-jmp? (s)
  (declare (xargs :guard (MA-state-p s)))
  (let ((LSU (MA-LSU s)) (ROB (MA-ROB s)))
    (let ((ROBE (nth-ROBE (ROB-head ROB) ROB)))
      (bs-and (ROBE-valid? ROBE)
	      (ROBE-complete? ROBE)
	      (b-ior (b-andc2 (ROBE-sync? ROBE)
			      (LSU-pending-writes? LSU))
		     (b-and (ROBE-branch? ROBE)
			    (b-xor (ROBE-br-predict? ROBE)
				   (ROBE-br-actual? ROBE))))))))
(defthm bitp-commit-jmp (bitp (commit-jmp? s)))
(in-theory (disable commit-jmp?))

; Enter-excpt? is raised if an excetion except for external interrupts
; is raised.
(defun enter-excpt? (s)
  (declare (xargs :guard (MA-state-p s)))
  (let ((robe (nth-ROBE (ROB-head (MA-ROB s)) (MA-ROB s)))
	(LSU (MA-LSU s)))
    (bs-and (ROBE-valid? robe)
	    (ROBE-complete? robe)
	    (excpt-raised? (ROBE-excpt robe))
	    (b-not (LSU-pending-writes? LSU)))))

(defthm bitp-enter-excpt (bitp (enter-excpt? s)))
(in-theory (disable enter-excpt?))

; Ex-intr? is raised if the external exception hanling starts.
(defun ex-intr? (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (let ((ROB (MA-ROB s)) (LSU (MA-LSU s)))
    (bs-and (ROB-empty? ROB)
	    (b-not (LSU-pending-writes? LSU))
	    (b-ior (ROB-exintr? ROB)
		   (MA-oracle-exintr orcl)))))

(defthm bitp-ex-intr (bitp (ex-intr? s orcl)))
(in-theory (disable ex-intr?))

(defun ex-intr-addr (s)
  (declare (xargs :guard (MA-state-p s)))
  (let ((DE0 (DQ-DE0 (MA-DQ s))) (IFU (MA-IFU s)) (pc (MA-pc s)))
    (b-if (DE-valid? DE0) (DE-pc DE0)
    (b-if (IFU-valid? IFU) (IFU-pc IFU)
	  pc))))

(defthm addr-ex-intr-addr
    (implies (MA-state-p s) (addr-p (ex-intr-addr s)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (MA-state-p s) (integerp (ex-intr-addr s))))))
(in-theory (disable ex-intr-addr))

(defun leave-excpt? (s)
  (declare (xargs :guard (MA-state-p s)))
  (let ((ROBE (nth-ROBE (ROB-head (MA-ROB s)) (MA-ROB s))))
    (bs-and (ROBE-valid? ROBE)
	    (ROBE-complete? ROBE)
	    (ROBE-rfeh? ROBE)
	    (b-not (LSU-pending-writes? (MA-LSU s))))))

(defthm bitp-leave-excpt (bitp (leave-excpt? s)))
(in-theory (disable leave-excpt?))


; If this line is raised, all entries in the pipeline are nullified.
; Flushing is usually taken when mispredicted branch, synchronization or
; a context switching takes place.
(defun flush-all? (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (bs-ior (commit-jmp? s)
	  (enter-excpt? s)
	  (ex-intr? s orcl)
	  (leave-excpt? s)))

(defthm bitp-flush-all (bitp (flush-all? s orcl)))
(in-theory (disable flush-all?))

; ROB-jmp-addr output the destination address of a ROB caused jump.
; This is valid when commit-jmp?, enter-excpt? or ex-intr? is raised.
; When leave-excpt? is raised, pc should get its new value from SR0.
; Notice that a conditional branch that is predicted to take place but
; actually does not also causes commit-jmp?.
(defun ROB-jmp-addr (ROB s orcl)
  (declare (xargs :guard (and (ROB-p ROB) (MA-state-p s) (MA-oracle-p orcl))))
  (let ((ROBE (nth-ROBE (ROB-head ROB) ROB)))
    (b-if (ex-intr? s orcl) #x30
    (b-if (enter-excpt? s)
	  (logapp 4 0 (excpt-type (ROBE-excpt ROBE)))
    (b-if (bs-and (ROBE-valid? ROBE)
		  (ROBE-complete? ROBE)
		  (b-orc2 (ROBE-sync? ROBE)
			  (ROBE-br-actual? ROBE)))
	  (addr (1+ (ROBE-pc ROBE)))
    (b-if (bs-and (ROBE-valid? ROBE)
		  (ROBE-complete? ROBE)
		  (ROBE-br-actual? ROBE))
 	  (addr (ROBE-val ROBE))
	  0))))))

(defthm addr-p-rob-jmp-addr
    (implies (and (ROB-p ROB) (MA-state-p s) (MA-oracle-p orcl))
	     (addr-p (ROB-jmp-addr ROB s orcl)))
  :hints (("goal" :in-theory (e/d (addr-p logapp* unsigned-byte-p* addr)
				  (LOGAPP-0)))))
(in-theory (disable ROB-jmp-addr))


; ROB-write-reg? is raised when ROB writes its result into the register file.
(defun ROB-write-reg? (ROB)
  (declare (xargs :guard (ROB-p ROB)))
  (let ((ROBE (nth-ROBE (ROB-head ROB) ROB)))
    (bs-and (ROBE-valid? ROBE)
	    (ROBE-complete? ROBE)
	    (ROBE-wb? ROBE)
	    (b-not (ROBE-wb-sreg? ROBE))
    	    (b-not (excpt-raised? (ROBE-excpt ROBE))))))

; ROB-write-sreg? is raised if ROB writes its result into the
; special register file.
(defun ROB-write-sreg? (ROB)
  (declare (xargs :guard (ROB-p ROB)))
  (let ((ROBE (nth-ROBE (ROB-head ROB) ROB)))
    (bs-and (ROBE-valid? ROBE)
	    (ROBE-complete? ROBE)
	    (ROBE-wb? ROBE)
	    (ROBE-wb-sreg? ROBE)
	    (b-not (excpt-raised? (ROBE-excpt ROBE))))))

; ROB-write-val returns the value to be written to the register file
; or the special regsiter file.  When enter-excpt? is on, this bus is
; used to output the program counter value of the instruction that
; caused the raised exception or the next instruction.
(defun ROB-write-val (ROB s)
  (declare (xargs :guard (and (ROB-p ROB) (MA-state-p s))))
  (let ((robe (nth-ROBE (ROB-head ROB) ROB)))
    (b-if (enter-excpt? s)
	  (b-if (bv-eqv 2 (excpt-type (ROBE-excpt robe)) 0)
		(word (1+ (ROBE-pc robe)))
		(word (ROBE-pc robe)))
	  (ROBE-val robe))))

; ROB-write-rid returns the register id to be written the write-back value
; into.  If ROB-write-reg? is on, this indexes a normal register.
; If ROB-write-sreg? is on, this specifies a special register.  We assume
; that both lines are not on simultaneously.
(defun ROB-write-rid (ROB)
  (declare (xargs :guard (ROB-p ROB)))
  (ROBE-dest (nth-ROBE (ROB-head ROB) ROB)))

; select-IU-RS0? is set is IU-RS0 is chosen as the open reservation station
; slot for the new instruction to go into.
(defun select-IU-RS0? (iu)
  (declare (xargs :guard (integer-unit-p iu)))
  (b-not (RS-valid? (IU-RS0 IU))))

(defthm bitp-select-IU-rs0 (bitp (select-IU-RS0? IU)))

(defun select-IU-RS1? (iu)
  (declare (xargs :guard (integer-unit-p iu)))
  (b-and (b-not (RS-valid? (IU-RS1 IU)))
	 (RS-valid? (IU-RS0 IU))))

(defthm bitp-select-IU-rs1 (bitp (select-IU-RS1? IU)))

; If there is an available reservation station in IU, this line is raised.
(defun IU-ready? (iu)
  (declare (xargs :guard (integer-unit-p iu)))
  (b-nand (RS-valid? (IU-RS0 IU)) (RS-valid? (IU-RS1 IU))))

(defthm bitp-IU-ready? (bitp (IU-ready? IU)))

(defun IU-RS0-issue-ready? (iu)
  (declare (xargs :guard (integer-unit-p iu)))
  (let ((RS0 (IU-RS0 IU)))
    (bs-and (RS-valid? RS0)
	    (RS-ready1? RS0)
	    (b-ior (RS-op RS0) (RS-ready2? RS0)))))

(defthm bitp-IU-rs0-issue-ready? (bitp (IU-RS0-issue-ready? IU)))

(defun IU-RS1-issue-ready? (iu)
  (declare (xargs :guard (integer-unit-p iu)))
  (let ((RS1 (IU-RS1 IU)))
    (bs-and (RS-valid? RS1)
	    (RS-ready1? RS1)
	    (b-not (IU-RS0-issue-ready? IU))
	    (b-ior (RS-op RS1) (RS-ready2? RS1)))))

(defthm bitp-IU-rs1-issue-ready? (bitp (IU-RS1-issue-ready? IU)))

(defun IU-output-dest (IU)
  (declare (xargs :guard (integer-unit-p IU)))
  (b-if (IU-RS0-issue-ready? IU) (RS-dest (IU-RS0 IU))
        (b-if (IU-RS1-issue-ready? IU) (RS-dest (IU-RS1 IU))
	      0)))

(defun IU-output-val (IU)
  (declare (xargs :guard (and (integer-unit-p IU))))
  (let ((RS0 (IU-RS0 IU)) (RS1 (IU-RS1 IU)))
    (b-if (IU-RS0-issue-ready? IU)
	  (b-if (RS-op RS0)
		(RS-val1 RS0)
		(word (+ (RS-val1 RS0) (RS-val2 RS0))))
    (b-if (IU-RS1-issue-ready? IU)
	  (b-if (RS-op RS1)
		(RS-val1 RS1)
		(word (+ (RS-val1 RS1) (RS-val2 RS1))))
	  0))))

; select-MU-RS0? is set if MU-RS0 is chosen as the open reservation station
; slot for the new instruction to go into.
(defun select-MU-RS0? (MU)
  (declare (xargs :guard (mult-unit-p MU)))
  (b-not (RS-valid? (MU-RS0 MU))))

(defthm bitp-select-MU-RS0 (bitp (select-MU-RS0? MU)))

(defun select-MU-RS1? (MU)
  (declare (xargs :guard (mult-unit-p MU)))
  (b-and (b-not (RS-valid? (MU-RS1 MU)))
	 (RS-valid? (MU-RS0 MU))))

(defthm bitp-select-MU-RS1 (bitp (select-MU-RS1? MU)))

; IF there is an available reservation station in mult-unit,
; this line is raised.
(defun MU-ready? (MU)
  (declare (xargs :guard (mult-unit-p MU)))
  (b-nand (RS-valid? (MU-RS0 MU)) (RS-valid? (MU-RS1 MU))))

(defthm bitp-MU-ready (bitp (MU-ready? MU)))

(defun MU-RS0-issue-ready? (MU)
  (declare (xargs :guard (mult-unit-p MU)))
  (let ((RS0 (MU-RS0 MU)))
    (bs-and (RS-valid? RS0) (RS-ready1? RS0) (RS-ready2? RS0))))

(defthm bitp-MU-RS0-issue-ready (bitp (MU-RS0-issue-ready? MU)))

(defun MU-RS1-issue-ready? (MU)
  (declare (xargs :guard (mult-unit-p MU)))
  (let ((RS1 (MU-RS1 MU)))
    (bs-and (RS-valid? RS1) (RS-ready1? RS1) (RS-ready2? RS1)
	    (b-not (MU-RS0-issue-ready? MU)))))

(defthm bitp-MU-RS1-issue-ready (bitp (MU-RS1-issue-ready? MU)))

;; We abstract away the implementation of the multiplier.
;; The real definition of the multiplier is not given, but a witness
;; functions are provided instead.  The witness functions are necessary to
;; enable us to run the specification and avoid an inconsistency of the
;; theory.
(encapsulate-impl multiplier
    ((ML1-output (ra rb) t)
     (ML2-output (data) t)
     (ML3-output (data) t))

((defun ML1-output (ra rb)
  (declare (xargs :guard (and (word-p ra) (word-p rb))))
  (cons ra rb))
 (defun ML2-output (data)
   (declare (xargs :guard t))
   data)
 (defun ML3-output (data)
   (declare (xargs :guard t))
   (if (consp data)
       (word (* (nfix (car data)) (nfix (cdr data))))
       0)))

((defthm ML-output-correct
    (implies (and (word-p ra) (word-p rb))
	     (equal (ML3-output (ML2-output (ML1-output ra rb)))
		    (word (* ra rb)))))
(defthm ML3-output-correct
    (word-p (ML3-output data))))
)


; select-BU-RS0? is set if BU-RS0 is chosen as the open reservation station
; slot for the new instruction to go into.
(defun select-BU-RS0? (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (b-not (Bu-RS-valid? (BU-RS0 BU))))

(defthm bitp-select-BU-RS0? (bitp (select-BU-RS0? BU)))

(defun select-BU-RS1? (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (b-and (b-not (BU-RS-valid? (BU-RS1 BU)))
	 (BU-RS-valid? (BU-RS0 BU))))

(defthm bitp-select-BU-RS1? (bitp (select-BU-RS1? BU)))

; IF there is an available reservation station in BU-unit,
; this line is raised.
(defun BU-ready? (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (b-nand (BU-RS-valid? (BU-RS0 BU)) (BU-RS-valid? (BU-RS1 BU))))

(defthm bitp-BU-ready (bitp (BU-ready? BU)))

(defun BU-RS0-issue-ready? (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (let ((RS0 (BU-RS0 BU)))
    (b-and (BU-RS-valid? RS0) (BU-RS-ready? RS0))))

(defthm bitp-BU-RS0-issue-ready (bitp (BU-RS0-issue-ready? BU)))

(defun BU-RS1-issue-ready? (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (let ((RS1 (BU-RS1 BU)))
    (bs-and (BU-RS-valid? RS1) (BU-RS-ready? RS1)
	    (b-not (BU-RS0-issue-ready? BU)))))

(defthm bitp-BU-RS1-issue-ready (bitp (BU-RS1-issue-ready? BU)))

(defun BU-output-dest (BU)
  (declare (xargs :guard (and (branch-unit-p BU))))
  (b-if (BU-RS0-issue-ready? BU) (BU-RS-dest (BU-RS0 BU))
        (b-if (BU-RS1-issue-ready? BU) (BU-RS-dest (BU-RS1 BU))
	      0)))

(defun BU-output-val (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (b-if (BU-RS0-issue-ready? BU)
	(b-if (bv-eqv *word-size* (BU-RS-val (BU-RS0 BU)) 0)
	      #xFFFF 0)
  (b-if (BU-RS1-issue-ready? BU)
	(b-if (bv-eqv *word-size* (BU-RS-val (BU-RS1 BU)) 0)
	      #xFFFF 0)
	0)))

; select-LSU-RS0? is set if LSU-RS0 is chosen as the open reservation station
; slot for the new instruction to go into.  This value is valid only if
; LSU-ready? is 1.
(defun select-LSU-RS0? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (b-if (LSU-rs1-head? LSU)
	(LSU-RS-valid? (LSU-RS1 LSU))
	(b-not (LSU-RS-valid? (LSU-RS0 LSU)))))

(defthm bitp-select-LSU-RS0?
    (implies (load-store-unit-p LSU) (bitp (select-LSU-RS0? LSU))))
(in-theory (disable select-LSU-RS0?))

(defun select-LSU-RS1? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (b-if (LSU-rs1-head? LSU)
	(b-not (LSU-RS-valid? (LSU-RS1 LSU)))
	(LSU-RS-valid? (LSU-RS0 LSU))))

(defthm bitp-select-LSU-RS1?
    (implies (load-store-unit-p LSU) (bitp (select-LSU-RS1? LSU))))
(in-theory (disable select-LSU-RS1?))

; IF there is an available reservation station in LSU-unit,
; this line is raised.
(defun LSU-ready? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (b-if (LSU-rs1-head? LSU)
	(b-not (LSU-RS-valid? (LSU-RS0 LSU)))
	(b-not (LSU-RS-valid? (LSU-RS1 LSU)))))

(defthm bitp-LSU-ready? (bitp (LSU-ready? LSU)))
(in-theory (disable LSU-ready?))

; LSU-RS0-issue-ready? is on when the load store instruction in RS0 is
; ready to be issued.
; RS0 should be a valid and necessary operands must be ready.
; Also the order of instruction issues are important.  IF RS1 contains a
; valid instruction and the next reservation flag points to RS1, it means
; that RS1 contains an earlier instruction than RS0.
(defun LSU-RS0-issue-ready? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (let ((RS0 (LSU-RS0 LSU))
	(RS1 (LSU-RS1 LSU)))
    (bs-and (LSU-RS-valid? RS0)
	    (b-orc1 (LSU-RS-ld-st? RS0) (LSU-RS-rdy3? RS0))
	    (LSU-RS-rdy1? RS0)
	    (b-ior (LSU-RS-op RS0) (LSU-RS-rdy2? RS0))
	    (b-nand (LSU-rs1-head? LSU) (LSU-RS-valid? RS1)))))

(defthm bitp-LSU-RS0-issue-ready? (bitp (LSU-RS0-issue-ready? LSU)))
(in-theory (disable LSU-RS0-issue-ready?))


; LSU-RS1-issue-ready? is on when the load store instruction in RS1 is ready
; to be issued.
; RS0 should be a valid and necessary operands must be ready.
; Also the order of instruction issues are important.  IF RS0 contains a
; valid instruction and the next reservation flag points to RS0, it means
; that RS0 contains an earlier instruction than RS1.
(defun LSU-RS1-issue-ready? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (let ((RS0 (LSU-RS0 LSU))
	(RS1 (LSU-RS1 LSU)))
    (bs-and (LSU-RS-valid? RS1)
	    (b-orc1 (LSU-RS-ld-st? RS1) (LSU-RS-rdy3? RS1))
	    (LSU-RS-rdy1? RS1)
	    (b-ior (LSU-RS-op RS1) (LSU-RS-rdy2? RS1))
	    (b-orc2 (LSU-rs1-head? LSU) (LSU-RS-valid? RS0)))))

(defthm bitp-LSU-RS1-issue-ready? (bitp (LSU-RS1-issue-ready? LSU)))

; If the access addresss for the load instruction in a read buffer entry and
; the store instruction in a write buffer entry match, we can use the
; stored value as the result of the load instruction.  LSU-address-match?
; is on if there is a store instruction in the write buffer, which has the
; same access address and has its completed? flag on.
(defun LSU-address-match? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (let ((rbuf (LSU-rbuf LSU))
	(wbuf0 (LSU-wbuf0 LSU))
	(wbuf1 (LSU-wbuf1 LSU)))
    (b-ior (bs-and (bv-eqv *addr-size* (rbuf-addr rbuf) (wbuf-addr wbuf0))
		   (rbuf-valid? rbuf)
		   (wbuf-valid? wbuf0)
		   (wbuf-complete? wbuf0))
	   (bs-and (bv-eqv *addr-size* (rbuf-addr rbuf) (wbuf-addr wbuf1))
		   (rbuf-valid? rbuf)
		   (wbuf-valid? wbuf1)
		   (wbuf-complete? wbuf1)))))

(defthm bitp-LSU-address-match (bitp (LSU-address-match? LSU)))
(in-theory (disable LSU-address-match?))


; If the addresses of load and store instruction in buffers match, we can
; use the value for the strore instruction as the result of the load
; instruction. LSU-forward-wbuf associatively search the write buffer
; for the store instruction entry with a maching address, and returns the
; store value.  If there are more than one matching entry the value of the
; latest completed instruction is returned.
(defun LSU-forward-wbuf (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (let ((rbuf (LSU-rbuf LSU))
	(wbuf0 (LSU-wbuf0 LSU))
	(wbuf1 (LSU-wbuf1 LSU)))
    (b-if (bs-and (bv-eqv *addr-size* (rbuf-addr rbuf) (wbuf-addr wbuf1))
		  (wbuf-valid? wbuf1)
		  (wbuf-complete? wbuf1))
	  (wbuf-val wbuf1)
    (b-if (bs-and (bv-eqv *addr-size* (rbuf-addr rbuf) (wbuf-addr wbuf0))
		  (wbuf-valid? wbuf0)
		  (wbuf-complete? wbuf0))
	  (wbuf-val wbuf0)
	  0))))


; A memory stall occurs when the memory system does not respond for some
; reasons.  We don't model cache system or memory controller at all.
; Instead we give a very simplistic view of memory, which does or does
; not repspond to the memory access resquest nondeterministicly.
; LSU-read-stall? can be set only when the LSU-rbuf1 is occupied.
; LUS-read-stall? does not indicate that the read address violates the
; memory protection, or there is an read-write address match with
; a write buffer entry.
(defun LSU-read-stall? (LSU orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU) (MA-oracle-p orcl))))
  (b-andc2 (rbuf-valid? (LSU-rbuf LSU))
	   (MA-oracle-data orcl)))

(defthm bitp-LSU-read-stall? (bitp (LSU-read-stall? LSU orcl)))
(in-theory (disable LSU-read-stall?))

; When the read address is prohibited by the memory protection mechanism,
; this line is raised.  This line is only valid when rbuf-valid? is on.
;           violation    1  0
;   rbuf-valid?  1       1  0
;                0       x  x
(defun LSU-read-prohibited? (LSU mem su)
  (declare (xargs :guard (and (load-store-unit-p LSU) (mem-p mem)
			      (bitp su))))
  (b-nor (readable-addr? (rbuf-addr (LSU-rbuf LSU)) mem) su))

(defthm bitp-LSU-read-prohibited? (bitp (LSU-read-prohibited? LSU mem su)))
(in-theory (disable LSU-read-prohibited?))

; IF the store instruction at the head of the write buffer has commited
; and we are ready to write the value to the memory, release-wbuf0-ready?
; is set.
(defun release-wbuf0-ready? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (let ((wbuf0 (LSU-wbuf0 LSU))
	(rbuf (LSU-rbuf LSU)))
    (bs-and (b-not (rbuf-valid? rbuf))
	    (wbuf-valid? wbuf0) (wbuf-commit? wbuf0))))

(defthm bitp-release-wbuf0-ready? (bitp (release-wbuf0-ready? LSU)))
(in-theory (disable release-wbuf0-ready?))

; If the write buffer entry at wbuf0 is valid, but not yet completed,
; we may be ready to check its address and complete the instruction.
; The actual memory access takes place after the store instruction commits.
; check-wbuf0? is on when the entry is actually sent to next latch
; to complete.
(defun check-wbuf0? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (bs-and (b-not (rbuf-valid? (LSU-rbuf LSU)))
	  (wbuf-valid? (LSU-wbuf0 LSU))
	  (b-not (wbuf-complete? (LSU-wbuf0 LSU)))))

(defthm bitp-check-wbuf0? (bitp (check-wbuf0? LSU)))
(in-theory (disable check-wbuf0?))

(defun check-wbuf1? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (bs-and (b-not (rbuf-valid? (LSU-rbuf LSU)))
	  (wbuf-valid? (LSU-wbuf1 LSU))
	  (b-not (wbuf-complete? (LSU-wbuf1 LSU)))
	  (b-not (check-wbuf0? LSU))
	  (b-not (release-wbuf0-ready? LSU))))

(defthm bitp-check-wbuf1? (bitp (check-wbuf1? LSU)))
(in-theory (disable check-wbuf1?))

; LSU-write-stall? is set when a write buffer entry tries to write its result
; to the memory and the memory does not respond.  It is only set when
; Release-wbuf0-ready? is on.
(defun LSU-write-stall? (LSU orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU) (MA-oracle-p orcl))))
  (b-andc2 (release-wbuf0-ready? LSU) (MA-oracle-data orcl)))

(defthm bitp-LSU-write-stall? (bitp (LSU-write-stall? LSU orcl)))
(in-theory (disable LSU-write-stall?))


(defun release-wbuf0? (LSU orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU) (MA-oracle-p orcl))))
  (b-andc2 (release-wbuf0-ready? LSU) (LSU-write-stall? LSU orcl)))

(defthm bitp-release-wbuf0? (bitp (release-wbuf0? LSU orcl)))
(in-theory (disable release-wbuf0?))

; If the memory access is prohibited by the memory system,
; LSU-write-prohibited? is raised.  This is valid only when
; check-wbuf0? or check-wbuf1? is on.  The instruciton in the corresponding
; write buffer entry caused the write address violation.
(defun LSU-write-prohibited? (LSU mem su)
  (declare (xargs :guard (and (load-store-unit-p LSU) (mem-p mem) (bitp su))))
  (b-if (check-wbuf0? LSU)
	(b-nor (writable-addr? (wbuf-addr (LSU-wbuf0 LSU)) mem) su)
  (b-if (check-wbuf1? LSU)
	(b-nor (writable-addr? (wbuf-addr (LSU-wbuf1 LSU)) mem) su)
	0)))

(defthm bitp-LSU-write-prohibited? (bitp (LSU-write-prohibited? LSU mem su)))
(in-theory (disable LSU-write-prohibited?))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Dispatching of instructions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun dispatch-no-unit? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (DQ-ready-no-unit? (MA-DQ s))
	  (b-not (ROB-full? (MA-ROB s)))
	  (b-not (ROB-exintr? (MA-ROB s)))))

(defthm bitp-dispatch-no-unit? (bitp (dispatch-no-unit? s)))
(in-theory (disable dispatch-no-unit?))

(defun dispatch-to-IU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (DQ-ready-to-IU? (MA-DQ s))
	  (b-not (ROB-full? (MA-ROB s)))
	  (b-not (ROB-exintr? (MA-ROB s)))
	  (IU-ready? (MA-IU s))))

(defthm bitp-dispatch-to-IU (bitp (dispatch-to-IU? s)))
(in-theory (disable dispatch-to-IU?))

(defun dispatch-to-MU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (DQ-ready-to-MU? (MA-DQ s))
	  (b-not (ROB-full? (MA-ROB s)))
	  (b-not (ROB-exintr? (MA-ROB s)))
	  (MU-ready? (MA-MU s))))

(defthm bitp-dispatch-to-MU (bitp (dispatch-to-MU? s)))
(in-theory (disable dispatch-to-MU?))

(defun dispatch-to-LSU? (s)
  (declare (xargs :guard  (MA-state-p s)))
  (bs-and (DQ-ready-to-LSU? (MA-DQ s))
	  (b-not (ROB-full? (MA-ROB s)))
	  (b-not (ROB-exintr? (MA-ROB s)))
	  (LSU-ready? (MA-LSU s))))

(defthm bitp-dispatch-to-LSU? (bitp (dispatch-to-LSU? s)))
(in-theory (disable dispatch-to-LSU?))

(defun dispatch-to-BU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (DQ-ready-to-BU? (MA-DQ s))
	  (b-not (ROB-full? (MA-ROB s)))
	  (b-not (ROB-exintr? (MA-ROB s)))
	  (BU-ready? (MA-BU s))))

(defthm bitp-dispatch-to-BU (bitp (dispatch-to-BU? s)))
(in-theory (disable dispatch-to-BU?))

; This line is raised if any instruction is ready to dispatch.
; A dispatch actually takes place if the corresponding execution
; unit has an empty reservation station.
(defun dispatch-inst? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-ior (dispatch-no-unit? s)
	  (dispatch-to-IU? s)
	  (dispatch-to-MU? s)
	  (dispatch-to-LSU? s)
	  (dispatch-to-BU? s)))

(defthm bitp-dispatch-inst? (bitp (dispatch-inst? s)))
(in-theory (disable dispatch-inst?))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Common Data Bus Arbitration
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun CDB-for-LSU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (LSU-latch-valid? (LSU-lch (MA-LSU s))))

(defun CDB-for-MU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (b-not (CDB-for-LSU? s))
	  (MU-latch-valid? (MU-lch2 (MA-MU s)))))

(defun CDB-for-BU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (b-not (CDB-for-LSU? s))
	  (b-not (CDB-for-MU? s))
	  (b-ior (BU-RS0-issue-ready? (MA-BU s))
		 (BU-RS1-issue-ready? (MA-BU s)))))

(defun CDB-for-IU? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (b-not (CDB-for-LSU? s))
	  (b-not (CDB-for-BU? s))
	  (b-not (CDB-for-MU? s))
	  (b-ior (IU-RS0-issue-ready? (MA-IU s))
		 (IU-RS1-issue-ready? (MA-IU s)))))


(defthm bitp-CDB-for-LSU  (implies (MA-state-p s) (bitp (CDB-for-LSU? s))))
(defthm bitp-CDB-for-MU  (bitp (CDB-for-MU? s)))
(defthm bitp-CDB-for-BU  (bitp (CDB-for-BU? s)))
(defthm bitp-CDB-for-IU  (bitp (CDB-for-IU? s)))
(in-theory (disable CDB-for-LSU? CDB-for-MU? CDB-for-BU? CDB-for-IU?))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Definition of the issues of instructions at each execution unit.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; issue-IU-RS0 and issue-IU-RS1 are set if the instruction in the
; corresponding reservation station is issued to the execution unit.
; Otherwise, cleared.  Notice issue-IU-RS0? is on, RS-valid? for RS0 is on.
; Similarly with RS1.
(defun issue-IU-RS0? (iu s)
  (declare (xargs :guard (and (integer-unit-p iu) (MA-state-p s))))
  (b-and (IU-RS0-issue-ready? IU) (CDB-for-IU? s)))

(defthm bitp-issue-IU-RS0  (bitp (issue-IU-RS0? IU s)))

(defun issue-IU-RS1? (iu s)
  (declare (xargs :guard (and (integer-unit-p iu) (MA-state-p s))))
  (bs-and (IU-RS1-issue-ready? IU) (CDB-for-IU? s)))

(defthm bitp-issue-IU-RS1 (bitp (issue-IU-RS1? IU s)))

(defun issue-MU-RS0? (MU s)
  (declare (xargs :guard (and (mult-unit-p MU) (MA-state-p s))))
  (b-and (MU-RS0-issue-ready? MU)
	 (bs-ior (CDB-for-MU? s)
		 (b-not (MU-latch-valid? (MU-lch1 MU)))
		 (b-not (MU-latch-valid? (MU-lch2 MU))))))

(defthm bitp-issue-MU-RS0 (bitp (issue-MU-RS0? MU s)))

(defun issue-MU-RS1? (MU s)
  (declare (xargs :guard (and (mult-unit-p MU) (MA-state-p s))))
  (bs-and (MU-RS1-issue-ready? MU)
	  (bs-ior (CDB-for-MU? s)
		  (b-not (MU-latch-valid? (MU-lch1 MU)))
		  (b-not (MU-latch-valid? (MU-lch2 MU))))))

(defthm bitp-issue-MU-RS1 (bitp (issue-MU-RS1? MU s)))

(defun issue-BU-RS0? (BU s)
  (declare (xargs :guard (and (branch-unit-p BU) (MA-state-p s))))
  (b-and (BU-RS0-issue-ready? BU) (CDB-for-BU? s)))

(defthm bitp-issue-BU-RS0? (bitp (issue-BU-RS0? BU s)))

(defun issue-BU-RS1? (BU s)
  (declare (xargs :guard (and (branch-unit-p BU) (MA-state-p s))))
  (b-and (BU-RS1-issue-ready? BU) (CDB-for-BU? s)))

(defthm bitp-issue-BU-RS1 (bitp (issue-BU-RS1? BU s)))

(defun issue-LSU-RS0? (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU) (MA-state-p s)
			      (MA-oracle-p orcl))))
  (bs-and (LSU-RS0-issue-ready? LSU)
	  (b-if (LSU-RS-ld-st? (LSU-RS0 LSU))
		(b-ior (b-not (wbuf-valid? (LSU-wbuf1 LSU)))
		       (release-wbuf0? LSU orcl))
		(bs-ior (b-not (rbuf-valid? (LSU-rbuf LSU)))
			(LSU-address-match? LSU)
			(LSU-read-prohibited? LSU (MA-mem s)
					      (sregs-su (MA-sregs s)))
			(b-not (LSU-read-stall? LSU orcl))))))

(defthm bitp-issue-LSU-RS0? (bitp (issue-LSU-RS0? LSU s orcl)))
(in-theory (disable issue-LSU-RS0?))

(defun issue-LSU-RS1? (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (bs-and (LSU-RS1-issue-ready? LSU)
	  (b-if (LSU-RS-ld-st? (LSU-RS0 LSU))
		(b-ior (b-not (wbuf-valid? (LSU-wbuf1 LSU)))
		       (release-wbuf0? LSU orcl))
		(bs-ior (b-not (rbuf-valid? (LSU-rbuf LSU)))
			(LSU-address-match? LSU)
			(LSU-read-prohibited? LSU (MA-mem s)
					      (sregs-su (MA-sregs s)))
			(b-not (LSU-read-stall? LSU orcl))))))

(defthm bitp-issue-LSU-RS1? (bitp (issue-LSU-RS1? LSU s orcl)))
(in-theory (disable issue-LSU-RS1?))

; Common Data Bus should raise ready? flag when a datum is ready.
; CDB-dest posts the destination ROB entry index, while CDB-val is
; the result of the executed instruction.
(defun CDB-ready? (s)
  (declare (xargs :guard  (MA-state-p s)))
  (bs-ior (LSU-latch-valid? (LSU-lch (MA-LSU s)))
	  (MU-latch-valid? (MU-lch2 (MA-MU s)))
	  (BU-RS0-issue-ready? (MA-BU s))
	  (BU-RS1-issue-ready? (MA-BU s))
	  (IU-RS0-issue-ready? (MA-IU s))
	  (IU-RS1-issue-ready? (MA-IU s))))

(defthm bitp-CDB-ready   (bitp (CDB-ready? s)))
(in-theory (disable CDB-ready?))

(defun CDB-dest (s)
  (declare (xargs :guard (MA-state-p s)))
  (b-if (CDB-for-IU? s)
	(IU-output-dest (MA-IU s))
  (b-if (CDB-for-BU? s)
	(BU-output-dest (MA-BU s))
  (b-if (CDB-for-MU? s)
	(MU-latch-dest (MU-lch2 (MA-MU s)))
  (b-if (CDB-for-LSU? s)
	(LSU-latch-dest (LSU-lch (MA-LSU s)))
	0)))))

(defthm rob-index-p-CDB-dest
    (implies (MA-state-p s) (rob-index-p (CDB-dest s)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (MA-state-p s) (integerp (CDB-dest s))))))

(in-theory (disable CDB-dest))

(defun CDB-excpt (s)
  (declare (xargs :guard (MA-state-p s)))
  (b-if (CDB-for-LSU? s) (LSU-latch-excpt (LSU-lch (MA-LSU s))) 0))

(defthm excpt-flags-p-CDB-excpt
    (implies (MA-state-p s) (excpt-flags-p (CDB-excpt s))))
(in-theory (disable CDB-excpt))


(defun CDB-val (s)
  (declare (xargs :guard  (MA-state-p s)))
  (b-if (CDB-for-IU? s)
	(IU-output-val (MA-IU s))
  (b-if (CDB-for-BU? s)
	(BU-output-val (MA-BU s))
  (b-if (CDB-for-MU? s)
	(ML3-output (MU-latch-data (MU-lch2 (MA-MU s))))
  (b-if (CDB-for-LSU? s)
	(LSU-latch-val (LSU-lch (MA-LSU s)))
	0)))))

(defthm word-p-CDB-val
    (implies (MA-state-p s) (word-p (CDB-val s)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (MA-state-p s) (integerp (CDB-val s)))
	     :hints (("Goal" :in-theory (enable word-p))))))
(in-theory (disable CDB-val))

(defun CDB-ready-for? (idx s)
  (declare (xargs :guard (and (rob-index-p idx) (MA-state-p s))))
  (b-and (CDB-ready? s)
	 (bv-eqv *rob-index-size* idx (CDB-dest s))))

(defthm bitp-CDB-ready-for (bitp (CDB-ready-for? idx s)))
(in-theory (disable CDB-ready-for?))

(defun dispatch-cntlv (s)
  (declare (xargs :guard (MA-state-p s)))
  (DE-cntlv (DQ-DE0 (MA-DQ s))))

(defthm cntlv-p-dispatch-cntlv
    (implies (MA-state-p s) (cntlv-p (dispatch-cntlv s)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (MA-state-p s) (integerp (dispatch-cntlv s)))
	     :hints (("Goal" :in-theory (enable cntlv-p))))))
(in-theory (disable dispatch-cntlv))

(defun dispatch-dest-reg (s)
  (declare (xargs :guard (MA-state-p s)))
  (DQ-out-dest-reg (MA-DQ s)))

(defthm rname-p-dispatch-dest-reg
    (implies (MA-state-p s) (rname-p (dispatch-dest-reg s)))
  :rule-classes
  ((:rewrite)
   (:rewrite :corollary
	     (implies (MA-state-p s) (integerp (dispatch-dest-reg s)))
	     :hints (("Goal" :in-theory (enable rname-p))))))
(in-theory (disable dispatch-dest-reg))

(defun dispatch-excpt (s)
  (declare (xargs :guard (MA-state-p s)))
  (DE-excpt (DQ-DE0 (MA-DQ s))))

(defthm excpt-flags-p-dispatch-excpt
    (implies (MA-state-p s) (excpt-flags-p (dispatch-excpt s))))
(in-theory (disable dispatch-excpt))

(defun dispatch-pc (s)
  (declare (xargs :guard (MA-state-p s)))
  (DE-pc (DQ-DE0 (MA-DQ s))))

(defthm addr-p-dispatch-pc
    (implies (MA-state-p s) (addr-p (dispatch-pc s))))
(in-theory (disable dispatch-pc))

(defun dispatch-ready1? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-ior (DQ-out-ready1? (MA-DQ s))
	  (ROBE-complete? (nth-ROBE (DQ-out-robe1 (MA-DQ s)) (MA-ROB s)))
	  (CDB-ready-for? (DQ-out-robe1 (MA-DQ s)) s)))

(defthm bitp-dispatch-ready1? (bitp (dispatch-ready1? s)))
(in-theory (disable dispatch-ready1?))


; The value for a dispatched instruction come from three possible
; sources: the register file, ROB and CDB.  If CQ-out-ready2 line is
; on, the value stored in the register file is the correct value.  If
; the complete? flag of the corresponding ROB entry is on, the correct
; value is redirected from the entry.  If the CDB outputs the result
; of an instruction on which the dispatched instruction truly
; depends, the value on CDB is read.
(defun dispatch-val1 (s)
  (declare (xargs :guard (MA-state-p s)))
  (b-if (DQ-out-ready1? (MA-DQ s))
	(DQ-read-val1 (MA-DQ s) s)
	(b-if (ROBE-complete? (nth-ROBE (DQ-out-robe1 (MA-DQ s)) (MA-ROB s)))
	      (ROBE-val (nth-ROBE (DQ-out-robe1 (MA-DQ s)) (MA-ROB s)))
	      (b-if (CDB-ready-for? (DQ-out-robe1 (MA-DQ s)) s)
		    (CDB-val s)
		    0))))

(defthm word-p-dispatch-val1
  (implies (MA-state-p s) (word-p (dispatch-val1 s))))
(in-theory (disable dispatch-val1))

(defun dispatch-src1 (s)
  (declare (xargs :guard (MA-state-p s)))
  (DQ-out-robe1 (MA-DQ s)))

(defthm rob-index-p-dispatch-src1
    (implies (MA-state-p s) (rob-index-p (dispatch-src1 s))))
(in-theory (disable dispatch-src1))

(defun dispatch-ready2? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-ior (DQ-out-ready2? (MA-DQ s))
	  (ROBE-complete? (nth-ROBE (DQ-out-robe2 (MA-DQ s)) (MA-ROB s)))
	  (CDB-ready-for? (DQ-out-robe2 (MA-DQ s)) s)))

(defthm bitp-dispatch-ready2  (bitp (dispatch-ready2? s)))
(in-theory (disable dispatch-ready2?))

; The value for a dispatched instruction come from three possible sources:
; the register file, ROB and CDB.  If DQ-out-ready2 line is on, the value
; stored in the register file is the correct value.  Otherwise, we have to
; get the value from the renaming register in ROB or CDB.
(defun dispatch-val2 (s)
  (declare (xargs :guard (MA-state-p s)))
  (b-if (DQ-out-ready2? (MA-DQ s))
	(read-reg (DQ-out-reg2 (MA-DQ s)) (MA-regs s))
	(b-if (ROBE-complete? (nth-ROBE (DQ-out-robe2 (MA-DQ s)) (MA-ROB s)))
	      (ROBE-val (nth-ROBE (DQ-out-robe2 (MA-DQ s)) (MA-ROB s)))
	      (b-if (CDB-ready-for? (DQ-out-robe2 (MA-DQ s)) s)
		    (CDB-val s)
		    0))))

(defthm word-p-dispatch-val2
    (implies (MA-state-p s) (word-p (dispatch-val2 s))))
(in-theory (disable dispatch-val2))

(defun dispatch-src2 (s)
  (declare (xargs :guard (MA-state-p s)))
  (DQ-out-robe2 (MA-DQ s)))

(defthm rob-index-p-dispatch-src2
    (implies (MA-state-p s) (rob-index-p (dispatch-src2 s))))
(in-theory (disable dispatch-src2))

(defun dispatch-ready3? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-ior (DQ-out-ready3? (MA-DQ s))
	  (ROBE-complete? (nth-ROBE (DQ-out-robe3 (MA-DQ s)) (MA-ROB s)))
	  (CDB-ready-for? (DQ-out-robe3 (MA-DQ s)) s)))

(defthm bitp-dispatch-ready3 (bitp (dispatch-ready3? s)))
(in-theory (disable dispatch-ready3?))

(defun dispatch-val3 (s)
  (declare (xargs :guard (MA-state-p s)))
  (b-if (DQ-out-ready3? (MA-DQ s))
	(read-reg (DQ-out-reg3 (MA-DQ s)) (MA-regs s))
	(b-if (ROBE-complete? (nth-ROBE (DQ-out-robe3 (MA-DQ s)) (MA-ROB s)))
	      (ROBE-val (nth-ROBE (DQ-out-robe3 (MA-DQ s)) (MA-ROB s)))
        (b-if (CDB-ready-for? (DQ-out-robe3 (MA-DQ s)) s)
	      (CDB-val s)
	      0))))

(defthm word-p-dispatch-val3
    (implies (MA-state-p s) (word-p (dispatch-val3 s))))
(in-theory (disable dispatch-val3))

(defun dispatch-src3 (s)
  (declare (xargs :guard (MA-state-p s)))
  (DQ-out-robe3 (MA-DQ s)))

(defthm rob-index-p-dispatch-src3
    (implies (MA-state-p s) (rob-index-p (dispatch-src3 s))))
(in-theory (disable dispatch-src3))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  Step functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(deflabel begin-MA-step-functions)
(defun step-pc (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (leave-excpt? s) (addr (sregs-sr0 (MA-sregs s)))
  (b-if (bs-ior (commit-jmp? s)
		(enter-excpt? s)
		(ex-intr? s orcl))
	(rob-jmp-addr (MA-ROB s) s orcl)
  (b-if (IFU-branch-predict? (MA-IFU s) s orcl)
	(IFU-branch-target (MA-IFU s))
  (b-if (b-and (b-nand (IFU-valid? (MA-IFU s)) (DQ-full? (MA-DQ s)))
	       (MA-oracle-fetch orcl))
	(addr (1+ (MA-pc s)))
	(MA-pc s))))))

(defthm addr-p-step-pc
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (addr-p (step-pc s orcl))))
(in-theory (disable step-pc))


(defun step-regs (s)
  (declare (xargs :guard (and (MA-state-p s))))
  (b-if (ROB-write-reg? (MA-ROB s))
	(write-reg (ROB-write-val (MA-ROB s) s)
		   (ROB-write-rid (MA-ROB s))
		   (MA-regs s))
	(MA-regs s)))

(defthm regs-p-step-regs
    (implies (MA-state-p s)
	     (regs-p (step-regs s))))
(in-theory (disable step-regs))


(defun step-sregs (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (leave-excpt? s)
	(update-sregs (MA-sregs s) :su (logcar (sregs-sr1 (MA-sregs s))))
  (b-if (ex-intr? s orcl)
	(sregs 1 (word (ex-intr-addr s)) (word (sregs-su (MA-sregs s))))
  (b-if (enter-excpt? s)
	(sregs 1 (ROB-write-val (MA-ROB s) s)
	       (word (sregs-su (MA-sregs s))))
  (b-if (ROB-write-sreg? (MA-ROB s))
	(write-sreg (ROB-write-val (MA-ROB s) s)
		    (ROB-write-rid (MA-ROB s))
		    (MA-sregs s))
	(MA-sregs s))))))

(defthm sregs-p-step-sregs
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (sregs-p (step-sregs s orcl))))
(in-theory (disable step-sregs))


(defun step-IFU (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-and (IFU-valid? (MA-IFU s)) (DQ-full? (MA-DQ s)))
	(update-IFU (MA-IFU s)
		    :valid? (b-not (flush-all? s orcl)))
  (b-if (IFU-fetch-prohibited? (MA-pc s) (MA-mem s) (sregs-su (MA-sregs s)))
	(IFU (b-nor (IFU-branch-predict? (MA-IFU s) s orcl)
		    (flush-all? s orcl))
	     #b101 (MA-pc s) 0)
  (b-if (MA-oracle-fetch orcl)
	(IFU (b-nor (IFU-branch-predict? (MA-IFU s) s orcl)
		    (flush-all? s orcl))
	     0 (MA-pc s) (read-mem (MA-pc s) (MA-mem s)))
	(IFU 0 0 0 0)))))

(defthm IFU-p-step-IFU
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (IFU-p (step-IFU s orcl))))
(in-theory (disable step-IFU))


(defun DE3-out (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (DE-valid? (DQ-DE3 DQ))
	(DQ-DE3 DQ)
	(decode-output (MA-IFU s) s orcl)))

(defthm dispatch-entry-p-DE3-out
    (implies (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))
	     (dispatch-entry-p (DE3-out DQ s orcl))))

(in-theory (disable DE3-out))


(defun DE2-out (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (DE-valid? (DQ-DE2 DQ))
	(DQ-DE2 DQ)
	(DE3-out DQ s orcl)))

(defthm dispatch-entry-p-DE2-out
    (implies (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))
	     (dispatch-entry-p (DE2-out DQ s orcl))))

(in-theory (disable DE2-out))


(defun DE1-out (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (DE-valid? (DQ-DE1 DQ))
	(DQ-DE1 DQ)
	(DE2-out DQ s orcl)))

(defthm dispatch-entry-p-DE1-out
    (implies (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))
	     (dispatch-entry-p (DE1-out DQ s orcl))))

(in-theory (disable DE1-out))

(defun step-DE0 (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-orc1 (DE-valid? (DQ-DE0 DQ)) (dispatch-inst? s))
	(update-dispatch-entry (DE1-out DQ s orcl)
		   :valid? (b-andc2 (DE-valid? (DE1-out DQ s orcl))
				    (flush-all? s orcl)))
	(update-dispatch-entry (DQ-DE0 DQ)
		   :valid? (b-andc2 (DE-valid? (DQ-DE0 DQ))
				    (flush-all? s orcl)))))

(defun step-DE1 (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-ior (b-and (DE-valid? (DQ-DE1 DQ)) (dispatch-inst? s))
	       (bs-and (b-not (DE-valid? (DQ-DE1 DQ)))
		       (DE-valid? (DQ-DE0 DQ))
		       (b-not (dispatch-inst? s))))
	(update-dispatch-entry (DE2-out DQ s orcl)
		   :valid? (b-andc2 (DE-valid? (DE2-out DQ s orcl))
				    (flush-all? s orcl)))
	(update-dispatch-entry (DQ-DE1 DQ)
		   :valid? (b-andc2 (DE-valid? (DQ-DE1 DQ))
				    (flush-all? s orcl)))))

(defun step-DE2 (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-ior (b-and (DE-valid? (DQ-DE2 DQ)) (dispatch-inst? s))
	       (bs-and (b-not (DE-valid? (DQ-DE2 DQ)))
		       (DE-valid? (DQ-DE1 DQ))
		       (b-not (dispatch-inst? s))))
	(update-dispatch-entry (DE3-out DQ s orcl)
		   :valid? (b-andc2 (DE-valid? (DE3-out DQ s orcl))
				    (flush-all? s orcl)))
	(update-dispatch-entry (DQ-DE2 DQ)
		   :valid? (b-andc2 (DE-valid? (DQ-DE2 DQ))
				    (flush-all? s orcl)))))

(defun step-DE3 (DQ s orcl)
  (declare (xargs :guard (and (DQ-p DQ) (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-and (DE-valid? (DQ-DE3 DQ)) (dispatch-inst? s))
	(dispatch-entry 0 0 0 0 0 0 0 0 0)
  (b-if (bs-and (b-not (DE-valid? (DQ-DE3 DQ)))
		(DE-valid? (DQ-DE2 DQ))
		(b-not (dispatch-inst? s)))
	(update-dispatch-entry (decode-output (MA-IFU s) s orcl)
	   :valid? (b-andc2 (DE-valid? (decode-output (MA-IFU s) s orcl))
			    (flush-all? s orcl)))
	(update-dispatch-entry (DQ-DE3 DQ)
		   :valid? (b-andc2 (DE-valid? (DQ-DE3 DQ))
				    (flush-all? s orcl))))))

(defun step-reg-ref (reg-ref idx s orcl)
  (declare (xargs :guard (and (reg-ref-p reg-ref)
			      (rname-p idx)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (reg-ref (b-if (flush-all? s orcl)
		    0
	        (b-if (bs-and (dispatch-inst? s)
			      (cntlv-wb? (dispatch-cntlv s))
			      (b-not (cntlv-wb-sreg? (dispatch-cntlv s)))
			      (bv-eqv *rname-size*
				      (dispatch-dest-reg s)
				      idx))
		      1
		    (b-if (bs-and (ROB-write-reg? (MA-ROB s))
				  (bv-eqv *rname-size*
					  (ROB-write-rid (MA-ROB s))
					  idx)
				  (bv-eqv *rname-size*
					  (ROB-head (MA-ROB s))
					  (reg-ref-robe reg-ref)))
			  0 (reg-ref-wait? reg-ref))))
	      (b-if (bs-and (dispatch-inst? s)
			    (cntlv-wb? (dispatch-cntlv s))
			    (b-not (cntlv-wb-sreg? (dispatch-cntlv s)))
			    (bv-eqv *rname-size*
				    (dispatch-dest-reg s)
				    idx))
		    (ROB-tail (MA-ROB s))
		    (reg-ref-robe reg-ref))))

(defun step-reg-list (r-list idx s orcl)
  (declare (xargs :guard (and (reg-ref-listp r-list)
			      (rname-p idx)
			      (MA-state-p s)
			      (MA-oracle-p orcl))))
  (if (endp r-list)
      nil
      (cons (step-reg-ref (car r-list) idx s orcl)
	    (step-reg-list (cdr r-list) (rname (1+ idx)) s orcl))))

(defun step-reg-tbl  (r-list s orcl)
  (declare (xargs :guard (and (reg-tbl-p r-list)
			      (MA-state-p s)
			      (MA-oracle-p orcl))
		  :guard-hints (("Goal" :in-theory (enable reg-tbl-p)))))
  (step-reg-list r-list 0 s orcl))

(defthm reg-ref-listp-step-reg-list
    (implies (and (reg-ref-listp reg-tbl) (MA-state-p s) (MA-oracle-p orcl))
	     (reg-ref-listp (step-reg-list reg-tbl idx s orcl))))

(defthm len-step-reg-list
    (equal (len (step-reg-list reg-tbl idx s orcl))
	   (len reg-tbl)))

(defthm reg-tbl-p-step-reg-tbl
    (implies (and (reg-tbl-p reg-tbl)
		  (MA-state-p s) (MA-oracle-p orcl))
	     (reg-tbl-p (step-reg-tbl reg-tbl s orcl)))
  :hints (("goal" :in-theory (enable reg-tbl-p))))

(in-theory (disable step-reg-tbl))


(defun step-sreg-ref (sreg-ref idx s orcl)
  (declare (xargs :guard (and (reg-ref-p sreg-ref)
			      (rname-p idx)
			      (MA-state-p s)
			      (MA-oracle-p orcl))))
  (reg-ref (b-if (flush-all? s orcl) 0
		(b-if (bs-and (dispatch-inst? s)
			      (cntlv-wb? (dispatch-cntlv s))
			      (cntlv-wb-sreg? (dispatch-cntlv s))
			      (bv-eqv *rname-size*
				      (dispatch-dest-reg s)
				      idx))
		      1
		    (b-if (bs-and (ROB-write-sreg? (MA-ROB s))
				  (bv-eqv *rname-size*
					  (ROB-write-rid (MA-ROB s))
					  idx)
				  (bv-eqv *rname-size*
					  (ROB-head (MA-ROB s))
					  (reg-ref-robe sreg-ref)))
			  0
			  (reg-ref-wait? sreg-ref))))
	      (b-if (bs-and (dispatch-inst? s)
			    (cntlv-wb? (dispatch-cntlv s))
			    (cntlv-wb-sreg? (dispatch-cntlv s))
			    (bv-eqv *rname-size*
				    (dispatch-cntlv s)
				    idx))
		    (ROB-tail (MA-ROB s))
		    (reg-ref-robe sreg-ref))))

(defun step-sreg-tbl (sreg-tbl s orcl)
  (declare (xargs :guard (and (sreg-tbl-p sreg-tbl) (MA-state-p s)
			      (MA-oracle-p orcl))))
  (sreg-tbl (step-sreg-ref (sreg-tbl-sr0 sreg-tbl) 0 s orcl)
	    (step-sreg-ref (sreg-tbl-sr1 sreg-tbl) 1 s orcl)))

(defun step-DQ (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (DQ (step-DE0 (MA-DQ s) s orcl)
      (step-DE1 (MA-DQ s) s orcl)
      (step-DE2 (MA-DQ s) s orcl)
      (step-DE3 (MA-DQ s) s orcl)
      (step-reg-tbl (DQ-reg-tbl (MA-DQ s)) s orcl)
      (step-sreg-tbl (DQ-sreg-tbl (MA-DQ s)) s orcl)))

(defthm DQ-p-step-DQ
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (DQ-p (step-DQ s orcl))))
(in-theory (disable step-DQ))


;Commit-inst? is on when the instruction at the head of the ROB commits.
; If the instruction at the head of the ROB is an exception causing
; instruction or synchronizing instruction, we check whether there are
; pending writes in the write buffer before allowing it to commit.
(defun commit-inst? (s)
  (declare (xargs :guard (MA-state-p s)))
  (let ((ROB (MA-ROB s))
	(LSU (MA-LSU s)))
    (let ((robe (nth-robe (ROB-head ROB) ROB)))
      (bs-and (ROBE-valid? robe)
	      (ROBE-complete? robe)
	      (b-nand (b-ior (excpt-raised? (ROBE-excpt robe))
			     (ROBE-sync? robe))
		      (LSU-pending-writes? LSU))))))

(defthm bitp-commit-inst? (bitp (commit-inst? s)))
(in-theory (disable commit-inst?))

(defun ROBE-receive-inst? (ROB index s)
  (declare (xargs :guard (and (ROB-p rob) (rob-index-p index) (MA-state-p s))))
  (b-and (dispatch-inst? s)
	 (bv-eqv *rob-index-size* (ROB-tail ROB) index)))

(defthm bitp-ROBE-receive-inst?
    (bitp (ROBE-receive-inst? ROB index s)))
(in-theory (disable ROBE-receive-inst?))

(defun ROBE-receive-result? (ROB index s)
  (declare (xargs :guard (and (ROB-p rob) (rob-index-p index)
			      (MA-state-p s))))
  (bs-and (ROBE-valid? (nth-ROBE index ROB))
	  (CDB-ready-for? index s)))

(defthm bitp-ROBE-receive-result?
    (bitp (ROBE-receive-result? ROB index s)))
(in-theory (disable ROBE-receive-result?))

(defun step-ROBE (robe idx ROB s orcl)
  (declare (xargs :guard (and (ROB-entry-p robe)
			      (rob-index-p idx)
			      (ROB-p ROB) (MA-state-p s) (MA-oracle-p orcl))))
  (ROB-entry (b-if (b-ior (b-and (commit-inst? s)
				 (bv-eqv *rob-index-size* (ROB-head ROB) idx))
			  (flush-all? s orcl))
		   0
		   (b-if (robe-receive-inst? ROB idx s)
			 1
			 (ROBE-valid? robe))) ; valid?
	     (b-if (robe-receive-inst? ROB idx s)
		   (dispatch-no-unit? s)
		   (b-ior (ROBE-complete? robe)
			  (robe-receive-result? rob idx s))) ;complete?
	     (b-if (robe-receive-inst? rob idx s)
		   (dispatch-excpt s)
		   (b-if (robe-receive-result? rob idx s)
			 (CDB-excpt s)
			 (ROBE-excpt robe))) ;excpt
	     (b-if (robe-receive-inst? rob idx s)
		   (cntlv-wb? (dispatch-cntlv s))
		   (ROBE-wb? robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (cntlv-wb-sreg? (dispatch-cntlv s))
		   (ROBE-wb-sreg? robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (cntlv-sync? (dispatch-cntlv s))
		   (ROBE-sync? robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (logbit 3 (cntlv-exunit (dispatch-cntlv s)))
		   (ROBE-branch? robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (cntlv-rfeh? (dispatch-cntlv s))
		   (ROBE-rfeh? robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (cntlv-br-predict? (dispatch-cntlv s))
		   (ROBE-br-predict? robe))
	     (b-if (b-and (robe-receive-result? rob idx s)
			  (ROBE-branch? robe))
		   (logcar (CDB-val s))
		   (ROBE-br-actual? robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (dispatch-pc s)
		   (ROBE-pc robe))
	     (b-if (robe-receive-inst? ROB idx s)
		   (word (DE-br-target (DQ-DE0 (MA-DQ s))))
		   (b-if (b-andc2 (robe-receive-result? rob idx s)
				  (ROBE-branch? robe))
			 (CDB-val s)
			 (ROBE-val robe)))
	     (b-if (robe-receive-inst? ROB idx s)
		   (dispatch-dest-reg s)
		   (ROBE-dest robe))))

(defthm ROBE-p-step-robe
    (implies (and (ROB-entry-p robe)
		  (ROB-p ROB)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (ROB-entry-p (step-robe robe index ROB s orcl))))
(in-theory (disable step-robe))

(defun step-ROBE-list (entries index ROB s orcl)
  (declare (xargs :guard (and (ROBE-listp entries)
			      (rob-index-p index)
			      (ROB-p ROB) (MA-state-p s) (MA-oracle-p orcl))))
  (if (endp entries)
      nil
      (cons (step-ROBE (car entries) index ROB s orcl)
	    (step-ROBE-list (cdr entries) (rob-index (1+ index))
			    ROB s orcl))))

(defun step-ROB-entries (entries ROB s orcl)
  (declare (xargs :guard (and (ROB-entries-p entries)
			      (ROB-p ROB) (MA-state-p s) (MA-oracle-p orcl))
		  :guard-hints (("Goal" :in-theory (enable ROB-entries-p)))))
  (step-ROBE-list entries 0 ROB s orcl))

(defthm robe-lisp-step-robe-list
    (implies (and (robe-listp entries)
		  (rob-p rob)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (ROBE-listp (step-robe-list entries index rob s orcl))))

(defthm len-step-ROBE-list
    (equal (len (step-ROBE-list entries index ROB s orcl))
	   (len entries)))

(defthm ROB-entries-p-step-ROB-entries
    (implies (and (ROB-entries-p entries)
		  (ROB-p ROB)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (ROB-entries-p (step-ROB-entries entries ROB s orcl)))
  :hints (("Goal" :in-theory (enable rob-entries-p))))
(in-theory (disable step-ROB-entries))

(defun step-ROB (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (ROB (b-if (flush-all? s orcl) 0
	     (b-xor (ROB-flg (MA-ROB s))
		    (b-xor (b-and (commit-inst? s)
				  (logbit *rob-index-size*
					  (+ 1 (ROB-head (MA-ROB s)))))
			   (b-and (dispatch-inst? s)
				  (logbit *rob-index-size*
					  (+ 1 (ROB-tail (MA-ROB s))))))))
       (b-if (b-ior (enter-excpt? s) (ex-intr? s orcl)) 0
	     (b-ior (MA-oracle-exintr orcl)
		    (ROB-exintr? (MA-ROB s))))
       (b-if (flush-all? s orcl) 0
	     (b-if (commit-inst? s)
		   (rob-index (+ 1 (ROB-head (MA-ROB s))))
		   (ROB-head (MA-ROB s))))
       (b-if (flush-all? s orcl) 0
	     (b-if (dispatch-inst? s)
		   (rob-index (+ 1 (ROB-tail (MA-ROB s))))
		   (ROB-tail (MA-ROB s))))
       (step-ROB-entries (ROB-entries (MA-ROB s)) (MA-ROB s) s orcl)))

(defthm ROB-p-step-ROB
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (ROB-p (step-ROB s orcl))))
(in-theory (disable step-ROB))

(defun step-IU-RS0 (iu s orcl)
  (declare (xargs :guard (and (integer-unit-p IU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS0 (IU-RS0 iu)))
    (RS (b-andc1 (flush-all? s orcl)
		 (b-if (RS-valid? rs0)
		       (b-not (issue-IU-RS0? iu s))
		       (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (b-not (logbit 0 (cntlv-operand (dispatch-cntlv s))))
	      (RS-op RS0))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (ROB-tail (MA-ROB s))
	      (RS-dest RS0))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (dispatch-ready1? s)
	      (b-ior (RS-ready1? RS0)
		     (CDB-ready-for? (RS-src1 RS0) s)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (dispatch-ready2? s)
	      (b-ior (RS-ready2? RS0)
		     (CDB-ready-for? (RS-src2 RS0) s)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (dispatch-val1 s)
	      (b-if (b-andc1 (RS-ready1? RS0) (CDB-ready-for? (RS-src1 RS0) s))
		    (CDB-val s)
		    (RS-val1 RS0)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (dispatch-val2 s)
	      (b-if (b-andc1 (RS-ready2? RS0) (CDB-ready-for? (RS-src2 RS0) s))
		    (CDB-val s)
		    (RS-val2 RS0)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (dispatch-src1 s)
	      (RS-src1 RS0))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS0? iu))
	      (dispatch-src2 s)
	      (RS-src2 RS0)))))

(defthm RS-p-step-IU-RS0
    (implies (and (integer-unit-p IU) (MA-state-p s) (MA-oracle-p orcl))
	     (RS-p (step-IU-RS0 IU s orcl))))
(in-theory (disable step-IU-RS0))


(defun step-IU-RS1 (iu s orcl)
  (declare (xargs :guard (and (integer-unit-p IU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS1 (IU-RS1 iu)))
    (RS (b-andc1 (flush-all? s orcl)
		 (b-if (RS-valid? rs1)
		       (b-not (issue-IU-RS1? iu s))
		       (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (b-not (logbit 0 (cntlv-operand (dispatch-cntlv s))))
	      (RS-op RS1))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (ROB-tail (MA-ROB s))
	      (RS-dest RS1))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (dispatch-ready1? s)
	      (b-ior (RS-ready1? RS1)
		     (CDB-ready-for? (RS-src1 RS1) s)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (dispatch-ready2? s)
	      (b-ior (RS-ready2? RS1)
		     (CDB-ready-for? (RS-src2 RS1) s)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (dispatch-val1 s)
	      (b-if (b-andc1 (RS-ready1? RS1) (CDB-ready-for? (RS-src1 RS1) s))
		    (CDB-val s)
		    (RS-val1 RS1)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (dispatch-val2 s)
	      (b-if (b-andc1 (RS-ready2? RS1) (CDB-ready-for? (RS-src2 RS1) s))
		    (CDB-val s)
		    (RS-val2 RS1)))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (dispatch-src1 s)
	      (RS-src1 RS1))
	(b-if (b-and (dispatch-to-IU? s) (select-IU-RS1? iu))
	      (dispatch-src2 s)
	      (RS-src2 RS1)))))

(defthm RS-p-step-IU-RS1
    (implies (and (integer-unit-p IU) (MA-state-p s) (MA-oracle-p orcl))
	     (RS-p (step-IU-RS1 IU s orcl))))
(in-theory (disable step-IU-RS1))


(defun step-IU (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (integer-unit (step-IU-RS0 (MA-IU s) s orcl)
		(step-IU-RS1 (MA-IU s) s orcl)))

(defthm integer-unit-p-step-IU
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (integer-unit-p (step-IU s orcl))))
(in-theory (disable step-IU))


(defun step-MU-RS0 (MU s orcl)
  (declare (xargs :guard (and (mult-unit-p MU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS0 (MU-RS0 MU)))
    (RS (b-andc1 (flush-all? s orcl)
		 (b-if (RS-valid? rs0)
		       (b-not (issue-MU-RS0? MU s))
		       (b-and (dispatch-to-MU? s)
			      (select-MU-RS0? MU))))
	0
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (ROB-tail (MA-ROB s))
	      (RS-dest RS0))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (dispatch-ready1? s)
	      (b-ior (RS-ready1? RS0)
		     (CDB-ready-for? (RS-src1 RS0) s)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (dispatch-ready2? s)
	      (b-ior (RS-ready2? RS0)
		     (CDB-ready-for? (RS-src2 RS0) s)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (dispatch-val1 s)
	      (b-if (b-andc1 (RS-ready1? RS0) (CDB-ready-for? (RS-src1 RS0) s))
		    (CDB-val s)
		    (RS-val1 RS0)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (dispatch-val2 s)
	      (b-if (b-andc1 (RS-ready2? RS0) (CDB-ready-for? (RS-src2 RS0) s))
		    (CDB-val s)
		    (RS-val2 RS0)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (dispatch-src1 s)
	      (RS-src1 RS0))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS0? MU))
	      (dispatch-src2 s)
	      (RS-src2 RS0)))))

(defun step-MU-RS1 (MU s orcl)
  (declare (xargs :guard (and (mult-unit-p MU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS1 (MU-RS1 MU)))
    (RS (b-andc1 (flush-all? s orcl)
		 (b-if (RS-valid? rs1)
		       (b-not (issue-MU-RS1? MU s))
		       (b-and (dispatch-to-MU? s)
			      (select-MU-RS1? MU))))
	0
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (ROB-tail (MA-ROB s))
	      (RS-dest RS1))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (dispatch-ready1? s)
	      (b-ior (RS-ready1? RS1)
		     (CDB-ready-for? (RS-src1 RS1) s)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (dispatch-ready2? s)
	      (b-ior (RS-ready2? RS1)
		     (CDB-ready-for? (RS-src2 RS1) s)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (dispatch-val1 s)
	      (b-if (b-andc1 (RS-ready1? RS1) (CDB-ready-for? (RS-src1 RS1) s))
		    (CDB-val s)
		    (RS-val1 RS1)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (dispatch-val2 s)
	      (b-if (b-andc1 (RS-ready2? RS1) (CDB-ready-for? (RS-src2 RS1) s))
		    (CDB-val s)
		    (RS-val2 RS1)))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (dispatch-src1 s)
	      (RS-src1 RS1))
	(b-if (b-and (dispatch-to-MU? s) (select-MU-RS1? MU))
	      (dispatch-src2 s)
	      (RS-src2 RS1)))))

(defun step-MU-lch1 (MU s orcl)
  (declare (xargs :guard (and (mult-unit-p MU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-ior (CDB-for-MU? s)
	       (b-not (MU-latch-valid? (MU-lch2 MU))))
	(b-if (issue-MU-RS0? MU s)
	      (MU-latch (b-not (flush-all? s orcl))
			(RS-dest (MU-RS0 MU))
			(ML1-output (RS-val1 (MU-RS0 MU))
				    (RS-val2 (MU-RS0 MU))))
	  (b-if (issue-MU-RS1? MU s)
		(MU-latch (b-not (flush-all? s orcl))
			  (RS-dest (MU-RS1 MU))
			    (ML1-output (RS-val1 (MU-RS1 MU))
					(RS-val2 (MU-RS1 MU))))
		(MU-latch 0 0 0)))
	(update-MU-latch (MU-lch1 MU)
		   :valid? (b-andc1 (flush-all? s orcl)
				    (MU-latch-valid? (MU-lch1 MU))))))

(defun step-MU-lch2 (MU s orcl)
  (declare (xargs :guard (and (mult-unit-p MU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-orc2 (CDB-for-MU? s)
		(MU-latch-valid? (MU-lch2 MU)))
	(MU-latch (b-and (MU-latch-valid? (MU-lch1 MU))
			 (b-not (flush-all? s orcl)))
		  (MU-latch-dest (MU-lch1 MU))
		  (ML2-output (MU-latch-data (MU-lch1 MU))))
	(update-MU-latch (MU-lch2 MU)
	   :valid? (b-andc1 (flush-all? s orcl)
			    (MU-latch-valid? (MU-lch2 MU))))))

(defun step-MU (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (mult-unit (step-MU-RS0 (MA-MU s) s orcl)
	     (step-MU-RS1 (MA-MU s) s orcl)
	     (step-MU-lch1 (MA-MU s) s orcl)
	     (step-MU-lch2 (MA-MU s) s orcl)))


(defthm mult-unit-p-step-MU
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (mult-unit-p (step-MU s orcl))))
(in-theory (disable step-MU))


(defun step-BU-RS0 (BU s orcl)
  (declare (xargs :guard (and (branch-unit-p BU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS0 (BU-RS0 BU)))
    (BU-RS (b-and (b-not (flush-all? s orcl))
		  (b-if (BU-RS-valid? RS0)
			(b-not (issue-BU-RS0? BU s))
			(b-and (dispatch-to-BU? s) (select-BU-RS0? BU))))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS0? BU))
		 (ROB-tail (MA-ROB s))
		 (BU-RS-dest RS0))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS0? BU))
		 (dispatch-ready3? s)
		 (b-ior (BU-RS-ready? RS0)
			(CDB-ready-for? (BU-RS-src RS0) s)))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS0? BU))
		 (dispatch-val3 s)
		 (b-if (b-andc1 (BU-RS-ready? RS0)
				(CDB-ready-for? (BU-RS-src RS0) s))
		       (CDB-val s)
		       (BU-RS-val RS0)))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS0? BU))
		 (dispatch-src3 s)
		 (BU-RS-src RS0)))))

(defun step-BU-RS1 (BU s orcl)
  (declare (xargs :guard (and (branch-unit-p BU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS1 (BU-RS1 BU)))
    (BU-RS (b-and (b-not (flush-all? s orcl))
		  (b-if (BU-RS-valid? RS1)
			(b-not (issue-BU-RS1? BU s))
			(b-and (dispatch-to-BU? s) (select-BU-RS1? BU))))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS1? BU))
		 (ROB-tail (MA-ROB s))
		 (BU-RS-dest RS1))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS1? BU))
		 (dispatch-ready3? s)
		 (b-ior (BU-RS-ready? RS1)
			(CDB-ready-for? (BU-RS-src RS1) s)))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS1? BU))
		 (dispatch-val3 s)
		 (b-if (b-andc1 (BU-RS-ready? RS1)
				(CDB-ready-for? (BU-RS-src RS1) s))
		       (CDB-val s)
		       (BU-RS-val RS1)))
	   (b-if (b-and (dispatch-to-BU? s) (select-BU-RS1? BU))
		 (dispatch-src3 s)
		 (BU-RS-src RS1)))))

(defun step-BU (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (branch-unit (step-BU-RS0 (MA-BU s) s orcl)
	       (step-BU-RS1 (MA-BU s) s orcl)))


(defthm branch-unit-p-step-BU
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (branch-unit-p (step-BU s orcl))))
(in-theory (disable step-BU))


(defun step-rs1-head? (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (issue-LSU-RS0? LSU s orcl) 1
  (b-if (issue-LSU-RS1? LSU s orcl) 0
	(LSU-rs1-head? LSU))))

(defthm bitp-step-rs1-head?
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s) (MA-oracle-p orcl))
	     (bitp (step-rs1-head? LSU s orcl))))
(in-theory (disable step-rs1-head?))

; Updating reservation stations in the memory unit.  Remember
; reservation stations in memory unit forms a queue as our machine
; accesses memory in-order.  RS0 is the head of the queue.
(defun step-LSU-RS0 (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS0 (LSU-RS0 LSU)))
    (LSU-RS (b-and (b-not (flush-all? s orcl))
		   (b-if (LSU-RS-valid? RS0)
			 (b-not (issue-LSU-RS0? LSU s orcl))
			 (b-and (dispatch-to-LSU? s)
				(select-LSU-RS0? LSU))))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (logbit 1 (cntlv-operand (dispatch-cntlv s)))
		  (LSU-RS-op RS0))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (cntlv-ld-st? (dispatch-cntlv s))
		  (LSU-RS-ld-st? RS0))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (ROB-tail (MA-ROB s))
		  (LSU-RS-dest RS0))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-ready3? s)
		  (b-ior (LSU-RS-rdy3? RS0)
			 (CDB-ready-for? (LSU-RS-src3 RS0) s)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-val3 s)
		  (b-if (CDB-ready-for? (LSU-RS-src3 RS0) s)
			(CDB-val s)
			(LSU-RS-val3 RS0)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-src3 s)
		  (LSU-RS-src3 RS0))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-ready1? s)
		  (b-ior (LSU-RS-rdy1? RS0)
			 (CDB-ready-for? (LSU-RS-src1 RS0) s)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-val1 s)
		  (b-if (b-andc1 (LSU-RS-rdy1? RS0)
				 (CDB-ready-for? (LSU-RS-src1 RS0) s))
			(CDB-val s)
			(LSU-RS-val1 RS0)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-src1 s)
		  (LSU-RS-src1 RS0))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-ready2? s)
		  (bs-ior (LSU-RS-rdy2? RS0)
			  (CDB-ready-for? (LSU-RS-src2 RS0) s)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-val2 s)
		  (b-if (b-andc1 (LSU-RS-rdy2? RS0)
				 (CDB-ready-for? (LSU-RS-src2 RS0) s))
			(CDB-val s)
			(LSU-RS-val2 RS0)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS0? LSU))
		  (dispatch-src2 s)
		  (LSU-RS-src2 RS0)))))

(defthm LSU-RS-p-step-LSU-RS0
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (LSU-RS-p (step-LSU-RS0 LSU s orcl))))
(in-theory (disable step-LSU-RS0))

(defun step-LSU-RS1 (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS1 (LSU-RS1 LSU)))
    (LSU-RS (b-and (b-not (flush-all? s orcl))
		   (b-if (LSU-RS-valid? RS1)
			 (b-not (issue-LSU-RS1? LSU s orcl))
			 (b-and (dispatch-to-LSU? s)
				(select-LSU-RS1? LSU))))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (logbit 1 (cntlv-operand (dispatch-cntlv s)))
		  (LSU-RS-op RS1))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (cntlv-ld-st? (dispatch-cntlv s))
		  (LSU-RS-ld-st? RS1))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (ROB-tail (MA-ROB s))
		  (LSU-RS-dest RS1))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-ready3? s)
		  (b-ior (LSU-RS-rdy3? RS1)
			 (CDB-ready-for? (LSU-RS-src3 RS1) s)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-val3 s)
		  (b-if (CDB-ready-for? (LSU-RS-src3 RS1) s)
			(CDB-val s)
			(LSU-RS-val3 RS1)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-src3 s)
		  (LSU-RS-src3 RS1))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-ready1? s)
		  (b-ior (LSU-RS-rdy1? RS1)
			 (CDB-ready-for? (LSU-RS-src1 RS1) s)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-val1 s)
		  (b-if (b-andc1 (LSU-RS-rdy1? RS1)
				 (CDB-ready-for? (LSU-RS-src1 RS1) s))
			(CDB-val s)
			(LSU-RS-val1 RS1)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-src1 s)
		  (LSU-RS-src1 RS1))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-ready2? s)
		  (b-ior (LSU-RS-rdy2? RS1)
			 (CDB-ready-for? (LSU-RS-src2 RS1) s)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-val2 s)
		  (b-if (b-andc1 (LSU-RS-rdy2? RS1)
				 (CDB-ready-for? (LSU-RS-src2 RS1) s))
			(CDB-val s)
			(LSU-RS-val2 RS1)))
	    (b-if (b-and (dispatch-to-LSU? s) (select-LSU-RS1? LSU))
		  (dispatch-src2 s)
		  (LSU-RS-src2 RS1)))))

(defthm LSU-RS-p-step-LSU-RS1
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (LSU-RS-p (step-LSU-RS1 LSU s orcl))))
(in-theory (disable step-LSU-RS1))

(defun step-rbuf (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS0 (LSU-RS0 LSU)) (RS1 (LSU-RS1 LSU)))
    (b-if (bs-and (rbuf-valid? (LSU-rbuf LSU))
		  (b-not (LSU-address-match? LSU))
		  (b-not (LSU-read-prohibited? LSU (MA-mem s)
					       (sregs-su (MA-sregs s))))
		  (LSU-read-stall? LSU orcl))
	  (update-read-buffer (LSU-rbuf LSU)
		:valid? (b-not (flush-all? s orcl)))
	  (b-if (b-andc2 (issue-LSU-RS0? LSU s orcl)
			 (LSU-RS-ld-st? RS0))
		(read-buffer (b-not (flush-all? s orcl))
			     (LSU-RS-dest RS0)
			     (b-if (LSU-RS-op RS0)
				   (addr (LSU-RS-val1 RS0))
				   (addr (+ (LSU-RS-val1 RS0)
					    (LSU-RS-val2 RS0)))))
	  (b-if (b-andc2 (issue-LSU-RS1? LSU s orcl)
			 (LSU-RS-ld-st? RS1))
		(read-buffer (b-not (flush-all? s orcl))
			     (LSU-RS-dest RS1)
			     (b-if (LSU-RS-op RS1)
				   (addr (LSU-RS-val1 RS1))
				   (addr (+ (LSU-RS-val1 RS1)
					    (LSU-RS-val2 RS1)))))
		(read-buffer 0 0 0))))))

(defthm read-buffer-p-step-rbuf
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (read-buffer-p (step-rbuf LSU s orcl))))
(in-theory (disable step-rbuf))

(defun step-LSU-lch (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (LSU-latch (b-and (b-not (flush-all? s orcl))
		    (bs-ior (b-and (rbuf-valid? (LSU-rbuf LSU))
				   (bs-ior (LSU-address-match? LSU)
					   (LSU-read-prohibited? LSU (MA-mem s)
					       (sregs-su (MA-sregs s)))
					   (b-not (LSU-read-stall? LSU orcl))))
			    (check-wbuf0? LSU)
			    (check-wbuf1? LSU)))
	     (b-if (b-ior (b-and (rbuf-valid? (LSU-rbuf LSU))
				 (LSU-read-prohibited? LSU (MA-mem s)
						    (sregs-su (MA-sregs s))))
			  (b-and (b-ior (check-wbuf0? LSU)
					(check-wbuf1? LSU))
				 (LSU-write-prohibited? LSU (MA-mem s)
						(sregs-su (MA-sregs s)))))
		   #b110 0)
	     (b-if (rbuf-valid? (LSU-rbuf LSU)) (rbuf-dest (LSU-rbuf LSU))
		   (b-if (check-wbuf0? LSU) (wbuf-dest (LSU-wbuf0 LSU))
			 (b-if (check-wbuf1? LSU) (wbuf-dest (LSU-wbuf1 LSU))
			       0)))
	     (b-if (b-and (rbuf-valid? (LSU-rbuf LSU))
			  (LSU-address-match? LSU))
		   (LSU-forward-wbuf LSU)
		   (b-if (bs-and (rbuf-valid? (LSU-rbuf LSU))
				 (b-not (LSU-read-prohibited? LSU (MA-mem s)
						 (sregs-su (MA-sregs s))))
				 (b-not (LSU-read-stall? LSU orcl)))
			 (read-mem (rbuf-addr (LSU-rbuf LSU)) (MA-mem s))
			 0))))

(defthm LSU-latch-p-step-LSU-lch
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (LSU-latch-p (step-LSU-lch LSU s orcl))))
(in-theory (disable step-LSU-lch))

(defun issued-write (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((RS0 (LSU-RS0 LSU)) (RS1 (LSU-RS1 LSU)))
    (b-if (issue-LSU-RS0? LSU s orcl)
	  (write-buffer (b-andc2 (LSU-RS-ld-st? RS0)
				 (flush-all? s orcl))
			0 0 (LSU-RS-dest RS0)
			(b-if (LSU-RS-op RS0)
			      (addr (LSU-RS-val1 RS0))
			      (addr (+ (LSU-RS-val1 RS0) (LSU-RS-val2 RS0))))
			(LSU-RS-val3 RS0))
    (b-if (issue-LSU-RS1? LSU s orcl)
	  (write-buffer (b-andc2 (LSU-RS-ld-st? RS1)
				 (flush-all? s orcl))
			0 0 (LSU-RS-dest RS1)
			(b-if (LSU-RS-op RS1)
			      (addr (LSU-RS-val1 RS1))
			      (addr (+ (LSU-RS-val1 RS1) (LSU-RS-val2 RS1))))
			(LSU-RS-val3 RS1))
	  (write-buffer 0 0 0 0 0 0)))))

(defthm write-buffer-p-issued-write
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (write-buffer-p (issued-write LSU s orcl))))
(in-theory (disable issued-write))

(defun wbuf1-output (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((wbuf1 (LSU-wbuf1 LSU)))
    (b-if (wbuf-valid? wbuf1)
	  (write-buffer
	   (b-andc2 (wbuf-valid? wbuf1)
		    (flush-all? s orcl))
	   (wbuf-complete? wbuf1)
	   (b-ior (wbuf-commit? wbuf1)
		  (b-and (commit-inst? s)
			 (bv-eqv *rob-index-size*
				 (ROB-head (MA-ROB s)) (wbuf-dest wbuf1))))
	   (wbuf-dest wbuf1)
	   (wbuf-addr wbuf1)
	   (wbuf-val wbuf1))
	  (issued-write LSU s orcl))))

(defthm write-buffer-p-wbuf1-output
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (write-buffer-p (wbuf1-output LSU s orcl))))
(in-theory (disable write-buffer-p))

(defun update-wbuf0 (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((wbuf0 (LSU-wbuf0 LSU)))
    (write-buffer (b-and (wbuf-valid? wbuf0)
			 (b-not (flush-all? s orcl)))
		  (b-ior (wbuf-complete? wbuf0) (check-wbuf0? LSU))
		  (b-ior (wbuf-commit? wbuf0)
			 (b-and (commit-inst? s)
				(bv-eqv *rob-index-size*
					(ROB-head (MA-ROB s))
					(wbuf-dest wbuf0))))
		  (wbuf-dest wbuf0)
		  (wbuf-addr wbuf0)
		  (wbuf-val wbuf0))))

(defthm write-buffer-p-update-wbuf0
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (write-buffer-p (update-wbuf0 LSU s orcl))))
(in-theory (disable update-wbuf0))

(defun update-wbuf1 (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((wbuf1 (LSU-wbuf1 LSU)))
    (write-buffer (b-and (wbuf-valid? wbuf1)
			 (b-not (flush-all? s orcl)))
		  (b-ior (wbuf-complete? wbuf1) (check-wbuf1? LSU))
		  (b-ior (wbuf-commit? wbuf1)
			 (b-and (commit-inst? s)
				(bv-eqv *rob-index-size*
					(ROB-head (MA-ROB s))
					(wbuf-dest wbuf1))))
		  (wbuf-dest wbuf1)
		  (wbuf-addr wbuf1)
		  (wbuf-val wbuf1))))

(defthm write-buffer-p-update-wbuf1
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (write-buffer-p (update-wbuf1 LSU s orcl))))
(in-theory (disable update-wbuf1))


(defun step-wbuf0 (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (b-if (b-orc1 (wbuf-valid? (LSU-wbuf0 LSU)) (release-wbuf0? LSU orcl))
	(wbuf1-output LSU s orcl)
	(update-wbuf0 LSU s orcl)))

(defthm write-buffer-p-step-wbuf0
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (write-buffer-p (step-wbuf0 LSU s orcl))))
(in-theory (disable step-wbuf0))

(defun step-wbuf1 (LSU s orcl)
  (declare (xargs :guard (and (load-store-unit-p LSU)
			      (MA-state-p s) (MA-oracle-p orcl))))
  (let ((wbuf0 (LSU-wbuf0 LSU)) (wbuf1 (LSU-wbuf1 LSU)))
    (b-if (b-ior (b-and (wbuf-valid? wbuf1) (release-wbuf0? LSU orcl))
		 (bs-and (b-not (wbuf-valid? wbuf1))
			 (wbuf-valid? wbuf0)
			 (release-wbuf0? LSU orcl)))
	  (issued-write LSU s orcl)
	  (update-wbuf1 LSU s orcl))))

(defthm write-buffer-p-step-wbuf1
    (implies (and (load-store-unit-p LSU)
		  (MA-state-p s)
		  (MA-oracle-p orcl))
	     (write-buffer-p (step-wbuf1 LSU s orcl))))
(in-theory (disable step-wbuf1))

(defun step-LSU (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (let ((LSU (MA-LSU s)))
    (load-store-unit (step-rs1-head? LSU s orcl)
		     (step-LSU-RS0 LSU s orcl)
		     (step-LSU-RS1 LSU s orcl)
		     (step-rbuf LSU s orcl)
		     (step-wbuf0 LSU s orcl)
		     (step-wbuf1 LSU s orcl)
		     (step-LSU-lch LSU s orcl))))

(defthm LSU-p-step-LSU
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (load-store-unit-p (step-LSU s orcl))))
(in-theory (disable step-LSU))

(defun step-mem (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (let ((mem (MA-mem s)) (LSU (MA-LSU s)))
    (b-if (release-wbuf0? LSU orcl)
	  (write-mem (wbuf-val (LSU-wbuf0 LSU))
		     (wbuf-addr (LSU-wbuf0 LSU))
		     mem)
	  mem)))

(defthm mem-p-step-mem
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (mem-p (step-mem s orcl))))
(in-theory (disable step-mem))

(deflabel end-MA-step-functions)

;; The next state function of the pipelined machine.
;; It takes a pipeline state and returns the state one clock cycle later.
(defun MA-step (s orcl)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-p orcl))))
  (MA-state (step-pc s orcl)
	    (step-regs s)
	    (step-sregs s orcl)
	    (step-IFU s orcl)
	    (step-DQ s orcl)
	    (step-ROB s orcl)
	    (step-IU s orcl)
	    (step-MU s orcl)
	    (step-BU s orcl)
	    (step-LSU s orcl)
	    (step-mem s orcl)))

(defthm MA-state-p-MA-step
    (implies (and (MA-state-p s) (MA-oracle-p orcl))
	     (MA-state-p (MA-step s orcl))))

(in-theory (disable MA-step))

;; MA-stepn runs the pipelined machine n cycles from the initial
;; state s.
(defun MA-stepn (s oracles n)
  (declare (xargs :guard (and (MA-state-p s) (integerp n) (>= n 0)
			      (MA-oracle-listp oracles)
			      (<= n (len oracles)))
		  :verify-guards nil))
  (if (zp n)
      s
      (MA-stepn (MA-step s (car oracles)) (cdr oracles) (1- n))))

(verify-guards MA-stepn)
(defthm MA-state-p-MA-stepn
    (implies (and (MA-state-p s) (MA-oracle-listp oracles)
		  (<= n (len oracles)))
	     (MA-state-p (MA-stepn s oracles n))))

(deflabel end-MA-def)

(deflabel begin-MA-flushed-def)
(defun IFU-empty? (IFU)
  (declare (xargs :guard (IFU-p IFU)))
  (b-not (IFU-valid? IFU)))

(defun DQ-empty? (DQ)
  (declare (xargs :guard (DQ-p DQ)))
  (bs-and (b-not (DE-valid? (DQ-DE0 DQ)))
	  (b-not (DE-valid? (DQ-DE1 DQ)))
	  (b-not (DE-valid? (DQ-DE2 DQ)))
	  (b-not (DE-valid? (DQ-DE3 DQ)))))

(defun IU-empty? (IU)
  (declare (xargs :guard (integer-unit-p IU)))
  (bs-and (b-not (RS-valid? (IU-rs0 IU)))
	  (b-not (RS-valid? (IU-rs1 IU)))))

(defun MU-empty? (MU)
  (declare (xargs :guard (mult-unit-p MU)))
  (bs-and (b-not (RS-valid? (MU-rs0 MU)))
	  (b-not (RS-valid? (MU-rs1 MU)))
	  (b-not (MU-latch-valid? (MU-lch1 MU)))
	  (b-not (MU-latch-valid? (MU-lch2 MU)))))

(defun BU-empty? (BU)
  (declare (xargs :guard (branch-unit-p BU)))
  (bs-and (b-not (BU-RS-valid? (BU-rs0 BU)))
	  (b-not (BU-RS-valid? (BU-rs1 BU)))))

(defun LSU-empty? (LSU)
  (declare (xargs :guard (load-store-unit-p LSU)))
  (bs-and (b-not (LSU-RS-valid? (LSU-rs0 LSU)))
	  (b-not (LSU-RS-valid? (LSU-rs1 LSU)))
	  (b-not (LSU-latch-valid? (LSU-lch LSU)))
	  (b-not (wbuf-valid? (LSU-wbuf0 LSU)))
	  (b-not (wbuf-valid? (LSU-wbuf1 LSU)))
	  (b-not (rbuf-valid? (LSU-rbuf LSU)))))

(defun exintr-flag? (s)
  (declare (xargs :guard (MA-state-p s)))
  (ROB-exintr? (MA-ROB s)))

(defun MA-flushed? (s)
  (declare (xargs :guard (MA-state-p s)))
  (bs-and (IFU-empty? (MA-IFU s))
	  (DQ-empty? (MA-DQ s))
	  (ROB-empty? (MA-ROB s))
	  (IU-empty? (MA-IU s))
	  (MU-empty? (MA-MU s))
	  (BU-empty? (MA-BU s))
	  (LSU-empty? (MA-LSU s))
	  (b-not (exintr-flag? s))))

(defthm bitp-MA-flushed? (bitp (MA-flushed? s)))

(deflabel end-MA-flushed-def)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  Number of Commits
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun num-commits (s oracles n)
  (declare (xargs :guard (and (MA-state-p s) (MA-oracle-listp oracles)
			      (integerp n) (<= 0 n))
		  :measure (nfix n)))
  (if (or (zp n) (endp oracles))
      0
      (b-if (b-ior (commit-inst? s)
		   (ex-intr? s (car oracles)))
	    (1+ (num-commits (MA-step s (car oracles))
				     (cdr oracles) (1- n)))
	    (num-commits (MA-step s (car oracles)) (cdr oracles) (1- n)))))



(deftheory MA-state-def
    (set-difference-theories (function-theory 'end-MA-state)
			     (function-theory 'begin-MA-state)))
(deftheory MA-step-functions
   (set-difference-theories (function-theory 'end-MA-step-functions)
			    (function-theory 'begin-MA-step-functions)))

(deftheory MA-def
   (set-difference-theories (function-theory 'end-MA-def)
			    (function-theory 'begin-MA-def)))

(deftheory low-level-MA-def
    (set-difference-theories (theory 'MA-def) '(ma-step MA-stepn)))

(deftheory MA-def-all
    (union-theories (theory 'MA-def)
		    (theory 'MA-state-def)))

(deftheory MA-flushed-def
    (set-difference-theories (universal-theory 'end-MA-flushed-def)
			     (universal-theory 'begin-MA-flushed-def)))


(in-theory (disable MA-flushed? IFU-empty? DQ-empty? IU-empty?
		    MU-empty? BU-empty? LSU-empty? exintr-flag?))

(in-theory (disable MA-def))


#|

Here is a simple example program for our sequential machine.

Our program calculates the factorial of the number at address #x800 and stores
it at address #x801.

Initial memory setting:

#x0: ST R0,(#x50)
#x1: LD R0,(#x3)
#x2: BZ R0, 0
#x3: 0

#x10: ST R0,(#x50)
#x11: LD R0,(#x13)
#x12: BZ R0, 0
#x13: 0

#x20: ST R0,(#x50)
#x21: LD R0,(#x23)
#x22: BZ R0, 0
#x23: 0

#x30: ST R0,(#x50)
#x31: LD R0,(#x33)
#x32: BZ R0, 0
#x33: 0

#x60: 0
#x61: 1
#x62: 2
#x63: -1

#x70: #x400
#x71: #x800

#x100: LD R15,(#x70) ; program base
#x101: LD R14,(#x71) ; data base
#x102: LD R0, (#x60) ; 0
#x103: LD R1, (#x61) ; 1
#x104: LD R2, (#x62) ; 2
#x105: LD R3, (#x63) ; -1
#x106: MTSR SR0,R15
#x107: MTSR SR1,R0
#x108: RFEH

Initial memory image:
#x400     LD R5,(R14+R0) ; R5 holds counter
#x401     ADD R6, R0, R1     ; R6 holds factorial. Initially 1.
Loop:
#x402:    Mul R6, R6, R5  ; coutner * fact -> fact
#x403:    ADD R5, R5, R3  ; decrement fact
#x404:    BZ  R5, Exit; if counter is zero, exit
#x405:    BZ  R0, Loop ; always jump to loop
EXIT:
#x406: ST  R6, (R14+R1)
#x407: SYNC
#x408: Trap

#x800: 5
#x801: 0
#x802: 5 ; Offset to Loop
#x803: 9 ; Offset ot Exit


How to run the program:
1. certify and compile all the proof scripts.
   (You may skip this, but the execution will be slow.)
2. Run Acl2.
3. Type command '(ld "MA-def.lisp")'.
4  Evaluate expressions below and set initial state s.
5. You can run the ISA machine for one cycle by
   (MA-step (@ s) (MA-oracle-p 0 1 1)).
   You can also run the machine for multiple cycles with ISA-stepn.
   For instance, if you want to run the machine 15 cycles, type:
     (assign oracle-list (make-list 15 :initial-element (MA-oracle 0 1 1)))
     (MA-stepn (@ s) 15 oracle-list).

6  Following macro may be useful to evaluate "expr" and set it to variable
   s, without printing the state of memory.


; Evaluate expression expr and set the result to s.
(defmacro eval-set-print-MA (s expr)
  `(pprogn (f-put-global ',s ,expr state)
           (mv nil
	     (list (MA-pc (f-get-global ',s state))
 	           (MA-regs (f-get-global ',s state))
	           (MA-sregs (f-get-global ',s state))
    	           (MA-DQ (f-get-global ',s state))
    	           (MA-ROB (f-get-global ',s state))
	           (MA-IU (f-get-global ',s state))
	           (MA-MU (f-get-global ',s state))
	           (MA-BU (f-get-global ',s state))
    	           (MA-LSU (f-get-global ',s state)))
	      state)))

; Function to be used in MA-step-seq
(defun make-MA-step-seq (orcl seq)
  (if (endp seq) nil
      (if (endp (cdr seq)) nil
	  (cons `(f-put-global ',(cadr seq) (MA-step (@ ,(car seq)) ,orcl)
		             state)
		(make-MA-step-seq orcl (cdr seq))))))

; Given an MA oracle and sequence of symbols, and execute MA-step one at a time
; and assigns its result to the symbol in the sequence.
; For instance, (MA-step-seq (@ orcl) s0 s1 s2 s3) assigns the result of
; applying MA-step to s0  to s1, the result of applying MA-step to s1 to s2,
; and so on.
(defmacro MA-step-seq (orcl &rest seq)
  (if (endp seq) nil
      `(pprogn
	,@(make-MA-step-seq orcl seq)
	(mv nil
	 (list (MA-pc (f-get-global ',(car (last seq)) state))
	  (MA-regs (f-get-global ',(car (last seq)) state))
	  (MA-sregs (f-get-global ',(car (last seq)) state))
	  (MA-ROB (f-get-global ',(car (last seq)) state)))
	  state))))


(defmacro pr-MA (s)
  `(list (MA-pc (@ ,s)) (MA-regs (@ ,s)) (MA-sregs (@ ,s))
    (MA-IFU (@ ,s)) (MA-DQ (@ ,s)) (MA-ROB (@ ,s)) (MA-IU (@ ,s))
    (MA-MU (@ ,s)) (MA-BU (@ ,s)) (MA-LSU (@ ,s))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Initial State Setting
(progn
(assign regs '(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign sregs (sregs 1 0 0))

(assign IFU (IFU 0 0 0 0))
(assign DE (dispatch-entry 0 0 0 0 0 0 0 0 0))
(assign reg-s (reg-ref 0 0))
(assign reg-tbl (make-list *num-regs* :initial-element (@ reg-s)))
(assign sreg-tbl (sreg-tbl (@ reg-s) (@ reg-s)))
(assign DQ (DQ (@ DE) (@ DE) (@ DE) (@ DE) (@ reg-tbl) (@ sreg-tbl)))

(assign ROBE (ROB-entry 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign entries (make-list *rob-size* :initial-element (@ ROBE)))
(assign ROB (ROB 0 0 0 0 (@ entries)))

(assign IU (integer-unit (RS 0 0 0 0 0 0 0 0 0) (RS 0 0 0 0 0 0 0 0 0)))
(assign MU (mult-unit (RS 0 0 0 0 0 0 0 0 0) (RS 0 0 0 0 0 0 0 0 0)
		      (MU-latch 0 0 0) (MU-latch 0 0 0)))
(assign BU (branch-unit (BU-RS 0 0 0 0 0) (BU-RS 0 0 0 0 0)))
(assign LSU (load-store-unit 0
			     (LSU-RS 0 0 0 0  0 0 0 0 0 0 0 0 0)
			     (LSU-RS 0 0 0 0  0 0 0 0 0 0 0 0 0)
			     (read-buffer 0 0 0)
			     (write-buffer 0 0 0 0 0 0)
			     (write-buffer 0 0 0 0 0 0)
			     (LSU-latch 0 0 0 0)))
(assign mem-alist '(
; Exception Handler
(#x0 . #x7050) ; ST R0,(#x50)
(#x1 . #x6003) ; LD R0,(#x3)
(#x2 . #x2000) ; BZ R0, 0
(#x3 . 0)
; Exception Handler
(#x10 . #x7050) ; ST R0,(#x50)
(#x11 . #x6013) ; LD R0,(#x13)
(#x12 . #x2000) ; BZ R0, 0
(#x13 . 0)
; Exception Handler
(#x20 . #x7050) ; ST R0,(#x50)
(#x21 . #x6023) ; LD R0,(#x23)
(#x22 . #x2000) ; BZ R0, 0
(#x23 . 0)

; Exception Handler
(#x30 . #x7050) ; ST R0,(#x50)
(#x31 . #x6033) ; LD R0,(#x33)
(#x32 . #x2000) ; BZ R0, 0
(#x33 . 0)

; Kernel Data Section
(#x60 .  0)
(#x61 . 1)
(#x62 . 2)
(#x63 . #xFFFF) ; -1
(#x70 . #x400)
(#x71 . #x800)
; Kernel Dispatching code
(#x100 . #x6F70) ; LD R15,(#x70) ; program base
(#x101 . #x6E71) ;  LD R14,(#x71) ; data base
(#x102 . #x6060) ;  LD R0, (#x60) ; 0
(#x103 . #x6161) ;  LD R1, (#x61) ; 1
(#x104 . #x6262) ; LD R2, (#x62) ; 2
(#x105 . #x6363) ; LD R3, (#x63) ; -1
(#x106 . #xAF00) ;  MTSR SR0,R15
(#x107 . #xA010) ;  MTSR SR1,R0
(#x108 . #x8000) ; #x103: RFEH
; Program
(#x400 . #x35E0) ;  LD R5,(R14+R0) ; R5 holds counter
(#x401 . #x0601) ;  ADD R6, R0, R1     ; R6 holds factorial. Initially 1.
; Loop:
(#x402 . #x1665) ;  Mul R6, R6, R5  ; coutner * fact -> fact
(#x403 . #x0553) ;  ADD R5, R5, R3  ; decrement fact
(#x404 . #x2502) ;  BZ  R5, Exit; if counter is zero, exit
(#x405 . #x20FD) ;  BZ  R0, Loop ; always jump to loop
; EXIT:
(#x406 . #x46E1) ; ST  R6, (R14+R1)
(#x407 . #x5000) ; SYNC
(#x408 . #xB000) ; Trap

; Data Section
(#x800 . 5)
(#x801 . 0)
(#x802 . 5) ; Offset to Loop
(#x803 . 9) ; Offset ot Exit
))


(assign mem (set-page-mode *read-only* 1 (compress1 'mem *init-mem*)))
(assign mem (set-page-mode *read-write* 2 (@ mem)))
(assign mem (compress1 'mem (load-mem-alist (@ mem-alist) (@ mem))))
(assign s (MA-state #x100 (@ regs) (@ sregs) (@ IFU) (@ DQ) (@ ROB)
		    (@ IU) (@ MU) (@ BU) (@ LSU) (@ mem)))

)
|#

