#|

Here is a simple example program for our sequential machine.

Our program calculates the factorial of the number at address #x800 and stores
it at address #x801.

Initial memory setting:

#x0: ST R0,(#x50)
#x1: LD R0,(#x3)
#x2: BZ R0, 0
#x3: 0

#x10: ST R0,(#x50)
#x11: LD R0,(#x13)
#x12: BZ R0, 0
#x13: 0

#x20: ST R0,(#x50)
#x21: LD R0,(#x23)
#x22: BZ R0, 0
#x23: 0

#x30: ST R0,(#x50)
#x31: LD R0,(#x33)
#x32: BZ R0, 0
#x33: 0

#x60: 0
#x61: 1
#x62: 2
#x63: -1

#x70: #x400
#x71: #x800

#x100: LD R15,(#x70) ; program base
#x101: LD R14,(#x71) ; data base
#x102: LD R0, (#x60) ; 0
#x103: LD R1, (#x61) ; 1
#x104: LD R2, (#x62) ; 2
#x105: LD R3, (#x63) ; -1
#x106: MTSR SR0,R15
#x107: MTSR SR1,R0
#x108: RFEH

Initial memory image:
#x400     LD R5,(R14+R0) ; R5 holds counter
#x401     ADD R6, R0, R1     ; R6 holds factorial. Initially 1.
Loop:
#x402:    Mul R6, R6, R5  ; coutner * fact -> fact
#x403:    ADD R5, R5, R3  ; decrement fact
#x404:    BZ  R5, Exit; if counter is zero, exit
#x405:    BZ  R0, Loop ; always jump to loop
EXIT:
#x406: ST  R6, (R14+R1)
#x407: SYNC
#x408: Trap

#x800: 5
#x801: 0
#x802: 5 ; Offset to Loop
#x803: 9 ; Offset ot Exit


How to run the program:
1. certify and compile all the proof scripts.
   (You may skip this, but the execution will be slow.)
2. Run Acl2.
3. Type command '(ld "MA-def.lisp")'.
4  Evaluate expressions below and set initial state s.
5. You can run the MA machine for one cycle by
   (MA-step (@ s) (MA-oracle-p 0 1 1)).
   You can also run the machine for multiple cycles with MA-stepn.
   For instance, if you want to run the machine 15 cycles, type:
     (assign oracle-list (make-list 15 :initial-element (MA-oracle 0 1 1)))
     (MA-stepn (@ s) 15 oracle-list).

6  Following macro may be useful to evaluate "expr" and set it to variable
   s, without printing the state of memory.

|#

; Evaluate expression expr and set the result to s.
(defmacro eval-set-print-MA (s expr)
  `(pprogn (f-put-global ',s ,expr state)
           (mv nil
	     (list (MA-pc (f-get-global ',s state))
 	           (MA-regs (f-get-global ',s state))
	           (MA-sregs (f-get-global ',s state))
    	           (MA-DQ (f-get-global ',s state))
    	           (MA-ROB (f-get-global ',s state))
	           (MA-IU (f-get-global ',s state))
	           (MA-MU (f-get-global ',s state))
	           (MA-BU (f-get-global ',s state))
    	           (MA-LSU (f-get-global ',s state)))
	      state)))

; Function to be used in MA-step-seq
(defun make-MA-step-seq (orcl seq)
  (if (endp seq) nil
      (if (endp (cdr seq)) nil
	  (cons `(f-put-global ',(cadr seq) (MA-step (@ ,(car seq)) ,orcl)
		             state)
		(make-MA-step-seq orcl (cdr seq))))))

; Given an MA oracle and sequence of symbols, and execute MA-step one at a time
; and assigns its result to the symbol in the sequence.
; For instance, (MA-step-seq (@ orcl) s0 s1 s2 s3) assigns the result of
; applying MA-step to s0  to s1, the result of applying MA-step to s1 to s2,
; and so on.
(defmacro MA-step-seq (orcl &rest seq)
  (if (endp seq) nil
      `(pprogn
	,@(make-MA-step-seq orcl seq)
	(mv nil
	 (list (MA-pc (f-get-global ',(car (last seq)) state))
	  (MA-regs (f-get-global ',(car (last seq)) state))
	  (MA-sregs (f-get-global ',(car (last seq)) state))
	  (MA-ROB (f-get-global ',(car (last seq)) state)))
	  state))))


(defmacro pr-MA (s)
  `(list (MA-pc (@ ,s)) (MA-regs (@ ,s)) (MA-sregs (@ ,s))
    (MA-IFU (@ ,s)) (MA-DQ (@ ,s)) (MA-ROB (@ ,s)) (MA-IU (@ ,s))
    (MA-MU (@ ,s)) (MA-BU (@ ,s)) (MA-LSU (@ ,s))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Initial State Setting
(progn
(assign regs '(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign sregs (sregs 1 0 0))

(assign IFU (IFU 0 0 0 0))
(assign DE (dispatch-entry 0 0 0 0 0 0 0 0 0))
(assign reg-s (reg-ref 0 0))
(assign reg-tbl (make-list *num-regs* :initial-element (@ reg-s)))
(assign sreg-tbl (sreg-tbl (@ reg-s) (@ reg-s)))
(assign DQ (DQ (@ DE) (@ DE) (@ DE) (@ DE) (@ reg-tbl) (@ sreg-tbl)))

(assign ROBE (ROB-entry 0 0 0 0 0 0 0 0 0 0 0 0 0))
(assign entries (make-list *rob-size* :initial-element (@ ROBE)))
(assign ROB (ROB 0 0 0 0 (@ entries)))

(assign IU (integer-unit (RS 0 0 0 0 0 0 0 0 0) (RS 0 0 0 0 0 0 0 0 0)))
(assign MU (mult-unit (RS 0 0 0 0 0 0 0 0 0) (RS 0 0 0 0 0 0 0 0 0)
		      (MU-latch 0 0 0) (MU-latch 0 0 0)))
(assign BU (branch-unit (BU-RS 0 0 0 0 0) (BU-RS 0 0 0 0 0)))
(assign LSU (load-store-unit 0
			     (LSU-RS 0 0 0 0  0 0 0 0 0 0 0 0 0)
			     (LSU-RS 0 0 0 0  0 0 0 0 0 0 0 0 0)
			     (read-buffer 0 0 0)
			     (write-buffer 0 0 0 0 0 0)
			     (write-buffer 0 0 0 0 0 0)
			     (LSU-latch 0 0 0 0)))
(assign mem-alist '(
; Exception Handler
(#x0 . #x7050) ; ST R0,(#x50)
(#x1 . #x6003) ; LD R0,(#x3)
(#x2 . #x2000) ; BZ R0, 0
(#x3 . 0)
; Exception Handler
(#x10 . #x7050) ; ST R0,(#x50)
(#x11 . #x6013) ; LD R0,(#x13)
(#x12 . #x2000) ; BZ R0, 0
(#x13 . 0)
; Exception Handler
(#x20 . #x7050) ; ST R0,(#x50)
(#x21 . #x6023) ; LD R0,(#x23)
(#x22 . #x2000) ; BZ R0, 0
(#x23 . 0)

; Exception Handler
(#x30 . #x7050) ; ST R0,(#x50)
(#x31 . #x6033) ; LD R0,(#x33)
(#x32 . #x2000) ; BZ R0, 0
(#x33 . 0)

; Kernel Data Section
(#x60 .  0)
(#x61 . 1)
(#x62 . 2)
(#x63 . #xFFFF) ; -1
(#x70 . #x400)
(#x71 . #x800)
; Kernel Dispatching code
(#x100 . #x6F70) ; LD R15,(#x70) ; program base
(#x101 . #x6E71) ;  LD R14,(#x71) ; data base
(#x102 . #x6060) ;  LD R0, (#x60) ; 0
(#x103 . #x6161) ;  LD R1, (#x61) ; 1
(#x104 . #x6262) ; LD R2, (#x62) ; 2
(#x105 . #x6363) ; LD R3, (#x63) ; -1
(#x106 . #xAF00) ;  MTSR SR0,R15
(#x107 . #xA010) ;  MTSR SR1,R0
(#x108 . #x8000) ; #x103: RFEH
; Program
(#x400 . #x35E0) ;  LD R5,(R14+R0) ; R5 holds counter
(#x401 . #x0601) ;  ADD R6, R0, R1     ; R6 holds factorial. Initially 1.
; Loop:
(#x402 . #x1665) ;  Mul R6, R6, R5  ; coutner * fact -> fact
(#x403 . #x0553) ;  ADD R5, R5, R3  ; decrement fact
(#x404 . #x2502) ;  BZ  R5, Exit; if counter is zero, exit
(#x405 . #x20FD) ;  BZ  R0, Loop ; always jump to loop
; EXIT:
(#x406 . #x46E1) ; ST  R6, (R14+R1)
(#x407 . #x5000) ; SYNC
(#x408 . #xB000) ; Trap

; Data Section
(#x800 . 5)
(#x801 . 0)
(#x802 . 5) ; Offset to Loop
(#x803 . 9) ; Offset ot Exit
))


(assign mem (set-page-mode *read-only* 1 (compress1 'mem *init-mem*)))
(assign mem (set-page-mode *read-write* 2 (@ mem)))
(assign mem (compress1 'mem (load-mem-alist (@ mem-alist) (@ mem))))
(assign s (MA-state #x100 (@ regs) (@ sregs) (@ IFU) (@ DQ) (@ ROB)
		    (@ IU) (@ MU) (@ BU) (@ LSU) (@ mem)))

)
