//
//  ContentView.swift
//  CD-Calendar
//
//  Created by bulko on 11/21/25.
//

import SwiftUI
import EventKit

let eventStore = EKEventStore()
var savedEventId: String!

struct ContentView: View {
    
    @State private var eventTitle = ""
    @State private var startDate = Date.now
    @State private var endDate = Date.now
    @State private var message = "No status"
    
    var body: some View {
        VStack {
            
            HStack {
                Text("Event Title")
                Spacer()
                TextField("Title", text: $eventTitle)
                    .frame(width: 200)
                    .padding(5)
                    .border(.gray)
            }
            .padding(.bottom, 50)
            
            DatePicker(selection: $startDate) {
                Text("Start")
            }
            
            DatePicker(selection: $endDate) {
                Text("End")
            }
            .padding(.bottom, 50)
            
            Button("Add Event") {
                addEventPressed()
            }
                .buttonStyle(.borderedProminent)
            
            Button("Delete Event") {
                removeEventPressed()
            }
                .buttonStyle(.borderedProminent)
                .padding(.bottom, 50)
            
            Text(message)
            
        }
        .padding()
    }
    
    func createEvent() {
        
        let event = EKEvent(eventStore: eventStore)
        
        event.title = eventTitle
        event.startDate = startDate
        event.endDate = endDate
        event.calendar = eventStore.defaultCalendarForNewEvents
        
        do {
            try eventStore.save(event, span: .thisEvent)
            savedEventId = event.eventIdentifier
            message = "Event added to calendar"
        } catch {
            message = "Event not added due to error"
        }
        
    }
    
    func addEventPressed() {
        
        if (EKEventStore.authorizationStatus(for: .event)) == .fullAccess {
            createEvent()
        } else {
            eventStore.requestFullAccessToEvents() {
                (granted, error) in
                if let error = error as NSError? {
                    message = "\(error.localizedDescription)"
                } else if granted {
                    createEvent()
                } else {
                    message = "User denied calendar access"
                }
            }
        }
        
    }
    
    func deleteEvent(eventIdentifier: String) {
        
        let eventToRemove = eventStore.event(withIdentifier: eventIdentifier)
        
        if eventToRemove != nil {
            
            do {
                try eventStore.remove(eventToRemove!, span: .thisEvent)
                message = "Event removed from calendar"
            } catch {
                message = "Event not removed from calendar"
            }
        }
        
    }
    
    func removeEventPressed() {
        
        if (EKEventStore.authorizationStatus(for: .event)) == .fullAccess {
            deleteEvent(eventIdentifier: savedEventId)
        } else {
            eventStore.requestFullAccessToEvents() {
                (granted, error) in
                if let error = error as NSError? {
                    message = "\(error.localizedDescription)"
                } else if granted {
                    deleteEvent(eventIdentifier: savedEventId)
                } else {
                    message = "User denied calendar access"
                }
            }
        }
        
    }
    
}



#Preview {
    ContentView()
}
