/*
* commit.c - commit stage implementation
*
* This file is part of the Alphasim tool suite written by
* Raj Desikan as part of the Bullseye project.
* Copyright (C) 1999 by Raj Desikan
* This source file is distributed "as is" in the hope that it will be
* useful.  It is distributed with no warranty, and no author or
* distributor accepts any responsibility for the consequences of its
* use. 
*
* Everyone is granted permission to copy, modify and redistribute
* this source file under the following conditions:
*
*    This tool set is distributed for non-commercial use only. 
*    Please contact the maintainer for restrictions applying to 
*    commercial use of these tools.
*
*    Permission is granted to anyone to make or distribute copies
*    of this source code, either as received or modified, in any
*    medium, provided that all copyright notices, permission and
*    nonwarranty notices are preserved, and that the distributor
*    grants the recipient permission for further redistribution as
*    permitted by this document.
*
*    Permission is granted to distribute this file in compiled
*    or executable form under the same conditions that apply for
*    source code, provided that either:
*
*    A. it is accompanied by the corresponding machine-readable
*       source code,
*    B. it is accompanied by a written offer, with no time limit,
*       to give anyone a machine-readable copy of the corresponding
*       source code in return for reimbursement of the cost of
*       distribution.  This written offer must permit verbatim
*       duplication by anyone, or
*    C. it is distributed by someone who received only the
*       executable form, and is accompanied by a copy of the
*       written offer of source code that they received concurrently.
*
* In other words, you are welcome to use, share and improve this
* source file.  You are forbidden to forbid anyone else to use, share
* and improve what you give them.
*
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include "alpha.h"                                                      
#include "regs.h"
#include "memory.h"
#include "resource.h"
#include "sim.h" 
#include "cache.h"
#include "loader.h"
#include "eventq.h"
#include "bpred.h"
#include "fetch.h" 
#include "slot.h"
#include "issue.h"
#include "map.h"
#include "writeback.h"
#include "commit.h"
#include "eventq.h"
#include "syscall.h"

int commit_width;

/* rename table for the commit stage */
int commit_int_table[MD_I_ARCH_REGS];
int commit_fp_table[MD_F_ARCH_REGS];

/* store queue */

int commit_sq_head;
int commit_sq_tail;
int commit_sq_num;
int commit_sq_nelem;
struct load_store_queue *commit_sq;
struct queue_elem *commit_elem;
struct rqueue_link *commitq_link;

counter_t *num_inst_committed;
counter_t commit_ctrl_flushes;
counter_t commit_trap_flushes;
counter_t cycles_lost;

void 
commit_stage_init(void){
  int i;
  for (i=0; i < MD_I_ARCH_REGS; i++){
    commit_int_table[i] = DNA;
    commit_fp_table[i] = DNA;
  }
  commit_sq_head=0;
  commit_sq_tail=0;
  commit_sq_num=0;
  commit_sq = (struct load_store_queue*)
    calloc(commit_sq_nelem, sizeof(struct load_store_queue));
  if (!commit_sq)
    fatal ("Out of virtual memory");
  for (i=0;i<commit_sq_nelem;i++)
    commit_sq[i].type=SQ;
  commit_elem = (struct queue_elem *)
    malloc(sizeof(struct queue_elem));
  if (!commit_elem)
    fatal("Out of virtual memory");
  commitq_link = (struct rqueue_link *)
    malloc(sizeof(struct rqueue_link));
  if (!commitq_link)
    fatal("Out of virtual memory");
}

void 
commit_stage(void){
  int n_committed=0;
  static struct dep_chain *dep_chain_pointer;    
  static enum md_opcode op;
  static md_inst_t inst;
  static md_addr_t addr;
  static byte_t temp_byte;
  static half_t temp_half;
  static word_t temp_word;
  static quad_t temp_quad;
  static int spec_mode=0;
  static int mispredict_penalty;
#if defined FUNC_DEBUG
  void my_sim_main(void);
  extern struct regs_t myregs;
  int i=0;
#endif
  struct queue_elem *elem;
  int branch_inst = FALSE;
  /* traverse the reorder buffer and commit instructions in order. Blow away
     everything on a mispredict */ 

  while (map_rb_num > 0 && n_committed < commit_width){
    if (map_rb[map_rb_head].completed == TRUE){
      /* This instruction shouldn't trap in a good program */
      if (map_rb[map_rb_head].inst_desc->trap == TRUE)
	panic("Retiring instruction trapped");
      if ((MD_OP_FLAGS(map_rb[map_rb_head].inst_desc->op) & F_CTRL)){
	if (n_committed != 0)
	  return;
	branch_inst = TRUE;
	sim_num_branches++;
	bpred_update(pred,
		     /* branch address */ 
		     map_rb[map_rb_head].inst_desc->regs_PC,
		     /* actual target address */
		     map_rb[map_rb_head].correctPC,
		     /* taken? */
		     (map_rb[map_rb_head].inst_desc->regs_PC+sizeof(md_inst_t))
		     != map_rb[map_rb_head].correctPC, 
		     /* pred taken? */ 
		     map_rb[map_rb_head].inst_desc->bpred_PC != 
		     (map_rb[map_rb_head].inst_desc->regs_PC + 
		      sizeof(md_inst_t)),
		     /* correct pred? */
		     map_rb[map_rb_head].inst_desc->bpred_PC == 
		     map_rb[map_rb_head].correctPC,
		     /* opcode */map_rb[map_rb_head].inst_desc->op,
		     /* dir predictor update pointer */
		     &(map_rb[map_rb_head].dir_update),
		     /* Inst desc */ map_rb[map_rb_head].inst_desc);
								      
	if (line_predictor) {
	  if (map_rb[map_rb_head].inst_desc->lpred_PC == 
	      map_rb[map_rb_head].correctPC) {
	    icache->line_pred_hits++;
	    if (icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].line_pred_hist < 3)
	      icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].line_pred_hist++;
	  }
	  else if (map_rb[map_rb_head].mispredict == TRUE) {
	    icache->line_pred_misses++;
	    if (icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].line_pred_hist)
	      icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].line_pred_hist--; 
	    if ((icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].line_pred_hist) < 2) {
	      if (map_rb[map_rb_head].correctPC % 
		(line_pred_width*sizeof(md_inst_t)) == 0) {
	      icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].next_addr = 
		  map_rb[map_rb_head].correctPC;
	      icache->line_pred_table[map_rb[map_rb_head].inst_desc->pred_set] [map_rb[map_rb_head].inst_desc->pred_blk_no] [map_rb[map_rb_head].inst_desc->pred_offset].line_pred_hist = line_pred_ini_value;
	      }
	    }
	  }
	}
      }
      
      /* check for mispredict */
      if (map_rb[map_rb_head].mispredict == TRUE || 
	  map_rb[map_rb_head].replaytrap == TRUE){
	/* problem here. There has been a mispredict or a replay trap
	   and now we need to squash instructions in the fetchbuffer, 
	   slotbuffer, mapbuffer, the issue queues, reorder buffer,
	   the ready queue, and the event queue. Also the resource
	   structures need to be reset */ 

	/* correct the PC */
	if (map_rb[map_rb_head].replaytrap == FALSE){
	  /* This is a misprediction */ 
	  sim_num_insn++;
	  commit_ctrl_flushes++;
	  
	  /* Jumps have a 10 cycle minimum penalty */
	  if (MD_IS_BRANCH(map_rb[map_rb_head].inst_desc->op) ||
	      MD_IS_INDIR(map_rb[map_rb_head].inst_desc->op)) {
	    mispredict_penalty = 
	      (sim_cycle+1>=map_rb[map_rb_head].inst_desc->time_stamp+10)?
	      sim_cycle+1:map_rb[map_rb_head].inst_desc->time_stamp+10;
	  }
	  else {
	    /* Branches have a 7 cycle minimum penalty */ 
	    mispredict_penalty=
	      (sim_cycle+1>=map_rb[map_rb_head].inst_desc->time_stamp+7)?
	      sim_cycle+1:map_rb[map_rb_head].inst_desc->time_stamp+7;
	  }
	  regs.regs_PC = map_rb[map_rb_head].correctPC;
#if defined FUNC_DEBUG
	  if (myregs.regs_PC != map_rb[map_rb_head].inst_desc->regs_PC){
	    panic("Mismatch1 at %ld",(long int)sim_num_insn);
	  }
	  my_sim_main();
          /*for (i=0;i<MD_I_ARCH_REGS-1;i++) {
	  if (myregs.regs_R[i] != regs.regs_R[i]){
            panic("Mismatch2 at %d",i);
	  }
          }*/
#endif
	  commit_flush_pipeline();
	
	  fetch_istall_buf.stall |= BRANCH_STALL;
	  fetch_istall_buf.resume=0;
	  eventq_queue_callback(mispredict_penalty, 
				(void *) fetch_resume_ifetch,
				(int) BRANCH_STALL);
	  return;
	}
	else {
	  /* This is a replay trap */
	  commit_trap_flushes++;
	  
	  /* Trap penalty is equale to load_trap_penalty from this cycle */
	  mispredict_penalty = sim_cycle + 
	    map_rb[map_rb_head].inst_desc->load_trap_penalty;
	  regs.regs_PC = map_rb[map_rb_head].inst_desc->regs_PC;
	  if (mispredict_penalty > 0) {
	    wb_trap_cycles_lost+=
	      mispredict_penalty - map_rb[map_rb_head].inst_desc->time_stamp;
	  }
	  commit_flush_pipeline();
	  
	  fetch_istall_buf.stall |= BRANCH_STALL;
	  fetch_istall_buf.resume=0;
	  eventq_queue_callback(mispredict_penalty, 
				(void *) fetch_resume_ifetch,
				(int) BRANCH_STALL);
	  return;
	}
      }
      /* No mispredict. Retire this instruction */
      else{
	/* If this is a load instruction, increment the head pointer */
	if (map_rb[map_rb_head].in_LQ == TRUE) {
	  assert(issue_lq_head == map_rb[map_rb_head].inst_desc->lq_no);
	  sim_num_refs++;
	  sim_num_loads++;
	  issue_lq_head = (issue_lq_head+1) % issue_lq_nelem;
	  issue_lq_num--;
	  map_rb[map_rb_head].in_LQ = FALSE;
	}
	/* If this is a store instruction, execute it here */
	else if (map_rb[map_rb_head].in_SQ == TRUE) {
	  int store_lat;
	  /* Execute the store instruction if this is first time execution */
	  if (commit_sq[map_rb[map_rb_head].inst_desc->sq_no].cachemiss ==
	      FALSE &&
	      commit_sq[map_rb[map_rb_head].inst_desc->sq_no].tlbmiss ==
	      FALSE &&
	      commit_sq[map_rb[map_rb_head].inst_desc->sq_no].mshrfull == 
	      FALSE) {
	    sim_num_refs++;
	    if (MD_VALID_ADDR(commit_sq[map_rb[map_rb_head].inst_desc->sq_no].addr))
	      writeback_exec_loadstore(sim_cycle, 
				       (int)map_rb[map_rb_head].inst_desc->sq_no, SQ);
	    else
	      panic("invalid address on a store");
	  }
	  
	  /* If we are not restarting after a TLB miss, MSHR Full, or 
	     MSHR Target Full */
	  /* Execute the store instruction */
	  /* Access memory for store inst */
	  if (dcache) {
	    cache_access_packet *c_packet;
	    if
	      (commit_sq[map_rb[map_rb_head].inst_desc->sq_no].tlbmiss
	       == TRUE) {
	      c_packet = 
		cache_create_access_packet(dcache, 
					   (Write | Restarted_access),
					   (commit_sq[map_rb[map_rb_head].inst_desc->sq_no].addr & ~3),
					   Virtual, 4, 
					   (void *)
					   (commit_sq + map_rb[map_rb_head].inst_desc->sq_no),
					   (void *)
					   cache_load_store_exec, 
					   (void *)
					   valid_sq,
					   (MSHR_STAMP_TYPE) map_rb[map_rb_head].inst_desc->inum);
	    }
	    else {
	      c_packet = 
		cache_create_access_packet(dcache, (Write | Pipeline_access),
					   (commit_sq[map_rb[map_rb_head].inst_desc->sq_no].addr & ~3),
					   Virtual, 4, 
					   (void *)(commit_sq + map_rb[map_rb_head].inst_desc->sq_no),
					   (void *)cache_load_store_exec, 
					   (void *)valid_sq,
					   (MSHR_STAMP_TYPE) map_rb[map_rb_head].inst_desc->inum);
	    }
	    /* access the cache if non-faulting */
	    store_lat = cache_timing_access(sim_cycle, c_packet);
	    /* If we don't have a cache hit find out what 
	       it was and take appropriate action */
	    if (store_lat <= 0) {
	      if (store_lat == CACHE_MISS) {
		commit_sq[map_rb[map_rb_head].inst_desc->sq_no].cachemiss = 
		  TRUE;
		commit_sq[map_rb[map_rb_head].inst_desc->sq_no].tlbmiss = 
		  FALSE;
	      }
	      
	      /* unset committed bit to indicate MSHR FULL */
	      else if ((store_lat == MSHRS_FULL) || (store_lat == TARGET_FULL)) {
		/* Cant continue. Mark this entry as false */
		commit_sq[map_rb[map_rb_head].inst_desc->sq_no].mshrfull = 
		  TRUE;
		map_rb[map_rb_head].completed = FALSE;
		return;
	      }
	      
	      /* unset commit bit to stall for TLB miss */
	      else if (store_lat == TLB_MISS) {
		/* Can't continue. Mark this store as incomplete */
		commit_sq[map_rb[map_rb_head].inst_desc->sq_no].tlbmiss = 
		  TRUE;
		map_rb[map_rb_head].completed = FALSE;
		return;
	      }
	      
	      else {
		assert (store_lat == BAD_ADDRESS);
		commit_sq_num--;
		commit_sq_head =  (commit_sq_head+1) % commit_sq_nelem;
		map_rb[map_rb_head].completed = TRUE;
	      }
	      
	    }
	  }
	  else {
	    /* no caches defined, just use op latency */
	    /* NOTE: If you want to have no caches and have these
	       loads go directly to some main memory bank, you
	       will have to add a call to mem_bank_access() here */
	    store_lat = 1;
	  }
	  if (CACHE_HIT(store_lat) || store_lat == CACHE_MISS) {
	    /* use computed cache access latency */
	    commit_sq_num--;
	    commit_sq_head = (commit_sq_head+1) % commit_sq_nelem;
	  }
	  /* If we sent a bad (misspeculated) address to the cache,
	   * just treat it like a cache hit (for now). */
	  else if (store_lat == BAD_ADDRESS) {
	    panic("Bad address on a store");
	  }
	  map_rb[map_rb_head].in_SQ = FALSE;
	}

	/* If this is a syscall instruction, execute it here */
	if (map_rb[map_rb_head].inst_desc->iflag == F_TRAP) {
	  sim_total_insn++;
	  regs.regs_R[MD_REG_ZERO] = 0;
	  regs.regs_F.d[MD_REG_ZERO] = 0;
	  inst = map_rb[map_rb_head].inst_desc->IR;
	  op = map_rb[map_rb_head].inst_desc->op;
	  regs.regs_PC = map_rb[map_rb_head].inst_desc->regs_PC;
	  regs.regs_NPC = map_rb[map_rb_head].inst_desc->regs_NPC;
	  elem = commit_elem;
	  elem->inst_desc = map_rb[map_rb_head].inst_desc;
	  switch(op) {
#define DEFINST(OP,MSK,NAME,OPFORM,RES,CLASS,O1,O2,I1,I2,I3)	\
      case OP:	\
      SYMCAT(OP,_IMPL);	\
      break;
#define DEFLINK(OP,MSK,NAME,MASK,SHIFT)	\
      case OP:	\
        panic("attempted to execute a linking opcode");
#define CONNECT(OP)
#define DECLARE_FAULT(FAULT)	\
      {break;}
#include "alpha.def"
      default:
        panic("trying to execute bogus inst");
	  }/*end switch*/
	  /* flush the pipeline */
	  regs.regs_PC = map_rb[map_rb_head].inst_desc->regs_NPC;
#if defined FUNC_DEBUG
	  my_sim_main();
#endif
	  sim_num_insn++;
	  fetch_istall_buf.resume=fetch_istall_buf.stall = 0;
	  sys_cycles_lost+=
	    sim_cycle - map_rb[map_rb_head].inst_desc->time_stamp;
	  commit_flush_pipeline();
	  return;
	}
	
	/* Wake up load instructions waiting on this store */
	if (map_rb[map_rb_head].stdeps) {
	  dep_chain_pointer = map_rb[map_rb_head].stdeps;
	  while (dep_chain_pointer != NULL){
	    dep_chain_pointer->qelem->st_wait_bit = FALSE;
	    if (OPERANDS_READY(dep_chain_pointer->qelem)){
	      commitq_link->qelem = dep_chain_pointer->qelem;
	      commitq_link->inum = map_rb[map_rb_head].inst_desc->inum;
	      if (commitq_link->qelem->inst_desc->dest_reg == INTEGER){
		issue_int_readyq_enqueue(commitq_link);
	      }
	      else{
		issue_fp_readyq_enqueue(commitq_link);
	      }
	    }
	    dep_chain_pointer = dep_chain_pointer->next;
	  }
	}

	
	if (map_rb[map_rb_head].inst_desc->dest_reg == INTEGER && 
	    map_rb[map_rb_head].inst_desc->out_phy_reg != DNA && 
	    map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
	  if (map_ir_mapping[map_rb[map_rb_head].inst_desc->out_arch_reg].phy_reg == 
	      map_rb[map_rb_head].inst_desc->out_phy_reg){
	    map_ir_mapping[map_rb[map_rb_head].inst_desc->out_arch_reg].phy_reg = map_rb[map_rb_head].inst_desc->out_arch_reg; 
	  }
	  /* Copy the physical register to the architectural
	     register. */
	  if (map_rb[map_rb_head].inst_desc->out_arch_reg != DNA) {
	    regs.regs_R[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
	      regs.regs_R[map_rb[map_rb_head].inst_desc->out_phy_reg];
	  }
	}
	else if (map_rb[map_rb_head].inst_desc->dest_reg == FLOATINGPT && 
		 map_rb[map_rb_head].inst_desc->out_phy_reg != DNA && 
		 map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
	  if (map_fr_mapping[map_rb[map_rb_head].inst_desc->out_arch_reg].phy_reg == map_rb[map_rb_head].inst_desc->out_phy_reg){
	    map_fr_mapping[map_rb[map_rb_head].inst_desc->out_arch_reg].phy_reg = map_rb[map_rb_head].inst_desc->out_arch_reg;
	  }
	  if (map_rb[map_rb_head].inst_desc->out_arch_reg != DNA) {
	    if (map_rb[map_rb_head].inst_desc->double_prec == TRUE)
	      regs.regs_F.d[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
		regs.regs_F.d[map_rb[map_rb_head].inst_desc->out_phy_reg]; 
	    else
	      regs.regs_F.q[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
		regs.regs_F.q[map_rb[map_rb_head].inst_desc->out_phy_reg];
	  }
	}
	
	while (map_rb[map_rb_head].op1_deps != NULL){
	  //map_rb[map_rb_head].op1_deps->inst_desc->in_phy_regs[0] 
	  //= map_rb[map_rb_head].inst_desc->out_arch_reg;
	  dep_chain_pointer = map_rb[map_rb_head].op1_deps;
	  map_rb[map_rb_head].op1_deps = 
	    map_rb[map_rb_head].op1_deps->next;
	  return_to_free_list(dep_chain_pointer);
	}
	
	while (map_rb[map_rb_head].op2_deps != NULL){
	  //map_rb[map_rb_head].op2_deps->inst_desc->in_phy_regs[1] = 
	  //map_rb[map_rb_head].inst_desc->out_arch_reg;
	  dep_chain_pointer = map_rb[map_rb_head].op2_deps;
	  map_rb[map_rb_head].op2_deps = 
	    map_rb[map_rb_head].op2_deps->next;
	  return_to_free_list(dep_chain_pointer);
	}
	
	while (map_rb[map_rb_head].stdeps != NULL){
	  dep_chain_pointer = map_rb[map_rb_head].stdeps;
	  map_rb[map_rb_head].stdeps = 
	    map_rb[map_rb_head].stdeps->next;
	  return_to_free_list(dep_chain_pointer);
	}
	while (map_rb[map_rb_head].cmovdeps != NULL){
	  dep_chain_pointer = map_rb[map_rb_head].cmovdeps;
	  map_rb[map_rb_head].cmovdeps = 
	    map_rb[map_rb_head].cmovdeps->next;
	  return_to_free_list(dep_chain_pointer);
	}
	/* Retire instruction and free the physical register */
	if (map_rb[map_rb_head].inst_desc->dest_reg == INTEGER && 
	    map_rb[map_rb_head].inst_desc->out_arch_reg != DNA && 
	    map_rb[map_rb_head].inst_desc->out_arch_reg != 31){
	  /* Free the previous mapping of the output register */
	  if(commit_int_table[map_rb[map_rb_head].inst_desc->out_arch_reg] !=
	     DNA && 
	     commit_int_table[map_rb[map_rb_head].inst_desc->out_arch_reg] 
	     != 31)
	    map_put_free_int_reg(commit_int_table[map_rb[map_rb_head].inst_desc->out_arch_reg]);
	  
	  /* update the integer commit table */ 
	  if (map_rb[map_rb_head].inst_desc->out_phy_reg != DNA &&
	      map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
	    commit_int_table[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
	      map_rb[map_rb_head].inst_desc->out_phy_reg;
	  }
	}
	
	else if (map_rb[map_rb_head].inst_desc->dest_reg == FLOATINGPT && 
		 map_rb[map_rb_head].inst_desc->out_arch_reg != DNA &&
		 map_rb[map_rb_head].inst_desc->out_arch_reg != 31){
	  if (commit_fp_table[map_rb[map_rb_head].inst_desc->out_arch_reg] 
	     != DNA &&
	      commit_fp_table[map_rb[map_rb_head].inst_desc->out_arch_reg] 
	      != 31){
	    map_put_free_fp_reg(commit_fp_table[map_rb[map_rb_head].inst_desc->out_arch_reg]); 
	  }
	  
	  if (map_rb[map_rb_head].inst_desc->out_phy_reg != DNA &&
	      map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
	    commit_fp_table[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
	      map_rb[map_rb_head].inst_desc->out_phy_reg;
	  }
	}
#if defined FUNC_DEBUG
	if (myregs.regs_PC != map_rb[map_rb_head].inst_desc->regs_PC){
	  panic("Mismatch3 at %ld",(long int)sim_num_insn);
	}
	my_sim_main();
	/*for (i=0;i<MD_I_ARCH_REGS-1;i++) {
	  if (myregs.regs_R[i] != regs.regs_R[i]){
            panic("Mismatch4 at %d",i);
	  }
          }*/
        /*for (i=0;i<MD_F_ARCH_REGS-1;i++){
	  if (myregs.regs_F.q[i] != regs.regs_F.q[i]){
	  panic("Mismatch4q at %d",i);
	  }
	  }*/
#endif
	sim_num_insn++;
	fetch_return_to_free_list(map_rb[map_rb_head].inst_desc);
	map_rb_head=(map_rb_head + 1) % (map_rb_nelem);
	map_rb_num--;
	n_committed++;
      }
    }
    else{
      /* Can't commit the instruction. Wait till next cycle */
      return;
    }
  }
}

/* Flush the pipeline */
void commit_flush_pipeline(void) {
  int i, index,j;
  struct rqueue_link *node, *prev;
  struct dep_chain *dep_chain_pointer;
  
  /* Recover RAS. equate vtos to rtos */
  pred->retstack.vtos = pred->retstack.rtos;
  
  /* Recover global and choice predictor indices */
  specgindex = gindex;
  speccindex = cindex;
  map_rb[map_rb_head].in_LQ = map_rb[map_rb_head].in_SQ = FALSE;
  /* First to be blown away is the fetch list */
  while (fetch_num > 0){
    fetch_return_to_free_list(fetch_data[fetch_head].inst_desc);
    fetch_head = (fetch_head + 1) & (fetch_ifq_size - 1);
    fetch_num--;
  }
  fetch_head=fetch_tail=0;
  
  /* blow away the slot list */
  while (slot_latch_num > 0){
    fetch_return_to_free_list(SL[slot_latch_head].inst_desc);
    slot_latch_head = (slot_latch_head + 1) & (slot_width - 1);
    slot_latch_num--;
  }
  slot_latch_head=slot_latch_tail=0;
  
  IQ->queue_pointer=0;
  IQ->queue_num = 0;

  FQ->queue_pointer=0;
  FQ->queue_num=0;
  /* blow away the integer ready queue */
  for (prev=NULL,node=int_ready_queue;node;prev=node,node=node->next);
  
  if (prev){
    prev->next = ready_queue_free_list;
    ready_queue_free_list = int_ready_queue;
    int_ready_queue=NULL;
  }
  
  /* blow away the fp ready queue */
  for (prev=NULL,node=fp_ready_queue;node;prev=node,node=node->next);
  
  if (prev){
    prev->next = ready_queue_free_list;
    ready_queue_free_list = fp_ready_queue;
    fp_ready_queue=NULL;
  }
  
  /* blow away the event queue */ 
  for (prev=NULL,node=event_queue;node;prev=node,node=node->next);
  
  if (prev){
    prev->next = event_queue_free_list;
    event_queue_free_list = event_queue;
    event_queue=NULL;
  }

  for(i=0;i<issue_lq_nelem;i++)
    issue_lq[i].tag = DNA;
  
  issue_lq_num=issue_lq_tail=issue_lq_head=0;
  
  for (i=0;i<commit_sq_nelem;i++)
    commit_sq[i].tag = DNA;
  
  commit_sq_num=commit_sq_tail=commit_sq_head=0;
  
  /* Blow away the writeback events in the global event queue */
  eventq_blow();
  
  /* blow away the reorder buffer */
  /* Copy physical register value of this instruction to the
     architectural register file (for mispredicted 
     bsr instructions) if this is not a replaytrap */
  if (map_rb[map_rb_head].inst_desc->out_arch_reg != DNA && 
      map_rb[map_rb_head].replaytrap == FALSE){ 
    if (map_rb[map_rb_head].inst_desc->dest_reg == INTEGER){
      regs.regs_R[map_rb[map_rb_head].inst_desc->out_arch_reg]=
	regs.regs_R[map_rb[map_rb_head].inst_desc->out_phy_reg];
    }
    else{ 
      if (map_rb[map_rb_head].inst_desc->double_prec == TRUE)
	regs.regs_F.d[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
	  regs.regs_F.d[map_rb[map_rb_head].inst_desc->out_phy_reg];
      else
	regs.regs_F.q[map_rb[map_rb_head].inst_desc->out_arch_reg] = 
	  regs.regs_F.q[map_rb[map_rb_head].inst_desc->out_phy_reg];
    }
  }
   
  if (map_rb[map_rb_head].inst_desc->dest_reg == INTEGER && 
      map_rb[map_rb_head].inst_desc->out_phy_reg != DNA &&
      map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
    map_put_free_int_reg(map_rb[map_rb_head].inst_desc->out_phy_reg);
  }
  else if (map_rb[map_rb_head].inst_desc->dest_reg == FLOATINGPT &&
	   map_rb[map_rb_head].inst_desc->out_phy_reg != DNA &&
	   map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
    map_put_free_fp_reg(map_rb[map_rb_head].inst_desc->out_phy_reg);
  }
  fetch_return_to_free_list(map_rb[map_rb_head].inst_desc);

  while (map_rb[map_rb_head].op1_deps != NULL){
    dep_chain_pointer = map_rb[map_rb_head].op1_deps;
    map_rb[map_rb_head].op1_deps = map_rb[map_rb_head].op1_deps->next;
    return_to_free_list(dep_chain_pointer);
  }
  while (map_rb[map_rb_head].op2_deps != NULL){
    dep_chain_pointer = map_rb[map_rb_head].op2_deps;
    map_rb[map_rb_head].op2_deps = map_rb[map_rb_head].op2_deps->next;
    return_to_free_list(dep_chain_pointer);
  }
  while (map_rb[map_rb_head].stdeps != NULL){
    dep_chain_pointer = map_rb[map_rb_head].stdeps;
    map_rb[map_rb_head].stdeps = map_rb[map_rb_head].stdeps->next;
    return_to_free_list(dep_chain_pointer);
  }
  while (map_rb[map_rb_head].cmovdeps != NULL){
    dep_chain_pointer = map_rb[map_rb_head].cmovdeps;
    map_rb[map_rb_head].cmovdeps = map_rb[map_rb_head].cmovdeps->next;
    return_to_free_list(dep_chain_pointer);
  }
  map_rb_head=(map_rb_head + 1) % (map_rb_nelem);
  map_rb_num--;
  while (map_rb_num > 0){
    map_rb[map_rb_head].in_LQ = map_rb[map_rb_head].in_SQ = FALSE;
    if (map_rb[map_rb_head].inst_desc->dest_reg == INTEGER && 
	map_rb[map_rb_head].inst_desc->out_phy_reg != DNA &&
	map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
      map_put_free_int_reg(map_rb[map_rb_head].inst_desc->out_phy_reg);
    }
    else if (map_rb[map_rb_head].inst_desc->dest_reg == FLOATINGPT &&
	     map_rb[map_rb_head].inst_desc->out_phy_reg != DNA &&
	     map_rb[map_rb_head].inst_desc->out_phy_reg != 31){
      map_put_free_fp_reg(map_rb[map_rb_head].inst_desc->out_phy_reg);
    }
    fetch_return_to_free_list(map_rb[map_rb_head].inst_desc);
    while (map_rb[map_rb_head].op1_deps != NULL){
      dep_chain_pointer = map_rb[map_rb_head].op1_deps;
      map_rb[map_rb_head].op1_deps = map_rb[map_rb_head].op1_deps->next;
      return_to_free_list(dep_chain_pointer);
    }
    while (map_rb[map_rb_head].op2_deps != NULL){
      dep_chain_pointer = map_rb[map_rb_head].op2_deps;
      map_rb[map_rb_head].op2_deps = map_rb[map_rb_head].op2_deps->next;
      return_to_free_list(dep_chain_pointer);
    }
    while (map_rb[map_rb_head].stdeps != NULL){
      dep_chain_pointer = map_rb[map_rb_head].stdeps;
      map_rb[map_rb_head].stdeps = map_rb[map_rb_head].stdeps->next;
      return_to_free_list(dep_chain_pointer);
    }
    while (map_rb[map_rb_head].cmovdeps != NULL){
      dep_chain_pointer = map_rb[map_rb_head].cmovdeps;
      map_rb[map_rb_head].cmovdeps = map_rb[map_rb_head].cmovdeps->next;
      return_to_free_list(dep_chain_pointer);
    }
    map_rb_head=(map_rb_head + 1) % (map_rb_nelem);
    map_rb_num--;
  }
  
  /* correct register mappings */
  for (i=0; i<MD_I_ARCH_REGS; i++){
    map_ir_mapping[i].phy_reg = i;
    map_ir_mapping[i].rbuf_no = DNA;
  }
  map_ir_mapping[32].phy_reg = 32;
  map_ir_mapping[32].rbuf_no = DNA;
  for (i = 0; i < MD_F_ARCH_REGS; i++){
    map_fr_mapping[i].phy_reg = i;
    map_fr_mapping[i].rbuf_no = DNA;
  }
  map_rb_head = map_rb_tail=0;

  /* return free registers from commit table */
  for (i=0; i < MD_I_ARCH_REGS; i++){
    if (commit_int_table[i] != DNA)
      map_put_free_int_reg(commit_int_table[i]);
    commit_int_table[i] = DNA;
  }
  for (i=0; i < MD_F_ARCH_REGS; i++){
    if (commit_fp_table[i] != DNA)
      map_put_free_fp_reg(commit_fp_table[i]);
    commit_fp_table[i] = DNA;
  }
  /* reset resource pool */
  for (index=0,i=0; i<N_ELT(res_fu_config); i++){
    for (j=0; j<res_fu_config[i].quantity; j++){
      res_fu_pool->resources[index].busy = FALSE;
      index++;
    }
  }
}

int valid_sq(struct load_store_queue *sq, unsigned int tag) {
  return sq->tag == tag;
}
