/*
* issue.c - issue stage implementation
*
* This file is part of the Alpha simulator tool suite written by
* Raj Desikan as part of the Bullseye project.
* Copyright (C) 1999 by Raj Desikan
*
* This source file is distributed "as is" in the hope that it will be
* useful.  It is distributed with no warranty, and no author or
* distributor accepts any responsibility for the consequences of its
* use. 
*
* Everyone is granted permission to copy, modify and redistribute
* this source file under the following conditions:
*
*    This tool set is distributed for non-commercial use only. 
*    Please contact the maintainer for restrictions applying to 
*    commercial use of these tools.
*
*    Permission is granted to anyone to make or distribute copies
*    of this source code, either as received or modified, in any
*    medium, provided that all copyright notices, permission and
*    nonwarranty notices are preserved, and that the distributor
*    grants the recipient permission for further redistribution as
*    permitted by this document.
*
*    Permission is granted to distribute this file in compiled
*    or executable form under the same conditions that apply for
*    source code, provided that either:
*
*    A. it is accompanied by the corresponding machine-readable
*       source code,
*    B. it is accompanied by a written offer, with no time limit,
*       to give anyone a machine-readable copy of the corresponding
*       source code in return for reimbursement of the cost of
*       distribution.  This written offer must permit verbatim
*       duplication by anyone, or
*    C. it is distributed by someone who received only the
*       executable form, and is accompanied by a copy of the
*       written offer of source code that they received concurrently.
*
* In other words, you are welcome to use, share and improve this
* source file.  You are forbidden to forbid anyone else to use, share
* and improve what you give them.
*
*/

#include <stdio.h>        
#include <stdlib.h>
#include <assert.h>
#include "alpha.h"
#include "regs.h"
#include "issue.h"
#include "writeback.h"
#include "resource.h"
#include "eventq.h"
#include "cache.h"
#include "bpred.h"
#include "fetch.h"
#include "map.h"


/* the issue queues are implemented as circular queues with a single pointer 
   because the map stage is solely 
   responsible for maintaining them. When an instruction is put on the ready 
   list, its issue queue entry is marked 
   with the cycle the entry should be removed from the list. The map stage 
   checks if the entry to which the pointer 
   is pointing is stale (i.e it was issued atleast ISSUE_HOLD cycles before) 
   and just overwrites it */ 

struct issue_window *IQ;	/* Integer issue queues */

struct issue_window *FQ;	/* FP issue queues */

struct rqueue_link *int_ready_queue;  /* integer ready queue */

struct rqueue_link *fp_ready_queue;   /* fp ready queue */ 

struct rqueue_link *event_queue;      /* event queue */


struct rqueue_link *ready_queue_free_list;  /* ready Q free list */

struct rqueue_link *event_queue_free_list;

int readyq_free_list_size;
int eventq_free_list_size;

/* integer issue width */
int issue_int_width;

/* fp issue width */
int issue_fp_width; 

/* Load Queue parameters */
int issue_lq_head;
int issue_lq_tail;
int issue_lq_nelem;
int issue_lq_num;
struct load_store_queue *issue_lq;

int int_reg_read_latency; /* Integer register file latency */
int fp_reg_read_latency; /* fp register file latency */
int full_bypass; /* If full bypassing is implemented for register read */
int int_bypass_allowed; /* Is integer result bypassing allowed? */
int fp_bypass_allowed; /* Is fp result bypassing allowed ?*/

/* If 1, disable slotting and clustering */
int issue_no_slot_clus;

void 
issue_stage_init(void){
   
  int i;
  /* allocate memory for the integer queue */
  IQ = (struct issue_window *) malloc(sizeof(struct issue_window));
  
  if (!IQ)
    fatal("out of virtual memory");
  
  /* alllocate memory for the fp queue */
  FQ = (struct issue_window *) malloc(sizeof(struct issue_window));
  
  if (!FQ)
    fatal("out of virtual memory");
  /* I am making this really big to incorporate collapsible issue queues. 
     We don't put a new entry in the IQ if there are already 
     map_int_issue_size entries in it. Same for FQ */
  IQ->queue_nelem = ISSUE_QUEUE_MAX_SIZE;
  FQ->queue_nelem = ISSUE_QUEUE_MAX_SIZE;
  IQ->queue_pointer=0;
  FQ->queue_pointer=0;
  IQ->queue_num=0;
  FQ->queue_num=0;
  /* Allocate memory for the IQ and FQ entries */
  IQ->window = (struct queue_elem *) 
    calloc(IQ->queue_nelem, sizeof(struct queue_elem));
  
  if (!IQ->window)
    fatal("Out of virtual memory");
  
  FQ->window = (struct queue_elem *) 
    calloc(FQ->queue_nelem, sizeof(struct queue_elem));
  
  if (!FQ->window)
    fatal("Out of virtual memory");
  
  
  readyq_free_list_size = eventq_free_list_size = 
    IQ->queue_nelem + FQ->queue_nelem;
  
  /* Initialize the ready queue free list */
  init_readyq_free_list();
  
  init_eventq_free_list();
  /* build the cluster resource pool */
  res_fu_pool = 
    res_create_pool("res-fu-pool", res_fu_config, N_ELT(res_fu_config)); 
  
  /* Initialize the load queue */
  issue_lq_head=0;
  issue_lq_tail=0;
  issue_lq_num=0;
  
  issue_lq = (struct load_store_queue *) 
    calloc(issue_lq_nelem, sizeof(struct load_store_queue));
  
  if (!issue_lq)
    fatal ("Out of virtual memory");
  
  for (i=0; i< issue_lq_nelem; i++)
    issue_lq[i].type = LQ;
}

/* Issue instructions to the func units in each of the clusters. 
   The  instructions are issued only when the 
   operands are available in the cluster and the funcional unit is free. 
   If the forwarded value is available in the 
   cluster, it is aquired without reading the register file. The head
   pointer points to the topmost entry of the issue queue. */

void 
issue_stage(){
  struct rqueue_link *prev, *node, *next_node;
  int  n_int_issued=0, n_fp_issued=0;
  struct res_template *fu;
  int lower_zero_assigned = FALSE;
  int upper_one_assigned = FALSE;
  
  /* Issue instructions from the fp issue queue first */

  for (prev=NULL,node=fp_ready_queue; node && n_fp_issued <
	 issue_fp_width; node = next_node){
    next_node = node->next;
    /* issue the instruction to a functional unit */
    
    if (ALPHA_OP_FUCLASS(node->qelem->inst_desc->op)) {
      /* fp loads use integer pipe. Lock functional unit and charge for
	 integer issue rather than fp issue */
      if (map_rb[node->qelem->inst_desc->r_buf_no].in_LQ == TRUE) {
	node->qelem->inst_desc->sub_clus_req = LOWER01;
	res_find_clus(res_fu_pool, node);
	if (node->qelem->clus[0] == TRUE && 
	    lower_zero_assigned == FALSE) {
	  node->qelem->inst_desc->sub_clus_req = LOWER0;
	  lower_zero_assigned = TRUE;
	}
	else if(node->qelem->clus[1] == TRUE) {
	  node->qelem->inst_desc->sub_clus_req = LOWER1;
	}
	else {
	  prev = node;
	  continue;
	}
      }
      /* special case if instruction is ITOF. We use integer functional
       unit and don't  charge for fp issue */
      else if (MD_IS_ITOF(node->qelem->inst_desc->op)) {
	node->qelem->inst_desc->sub_clus_req = LOWER01;
	res_find_clus(res_fu_pool, node);
	if (node->qelem->clus[0] == TRUE && 
	    lower_zero_assigned == FALSE) {
	  node->qelem->inst_desc->sub_clus_req = LOWER0;
	  lower_zero_assigned = TRUE;
	}
	else if(node->qelem->clus[1] == TRUE) {
	  node->qelem->inst_desc->sub_clus_req = LOWER1;
	}
	else {
	  prev = node;
	  continue;
	}
      }
      else {
	node->qelem->inst_desc->sub_clus_req = UPPERLOWER;
      }
      fu = res_get(res_fu_pool,node);
      if (fu){
	node->qelem->issued = TRUE;
	node->qelem->retire_cycle = sim_cycle + ISSUE_HOLD+1;
	/* reserve the functional unit */
	if (fu->master->busy > sim_cycle)
	  panic("functional unit already in use");
	if (MD_IS_ITOF(node->qelem->inst_desc->op)) {
	   fu->master->busy = fu->issuelat + sim_cycle + 3;
	   assert(sim_cycle >= node->qelem->issue_cycle+1);
	   
	   if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	     queue_event(node, sim_cycle + fu->oplat + 3);
	   else {
	     queue_event(node, sim_cycle + fu->oplat + 3 + 
			 int_reg_read_latency-
			 (sim_cycle-(node->qelem->issue_cycle+1)));
	   }
	}
	else if (map_rb[node->qelem->inst_desc->r_buf_no].in_LQ == TRUE) {
	/* Schedule functional unit release event */
	  fu->master->busy = fu->issuelat + sim_cycle;
	  node->inum = node->qelem->inst_desc->inum;
	  assert(sim_cycle >= node->qelem->issue_cycle+1);
	  if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	    queue_event(node, sim_cycle + fu->oplat);
	  else {
	    queue_event(node, sim_cycle + fu->oplat + 
			int_reg_read_latency - 
			(sim_cycle-(node->qelem->issue_cycle+1)));
	  }
	}
	else {
	/* Schedule functional unit release event */
	  fu->master->busy = fu->issuelat + sim_cycle;
	  node->inum = node->qelem->inst_desc->inum;
	  assert(sim_cycle >= node->qelem->issue_cycle+1);
	  if (node->qelem->issue_cycle + fp_reg_read_latency < sim_cycle)
	    queue_event(node, sim_cycle + fu->oplat);
	  else {
	    queue_event(node, sim_cycle + fu->oplat + 
			fp_reg_read_latency - 
			(sim_cycle-(node->qelem->issue_cycle+1)));
	  }
	}
	sim_total_insn++;
	if (map_rb[node->qelem->inst_desc->r_buf_no].in_LQ == TRUE 
	    || MD_IS_ITOF(node->qelem->inst_desc->op)) {
	  eventq_queue_callback(sim_cycle+3, 
				(void *) issue_int_dec, 0);
	  n_int_issued++;
	}
	else {
	  eventq_queue_callback(sim_cycle+3, 
				(void *) issue_fp_dec, 0);
	  n_fp_issued++;
	}
	/* release this entry from the ready queue */
	if (!prev){
	  fp_ready_queue = next_node;
	  readyq_return_to_free_list(node);
	}
	else {
	  prev->next = next_node;
	  readyq_return_to_free_list(node);
	}
      }
      else{
	/*cannot release this entry from the ready list */
	prev=node;
      }
    }
    else /* does not require a functional unit */{
      node->qelem->issued = TRUE;
      node->inum = node->qelem->inst_desc->inum;
      assert(sim_cycle >= node->qelem->issue_cycle+1);
      if (node->qelem->issue_cycle + fp_reg_read_latency < sim_cycle)
	queue_event(node, sim_cycle + 1);
      else {
	queue_event(node, sim_cycle + 1 + fp_reg_read_latency - 
		    (sim_cycle-(node->qelem->issue_cycle+1)));
      }
      sim_total_insn++;
      node->qelem->retire_cycle = sim_cycle + ISSUE_HOLD+1;
      n_fp_issued++;
      
      /* Remove this entry from issue queue after two cycles */
      eventq_queue_callback(sim_cycle+3, 
			    (void *) issue_fp_dec, 0);
      
      /* release this entry from the ready queue */
      if (!prev){
	fp_ready_queue = next_node;
	readyq_return_to_free_list(node);
	}
      else {
	prev->next = next_node;
	readyq_return_to_free_list(node);
      }
    }
  }
  /* visit all ready instructions i.e., insts whose register input 
     dependencies have been satisfied, stop issue when 
     no more instructions are available or issue bandwidth is
     exhausted. The ready list is not copied and blown off.  
     Instead each instruction is checked and if it can be issued, it
     is removed from the list. */ 
  
  for (prev=NULL,node=int_ready_queue; node &&  n_int_issued <
	 issue_int_width; node = next_node){
    next_node = node->next;
    /* issue the instruction to a functional unit. the RdPort condition is
       required for the simulator to work with early instruction retire 
       disabled */
    
    if (ALPHA_OP_FUCLASS(node->qelem->inst_desc->op) && 
	ALPHA_OP_FUCLASS(node->qelem->inst_desc->op) != RdPort) { 
      if (node->qelem->inst_desc->src_reg1 == FLOATINGPT)
	node->qelem->inst_desc->sub_clus_req = LOWER01;
      else if (node->qelem->inst_desc->slot_clus_assigned == LOWER) {
	if ((MD_IS_RETURN(node->qelem->inst_desc->op) || 
	     MD_IS_BRANCH(node->qelem->inst_desc->op) || 
	     MD_IS_INDIR(node->qelem->inst_desc->op)))
	  node->qelem->inst_desc->sub_clus_req = LOWER0;
	else
	  node->qelem->inst_desc->sub_clus_req = LOWER01;
      }
      else {
	if (ALPHA_OP_FUCLASS(node->qelem->inst_desc->op) == IntMULT)
	  node->qelem->inst_desc->sub_clus_req = UPPER1;
	else
	  node->qelem->inst_desc->sub_clus_req = UPPER01;
      }
      
      /* If static slotting is disabled, we can send simple ALU instructions 
	 to any subcluster */
#ifdef FLEXIBLE_SIM
      if (issue_no_slot_clus){
	enum md_opcode op = node->qelem->inst_desc->op;
	enum alpha_fu_class fu_req = ALPHA_OP_FUCLASS(op);
	if (fu_req == IntALU && 
	    (MD_IS_RETURN(op) || MD_IS_BRANCH(op) || MD_IS_INDIR(op))) 
	  node->qelem->inst_desc->sub_clus_req = LOWER0;
	else if(MD_IS_ITOF(op) || MD_IS_FTOI(op))
	  node->qelem->inst_desc->sub_clus_req = LOWER01;
	else if (fu_req == IntALU && 
		 ((MD_OP_FLAGS(op) & (F_CTRL|F_COND)) == (F_CTRL|F_COND)|| 
		  MD_IS_SHIFT(op))) 
	  node->qelem->inst_desc->sub_clus_req = UPPER01;
	else if (ALPHA_OP_FUCLASS(op) == IntMULT)
	  node->qelem->inst_desc->sub_clus_req = UPPER1;
	else if (map_rb[node->qelem->inst_desc->r_buf_no].in_LQ == TRUE ||
		 map_rb[node->qelem->inst_desc->r_buf_no].in_SQ == TRUE) {
	  node->qelem->inst_desc->sub_clus_req = LOWER01;
	}
	else
	  node->qelem->inst_desc->sub_clus_req = UPPERLOWER;
      }
#endif

      /* find which clusters are ready to execute this instruction */
      res_find_clus(res_fu_pool, node);

      if (node->qelem->inst_desc->src_reg1 == FLOATINGPT)
	node->qelem->inst_desc->sub_clus_req = LOWER01;
      else if (node->qelem->inst_desc->slot_clus_assigned == LOWER) {
	if ((MD_IS_RETURN(node->qelem->inst_desc->op) || 
	     MD_IS_BRANCH(node->qelem->inst_desc->op) || 
	     MD_IS_INDIR(node->qelem->inst_desc->op))) {
	  if (node->qelem->clus[0] == TRUE) {
	    node->qelem->inst_desc->sub_clus_req = LOWER0;
	    lower_zero_assigned = TRUE; 
	  }
	  else {
	    prev = node;
	    continue;
	  }
	}
	else {
	  if (node->qelem->clus[0] == TRUE && 
	      lower_zero_assigned == FALSE) {
	    node->qelem->inst_desc->sub_clus_req = LOWER0;
	    lower_zero_assigned = TRUE;
	  }
	  else if(node->qelem->clus[1] == TRUE) 
	    node->qelem->inst_desc->sub_clus_req = LOWER1;
	  else {
	    prev = node;
	    continue;
	  }
	}
      }
      else {
	if (ALPHA_OP_FUCLASS(node->qelem->inst_desc->op) == IntMULT) {
	  node->qelem->inst_desc->sub_clus_req = UPPER1;
	  upper_one_assigned = TRUE;
	}
	else {
	  if (node->qelem->clus[1] == TRUE &&
	      upper_one_assigned == FALSE) {
	    node->qelem->inst_desc->sub_clus_req = UPPER1;
	    upper_one_assigned = TRUE;
	  }
	  else if(node->qelem->clus[0] == TRUE)
	    node->qelem->inst_desc->sub_clus_req = UPPER0;
	  else {
	    prev = node;
	    continue;
	  }
	}
      }
      
      
      fu = res_get(res_fu_pool,node);
      
      if (fu){
	node->qelem->issued = TRUE;
	node->qelem->retire_cycle = sim_cycle + ISSUE_HOLD+1;
	/* reserve the functional unit */
	if (fu->master->busy > sim_cycle)
	  panic("functional unit already in use");
	
	/* Schedule functional unit release event  */
	/* If this is a jump instruction, add 6 cycles to latency */
	node->inum = node->qelem->inst_desc->inum;
	if ((MD_IS_INDIR(node->qelem->inst_desc->op))){
	  fu->master->busy = fu->issuelat + sim_cycle + 6;
	  assert(sim_cycle >= node->qelem->issue_cycle+1);
	  if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	    queue_event(node, sim_cycle + fu->oplat + 6);
	  else {
	    queue_event(node, sim_cycle + fu->oplat + 6 + 
			int_reg_read_latency-
			(sim_cycle-(node->qelem->issue_cycle+1)));
	  }
	}
	else if (MD_IS_BRANCH(node->qelem->inst_desc->op) ||
		 MD_IS_FTOI(node->qelem->inst_desc->op)) {
	  fu->master->busy = fu->issuelat + sim_cycle + 2;
	  assert(sim_cycle >= node->qelem->issue_cycle+1);
	  /* If there is full or partial bypass */
	  if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	    queue_event(node, sim_cycle + fu->oplat + 2);
	  else {
	    queue_event(node, sim_cycle + fu->oplat + 2 + 
			int_reg_read_latency-
			(sim_cycle-(node->qelem->issue_cycle+1)));
	  }
	}
	 /* If instructions is a CMOV fake its latency by adding 
	    an extra cycle */
	else if (MD_IS_CMOV(node->qelem->inst_desc->op)) {
	  fu->master->busy = fu->issuelat + sim_cycle + 1;
	  assert(sim_cycle >= node->qelem->issue_cycle+1);
	  if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	    queue_event(node, sim_cycle + fu->oplat + 1);
	  else {
	    queue_event(node, sim_cycle + fu->oplat + 1 + 
			int_reg_read_latency -
			(sim_cycle-(node->qelem->issue_cycle+1)));
	  }
	}
	else {
	  fu->master->busy = fu->issuelat + sim_cycle;
	  assert(sim_cycle >= node->qelem->issue_cycle+1);
	  if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	    queue_event(node, sim_cycle + fu->oplat);
	  else {
	    queue_event(node, sim_cycle + fu->oplat + 
			int_reg_read_latency -
			(sim_cycle-(node->qelem->issue_cycle+1)));
	  }
	}
	sim_total_insn++;
	n_int_issued++;
	/* If this is an fp store or FTOI instruction, fake communication
	   between IQ and FQ by removing the entry from the FQ */
	if (MD_IS_FTOI(node->qelem->inst_desc->op) || 
	    node->qelem->inst_desc->fpstore == TRUE) {
	  eventq_queue_callback(sim_cycle+3, 
			      (void *) issue_fp_dec, 0);
	}
	eventq_queue_callback(sim_cycle+3, 
			      (void *) issue_int_dec, 0);
	/* release this entry from the ready queue */
	if (!prev){
	  int_ready_queue = next_node;
	  readyq_return_to_free_list(node);
	}
	else {
	  prev->next = next_node;
	  readyq_return_to_free_list(node);
	}
      }
      else{
	/* cannot release this entry from the ready list */
	prev=node;
      }
    }
    else /* does not require a functional unit */{
      node->qelem->issued = TRUE;
      node->inum = node->qelem->inst_desc->inum;
      assert(sim_cycle >= node->qelem->issue_cycle+1);
      if (node->qelem->issue_cycle + int_reg_read_latency < sim_cycle)
	queue_event(node, sim_cycle + 1 );
      else {
	queue_event(node, sim_cycle + 1 + int_reg_read_latency -
		    (sim_cycle-(node->qelem->issue_cycle+1)));
      }

      sim_total_insn++;
      node->qelem->retire_cycle = sim_cycle + ISSUE_HOLD;
      n_int_issued++;
      eventq_queue_callback(sim_cycle+3, 
				  (void *) issue_int_dec, 0);
      /* release this entry from the ready queue */
      if (!prev){
	int_ready_queue = next_node;
	readyq_return_to_free_list(node);
      }
      else {
	prev->next = next_node;
	readyq_return_to_free_list(node);
      }
    }
  }
}

void issue_int_readyq_enqueue(struct rqueue_link *node){
  struct rqueue_link *new_node, *prev, *rqueue;
  /* get a free ready list node */
  new_node = get_free_ready_node(node);
  new_node->qelem = node->qelem;

  /* The ready queue is time sorted. This for loop inserts this
     instruction in the correct position in the queue */
  for (prev=NULL, rqueue=int_ready_queue;
       rqueue && ((node->qelem->inst_desc->inum > 
		   rqueue->qelem->inst_desc->inum && 
		   node->qelem->inst_desc->inum - 
		   rqueue->qelem->inst_desc->inum < map_rb_nelem) || 
		  (node->qelem->inst_desc->inum < 
		   rqueue->qelem->inst_desc->inum &&
		   rqueue->qelem->inst_desc->inum -
		   node->qelem->inst_desc->inum > map_rb_nelem));
       prev=rqueue,rqueue=rqueue->next);
  if (prev){
    /* insert middle or end */
    new_node->next = prev->next;
    prev->next = new_node;
  }
  else{
    /* insert at beginning */
    new_node->next = int_ready_queue;
    int_ready_queue = new_node;
  }
}

void issue_fp_readyq_enqueue(struct rqueue_link *node){
  struct rqueue_link *new_node, *prev, *rqueue;
  
  /* get a free ready list node */
  new_node = get_free_ready_node(node);
  new_node->qelem = node->qelem;
  
  /* The ready queue is time sorted. This for loop inserts this
     instruction in the correct position in the queue */
  for (prev=NULL, rqueue=fp_ready_queue;
       rqueue && ((node->qelem->inst_desc->inum > 
		   rqueue->qelem->inst_desc->inum && 
		   node->qelem->inst_desc->inum - 
		   rqueue->qelem->inst_desc->inum < map_rb_nelem) || 
		  (node->qelem->inst_desc->inum < 
		   rqueue->qelem->inst_desc->inum &&
		   rqueue->qelem->inst_desc->inum -
		   node->qelem->inst_desc->inum > map_rb_nelem)); 
       prev=rqueue,rqueue=rqueue->next);
  if (prev){
    /* insert middle or end */
    new_node->next = prev->next;
    prev->next = new_node;
  }
  else{
    /* insert at beginning */
    new_node->next = fp_ready_queue;
    fp_ready_queue = new_node;
  }
  
}
/* gets a node from the ready list. The details of the node to be
   enqueued are copied onto the node retrieved from the free list
   before returning */ 
struct rqueue_link *
get_free_ready_node(struct rqueue_link *node) {
  struct rqueue_link *new_node;
  
  /* if free list is exhausted, add some more entries */
  if(!ready_queue_free_list){
    init_readyq_free_list();
  }
  new_node = ready_queue_free_list;
  ready_queue_free_list = ready_queue_free_list->next;
  new_node->next = NULL;
  return new_node;
}

struct rqueue_link *
get_free_event_node(struct rqueue_link *node) {
  struct rqueue_link *new_node;
  
  /* if free list is exhausted, add some more entries */
  if(!event_queue_free_list){
    init_eventq_free_list();
  }
  new_node = event_queue_free_list;
  event_queue_free_list = event_queue_free_list->next;
  new_node->next = NULL;
  return new_node;
}

void init_readyq_free_list(){
  int i;
  struct rqueue_link *node;
  ready_queue_free_list = NULL;
  warn("increasing ready queue free list size");
  for (i=0; i<readyq_free_list_size ;i++){
    node = (struct rqueue_link *) malloc(sizeof(struct rqueue_link));
    if (!node)
      fatal("Out of virtual memory");
    node->next = ready_queue_free_list;
    ready_queue_free_list = node;
  }
}

void init_eventq_free_list(){
  int i;
  struct rqueue_link *node;
  event_queue_free_list = NULL;
  warn("increasing event queue free list size");
  for (i=0; i<eventq_free_list_size ;i++){
    node = (struct rqueue_link *) malloc(sizeof(struct rqueue_link));
    if (!node)
      fatal("Out of virtual memory");
    node->qelem = (struct queue_elem *) malloc(sizeof(struct queue_elem));
    if (!node->qelem)
      fatal("Out of virtual memory");
    node->next = event_queue_free_list;
    event_queue_free_list = node;
  }
}

void readyq_return_to_free_list(struct rqueue_link *node){
  node->next = ready_queue_free_list;
  ready_queue_free_list = node;
}

/* Insert an event for rqueue into the event queue. Event queue is
   sorted from earliest to latest event. Event and  
   associated side-effects will be apparent at the start of cycle WHEN */

void queue_event(struct rqueue_link *queue, tick_t when){
  struct rqueue_link *prev, *ev, *new_ev;
  
  
  if (when <= sim_cycle){
    panic("event occured in the past");
  }
  
  /* get a free event record */

  new_ev = get_free_event_node(queue);
  /* Copy the contents of the issue queue. The contents have to be
     copied because map will overwrite the contents of  
     the issue queue entry after ISSUE_HOLD number of cycles and if we
     have only a pointer in the eventq it will contain  
     the wrong data */
  assert(new_ev->qelem != NULL);
  *(new_ev->qelem) = *(queue->qelem);
  new_ev->x.when = when;
  new_ev->inum = new_ev->qelem->inst_desc->inum;
  
  /* locate insertion point */
  for (prev=NULL, ev=event_queue;ev && ev->x.when < when; prev=ev, ev=ev->next);
  if (prev){
    /* insert middle or end */
    new_ev->next = prev->next;
    prev->next = new_ev;
  }
  else{
    /* insert at the beginning */
    new_ev->next = event_queue;
    event_queue = new_ev;
  }
}

/* return the next event that has already occured, returns NULL when
   there are no remaining events or all remaining  
   events are in the future */

struct queue_elem *
issue_next_event(void){
  struct rqueue_link *ev;
  struct queue_elem *qelem;
  
  if (event_queue && event_queue->x.when <= sim_cycle){
    /* unlink and return first event on priority list */
    ev = event_queue;
    event_queue = event_queue->next;
      
    /* event still valid? */
    if (EVENT_VALID(ev)){
      qelem = ev->qelem;
      
      /* reclaim event record */
      EVENT_FREE(ev);
      
      /* is valid, return the Q part */
      return qelem;
    }
    else{
      /* receiving instruction was squashed, return next event */
      EVENT_FREE(ev);
      return issue_next_event();
    }
  }
  else{
    /* no event or no event is ready */
    return NULL;
  }
}

/* Decrement the number of int queue entries */
void issue_int_dec(tick_t now, int x) {
  assert(IQ->queue_num > 0);
  IQ->queue_num--;
}

/* Decrement the number of fp queue entries */
void issue_fp_dec(tick_t now, int x) {
  assert(FQ->queue_num > 0);
  FQ->queue_num--;
}

