/*
 * sim-fast.c - sample fast functional simulator implementation (for functional
 * simulator debugging)
 *
 * This file is part of the Alpha simulator tool suite written by
 * Raj Desikan as part of the Bullseye project.
 * It has been written by extending the SimpleScalar tool suite written by
 * Todd M. Austin as a part of the Multiscalar Research Project.
 *  
 * 
 * Copyright (C) 1994, 1995, 1996, 1997, 1998 by Todd M. Austin
 *
 * Copyright (C) 1999 by Raj Desikan
 *
 * This source file is distributed "as is" in the hope that it will be
 * useful.  The tool set comes with no warranty, and no author or
 * distributor accepts any responsibility for the consequences of its
 * use. 
 * 
 * Everyone is granted permission to copy, modify and redistribute
 * this tool set under the following conditions:
 * 
 *    This source code is distributed for non-commercial use only. 
 *    Please contact the maintainer for restrictions applying to 
 *    commercial use.
 *
 *    Permission is granted to anyone to make or distribute copies
 *    of this source code, either as received or modified, in any
 *    medium, provided that all copyright notices, permission and
 *    nonwarranty notices are preserved, and that the distributor
 *    grants the recipient permission for further redistribution as
 *    permitted by this document.
 *
 *    Permission is granted to distribute this file in compiled
 *    or executable form under the same conditions that apply for
 *    source code, provided that either:
 *
 *    A. it is accompanied by the corresponding machine-readable
 *       source code,
 *    B. it is accompanied by a written offer, with no time limit,
 *       to give anyone a machine-readable copy of the corresponding
 *       source code in return for reimbursement of the cost of
 *       distribution.  This written offer must permit verbatim
 *       duplication by anyone, or
 *    C. it is distributed by someone who received only the
 *       executable form, and is accompanied by a copy of the
 *       written offer of source code that they received concurrently.
 *
 * In other words, you are welcome to use, share and improve this
 * source file.  You are forbidden to forbid anyone else to use, share
 * and improve what you give them.
 *
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/*
 * This file implements a very fast functional simulator.  This functional
 * simulator implementation is much more difficult to digest than the simpler,
 * cleaner sim-safe functional simulator.  By default, this simulator performs
 * no instruction error checking, as a result, any instruction errors will
 * manifest as simulator execution errors, possibly causing sim-fast to
 * execute incorrectly or dump core.  Such is the price we pay for speed!!!!
 *
 * The following options configure the bag of tricks used to make sim-fast
 * live up to its name.  For most machines, defining all the options results
 * in the fastest functional simulator.
 */

/* don't count instructions flag, enabled by default, disable for inst count */
#undef NO_INSN_COUNT

#ifdef __GNUC__
/* faster dispatch mechanism, requires GNU GCC C extensions, CAVEAT: some
   versions of GNU GCC core dump when optimizing the jump table code with
   optimization levels higher than -O1 */
/* #define USE_JUMP_TABLE */
#endif /* __GNUC__ */

#include "host.h"
#include "misc.h"
#include "alpha.h"
#include "ssregs.h"
#include "ssmemory.h"
#include "ssloader.h"
#include "sssyscall.h"
#include "sim.h"
#include "issue.h"

/* simulated registers */
struct myregs_t myregs;

struct mymem_t *mymem = NULL; 

void 
my_sim_init(void)
{
  my_regs_init(&myregs);
  mymem = my_mem_create("mymem");
  my_mem_init(mymem);
}

void 
my_sim_load_prog(char *fname,
		  int argc, char **argv,
		  char **envp)
{
    my_ld_load_prog(fname, argc, argv, envp, &myregs, mymem, TRUE);           
}

/*
 * configure the execution engine
 */

#undef SET_NPC
#undef CPC
#undef GPR
#undef SET_GPR
#undef FPR_L
#undef SET_FPR_L
#undef FPR_F
#undef SET_FPR_F
#undef FPR_D
#undef SET_FPR_D
#undef SET_HI
#undef HI
#undef SET_LO
#undef LO
#undef FCC
#undef SET_FCC
#undef FPR_Q
#undef SET_FPR_Q
#undef FPR
#undef SET_FPR
#undef FPCR
#undef SET_FPCR
#undef UNIQ
#undef SET_UNIQ
#undef READ_BYTE
#undef READ_HALF
#undef READ_WORD
#undef READ_QUAD
#undef WRITE_BYTE
#undef WRITE_HALF
#undef WRITE_WORD
#undef WRITE_QUAD


/* next program counter */
#define SET_NPC(EXPR)		(myregs.regs_NPC = (EXPR))

/* current program counter */
#define CPC			(myregs.regs_PC)

/* general purpose registers */
#define GPR(N)			(myregs.regs_R[N])
#define SET_GPR(N,EXPR)		(myregs.regs_R[N] = (EXPR))

#if defined(TARGET_PISA)

/* floating point registers, L->word, F->single-prec, D->double-prec */
#define FPR_L(N)		(myregs.regs_F.l[(N)])
#define SET_FPR_L(N,EXPR)	(myregs.regs_F.l[(N)] = (EXPR))
#define FPR_F(N)		(myregs.regs_F.f[(N)])
#define SET_FPR_F(N,EXPR)	(myregs.regs_F.f[(N)] = (EXPR))
#define FPR_D(N)		(myregs.regs_F.d[(N) >> 1])
#define SET_FPR_D(N,EXPR)	(myregs.regs_F.d[(N) >> 1] = (EXPR))

/* miscellaneous register accessors */
#define SET_HI(EXPR)		(myregs.regs_C.hi = (EXPR))
#define HI			(myregs.regs_C.hi)
#define SET_LO(EXPR)		(myregs.regs_C.lo = (EXPR))
#define LO			(myregs.regs_C.lo)
#define FCC			(myregs.regs_C.fcc)
#define SET_FCC(EXPR)		(myregs.regs_C.fcc = (EXPR))

#elif defined(TARGET_ALPHA)

/* floating point registers, L->word, F->single-prec, D->double-prec */
#define FPR_Q(N)		(myregs.regs_F.q[N])
#define SET_FPR_Q(N,EXPR)	(myregs.regs_F.q[N] = (EXPR))
#define FPR(N)			(myregs.regs_F.d[N])
#define SET_FPR(N,EXPR)		(myregs.regs_F.d[N] = (EXPR))

/* miscellaneous register accessors */
#define FPCR			(myregs.regs_C.fpcr)
#define SET_FPCR(EXPR)		(myregs.regs_C.fpcr = (EXPR))
#define UNIQ			(myregs.regs_C.uniq)
#define SET_UNIQ(EXPR)		(myregs.regs_C.uniq = (EXPR))

#else
#error No ISA target defined...
#endif

/* precise architected memory state accessor macros */
#define READ_BYTE(SRC, FAULT)						\
  ((FAULT) = md_fault_none, MEM_READ_BYTE(mymem, (SRC)))
#define READ_HALF(SRC, FAULT)						\
  ((FAULT) = md_fault_none, MEM_READ_HALF(mymem, (SRC)))
#define READ_WORD(SRC, FAULT)						\
  ((FAULT) = md_fault_none, MEM_READ_WORD(mymem, (SRC)))
#ifdef HOST_HAS_QUAD
#define READ_QUAD(SRC, FAULT)						\
  ((FAULT) = md_fault_none, MEM_READ_QUAD(mymem, (SRC)))
#endif /* HOST_HAS_QUAD */

#define WRITE_BYTE(SRC, DST, FAULT)					\
  ((FAULT) = md_fault_none, MEM_WRITE_BYTE(mymem, (DST), (SRC)))
#define WRITE_HALF(SRC, DST, FAULT)					\
  ((FAULT) = md_fault_none, MEM_WRITE_HALF(mymem, (DST), (SRC)))
#define WRITE_WORD(SRC, DST, FAULT)					\
  ((FAULT) = md_fault_none, MEM_WRITE_WORD(mymem, (DST), (SRC)))
#ifdef HOST_HAS_QUAD
#define WRITE_QUAD(SRC, DST, FAULT)					\
  ((FAULT) = md_fault_none, MEM_WRITE_QUAD(mymem, (DST), (SRC)))
#endif /* HOST_HAS_QUAD */

/* system call handler macro */
#define SYSCALL(INST)	my_sys_syscall(&myregs, my_mem_access, mymem, INST, TRUE)

#ifndef NO_INSN_COUNT
#define INC_INSN_CTR()	sim_num_insn++
#else /* !NO_INSN_COUNT */
#define INC_INSN_CTR()	/* nada */
#endif /* NO_INSN_COUNT */

#ifdef TARGET_ALPHA
#define ZERO_FP_REG()	myregs.regs_F.d[MD_REG_ZERO] = 0.0
#else
#define ZERO_FP_REG()	/* nada... */
#endif

counter_t my_sim_num_insn=0;

void
my_sim_main(void)
{
#ifdef USE_JUMP_TABLE
  /* the jump table employs GNU GCC label extensions to construct an array
     of pointers to instruction implementation code, the simulator then uses
     the table to lookup the location of instruction's implementing code, a
     GNU GCC `goto' extension is then used to jump to the inst's implementing
     code through the op_jump table; as a result, there is no need for
     a main simulator loop, which eliminates one branch from the simulator
     interpreter - crazy, no!?!? */

  /* instruction jump table, this code is GNU GCC specific */
  static void *op_jump[/* max opcodes */] = {
    &&opcode_NA, /* NA */
#define DEFINST(OP,MSK,NAME,OPFORM,RES,FLAGS,O1,O2,I1,I2,I3)		\
    &&opcode_##OP,
#define DEFLINK(OP,MSK,NAME,MASK,SHIFT)					\
    &&opcode_##OP,
#define CONNECT(OP)
#include "ssmachine.def"
  };
#endif /* USE_JUMP_TABLE */

  /* register allocate instruction buffer */
  register md_inst_t inst;

  /* decoded opcode */
  register enum md_opcode op;

  
  /* must have natural byte/word ordering */
  if (sim_swap_bytes || sim_swap_words)
    fatal("sim: *fast* functional simulation cannot swap bytes or words");

#ifdef USE_JUMP_TABLE

  myregs.regs_NPC = myregs.regs_PC;

  /* load instruction */
  inst = __UNCHK_MEM_READ(mymem, myregs.regs_NPC, md_inst_t);

  /* jump to instruction implementation */
  MD_SET_OPCODE(op, inst);
  goto *op_jump[op];

#define DEFINST(OP,MSK,NAME,OPFORM,RES,FLAGS,O1,O2,I1,I2,I3)		\
  opcode_##OP:								\
    /* maintain $r0 semantics */					\
    myregs.regs_R[MD_REG_ZERO] = 0;					\
    ZERO_FP_REG();							\
									\
    /* keep an instruction count */					\
    /*INC_INSN_CTR();*/							\
									\
    /* locate next instruction */					\
    myregs.regs_PC = myregs.regs_NPC;					\
									\
    /* set up default next PC */					\
    myregs.regs_NPC += sizeof(md_inst_t);					\
									\
    /* execute the instruction */					\
    SYMCAT(OP,_IMPL);							\
									\
    /* get the next instruction */					\
    inst = __UNCHK_MEM_READ(mymem, myregs.regs_NPC, md_inst_t);		\
									\
    /* jump to instruction implementation */				\
    MD_SET_OPCODE(op, inst);						\
    goto *op_jump[op];

#define DEFLINK(OP,MSK,NAME,MASK,SHIFT)					\
  opcode_##OP:								\
    panic("attempted to execute a linking opcode");
#define CONNECT(OP)
#define DECLARE_FAULT(FAULT)						\
	  { /* uncaught... */ }
#include "ssmachine.def"

  opcode_NA:
    panic("attempted to execute a bogus opcode");

  /* should not get here... */
  panic("exited sim-fast main loop");

#else /* !USE_JUMP_TABLE */

  /* set up initial default next PC */
  myregs.regs_NPC = myregs.regs_PC + sizeof(md_inst_t);

  //while (TRUE)
  //{
      /* maintain $r0 semantics */
      myregs.regs_R[MD_REG_ZERO] = 0;
#ifdef TARGET_ALPHA
      myregs.regs_F.d[MD_REG_ZERO] = 0.0;
#endif /* TARGET_ALPHA */

      /* keep an instruction count */

      /* load instruction */
      inst = __UNCHK_MEM_READ(mymem, myregs.regs_PC, md_inst_t);

      /* decode the instruction */
      MD_SET_OPCODE(op, inst);

      /* execute the instruction */
      switch (op)
	{
#define DEFINST(OP,MSK,NAME,OPFORM,RES,FLAGS,O1,O2,I1,I2,I3)		\
	case OP:							\
	  SYMCAT(OP,_IMPL);						\
	  break;
#define DEFLINK(OP,MSK,NAME,MASK,SHIFT)					\
	case OP:							\
	  panic("attempted to execute a linking opcode");
#define CONNECT(OP)
#define DECLARE_FAULT(FAULT)						\
	  { /* uncaught... */break; }
#include "ssmachine.def"
	default:
	  panic("attempted to execute a bogus opcode");
	}
      //printf("executing %s for %lx in sim-fast\n",MD_OP_NAME(op),(long int)myregs.regs_PC);
      /* execute next instruction */
      my_sim_num_insn++;
      myregs.regs_PC = myregs.regs_NPC;
      myregs.regs_NPC += sizeof(md_inst_t);
      //}

#endif /* USE_JUMP_TABLE */
}
