/*
 * cache_common.c - cache module routines for functional simulation
 *
 * This file is a part of the SimpleScalar tool suite, originally
 * written by Todd M. Austin as a part of the Multiscalar Research 
 * Project. The file has been rewritten by Doug Burger, as a
 * part of the Galileo research project.  Alain Kagi has also 
 * contributed to this code.
 *
 * The tool suite is currently maintained by Doug Burger and Todd M. Austin.
 * 
 * Copyright (C) 1994, 1995, 1996, 1997 by Todd M. Austin
 *
 * This source file is distributed "as is" in the hope that it will be
 * useful.  The tool set comes with no warranty, and no author or
 * distributor accepts any responsibility for the consequences of its
 * use. 
 * 
 * Everyone is granted permission to copy, modify and redistribute
 * this tool set under the following conditions:
 * 
 *    This source code is distributed for non-commercial use only. 
 *    Please contact the maintainer for restrictions applying to 
 *    commercial use.
 *
 *    Permission is granted to anyone to make or distribute copies
 *    of this source code, either as received or modified, in any
 *    medium, provided that all copyright notices, permission and
 *    nonwarranty notices are preserved, and that the distributor
 *    grants the recipient permission for further redistribution as
 *    permitted by this document.
 *
 *    Permission is granted to distribute this file in compiled
 *    or executable form under the same conditions that apply for
 *    source code, provided that either:
 *
 *    A. it is accompanied by the corresponding machine-readable
 *       source code,
 *    B. it is accompanied by a written offer, with no time limit,
 *       to give anyone a machine-readable copy of the corresponding
 *       source code in return for reimbursement of the cost of
 *       distribution.  This written offer must permit verbatim
 *       duplication by anyone, or
 *    C. it is distributed by someone who received only the
 *       executable form, and is accompanied by a copy of the
 *       written offer of source code that they received concurrently.
 *
 * In other words, you are welcome to use, share and improve this
 * source file.  You are forbidden to forbid anyone else to use, share
 * and improve what you give them.
 *
 * INTERNET: dburger@cs.wisc.edu
 * US Mail:  1210 W. Dayton Street, Madison, WI 53706
 *
 * $Id: cache_common.c,v 1.1.6.1 2000/04/03 19:57:10 hrishi Exp $
 *
 * $Log: cache_common.c,v $
 * Revision 1.1.6.1  2000/04/03 19:57:10  hrishi
 * Added another file that was overlooked the last time
 *
 * Revision 1.3  1998/09/28 05:16:52  dburger
 * Sampling, subblocking, cache interface all work, memory leak and deadlock solved
 *
 * Revision 1.2  1998/09/16 21:10:22  dburger
 * Code apparently works w/ subblocking for sim-outorder AND sim-cache, no sampling yet
 *
 * Revision 1.1  1998/09/12 19:14:49  dburger
 * Code works, subblock support not yet put in
 *
 * Revision 1.3  1998/09/12 19:05:04  dburger
 * Code works for compress, doesn't support subblocking
 *
 * Revision 1.2  1998/09/12 19:04:00  dburger
 * Code works for compress, doesn't support subblocking
 *
 * Revision 1.1  1998/02/23 05:59:30  dburger
 * rs->blocking, cache_restart_access, istall_buf bugs fixed
 *
 *
 * Revision 1.1  1997/12/18 20:17:36  dburger
 * Just adding hamming/subblock stuff now
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>

#include "cache.h"
#include "eventq.h"
#include "loader.h"
#include "misc.h"
#include "tlb.h"
#include "bus.h"

/* Head to list of unused cache buffer entries */
cache_access_packet *cache_packet_free_list;

/* Here's a general function version of the above two macros */
unsigned int
count_valid_bits(struct cache *cp, unsigned int vector)
{
  int i, num = 0;

  for (i = 0; i < cp->subblock_ratio; i++)
    num += (CACHE_GET_SB_BIT(i, vector) != 0);
  return(num);
}

/* unlink BLK from the hash table bucket chain in SET */
void
unlink_htab_ent(struct cache *cp,		/* cache to update */
		struct cache_set *set,		/* set containing bkt chain */
		struct cache_blk *blk)		/* block to unlink */
{
  struct cache_blk *prev, *ent;
  int index = CACHE_HASH(cp, blk->tag);

  /* locate the block in the hash table bucket chain */
  for (prev=NULL,ent=set->hash[index];
       ent;
       prev=ent,ent=ent->hash_next)
    {
      if (ent == blk)
	break;
    }
  assert(ent);

  /* unlink the block from the hash table bucket chain */
  if (!prev)
    {
      /* head of hash bucket list */
      set->hash[index] = ent->hash_next;
    }
  else
    {
      /* middle or end of hash bucket list */
      prev->hash_next = ent->hash_next;
    }
  ent->hash_next = NULL;
}

/* insert BLK onto the head of the hash table bucket chain in SET */
void
link_htab_ent(struct cache *cp,		/* cache to update */
	      struct cache_set *set,	/* set containing bkt chain */
	      struct cache_blk *blk)	/* block to insert */
{
  int index = CACHE_HASH(cp, blk->tag);

  /* insert block onto the head of the bucket chain */
  blk->hash_next = set->hash[index];
  set->hash[index] = blk;
}

/* insert BLK into the order way chain in SET at location WHERE */
void
update_way_list(struct cache_set *set,	/* set contained way chain */
		struct cache_blk *blk,	/* block to insert */
		enum list_loc_t where)	/* insert location */
{
  /* unlink entry from the way list */
  if (!blk->way_prev && !blk->way_next)
    {
      /* only one entry in list (direct-mapped), no action */
      assert(set->way_head == blk && set->way_tail == blk);
      /* Head/Tail order already */
      return;
    }
  /* else, more than one element in the list */
  else if (!blk->way_prev)
    {
      assert(set->way_head == blk && set->way_tail != blk);
      if (where == Head)
	{
	  /* already there */
	  return;
	}
      /* else, move to tail */
      set->way_head = blk->way_next;
      blk->way_next->way_prev = NULL;
    }
  else if (!blk->way_next)
    {
      /* end of list (and not front of list) */
      assert(set->way_head != blk && set->way_tail == blk);
      if (where == Tail)
	{
	  /* already there */
	  return;
	}
      set->way_tail = blk->way_prev;
      blk->way_prev->way_next = NULL;
    }
  else
    {
      /* middle of list (and not front or end of list) */
      assert(set->way_head != blk && set->way_tail != blk);
      blk->way_prev->way_next = blk->way_next;
      blk->way_next->way_prev = blk->way_prev;
    }

  /* link BLK back into the list */
  if (where == Head)
    {
      /* link to the head of the way list */
      blk->way_next = set->way_head;
      blk->way_prev = NULL;
      set->way_head->way_prev = blk;
      set->way_head = blk;
    }
  else if (where == Tail)
    {
      /* link to the tail of the way list */
      blk->way_prev = set->way_tail;
      blk->way_next = NULL;
      set->way_tail->way_next = blk;
      set->way_tail = blk;
    }
  else
    panic("bogus WHERE designator");
}

/* parse policy */
enum cache_policy			/* replacement policy enum */
cache_char2policy(char c)		/* replacement policy as a char */
{
  switch (c) {
  case 'l': return LRU;
  case 'r': return Random;
  case 'f': return FIFO;
  default: fatal("bogus replacement policy, `%c'", c);
  }
}

/* parse policy */
enum cache_trans			/* replacement policy enum */
cache_string2trans(char *s)
{
  if (!mystricmp(s, "vivt"))
    return VIVT;
  else if (!mystricmp(s, "vipt"))
    return VIPT;
  else if (!mystricmp(s, "pipt"))
    return PIPT;
  else if (!mystricmp(s, "pivt"))
    fatal("Physically indexed, virtually tagged caches not supported!\n");
  else
    fatal("Unrecognized cache translation policy specified (VIVT, VIPT, PIPT are legal\n");
  return PIPT;	/* Default, should never happen */
}

/* print cache configuration */
void
cache_config(struct cache *cp,		/* cache instance */
	     FILE *stream)		/* output stream */
{
  fprintf(stream,
	  "cache %s: %d size, %d sets, %d byte blocks\n",
	  cp->name, cp->assoc * cp->nsets * cp->bsize, cp->nsets, cp->bsize);
  fprintf(stream,
	  "cache %s: %d-way, %s replacement policy, write-back\n",
	  cp->name, cp->assoc,
	  cp->policy == LRU ? "LRU"
	  : cp->policy == Random ? "Random"
	  : cp->policy == FIFO ? "FIFO"
	  : (abort(), ""));
  if (cp->prefetch)
    fprintf(stream, "cache: %s: prefetching...\n", cp->name);
  else
    fprintf(stream, "cache: %s: no prefetch\n", cp->name);
}

struct cache_blk *
find_blk_match_no_jump(struct cache *cp,
		       md_addr_t set,
		       md_addr_t tag)
{
  struct cache_blk *repl;

  if (!cp->hsize)
  {
    for (repl = cp->sets[set].way_head; repl; repl=repl->way_next)
    {
      if ((repl->tag == tag) && (repl->status & CACHE_BLK_VALID))
      break;
    }
  }
  else
  {
    int hindex = CACHE_HASH(cp, tag);

    for (repl=cp->sets[set].hash[hindex]; repl; repl=repl->hash_next)
    {
      if (repl->tag == tag && (repl->status & CACHE_BLK_VALID))
      break;
    }
  }
  return(repl);
}

void *
cache_follow(struct cache *cp, 
             md_addr_t addr, 
             enum resource_type *type)
{
  int index = 0;
  void *next_cache;

  if (cp->num_resources > 1)
  {
    switch(cp->resource_code) {
    case 0:
      break;
    default:    
      break;
    }
  }
  *type = cp->resource_type[index];
  next_cache = (void *)cp->resources[index];

  if (*type == Bus)
  {
      struct bus *bus = (struct bus *)cp->resources[index];
      *type = bus->resource_type[index];
      assert(*type != Bus);
      next_cache = (void *)bus->resources[index];
  }

  return next_cache;
}

/* Allocates CACHE_ACCESS_PACKET_FREE_LIST entries of type cache_access_packet
   onto the appropriate free list */
void increase_cache_packet_free_list()
{
  int i;
  cache_access_packet *temp;
  static int free_flag = 0;

  if (free_flag)
    fprintf(stderr, "Warning: Increasing cache_packet_free_list\n");

  free_flag++;
  cache_packet_free_list = NULL;

  for (i=0; i<CACHE_ACCESS_PACKET_FREE_LIST; i++)
    {
      temp = (cache_access_packet *) malloc(sizeof(cache_access_packet));
      if (temp == NULL)
	fatal("Malloc on cache access packet free list expansion failed");
      temp->next = cache_packet_free_list;
      cache_packet_free_list = temp;
    }
}

/* Create a "cache access packet", which is used to hold the transient information
   for a specific access to a specific cache.  Makes blocking the request (and 
   subsequently restarting) much cleaner. Could differentiate between functional
   and timing versions by not initializing the last three fields in the functional
   version (more efficient), but that's probably not a lot of overhead.  */
inline cache_access_packet *
cache_create_access_packet(void *mp, 			/* Pointer to level in the memory hierarchy */
			   unsigned int cmd,
			   md_addr_t addr, 
			   enum trans_cmd vorp,
			   int nbytes, 
			   void *obj, 
			   RELEASE_FN_TYPE release_fn,
			   VALID_FN_TYPE valid_fn, 
			   MSHR_STAMP_TYPE stamp)
{
  cache_access_packet *temp;

  /* If free list is exhausted, add some more entries */
  if (!cache_packet_free_list)
    {
      increase_cache_packet_free_list();
    }

  temp = cache_packet_free_list;
  cache_packet_free_list = cache_packet_free_list->next;

  /* Initialize fields of buffering function */
  temp->cp = mp;
  temp->cmd = cmd;
  temp->addr = addr;
  temp->vorp = vorp;
  temp->nbytes = nbytes;
  temp->obj = obj;
  temp->release_fn = release_fn;
  temp->valid_fn = valid_fn;
  temp->stamp = stamp;

  return(temp);
}

void
cache_free_access_packet(cache_access_packet *buf)
{
  buf->next = cache_packet_free_list;
  cache_packet_free_list = buf;
}

/* return non-zero if block containing address ADDR is contained in cache
   CP, this interface is used primarily for debugging and asserting cache
   invariants. */
/* Note that this scheme won't work in the context of VIPT caches ... in
   which case either both addresses will need to be passed, or the function
   should translate the virtual address */
int					/* non-zero if access would hit */
cache_probe(struct cache *cp,		/* cache instance to probe */
	    md_addr_t addr)		/* address of block to probe */
{
  md_addr_t tag = CACHE_TAG(cp, addr);
  md_addr_t set = CACHE_SET(cp, addr);
  md_addr_t sb = CACHE_SB_TAG(cp, addr);
  struct cache_blk *blk;

  /* permissions are checked on cache misses */

  if (cp->hsize)
  {
    /* higly-associativity cache, access through the per-set hash tables */
    int hindex = CACHE_HASH(cp, tag);
    
    for (blk=cp->sets[set].hash[hindex];
	 blk;
	 blk=blk->hash_next)
    {	
      if ((blk->tag == tag && (blk->status & CACHE_BLK_VALID)) &&
	  ((!IS_BLOCK_SUBBLOCKED(blk)) || 
	   (CACHE_GET_SB_BIT(sb, blk->sb_valid))))
	  return TRUE;
    }
  }
  else
  {
    /* low-associativity cache, linear search the way list */
    for (blk=cp->sets[set].way_head;
	 blk;
	 blk=blk->way_next)
    {
      if ((blk->tag == tag && (blk->status & CACHE_BLK_VALID)) &&
	  ((!IS_BLOCK_SUBBLOCKED(blk)) || 
	   (CACHE_GET_SB_BIT(sb, blk->sb_valid))))
	  return TRUE;
    }
  }
  
  /* cache block not found */
  return FALSE;
}


