/*
 * rambus.h - rambus timing module interface
 *
 * This file is a part of the SimpleScalar tool suite, written by
 * Todd M. Austin as a part of the Multiscalar Research Project.
 * The file has been substantially modified by Doug Burger, as a
 * part of the Galileo research project.  Alain Kagi has also 
 * contributed to this code.
 *  
 * The tool suite is currently maintained by Doug Burger and Todd M. Austin.
 * 
 * Copyright (C) 1994, 1995, 1996, 1997 by Todd M. Austin
 *
 * This source file is distributed "as is" in the hope that it will be
 * useful.  The tool set comes with no warranty, and no author or
 * distributor accepts any responsibility for the consequences of its
 * use. 
 * 
 * Everyone is granted permission to copy, modify and redistribute
 * this tool set under the following conditions:
 * 
 *    This source code is distributed for non-commercial use only. 
 *    Please contact the maintainer for restrictions applying to 
 *    commercial use.
 *
 *    Permission is granted to anyone to make or distribute copies
 *    of this source code, either as received or modified, in any
 *    medium, provided that all copyright notices, permission and
 *    nonwarranty notices are preserved, and that the distributor
 *    grants the recipient permission for further redistribution as
 *    permitted by this document.
 *
 *    Permission is granted to distribute this file in compiled
 *    or executable form under the same conditions that apply for
 *    source code, provided that either:
 *
 *    A. it is accompanied by the corresponding machine-readable
 *       source code,
 *    B. it is accompanied by a written offer, with no time limit,
 *       to give anyone a machine-readable copy of the corresponding
 *       source code in return for reimbursement of the cost of
 *       distribution.  This written offer must permit verbatim
 *       duplication by anyone, or
 *    C. it is distributed by someone who received only the
 *       executable form, and is accompanied by a copy of the
 *       written offer of source code that they received concurrently.
 *
 * In other words, you are welcome to use, share and improve this
 * source file.  You are forbidden to forbid anyone else to use, share
 * and improve what you give them.
 *
 * INTERNET: dburger@cs.wisc.edu
 * US Mail:  1210 W. Dayton Street, Madison, WI 53706
 *
 * This file was written by Wei-fen Lin at the University of Michigan
 *
 * $Id: rambus.h,v 1.1.4.1 2000/04/03 19:48:07 hrishi Exp $
 *
 * Revision 1.1  1998/10/04  18:50:23  dburger
 * Initial revision
 *
 *
 */

/* The RAMBUS model is built up based on the datasheet,
"64-72Mbit Direct RDRAM (256K*16/18*16d)", which is issued by RAMBUS Inc.
on 3/12/98. The document is available at 
http://www.rambus.com/html/direct_documentation.html */


#ifndef __RAMBUS_H
#define __RAMBUS_H

/* Define timing parameters. See page43 on datasheet */
#define _tRC 32
#define _tRAS 24
#define _tPACKET 4
#define _tRCD 9
#define _tCAC 8
#define _tRDP 4
#define _tRP 8
#define _tCWD 8 // It's 6 in data sheet

/* ---------------------
	RDRAM configurations
	
A channel is 16-bit wide and can fetch 16 bits per 1.25ns.
n channels has bandwidth 16*n bits and can fetch 16*n bits per 1.25ns */

/* the number of channels (Has to be the power of 2 )*/
#define rdram_channel_num 4 

/* the total number of devices (This has to be multiple of rdram_channel_num) */
#define rdram_device_num 32 

/* the number of devices per channel ( has to be the power of 2) */
#define rdram_device_per_channel (rdram_device_num/rdram_channel_num)


/*--------------------------------- 
 	 Definition of Queue
----------------------------------*/

typedef struct _QITEM{
		int DevIndex;
		int BankIndex;
		int RowIndex;
		int ColIndex;
		int Size;
		tick_t EndTime;
		int Type;		
		struct _QITEM *Next;		
}QITEM;


typedef struct{
		
		QITEM *Head,*Tail;
		int Size;/* the size of the queue*/

}QUEUE;


/*-----------------------------------
   Definition of RDRAM structures
-------------------------------------*/

typedef enum{ IDLE, BANK0, BANK1}SENSEAMPSTATUS;

typedef struct{
		SENSEAMPSTATUS Status; /* Idle or BANK0 or BANK1 */
		int RowIndex;
}RDRAM_SENSEAMP;

typedef struct{
		RDRAM_SENSEAMP *RAmp;//SA0, SA1_2, SA3_4, SA5_6, SA7_8,SA9_10, SA11_12, SA13_14, SA_15 	
		RDRAM_SENSEAMP *LAmp;//SA0_1, SA2_3, SA4_5, SA6_7, SA8_9, SA10_11,SA12_13,SA14_15	
}RDRAM_DEVICE;
 

/*  the relatin to the previous RDRAM access */
 
typedef enum {	RDRAMFIRSTACCESS, // this can be conbined with SAFIRSTACCESS
					SAFIRSTACCESS, // first time touch the senseamp 
					DIFFDEV_SAACT, // different device, activated senseamp 
					DIFFDEV_SANONACT, // different device, nonactivated senseamp 
				   SAMEDEV_NONADJBANK_SAACT, // same device, nonadjacent bank,activated senseamp 
					SAMEDEV_NONADJBANK_SANONACT,// same device, nonadjacent bank, nonactivated senseamp *
					SAMEDEV_ADJBANK,// same device , adjacent bank 
				   SAMEDEV_SAMEBANK_DIFFROW, // same device, same bank, different row 
					SAMEDEV_SAMEBANK_SAMEROW  // same device, same bank, same row 
				}RELATION;


typedef struct{

		RDRAM_DEVICE *Device;	
		int No_of_Device;	
                int tRC, tRAS, tPACKET, tRCD, tCAC, tRDP, tRP, tCWD;
		struct {
			int DevIndex;
			int BankIndex;
			int RowIndex;
			int Size;
			int Type;
			tick_t EndTime;
		}PreAccess;
		QUEUE *Process;
		struct{
		      SS_COUNTER_TYPE Access[SAMEDEV_SAMEBANK_SAMEROW+1];		
		      SS_COUNTER_TYPE ReadRefs;
		      SS_COUNTER_TYPE WriteRefs;
		      SS_COUNTER_TYPE TlbRefs;
		      SS_COUNTER_TYPE PreRefs;
		      SS_COUNTER_TYPE ReadHits;/* including Read,Tlb, and Prefetch */
		      SS_COUNTER_TYPE WriteHits;/* including Write */
		      SS_COUNTER_TYPE IdleTime;
		      SS_COUNTER_TYPE AveReadQueuingTime;/*including Read,Tlb, and Prefetch */
		      SS_COUNTER_TYPE AveWriteQueuingTime;/* including Write */			
		}Stats;

}_RAMBUS;
				

/* Exported interface functions */

void rambus_init(_RAMBUS *rb, int cycle_ratio);
tick_t rambus_access(_RAMBUS *rb,int addr,int size, int type,tick_t now);
void rambus_timing_reg_stats( _RAMBUS *rb, struct stat_sdb_t *sdb, char *name);


#endif




