# SimpleScalar Visualization Engine
# Graph generation Functions
#
# *** PUT COPYRIGHT STUFF HERE ***
#=======================================================================#

package require Tk 8.0

# Certain global arrays that have several graph options.

global IntegerGraphOptions
global FloatGraphOptions
global TextGraphOptions
global ColorGraphOptions 
global GraphOptionValue
global ColorGraphOptionValue
global GraphNames

# These variables control modal dialog windows and the points that are
# yet to be displayed.

global conf_close LastPointDisplayed ResizeCount

# Certain graph display control parameters.

global Xdist Ydist HashLineWidth UnitLineWidth GLineColor GTextColor GBgColor LabelToAxis MAX_XSIZE

# All the procedures for displaying graphs.

proc init_graph {} {

    # NOTE : Option names should not include the character |

    global IntegerGraphOptions FloatGraphOptions TextGraphOptions
    global ColorGraphOptions 
    global GraphNames MAX_XSIZE ResizeCount
    global Xdist Ydist HashLineWidth UnitLineWidth GLineColor GTextColor GBgColor LabelToAxis
    
    set IntegerGraphOptions [list x-hash y-hash]
    set FloatGraphOptions [list width(cm) height(cm) x-scale y-scale]
    set TextGraphOptions [list x-label y-label]
    set ColorGraphOptions [list "line color" "axes color"]
    set GraphNames [list "ipc" "dl1misses" \
	    "il1misses" "bpredict"]

    # Several configuration distances in pixels.
    
    set Xdist 40
    set Ydist 40
    set HashLineWidth 3
    set UnitLineWidth 6
    set LabelToAxis 10

    # The maximum x-span of the graph in cms.

    set MAX_XSIZE 200

    # Initialize ResizeCount to 0.

    foreach label $GraphNames {

	set ResizeCount($label) 0

    }

}


# Returns 1 if graph displayed successfully. Else, returns 0.

proc display_graph {label posx posy {ReadConfigFile 1} } {

    global GraphOptionValue
    global ColorGraphOptionValue
    global Xdist Ydist HashLineWidth UnitLineWidth GTextColor GBgColor LabelToAxis
    global GLineColor GLabelColor
    global LastPointDisplayed MAX_XSIZE
    set GLabelFont "-*-Helvetica-Bold-R-Normal--*-120-*"

    # these are defined in stat.tcl to help place the location
    # of the statistics window
    global xOffset
    global yOffset

    # An ugly way to determine what the X,Y offsets are
    set unknown [expr [eval string compare $posx "Unk"] == 0]
    set def   [expr [ info exists posx]]
    set undef [expr !$def]
    if {$unknown || $undef} {
	set X $xOffset
    } else {
	set X $posx
    }
    # *whew* do it again:
    set unknown [expr [eval string compare $posy "Unk"] == 0]
    set def   [expr [ info exists posy]]
    set undef [expr !$def]
    if {$unknown || $undef} {
	set Y +${yOffset}
    } else {
	set Y $posy
    }
    set nextWindowPlacement 100

    # Get the resolution
    set PixelsPerPoint [tk scaling -displayof .]
    set PixelsPerCm [expr 28.8*$PixelsPerPoint]
    
    # Get the corresponding statistic name.
    
    switch -exact $label {
	"ipc" { 
		set StatName sim_IPC 
		set nextWindowPlacement 100
	}
	"dl1misses" {
	    # set CacheName [ss::getOption "-cache:dcache"]
	    set CacheName "dl1"
	    set StatName "${CacheName}.miss_rate"
	    set nextWindowPlacement 100
	}
	"il1misses" {
	    # set CacheName [ss::getOption "-cache:icache"]
	    set CacheName "il1"
	    set StatName "${CacheName}.miss_rate"
	    set nextWindowPlacement 100
	}
	"bpredict" { 
	    set PredictorType [ss::getOption "-bpred"]
	    set PredictorType "bpred_${PredictorType}"
	    set StatName "${PredictorType}.bpred_dir_rate"
	    set nextWindowPlacement 100
	}
    }
    set screenHigh [winfo vrootheight .]
    set yhi [expr $yOffset + $nextWindowPlacement]
    set screenOverlap [expr $yhi - $screenHigh]
    set halfNextWindowHeight [expr $nextWindowPlacement * 0.5]
    # puts "overlap: $screenOverlap; half height: $halfNextWindowHeight"
    if { $screenOverlap  > $halfNextWindowHeight } {
        # it is time to re-set and re-position at the top of the screen
	set yOffset 4
	set xOffset "-40"
    } else {
	set yOffset [expr $yOffset + $nextWindowPlacement]
    }
    # Display the graph window at the appropriate place with the appropriate
    # buttons.
    
    if { [winfo exists ".${label}_graph"] } {
		wm withdraw .${label}_graph 
		wm deiconify ".${label}_graph"
    } else {
	set LastPointDisplayed($label) 0
	set Width [winfo width .]
	set Height [winfo height .]
	
	if { $ReadConfigFile == 1 } {

	    if { [read_graph_config $label] == 0 } {
		
		return 0
	    }
	}

	# Read up all the graph configuration parameters into temporaries.
	set Width $GraphOptionValue(${label},width\(cm\))
	set Height $GraphOptionValue(${label},height\(cm\))
	set Xhash $GraphOptionValue(${label},x-hash)
	set Yhash $GraphOptionValue(${label},y-hash)
	set Xlabel $GraphOptionValue(${label},x-label)
	set Ylabel $GraphOptionValue(${label},y-label)
	set Xscale $GraphOptionValue(${label},x-scale)
	set Yscale $GraphOptionValue(${label},y-scale)
	set PixelsPerXHash [expr $PixelsPerCm/$Xhash]
	set PixelsPerYHash [expr $PixelsPerCm/$Yhash]
	# New!
	set LineColor $GraphOptionValue(${label},line color)
	set AxesColor $GraphOptionValue(${label},axes color)
	if {[info exists LineColor]} {
	} else {
	    set LineColor $GLineColor
	}
	if {[info exists AxesColor]} {
	} else {
	    set AxesColor $GLabelColor
	}
	set Width [expr $PixelsPerCm*$Width]
	set Height [expr $PixelsPerCm*$Height]

	toplevel ".${label}_graph" -background $GBgColor
	wm title ".${label}_graph" $Ylabel
	wm transient .${label}_graph .
	
	# Create various TK widgets necessary for displaying the graph.
	
	# The display frame that holds the canvas graph widgets 
	# and the scrollbar.
	frame .${label}_graph.display -relief flat -background $GBgColor \
		-borderwidth 0
	
	# Three canvas items. Two are static and the third is bound
	# to the scrollbar.
	
        ###################	
	canvas .${label}_graph.display.graph -width [expr $Width-$Ydist-10] \
		-height [expr $Height-$Xdist+16] -bg $GBgColor -bd 0 \
		-xscrollcommand [list .${label}_graph.display.xscrollbar set] \
		-relief flat
		# fix bug
	.${label}_graph.display.graph configure -confine 0 \
		-scrollregion [list 60 0 "${MAX_XSIZE}c" $Height]
        ###################	
	
	# The two static canvas widgets.
	canvas .${label}_graph.display.yaxis -width [expr $Ydist+10] \
		-height $Height \
		-bg $GBgColor -bd 0 -relief flat
	canvas .${label}_graph.display.xaxis -width [expr $Width-$Ydist-10] \
		-height [expr $Xdist-15] -bg $GBgColor -bd 0 -relief flat
        ###################	
	# The scrollbar widget.
	scrollbar .${label}_graph.display.xscrollbar \
		-orient horizontal \
		-width 10 \
		-activerelief groove \
		-elementborderwidth 1 \
		-command [list .${label}_graph.display.graph xview]

        ###################	
	# The controls frame that holds the control buttons.
	frame .${label}_graph.controls -relief flat -borderwidth 0 \
		-background $GBgColor
	button .${label}_graph.controls.config -text "Configure..." \
		-command [list edit_config_data $label]
	button .${label}_graph.controls.hide -text "Close" \
		-command [list destroy ".${label}_graph"]
	
	# Pack all the widgets.
	    
	pack .${label}_graph.display.yaxis -side left -fill both

	pack .${label}_graph.display.xaxis -side bottom -fill both

	pack .${label}_graph.display.graph -side top -fill both

	pack .${label}_graph.display -side top -fill both

	pack .${label}_graph.controls.config -side right -anchor e -padx 120
	pack .${label}_graph.controls.hide -side left -anchor e -padx 120
	pack .${label}_graph.controls -side bottom -fill x -ipadx 5 -ipady 5
	pack .${label}_graph.display.xscrollbar -side bottom -fill both
	
	#  OLD: wm geometry .${label}_graph +50+100
	# use X Windows geometry style--always include the
	# leading minus sign or leading plus sign
	#
	# puts "X:$X;Y:$Y."
	set res [wm geometry .${label}_graph "=$X$Y" ]
	
	# If the graph window is resized, then scale the graph accordingly.
	
	# Display the actual graph data.
	# Draw the axes and display their labels.
	
	set Graph .${label}_graph.display.graph
	set YAxis .${label}_graph.display.yaxis
	set XAxis .${label}_graph.display.xaxis
	
	$YAxis create line [expr $Ydist+7] 0 [expr $Ydist+7] $Height \
		-arrow both \
		-fill $AxesColor
	$YAxis create line 0 [expr $Height-$Xdist] [expr $Ydist+10] \
		[expr $Height-$Xdist] -arrow first \
		-fill $AxesColor
	$Graph create line 0 [expr $Height-$Xdist] "${MAX_XSIZE}c" \
		[expr $Height-$Xdist] -arrow last \
		-fill $AxesColor
	$YAxis create text [expr $Ydist-$LabelToAxis] \
		[expr $Height-$Xdist+$LabelToAxis] \
		-fill $AxesColor \
		-text "0"
	
	set Factor 1
	for { set NewText $Xscale } { $NewText >= 10 } \
		{ set NewText [expr $NewText/10] } {
	    set Factor [expr $Factor*10]
	}
	
	$XAxis create text [expr $Width/2] 10 \
		-fill $LineColor \
		-font $GLabelFont \
		-text "$Xlabel ( x $Factor )" -justify center

	# Display all the units and hashes on the X-axis
	
	set i 0
	set DisplayedText [expr round($NewText)]
	
	for { set NextXUnit [expr $i+$PixelsPerCm] } \
		{ $i <= [expr $MAX_XSIZE*$PixelsPerCm-20] } \
		{ set i [expr $i+$PixelsPerXHash] } {
	    
	    if { $NextXUnit > $i } { 
		
		$Graph create line $i \
			[expr $Height-$Xdist-$HashLineWidth/2] \
			$i [expr $Height-$Xdist+$HashLineWidth/2] \
			-fill $AxesColor
	    } else {
		
		$Graph create line $i \
			[expr $Height-$Xdist-$UnitLineWidth/2] \
			$i [expr $Height-$Xdist+$UnitLineWidth/2] \
			-fill $AxesColor
		$Graph create text $i \
			[expr $Height-$Xdist+$LabelToAxis] \
			-fill $AxesColor \
			-text $DisplayedText
		set DisplayedText [expr round($DisplayedText+$NewText)]
		set NextXUnit [expr $i+$PixelsPerCm]
		
	    }
	    
	}
	
	set Factor 1
	for { set NewText $Yscale } { $NewText >= 10 } \
		{ set NewText [expr $NewText/10] } {
	    set Factor [expr $Factor*10]
	}
	
        set curY 20
	set curIndex 0
	set charactersRemaining [string length $Ylabel]
	while {$charactersRemaining > 0} {
	    set thisChar [string index $Ylabel $curIndex]
	    # write it
	    $YAxis create text 10 $curY \
		-fill $LineColor \
		-font $GLabelFont \
		-text "$thisChar" -justify center
	    set curY [expr $curY + 15 ]
	    set curIndex [expr $curIndex + 1]
	    set charactersRemaining [expr $charactersRemaining - 1]
	}
	# Nah--don't bother
	# set curY [expr $curY + 15 ]
	# $YAxis create text 10 $curY -text "(x" \
		# -fill $GLabelColor -font $GLabelFont
	# set curY [expr $curY + 15 ]
	# $YAxis create text 10 $curY -text "$Factor)" \
		# -fill $GLabelColor -font $GLabelFont
	
	# Display all the units and hashes on the Y-axis
	
	set i [expr $Height-$Xdist]
	set DisplayedText $NewText
	set DisplayedText [format "%.2f" $DisplayedText]
	
	for { set NextYUnit [expr $i-$PixelsPerCm] } \
		{ $i >=  20 } \
		{ set i [expr $i-$PixelsPerYHash] } {
	    
	    if { $NextYUnit < $i } { 
		
		$YAxis create line [expr $Xdist-$HashLineWidth/2+7] $i \
			[expr $Xdist+$HashLineWidth/2+7] $i \
			-fill $AxesColor
	    } else {
		
		$YAxis create line [expr $Xdist-$UnitLineWidth/2+7] $i \
			[expr $Xdist+$UnitLineWidth/2+7] $i \
			-fill $AxesColor
		$YAxis create text [expr $Ydist-2*$LabelToAxis+7] $i \
			-fill $AxesColor 	\
			-text $DisplayedText
		set DisplayedText [expr $DisplayedText+$NewText]
		set DisplayedText [format "%.2f" $DisplayedText]
		set NextYUnit [expr $i-$PixelsPerCm]
		
	    }
	    
	}
	
	if { [llength $::allCycles] >= 4 } {
	    
	    # Display all the collected values till the current cycle.
	    
	    for { set CycleIndex 1} \
		    { $CycleIndex < [expr {[llength $::allCycles]-2}] } \
		    { set CycleIndex [expr {$CycleIndex+1}] } {
		
		set prevY [stat_value ::allStatistics $StatName $CycleIndex]
		set currY [stat_value ::allStatistics $StatName \
			[expr $CycleIndex+1] ]
		set prevX [stat_cycle ::allCycles $CycleIndex]
		set currX [stat_cycle ::allCycles [expr $CycleIndex+1] ]
		
		if { [string compare $prevY ""] != 0 && \
			[string compare $currY ""] != 0 } {
		    
		    $Graph create line \
			    [expr {$prevX*$PixelsPerCm/$Xscale}] \
			    [expr {$Height-$prevY*$PixelsPerCm/$Yscale \
			    -$Xdist}] \
			    [expr {$currX*$PixelsPerCm/$Xscale}] \
			    [expr {$Height-$currY*$PixelsPerCm/$Yscale \
			    -$Xdist}] \
			    -width 2  -fill $LineColor\
			    -smooth 1 -joinstyle round -splinesteps 4
		    
		}
		
		set LastPointDisplayed($label) $CycleIndex
		
	    }
	}
    }

    return 1

}
 

# This proc updates the graph dynamically to show the results as simulation
# proceeds.

proc display_graph_data {} {

    global GraphNames
    global GraphOptionValue
    global ColorGraphOptionValue
    global GLineColor GLabelColor
    global Xdist Ydist LastPointDisplayed

    # Display the actual curve every time the display rate is reached.
    
    foreach GraphName $GraphNames {
	
	if { [winfo exists ".${GraphName}_graph"] } {
	    
	    # wm withdraw .${GraphName}_graph 
	    wm deiconify .${GraphName}_graph 
	    # Get the graph options.
	    # New!
	    set LineColor $GraphOptionValue(${GraphName},line color)
	    set AxesColor $GraphOptionValue(${GraphName},axes color)
	    if {[info exists LineColor]} {
	    } else {
		set LineColor $GLineColor
	    }
	    if {[info exists AxesColor]} {
	    } else {
		set AxesColor $GLabelColor
	    }
	    set Height $GraphOptionValue(${GraphName},height\(cm\))
	    set Xscale $GraphOptionValue(${GraphName},x-scale)
	    set Yscale $GraphOptionValue(${GraphName},y-scale)
	    set PixelsPerPoint [tk scaling -displayof ".${GraphName}_graph"]
	    set PixelsPerCm [expr 28.8*$PixelsPerPoint]
	    set Height [expr $PixelsPerCm*$Height]
	    
	    # Get the corresponding statistic name.
	    
	    switch -exact $GraphName {
		
		"ipc" { set StatName sim_IPC}
		
		"dl1misses" {
		    # set CacheName [ss::getOption "-cache:dcache"]
	    	    set CacheName "dl1"
		    set StatName "${CacheName}.miss_rate"
		}
		"il1misses" {
		    # set CacheName [ss::getOption "-cache:icache"]
	    	    set CacheName "il1"
		    set StatName "${CacheName}.miss_rate"
		}
		"bpredict" { 
		    set PredictorType [ss::getOption "-bpred"]
		    set PredictorType "bpred_${PredictorType}"
		    set StatName "${PredictorType}.bpred_dir_rate"
		}
	    }
	    
	    # Get the endpoints of the line that needs to be drawn.
	    
	    if { [llength $::allCycles] > 3} {
		
		for { set CycleIndex \
			[expr {$LastPointDisplayed($GraphName)+1}] } \
			{ $CycleIndex < [expr {[llength $::allCycles]-2}] } \
			{ set CycleIndex [expr {$CycleIndex+1}] } {
		    
		    set prevY [stat_value ::allStatistics $StatName \
			    $CycleIndex]
		    set currY [stat_value ::allStatistics $StatName \
			    [expr $CycleIndex+1] ]
		    set prevX [stat_cycle ::allCycles $CycleIndex]
		    set currX [stat_cycle ::allCycles [expr $CycleIndex+1] ]
		    
		    # Draw the line corresponding to the latest figures.
		    
		    if { [string compare $prevY ""] != 0 && \
			    [string compare $currY ""] != 0 } {
			
			.${GraphName}_graph.display.graph create line \
				[expr {$prevX*$PixelsPerCm/$Xscale}] \
				[expr $Height-$prevY*$PixelsPerCm/$Yscale \
				-$Xdist] \
				[expr {$currX*$PixelsPerCm/$Xscale}] \
				[expr $Height-$currY*$PixelsPerCm/$Yscale \
				-$Xdist] \
				-width 2 \
				-fill $LineColor -smooth 1 \
				-joinstyle round \
				-splinesteps 4
		    }
		    set LastPointDisplayed($GraphName) $CycleIndex
		}
	    }
	}
    }
    return 1
}


# Returns 1 if config data was successfully read from the file. Else,
# returns 0.

proc read_graph_config {label} {
    
    global IntegerGraphOptions
    global FloatGraphOptions
    global TextGraphOptions
    global ColorGraphOptions 
    global GraphOptionValue
    global ColorGraphOptionValue
    global MYHOME

    set FileName "${MYHOME}/.ss-viz/${label}.gconf"

    if { [ catch { set "${label}_rfid" [open $FileName r] } ] } {

	tk_messageBox -parent . -title Error -type ok -icon error \
		-message "Couldnt open the file $FileName for reading"
	return 0

    }
    set AllGraphOptions [concat $FloatGraphOptions $IntegerGraphOptions \
	    $TextGraphOptions ]
    
    set Command "gets \${${label}_rfid} OptionString"
    while { [eval $Command] >= 0 } {

	regexp {^\|(.*)\|(.*)\|$} $OptionString x OptionName \
		OptionValue

	# If an invalid option name is encountered, report the error and
	# don't show the graph.

	if { [lsearch $AllGraphOptions $OptionName] == -1} {

	    if { [lsearch $ColorGraphOptions $OptionName] == -1} {
		tk_messageBox -parent . -title Error -type ok -icon error \
			-message "Option $OptionName is an invalid option. Cannot display the $label graph"
		return 0
	    } else {
		# save for writing out the config values
		set ColorGraphOptionValue($OptionName) $OptionValue
	    }
	}
	# puts "graph $label: $OptionName=$OptionValue"
	set GraphOptionValue(${label},${OptionName}) $OptionValue
    }

    set Command "close \${${label}_rfid}"
    eval $Command

    # If erroneous option values are found, force the user to correct them.
    set ErrorMesg "No error"
    if { ![config_values_ok $label ErrorMesg] } {

	tk_messageBox -parent . -title Error -type ok -icon error \
		-message $ErrorMesg	
	
	edit_config_data $label

    }
    return 1
}

# Returns 1 if the options are saved successfully. Else, returns 0.

proc set_graph_config {label} {

    global FloatGraphOptions
    global IntegerGraphOptions
    global TextGraphOptions
    global ColorGraphOptions 
    global ColorGraphOptionValue
    global MYHOME

    set DisplayFrame ".${label}_graph.configwin.datafr"
    set AllGraphOptions [concat $FloatGraphOptions $IntegerGraphOptions \
	    $TextGraphOptions ]

    set FileName "${MYHOME}/.ss-viz/${label}.gconf"
    
    if { [ catch { set "${label}_wfid" [open $FileName w] } ] } {

	tk_messageBox -parent . -title Error -type ok -icon error \
		-message "Couldn't save options - unable to open file $FileName for writing"
	return 0

    }

    foreach Option $AllGraphOptions {

	set Value [${DisplayFrame}.${Option}_ent get]
	set Command "puts \${${label}_wfid} |\${Option}|\${Value}|"
	eval $Command

    }
    # special case for now--we have to write out the original 
    # color values because we don't give the user a chance to edit 
    # them
    foreach Option $ColorGraphOptions {
	set Value $ColorGraphOptionValue($Option) 
	set Command "puts \${${label}_wfid} |\${Option}|\${Value}|"
	eval $Command
    }
    set Command "close \${${label}_wfid}"
    eval $Command
    return 1
}

proc edit_config_data {label} {

    global IntegerGraphOptions
    global FloatGraphOptions
    global TextGraphOptions
    global ColorGraphOptions 
    global GraphOptionValue
    global ColorGraphOptionValue
    global conf_close
    
    toplevel .${label}_graph.configwin

    set DisplayFrame ".${label}_graph.configwin.datafr"
    set ButtonFrame ".${label}_graph.configwin.buttonfr"
    frame $DisplayFrame -relief flat
    
    # add $ColorGraphOptions below
    set AllGraphOptions [concat $FloatGraphOptions $IntegerGraphOptions \
	    $TextGraphOptions]

    foreach Option $AllGraphOptions {

	label "${DisplayFrame}.${Option}_lab" -text $Option
	entry "${DisplayFrame}.${Option}_ent" -width 15 -justify center
	"${DisplayFrame}.${Option}_ent" insert 0 \
		$GraphOptionValue(${label},${Option});
    }

    frame $ButtonFrame -relief groove -borderwidth 2

    button $ButtonFrame.ok -text "OK" -command [list save_config_data \
	    $label conf_close $DisplayFrame]

    button $ButtonFrame.cancel -text "Cancel" \
	    -command [list set conf_close 0]

    set Row 0
    set Column 0
    for {set i 0 } { $i < [llength $AllGraphOptions] } { set i [expr $i+2] } {

	set Option1 [lindex $AllGraphOptions $i]
	if { [catch {set Option2 [lindex $AllGraphOptions [expr $i+1]] } ] } {

	    grid config "${DisplayFrame}.${Option1}_lab" \
		    -row $Row -column $Column

	    set Column [expr $Column+1]

	    grid config "${DisplayFrame}.${Option1}_ent" \
		    -row $Row -column $Column

	} else {
	    grid config "${DisplayFrame}.${Option1}_lab" \
		    -row $Row -column $Column

	    set Column [expr $Column+1]
		    
	    grid config "${DisplayFrame}.${Option1}_ent"\
		    -row $Row -column $Column

	    set Column [expr $Column+1]

	    grid config "${DisplayFrame}.${Option2}_lab" \
		    -row $Row -column $Column

	    set Column [expr $Column+1]
		    
	    grid config "${DisplayFrame}.${Option2}_ent" \
		    -row $Row -column $Column
		    
	}

	set Row [expr $Row+1]
	set Column 0

    }

    grid configure $ButtonFrame.ok $ButtonFrame.cancel
    
    pack $DisplayFrame $ButtonFrame -side top
    # set firstChar [string index $label 0]
    # set firstChar [string toupper $firstChar]
    # set restOfLabel [string range $label 1 "end"]
    # set labelTitle ${firstChar}${restOfLabel}
    # set labelTitle [string toupper $label]
    # wm title .${label}_graph.configwin "$labelTitle Values"
    set Ylabel $GraphOptionValue(${label},y-label)
    wm title ".${label}_graph.configwin" "$Ylabel Values"

    tkwait variable conf_close
    destroy .${label}_graph.configwin

    if {$conf_close == 1} {

	# Configuration might have changed. Redisplay the graph so that
	# the new configuration is reflected.

	set RootX [winfo rootx ".${label}_graph"]
        set SPlusI  [string first "+" $RootX]
	if { $SPlusI < 0 } {
	       # puts "Need an X plus"
	       set RootX \+$RootX
        }
	set RootY [winfo rooty ".${label}_graph"]
        set SPlusI  [string first "+" $RootY]
	if { $SPlusI < 0 } {
	       # puts "Need an Y plus"
	       set RootY \+$RootY
        }
	destroy ".${label}_graph"
	display_graph $label $RootX $RootY
    }
}

proc config_values_ok {label Message} {

    global IntegerGraphOptions
    global FloatGraphOptions
    global TextGraphOptions
    global ColorGraphOptions 
    global GraphOptionValue
    global ColorGraphOptionValue

    upvar $Message ErrorLink

    foreach Option $ColorGraphOptions {
	set ValueIndex "${label},${Option}"
	# Note: $ValueIndex looks like:
	# "ipc,line color"
	# "ipc,axes color"
    }
    foreach Option $IntegerGraphOptions {
	set ValueIndex "${label},${Option}"
	if {![string is integer -strict $GraphOptionValue($ValueIndex)]} {
	    set ErrorLink "Option $Option for $label has to be an integer"
	    return 0
	}

    }
    foreach Option $FloatGraphOptions {

	set ValueIndex "${label},${Option}"
	if {![string is double -strict $GraphOptionValue($ValueIndex)]} {
	    set ErrorLink "Option $Option for $label has to be a float"
	    return 0
	}

    }
    foreach Option $TextGraphOptions {

	set ValueIndex "${label},${Option}"
	if {![string is ascii -strict $GraphOptionValue($ValueIndex)]} {
	    set ErrorLink "Option $Option for $label has to be some text"
	    return 0
	}

    }
    set Width $GraphOptionValue(${label},width\(cm\))
    set maxWidth 30
    if {$Width > $maxWidth} {
	set $GraphOptionValue(${label},width\(cm\)) $maxWidth
	set ErrorLink "Sorry:  The maximum allowable width is $maxWidth centimeters."
	return 0
    }
    set Height $GraphOptionValue(${label},height\(cm\))
    set maxHeight 20
    if {$Height > $maxHeight} {
	set $GraphOptionValue(${label},height\(cm\)) $maxHeight
	    set ErrorLink "Sorry:  The maximum allowable height is $maxHeight"
	    return 0
	}
	return 1
    }

    proc save_config_data {label conf Display} {

	global GraphOptionValue
	global ColorGraphOptionValue
	global IntegerGraphOptions
	global FloatGraphOptions
	global TextGraphOptions
	global ColorGraphOptions 

	upvar $conf conf_handle

	set ErrorMesg "No error"
	set AllGraphOptions [concat $FloatGraphOptions $IntegerGraphOptions \
		$TextGraphOptions]

	#Set GraphOptionValue from the widget to reflect the new values.

	foreach Option $AllGraphOptions {

	    set Value [${Display}.${Option}_ent get]
	    set GraphOptionValue(${label},${Option}) $Value

	}
	if {![config_values_ok $label ErrorMesg]} {
	    
	    # Reports an incorrect config entry
	    
	    tk_messageBox -parent .${label}_graph.configwin -title Error \
		    -type ok -icon error -message $ErrorMesg

	    # Reset GraphOptionValue from the file.

	    read_graph_config $label
	    
	} else {
	    
	    # Writes the config data to the appropriate file.
	    
	    set_graph_config $label
	    set conf_handle 1
	}
    }

    # Resizes the graph for Configure events. Not used as of now. Might be 
    # useful in the future.

    proc resize_graph {label NewWidth NewHeight} {

	global ResizeCount GraphOptionValue
	global Xdist Ydist GBgColor MAX_XSIZE HashLineWidth UnitLineWidth
	global LastPointDisplayed LabelToAxis

	if { $ResizeCount($label) == 0 } {
	    # puts "Resize count is 0 now."
	    set ResizeCount($label) 1
	    return

	} else {
	    # puts "Resize count is 1 now"
	    # Set GraphOptionValue to reflect the new parameters.
	    
	    set GraphOptionValue(${label},width\(cm\)) $NewWidth
	    set GraphOptionValue(${label},height\(cm\)) $NewHeight

	    set Width $NewWidth
	    set Height $NewHeight
	    # puts "$Height $Width"
	    # Redisplay the graph without reading up the options from the 
	    # config file.

	    destroy .${label}_graph.display

	    set PixelsPerPoint [tk scaling -displayof .]
	    set PixelsPerCm [expr 28.8*$PixelsPerPoint]
	    set Width [expr $PixelsPerCm*$Width]
	    set Height [expr $PixelsPerCm*$Height]

	    set Xhash $GraphOptionValue(${label},x-hash)
	    set Yhash $GraphOptionValue(${label},y-hash)
	    set Xlabel $GraphOptionValue(${label},x-label)
	    set Ylabel $GraphOptionValue(${label},y-label)
	    set Xscale $GraphOptionValue(${label},x-scale\(cm\))
	    set Yscale $GraphOptionValue(${label},y-scale\(cm\))
	    set PixelsPerXHash [expr $PixelsPerCm/$Xhash]
	    set PixelsPerYHash [expr $PixelsPerCm/$Yhash]
	    
	    # Create various TK widgets necessary for displaying the graph.
	    
	    # The display frame that holds the canvas graph widgets 
	    # and the scrollbar.
	    frame .${label}_graph.display -relief sunken -background $GBgColor \
		    -borderwidth 3
	    
	    # Three canvas items. Two are static and the third is bound
	    # to the scrollbar.
	    
	    # The canvas that is bound to the scrollbar.
	    canvas .${label}_graph.display.graph -width [expr $Width-$Ydist-10] \
		    -height [expr $Height-$Xdist+15] -bg $GBgColor -bd 0 \
		    -xscrollcommand [list .${label}_graph.display.xscrollbar set] \
		    -relief flat
	    .${label}_graph.display.graph configure -confine 0 \
		    -scrollregion [list 0 0 "${MAX_XSIZE}c" $Height]
	    
	    # The two static canvas widgets.
	    canvas .${label}_graph.display.yaxis -width [expr $Ydist+10] \
		    -height $Height \
		    -bg $GBgColor -bd 0 -relief flat
	    canvas .${label}_graph.display.xaxis -width [expr $Width-$Ydist-10] \
		    -height [expr $Xdist-15] -bg $GBgColor -bd 0 -relief flat
	    
	    # The scrollbar widget.
	    scrollbar .${label}_graph.display.xscrollbar -orient horizontal \
		    -command [list .${label}_graph.display.graph xview]
	    .${label}_graph.display.xscrollbar configure -width 3m
	    

	    pack .${label}_graph.display.xscrollbar -side bottom -fill x
	    pack .${label}_graph.display.yaxis -side left -fill y 
	    pack .${label}_graph.display.xaxis -side bottom -fill x 
	    pack .${label}_graph.display.graph -side top -fill x -fill y
	    pack .${label}_graph.display -side top -fill x -fill y

	    set Graph .${label}_graph.display.graph
	    set YAxis .${label}_graph.display.yaxis
	    set XAxis .${label}_graph.display.xaxis
	    
	    $YAxis create line [expr $Ydist+7] 0 [expr $Ydist+7] $Height \
		    -arrow both \
		    -fill $AxesColor 
	    $YAxis create line 0 [expr $Height-$Xdist] [expr $Ydist+10] \
		    [expr $Height-$Xdist] -arrow first \
		    -fill $AxesColor 
	    $Graph create line 0 [expr $Height-$Xdist] "${MAX_XSIZE}c" \
		    [expr $Height-$Xdist] -arrow last \
		    -fill $AxesColor 
	    $YAxis create text [expr $Ydist-$LabelToAxis] \
		    [expr $Height-$Xdist+$LabelToAxis] \
		    -fill $AxesColor \
		    -text "0"
	    set Factor 1
	    for { set NewText $Xscale } { $NewText >= 10 } \
		    { set NewText [expr $NewText/10] } {
		set Factor [expr $Factor*10]
	    }
	    
	    $XAxis create text [expr $Width/2] 10 \
		    -fill $AxesColor \
		    -text "$Xlabel ( x $Factor )" -justify center
	    
	    # Display all the units and hashes on the X-axis
	    
	    set i 0
	    set DisplayedText [expr round($NewText)]
	    
	    for { set NextXUnit [expr $i+$PixelsPerCm] } \
		    { $i <= [expr $MAX_XSIZE*$PixelsPerCm-20] } \
		    { set i [expr $i+$PixelsPerXHash] } {
		
		if { $NextXUnit > $i } { 
		    
		    $Graph create line $i [expr $Height-$Xdist-$HashLineWidth/2] \
			$i [expr $Height-$Xdist+$HashLineWidth/2] \
			-fill $AxesColor 
		    
		} else {
		    
		    $Graph create line $i [expr $Height-$Xdist-$UnitLineWidth/2] \
			    $i [expr $Height-$Xdist+$UnitLineWidth/2] \

			    -fill $AxesColor 
		    $Graph create text $i [expr $Height-$Xdist+$LabelToAxis] \
			    -fill $AxesColor 
			    -text $DisplayedText
		    set DisplayedText [expr round($DisplayedText+$NewText)]
		    set NextXUnit [expr $i+$PixelsPerCm]
		}
	    }
	    
	    # Display all the units and hashes on the Y-axis
	    
	    set Factor 1
	    for { set NewText $Yscale } { $NewText >= 10 } \
		    { set NewText [expr $NewText/10] } {
		set Factor [expr $Factor*10]
	    }
	    
	    $YAxis create text 10 10 \
		-fill $LineColor 
		-text "x $Factor"
	    
	    set i [expr $Height-$Xdist]
	    set DisplayedText $NewText
	    set DisplayedText [format "%.2f" $DisplayedText]
	    
	    for { set NextYUnit [expr $i-$PixelsPerCm] } \
		    { $i >=  20 } \
		    { set i [expr $i-$PixelsPerYHash] } {
		
		if { $NextYUnit < $i } { 
		    
		    $YAxis create line [expr $Xdist-$HashLineWidth/2+7] $i \
			    [expr $Xdist+$HashLineWidth/2+7] $i \

			    -fill $AxesColor 
		    
		} else {
		    
		    $YAxis create line [expr $Xdist-$UnitLineWidth/2+7] $i \
			    [expr $Xdist+$UnitLineWidth/2+7] $i \
			    -fill $AxesColor
		    $YAxis create text [expr $Ydist-2*$LabelToAxis+7] $i \
			    -text $DisplayedText \
			    -fill $LineColor
		    set DisplayedText [expr $DisplayedText+$NewText]
		    set DisplayedText [format "%.2f" $DisplayedText]
		    set NextYUnit [expr $i-$PixelsPerCm]
		    
		}
		
	    }

	    # Get the corresponding statistic name.
	
	    switch -exact $label {
	    
		"ipc" { set StatName sim_IPC }
		
		"miss rate (dl1)" {
		    
		    set CacheName [ss::getOption "-cache:dcache"]
		    set StatName "${CacheName}.miss_rate"
		    
		}
		
		"miss rate (il1)" {
		    
		    set CacheName [ss::getOption "-cache:icache"]
		    set StatName "${CacheName}.miss_rate"
		    
		}
		
		"branch prediction" { 
		    
		    set PredictorType [ss::getOption "-bpred"]
		    set PredictorType "bpred_${PredictorType}"
		    set StatName "${PredictorType}.bpred_dir_rate"
		    
		}
		
	    }
	    
	    if { [llength $::allCycles] >= 4 } {
		
		# Display all the collected values till the current cycle.
		
		for { set CycleIndex 1} \
			{ $CycleIndex < [expr {[llength $::allCycles]-2}] } \
			{ set CycleIndex [expr {$CycleIndex+1}] } {
		    
		    set prevY [stat_value ::allStatistics $StatName $CycleIndex]
		    set currY [stat_value ::allStatistics $StatName \
			    [expr $CycleIndex+1] ]
		    set prevX [stat_cycle ::allCycles $CycleIndex]
		    set currX [stat_cycle ::allCycles [expr $CycleIndex+1] ]
		    
		    if { [string compare $prevY ""] != 0 && \
			    [string compare $currY ""] != 0 } {
			
			$Graph create line \
				[expr {$prevX*$PixelsPerCm/$Xscale}] \
				[expr {$Height-$prevY*$PixelsPerCm/$Yscale \
				-$Xdist}] \
				[expr {$currX*$PixelsPerCm/$Xscale}] \
				[expr {$Height-$currY*$PixelsPerCm/$Yscale \
				-$Xdist}] \
				-fill red -smooth 1 \
				-joinstyle round -splinesteps 4
		    }
		    set LastPointDisplayed($label) $CycleIndex
		}
	    }
	    update idletasks
	    update

	    bind .${label}_graph.display <Configure> \
		    [list resize_graph $label %w %h]

	}
    }
    proc reset_count {label} {
	global ResizeCount
	# puts "Resize Count is reset to 0"
	set ResizeCount($label) 0
    }

proc hide_graph {label} {
    if { [winfo exists .${label}_graph] } {
	wm withdraw .${label}_graph 
	update idletasks
	update
    }
}
