# SimpleScalar Visualization Engine
# Step Bar Functions
#
# The step bar has buttons to advance the simulation engine.  It has a
# "single-step" button, a "take a few steps" button, a "step until all
# done" button, and a "stop" button.  It also has a cycle number
# display and some data entry fields.  The data entries specify the
# sample collection and display update periods.
#
# *** PUT COPYRIGHT STUFF HERE ***
#=======================================================================#

package require Tk 8.0

global arrowColor
global backArrowColor
global stopArrowColor

# source "$env(VISUAL_SOURCE_DIR)/toolbutton.tcl"
source "ui/toolbutton.tcl"

#-----------------------------------------------------------------------#
# Stepbar options:
#
option add *Stepbar.Period.Entry.background white widgetDefault
option add *Stepbar.Period.Entry.width      10    widgetDefault

proc stepbar_create {window} {
    frame $window -class Stepbar

    # The three stepping buttons are called "one" (to single-step),
    # "few" (for "step a few cycles"), and "all" (for "step until all
    # done").  There's also an "end" button which ends the current
    # stepping cycle.  They are tool buttons, which mean they are
    # labeled using a paint routine and invoke an action routine when
    # pressed.
    #
    toolbutton_create $window.one paint_one step_one
    toolbutton_create $window.few paint_few step_few
    toolbutton_create $window.all paint_all step_all
    toolbutton_create $window.end paint_end step_end

    pack $window.one $window.few $window.all $window.end -side left
    # pack $window.all $window.few $window.end -side left


    # The "period" megawidget is a two-by-two table of entry values.  The
    # rows are for the "collect" and "display" intervals, while the
    # columns are for the "rate" and "next" values.  The idea is that,
    # e.g., data is collected from the simulator every "collect/rate"
    # cycles and the next collection point is scheduled just after the
    # "collect/next" cycle number.  Similarly the "display" values are
    # used to schedule display updates by cycles.
    #
    frame $window.period -class Period

    # make entry easy for first six digits
    entry $window.period.collectRate -justify center -width 7

    label $window.period.collectLabel -text "Number of cycles per step:"
    label $window.period.collectRateLabel -text ""

    grid $window.period.collectLabel     -column 0 -row 0 -sticky e
    grid $window.period.collectRateLabel -column 1 -row 0 -sticky e
    grid $window.period.collectRate      -column 2 -row 0 -sticky e

    $window.period.collectRate insert 0 "500"

    foreach e {collectRate } {
	set entry $window.period.$e
	bind $entry <FocusOut> [list stepentry_validate $entry]
	bind $entry <Return>   [list stepentry_leave $entry]
    }
    pack $window.period -side left

    # The "cycles" megawidget displays the number of simulated cycles.
    #
    frame $window.cycles -class Cycles

    label $window.cycles.value -text "0"
    label $window.cycles.label -text "Total cycles stepped:"

    grid $window.cycles.label -column 0 -row 0 -sticky w
    grid $window.cycles.value -column 1 -row 0 -sticky w

    pack $window.cycles -side left

    return $window
}

#-----------------------------------------------------------------------#
# Functions for data entry fields in stepbar:
#
proc stepentry_leave {entry} {
    $entry selection clear
    focus .
    return $entry
}

proc stepentry_validate {entry} {
    set value [$entry get]

    if {[string is integer -strict $value]} {
	$entry configure -background white
    } else {
	$entry configure -background red
	status_message .status "Data values must be integers"
    }

    return $entry
}

#-----------------------------------------------------------------------#
# Functions to calculate step amount:
#
proc stepfield_next {window abbr cycle} {
    if {[string compare $abbr "display"] == 0} {
	#
	# we should never hit here because the "display" counter
	# is gone, but we do because of some vestigial 
	# initialization code
	#
	# puts "give me a BREAK"
	return 0
    }
    set next [${window}.period.${abbr}Rate get]
    # puts "step increment is now $next"
    return $next
}

#-----------------------------------------------------------------------#
# The actual stepping functions:
#
proc step_all {button} {
    # infinity? what's the biggest number we've got?
    # how about 2^31 or 2147483648
    stepper_step infinity
    return $button
}
proc step_end {button} {
    set ::StopSign 1
    return $button
}
proc step_few {button} {
    set current [ss::getStatistic "sim_cycle"]
    set collect [stepfield_next .stepbar collect $current] 
    # puts "Step size: $collect. Current cycle: $current"
    # stepper_step [expr {$collect-$current}]
    stepper_step $collect
    return $button
}

proc step_one {button} {

    stepper_step 1
    return $button
}

#-----------------------------------------------------------------------#
# paint functions to label step buttons:
#
proc paint_all {canvas xlo xhi ylo yhi} {
    global stopArrowColor
    set wide [expr {0.1*($xhi-$xlo)}]
    set barWidth 4

    $canvas create line $xhi $ylo $xhi $yhi \
	     -fill $stopArrowColor -width $barWidth

    paint_few $canvas $xlo $xhi $ylo $yhi

    return $canvas
}
proc paint_end {canvas xlo xhi ylo yhi} {
    global stopArrowColor

    set firstMark [expr $xlo + 12 ]
    set barWidth 4
    set gap 7

    $canvas create line $firstMark $ylo $firstMark $yhi \
	    -fill $stopArrowColor -width $barWidth

    set firstMark [expr $firstMark + $gap ]

    $canvas create line $firstMark $ylo $firstMark $yhi \
	    -fill $stopArrowColor -width $barWidth

    return $canvas
}

proc old_paint_end {canvas xlo xhi ylo yhi} {
    # the famous stop sign
    set div [expr {2.0+sqrt(2.0)}]

    set dx [expr {($xhi-$xlo)/$div}]
    set dy [expr {($yhi-$ylo)/$div}]

    set xdn [expr {$xlo+$dx}]
    set xup [expr {$xhi-$dx}]

    set ydn [expr {$ylo+$dy}]
    set yup [expr {$yhi-$dy}]

    $canvas create polygon \
	    $xlo $yup  $xdn $yhi  $xup $yhi  $xhi $yup \
	    $xhi $ydn  $xup $ylo  $xdn $xlo  $xlo $ydn \
	    -fill red

    return $canvas
}

proc paint_few {canvas xlo xhi ylo yhi} {
    global arrowColor
    paint_one $canvas $xlo $xhi $ylo $yhi

    set head [expr {0.5*($xhi-$xlo)}]
    set wide [expr {0.6*($yhi-$ylo)}]
    set tips [expr {0.5*($yhi-$ylo-$wide)}]

    set xto [expr {$xlo+0.7*($xhi-$xlo)}]
    set ymid [expr {0.5*($ylo+$yhi)}]

    $canvas create line $xlo $ymid $xto $ymid \
	    -arrow last -arrowshape [list $head $head $tips] \
	    -fill $arrowColor -width $wide

    return $canvas
}
proc paint_one {canvas xlo xhi ylo yhi} {
    global backArrowColor
    set head [expr {0.5*($xhi-$xlo)}]
    set wide [expr {0.6*($yhi-$ylo)}]
    set tips [expr {0.5*($yhi-$ylo-$wide)}]

    set ymid [expr {0.5*($ylo+$yhi)}]

    $canvas create line [expr $xlo + 3 ] $ymid [expr $xhi - 1] $ymid \
	    -arrow last -arrowshape [list $head $head $tips] \
	    -fill $backArrowColor -width $wide

    return $canvas
}

