// Parallelizer Module :

// Author : Ramakrishna Kotla, kotla@cs.utexas.edu
// LASR, Dept. of Computer Science, Univ. of Texas, Austin
// Date : 5th July 2003 
  

// Description of CBASE architecture: 
// ---------------------------------
// CBASE architecture has 3 stages. First stage consists of BASE
// library, which primarily deals with 3 phase agreement protocol, 
// checkpointing, view change and recovery. It does everything 
// same as the existing BASE except that it does not execute 
// the request and instead enqueues the request in the input 
// queue of the Parallelizer module. Parallelizer is the second 
// stage of CBASE, which does application specific (NFS in our 
// case) concurrency analysis of the requests to extract 
// concurrency among the requests in the input queue and enqueues 
// the requests that can be  executed concurrently in the output 
// queue. Execution stage, which is the third stage, dequeues the 
// requests and executes the requests based on some execution model 
// (thread per request,bounded thread pool, single threaded/
//  event based models).  
 
// Functionality of Parallelizer module
// ------------------------------------
//
// It has three components : Input Queue, Parallelizer and Output
// Queue. 
// 
// Input Queue  : BASE enqueues the requests into the input queue. 
// 
// Parallelizer : Dequeues requests from the input queue and does
// concurrency analysis to update the dependency graph of all the
// outstanding requests. All the requests that are not dependent 
// on other requests are dequed from the input queue and enqueued
// to the output queue. Concurrency analysis is application 
// specific and hence needs to be modified for supporting a new
// service. It is a modular design and involves just adding a 
// new concurrency analysis function for every new service that 
// is supported and register the function during initialzation.
//
// Output Queue : It consists of requests that are ready to be 
// executed concurrently. These requests are dequeued by the 
// execution stage to execute requests. 
//

#ifndef _PARALLELIZER_h
#define _PARALLELIZER_h 1
#include "RequestQueue.h"

  

// Parallelizer Stage 
class Parallelizer {

private :

  // Req queue : queue of  requests. Shared by threads in
  // "Exec" stage and thread in "Parallelizer" stage.
  RequestQueue req_queue;

  // KOTLA : To do have a ready queue requests that can be 
  // executed parallely as perfromance opt. In such case we dont have
  // to look at all the requests in req_queue to find
  // a ready request.  

  // Next checkpoint to be taken by BASE
  int next_chkpt; 

  // This function checks the dependencies of the incoming request 
  bool parallelize();  

  // Function pointer that reads the service specific state
  int (*read_state)(Byz_req, int *, int *);  

  // Concurrency Matrix for nfs
  static int conc_matrix[18][18][2];

  // Check if this_req is dependent on enqueued request
  bool is_dependent( Req *this_req, Req *other_req);

public :
  
  Parallelizer(Seqno chkpt, int max_elem):req_queue(chkpt,max_elem) { 
    next_chkpt = chkpt;
    read_state = NULL;
  }

  // Enqueues request into the req queue. Returns true if it 
  // successfully enqueues the request.
  bool insert(int, Request_id, Seqno, Thread_arg *); 

  // Return next ready request to "EXEC" stage thread 
  // Returns NULL if there is no request ready.
  Req* next_request(); 

  // Remove the request from the request queue 
  // "EXEC" stage calls this function once the request is executed
  // Retuns false if the queue is empty.
  bool remove(Req *);
  
  // Sync up the requests at the next_chkpt
  void sync(Seqno chkpt);    

  // Clear the parallelizer
  void clear (Seqno chkpt);
 
  // Register read_state function
  void register_read_state(int (*read)(Byz_req, int *, int *));
};


#endif // _PARALLELIZER_H
