//                              -*- Mode: Verilog -*-
// Filename        : lc35_top.v
// Description     : LC3.5 top level with datapath and control
// Author          : Stephen W. Keckler
// Created On      : Sun Oct 12 16:36:39 2003
// Last Modified By: .
// Last Modified On: .
// Update Count    : 0
// Status          : Unknown, Use with caution!


`include "lc35.h"

module lc35_top (clk, reset);
   input clk, reset;
   
   // Datapath interface signals
   wire  pc_ld, w_newpc;
   wire  ir_ld;
   
   wire  rr_ld, sr_sel;
   wire  link_reg, w_regfile_write;
   reg 	 op1_mux;
   reg [2:0] op2_mux;
   
   wire      x_ld;
   wire      m_ld, m_sel;
   wire      m_write;
   
   wire [3:0] ir_opcode;
   wire       ir_imm, ir_jsr;
   wire [2:0] ir_brnzp, dp_nzp;
   
   
   // Internal control signals
   
   reg [2:0]  stage_in;
   wire [2:0] stage;
   
   wire [3:0] opcode;
   wire       imm, jsr;
   wire [2:0] brnzp, nzp;
   wire       branch_taken;
   
   // LC 3.5 Datapath 
   datapath lc35_dp(clk, reset_pc,
		    pc_ld, w_newpc, 
		    ir_ld, 
		    sr_sel, rr_ld, 
		    link_reg, w_regfile_write, 
		    op1_mux, op2_mux, x_ld,
		    m_ld, m_sel, m_write,
		    ir_opcode, ir_imm, ir_jsr, ir_brnzp, dp_nzp);
   
   // LC 3.5 state machine
   
   always @(reset or stage)
     begin
	if(reset) stage_in = `RESET;
	else begin
	   case(stage)
	     `RESET: stage_in = `FETCH;
	     `FETCH: stage_in = `RREAD;
	     `RREAD: stage_in = `EXEC;
	     `EXEC:  stage_in = `MEMORY;
	     `MEMORY: stage_in = `WRITE;
	     `WRITE:  stage_in = `FETCH;
	     default: stage_in = `FETCH;
	   endcase
	end // else: !if(reset)
     end
   
   dff #3 stage_reg(clk, stage_in, stage);
   
   // FETCH control
   
   dff_ld #4 instr_reg(clk, ir_ld, ir_opcode, opcode);
   dff_ld #1 instr_imm(clk, ir_ld, ir_imm, imm);
   dff_ld #1 instr_jsr(clk, ir_ld, ir_jsr, jsr);
   dff_ld #3 instr_nzp(clk, ir_ld, ir_brnzp, brnzp);
   
   assign ir_ld = (stage == `FETCH);
   assign
	 w_newpc = (opcode == `BR && branch_taken) || (opcode == `JMP) || 
	 (opcode == `JSR) || (opcode == `TRAP);
   
   // RREAD control
   
   assign sr_sel = (opcode == `ST) || (opcode == `STR);
   assign rr_ld = (stage == `RREAD);
   
   // EXEC control
   
   assign x_ld = (stage == `EXEC);
   
   always @(opcode or jsr)
     case(opcode)
       `LEA, `LD, `ST, `BR: op1_mux = `OP1_SELECT_PC;
       `JSR: op1_mux = jsr ? `OP1_SELECT_PC : `OP1_SELECT_REG;
       default: op1_mux = `OP1_SELECT_REG;
     endcase // case(instr)
   
   always @(opcode or imm or jsr)
     begin
	if(imm && (opcode == `ADD || opcode == `SUB || opcode == `AND ||  // SEXT 5
		   opcode == `OR  || opcode == `ASH || opcode == `LSH))
	  op2_mux = `OP2_SELECT_imm5;
	else if(opcode == `LDR || opcode == `STR || (opcode == `JSR && ~jsr))  // SEXT 6
	  op2_mux = `OP2_SELECT_offset6;
	else if(opcode == `LEA || opcode == `LD || opcode == `ST ||  // SEXT 9
		opcode == `BR || opcode == `TRAP) 
	  op2_mux = `OP2_SELECT_offset9;
	else if (opcode == `JSR && jsr) // SEXT 11
	  op2_mux = `OP2_SELECT_offset11;
	else  // register file output
	  op2_mux = `OP2_SELECT_REG;
     end
   
   // MEM control
   
   assign m_ld = (stage == `MEMORY);
   assign m_sel = (opcode == `LD || opcode == `LDR || opcode == `TRAP);
   assign m_write = (stage == `MEMORY) && (opcode == `ST || opcode == `STR);
   
   // WRITE control
   
   dff_ld #3 ff_nzp(clk, w_regfile_write, dp_nzp, nzp);
   
   assign
	 w_regfile_write = (stage == `WRITE) &&
	 (opcode == `ADD || opcode == `SUB || opcode == `AND ||
	  opcode == `OR  || opcode == `ASH || opcode == `LSH ||
	  opcode == `NOT || opcode == `LEA || opcode == `LD ||
	  opcode == `LDR || opcode == `JSR || opcode == `TRAP);
   assign pc_ld = (stage == `RESET) || (stage == `WRITE);
   assign reset_pc = (stage == `RESET);
   assign branch_taken = |(brnzp & nzp);
   assign link_reg = (opcode == `JSR) || (opcode == `TRAP);
   
endmodule
