; Challenge: Define rotate for a list and prove that if you rotate n steps,
; where n is its length, then you get the list back.

(defun rotate (n x)
  (if (zp n)
      x
    (rotate (- n 1)
            (append (cdr x)
                    (list (car x))))))

(defthm rotate-len ; fails without user assistance
  (implies (true-listp x)
           (equal (rotate (len x) x)
                  x)))

; Clever proof plan: It suffices to prove the following lemma.

(defthm rotate-append ; fails to prove
  (implies (and (true-listp x)
                (true-listp y))
           (equal (rotate (len x) (append x y))
                  (append y x)))
  :rule-classes nil)

; That fails, so let's think about how this might be proved by induction.

; Base case: x is not a cons.  Then it's pretty trivial.

; Inductive step: x is a cons and we can assume that the lemma holds with x
; replaced by (cdr x) and y replaced however we wish.

; Note that by definition,
;   (rotate n (append x y))
; will expand to
;   (rotate (- n 1)
;           (append (append (cdr x) y)
;                   (list (car (append x y)))))
; so instead of a second argument of (append x y) we concatenate
; (cdr x), y, and (list (car x)).

; Let's stop thinking and give ACL2 a hint, that for the inductive
; step it should assume the lemma holds with:

;   replace x by (cdr x)
;   replace y by (append y (list (car x)))

(defun rotate-append-induction (x y)
  (if (consp x)
      (rotate-append-induction (cdr x)
                               (append y (list (car x))))
    y))

(defthm rotate-append ; fails; can you see how the checkpoint provides a clue?
  (implies (and (true-listp x)
                (true-listp y))
           (equal (rotate (len x) (append x y))
                  (append y x)))
  :hints (("Goal" :induct (rotate-append-induction x y)))
  :rule-classes nil)

(defthm append-assoc
  (equal (append (append x y) z)
         (append x y z)))

(defthm rotate-append
  (implies (and (true-listp x)
                (true-listp y))
           (equal (rotate (len x) (append x y))
                  (append y x)))
  :hints (("Goal" :induct (rotate-append-induction x y)))
  :rule-classes nil)

(defthm rotate-len
  (implies (true-listp x)
           (equal (rotate (len x) x)
                  x))
  :hints (("Goal" :use ((:instance rotate-append (y nil))))))

; Also see rotate-alt.lsp for a more direct proof for which
; ACL2 requires yet more help.
