/*
 *  $Id: ibsend.c,v 1.17 1997/01/07 01:45:29 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      See COPYRIGHT in top-level directory.
 */


#include "mpiimpl.h"
#ifdef MPI_ADI2
#include "reqalloc.h"
#endif

/*@
    MPI_Ibsend - Starts a nonblocking buffered send

Input Parameters:
. buf - initial address of send buffer (choice) 
. count - number of elements in send buffer (integer) 
. datatype - datatype of each send buffer element (handle) 
. dest - rank of destination (integer) 
. tag - message tag (integer) 
. comm - communicator (handle) 

Output Parameter:
. request - communication request (handle) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_TAG
.N MPI_ERR_RANK
.N MPI_ERR_BUFFER

@*/
int MPI_Ibsend( buf, count, datatype, dest, tag, comm, request )
void             *buf;
int              count;
MPI_Datatype     datatype;
int              dest;
int              tag;
MPI_Comm         comm;
MPI_Request      *request;
{
    int         mpi_errno = MPI_SUCCESS;
    static char myname[] = "MPI_IBSEND";

#ifdef MPI_ADI2
    int         psize;
    void        *bufp;
    MPIR_SHANDLE *shandle;
    struct MPIR_COMMUNICATOR *comm_ptr;
    struct MPIR_DATATYPE     *dtype_ptr;

    TR_PUSH(myname);
    comm_ptr = MPIR_GET_COMM_PTR(comm);
    MPIR_TEST_MPI_COMM(comm,comm_ptr,comm_ptr,myname);

    dtype_ptr = MPIR_GET_DTYPE_PTR(datatype);
    MPIR_TEST_DTYPE(datatype,dtype_ptr,comm_ptr,myname);

    if (MPIR_TEST_COUNT(comm,count) ||
	MPIR_TEST_SEND_RANK(comm_ptr,dest) || MPIR_TEST_SEND_TAG(comm,tag))
	return MPIR_ERROR( comm_ptr, mpi_errno, myname );

    MPIR_ALLOC(*request,(MPI_Request)MPID_SendAlloc(),
	       comm_ptr, MPI_ERR_EXHAUSTED,myname);
    shandle = &(*request)->shandle;
    MPID_Request_init( shandle, MPIR_SEND );

    MPIR_REMEMBER_SEND((&(*request)->shandle),buf, count, datatype, dest, tag,
		       comm_ptr);

    if (dest == MPI_PROC_NULL) {
	(*request)->shandle.is_complete = 1;
	return MPI_SUCCESS;
    }

    /* Allocate space if needed */
    MPI_Pack_size( count, datatype, comm, &psize );
    MPIR_CALL(MPIR_BsendAlloc( psize, *request, &bufp ),comm_ptr, myname );
    /* Information stored in the bsend part by BsendAlloc */

    MPIR_IbsendDatatype( comm_ptr, buf, count, dtype_ptr, 
			 comm_ptr->local_rank, tag, comm_ptr->send_context, 
			 comm_ptr->lrank_to_grank[dest], *request, 
			 &mpi_errno );
    TR_POP;
#else
    int err;

    /* We'll let MPI_Bsend_init routine detect the errors */
    err = MPI_Bsend_init( buf, count, datatype, dest, tag, comm, request );
    if (err)
	return err;
    
    (*request)->shandle.persistent = 0;
    MPIR_BsendPersistent( *request, 0 );

    if (dest != MPI_PROC_NULL) {
	return MPI_Start( request );
	}

    /*
       This must create a completed request so that we can wait on it
     */
    MPID_Set_completed( comm_ptr->ADIctx, *request );
    (*request)->shandle.active     = 1;
#endif
    MPIR_RETURN( comm_ptr, mpi_errno, myname );
}
