
; reader.lisp

; Events to read DIMAC-formated files.  For example usage, see the
; final events of this file.

(in-package "ACL2")

(include-book "tools/bstar" :dir :system)
(include-book "std/util/define" :dir :system)
(include-book "std/io/read-file-bytes" :dir :system)
(include-book "std/strings/top" :dir :system)

;; (define symbol-append
;;   ((x symbolp)
;;    (y symbolp))
;;   :returns (x+y symbolp)
;;   (intern-in-package-of-symbol (string-append (symbol-name x)
;;                                               (symbol-name y))
;;                                x))


;; (defmacro error-symbol (symbol)
;;   `(symbol-append __function__ (symbol-append '--- ,symbol)))

(define empty-stringp ;emptyp? shorter
  ((string stringp))
  :returns (empty booleanp)
  (null (explode string))

  ///

  (defthm consp-append1
    (implies (consp x)
             (consp (append x y)))
    :rule-classes :type-prescription)

  (defthm consp-append2
    (implies (consp y)
             (consp (append x y)))
    :rule-classes :type-prescription)

  (defthm consp-implies-not-empty-stringp-implode
    (implies (consp x)
             (not (empty-stringp (implode x))))))


(defconst *empty-string* "")
;(defconst *es* "") ;shorter???


(define create-error-string
  ((function-name symbolp)
   (message stringp)
   (old-error stringp))
  :returns (new-error stringp)
  (b* ((new-error (symbol-name function-name))
       (new-error (string-append new-error " :: "))
       (new-error (string-append new-error message))
       (new-error (implode (append (explode new-error)
                                   (list #\Newline))))
       (new-error (string-append new-error old-error)))
      new-error)

  ///
  (defthm not-empty-stringp-create-error-string
    (not (empty-stringp (create-error-string function-name
                                             message
                                             old-error)))))

(defmacro error-string-define (message
                               &optional (old-error '*empty-string*))
  `(create-error-string __function__ ,message ,old-error))



(define char-list-to-integer-list
  ((char-list character-listp))
  :returns (integer-list integer-listp :hyp :guard)
  (if (atom char-list)
      char-list
    (cons (char-code (car char-list))
          (char-list-to-integer-list (cdr char-list)))))

(define string-to-integer-list
  ((string stringp))
  :returns (integer-list integer-listp :hyp :guard)
  (char-list-to-integer-list (explode string)))


(define read-line1
  ((content integer-listp))
  :returns (mv (err stringp)
               (line integer-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            nil
            content))
       ((if (equal (car content) (char-code #\Newline)))
        (mv *empty-string* (list (car content)) (cdr content)))
       ((mv err line new-content)
        (read-line1 (cdr content))))
      (mv err (cons (car content) line) new-content))

  ///
  (defthm read-line1-no-error-implies-smaller-content
    (implies (empty-stringp (mv-nth 0 (read-line1 content)))
             (< (len (mv-nth 2 (read-line1 content)))
                (len content)))
    :rule-classes :linear))

(define skip-line
  ((content integer-listp))
  :returns (mv (err stringp)
               (new-content integer-listp :hyp :guard))
  (b* (((mv err ?line new-content)
        (read-line1 content)))
      (mv err new-content))

  ///
  (defthm skip-line-no-error-implies-smaller-content
    (implies (empty-stringp (mv-nth 0 (skip-line content)))
             (< (len (mv-nth 1 (skip-line content)))
                (len content)))
    :rule-classes :linear))

(define trim-whitespace
  ((content integer-listp))
  :returns (new-content integer-listp :hyp :guard)
  (b* (((if (atom content))
        content)
       ((if (member (car content) (list (char-code #\Space)
                                        (char-code #\Tab)
                                        (char-code #\Newline))))
        (trim-whitespace (cdr content))))
      content)

  ///
  (defthm trim-whitespace-at-least-as-small-content
    (<= (len (trim-whitespace content))
        (len content))
    :rule-classes :linear))


(define char-code-to-digit
  ((char-code integerp))
  :returns (mv (err stringp)
               (digit integerp :hyp :guard))
  (b* ((adjusted-char-code (- char-code 48))
       ((if (and (<= 0 adjusted-char-code)
                 (<= adjusted-char-code 9)))
        (mv *empty-string* adjusted-char-code)))
      (mv (error-string-define "Not a digit.")
          char-code))

  ///
  (defthm char-code-to-digit-no-error-implies-digit
    (implies (and (integerp char-code)
                  (empty-stringp (mv-nth 0 (char-code-to-digit char-code))))
             (and (natp (mv-nth 1 (char-code-to-digit char-code)))
                  (<= 0 (mv-nth 1 (char-code-to-digit char-code)))
                  (<= (mv-nth 1 (char-code-to-digit char-code)) 9)))
    :rule-classes :forward-chaining))


(define parse-natural-my
  ((content integer-listp)
   &optional
   ((base natp) '0))
  :returns (mv (err stringp)
               (num natp :hyp :guard :rule-classes :type-prescription)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            base
            content))
       ((mv err digit)
        (char-code-to-digit (car content)))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Bad char." err)
            base
            content))
       ((mv ?err num new-content)
        (parse-natural-my (cdr content) (+ (* base 10) digit))))
      (mv *empty-string* num new-content))

  ///
  (defthm parse-natural-my-fn-at-least-as-small-content
    (<= (len (mv-nth 2 (parse-natural-my-fn content base)))
        (len content))
    :rule-classes :linear)

  (defthm parse-natural-my-at-least-as-small-content
    (<= (len (mv-nth 2 (parse-natural-my content)))
        (len content))
    :rule-classes :linear)

  (defthm parse-natural-my-fn-no-error-implies-smaller-content
    (implies (empty-stringp (mv-nth 0 (parse-natural-my-fn content base)))
             (< (len (mv-nth 2 (parse-natural-my-fn content base)))
                (len content)))
    :rule-classes :linear)

  (defthm parse-natural-my-no-error-implies-smaller-content
    (implies (empty-stringp (mv-nth 0 (parse-natural-my content)))
             (< (len (mv-nth 2 (parse-natural-my content)))
                (len content)))
    :rule-classes :linear))



(define parse-integer-my
  ((content integer-listp))
  :returns (mv (err stringp)
               (num integerp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            0
            content))
       (signed (equal (car content) (char-code #\-)))
       (sign-mult (if signed -1 1))
       (content (if signed (cdr content) content))
       ((mv err num new-content)
        (parse-natural-my content)))
      (mv err (* sign-mult num) new-content))

  ///
  (defthm parse-integer-my-at-least-as-small-content
    (<= (len (mv-nth 2 (parse-integer-my content)))
        (len content))
    :rule-classes :linear)

  (defthm parse-integer-my-no-error-implies-smaller-content
    (implies (empty-stringp (mv-nth 0 (parse-integer-my content)))
             (< (len (mv-nth 2 (parse-integer-my content)))
                (len content)))
    :rule-classes :linear))


(define parse-char
  ((content integer-listp)
   (char characterp))
  :returns (mv (success symbolp)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv nil content))
       ((if (equal (car content) (char-code char)))
        (mv t (cdr content))))
      (mv nil content))

  ///
  (defthm parse-char-success-implies-smaller-content
    (implies (mv-nth 0 (parse-char content char))
             (< (len (mv-nth 1 (parse-char content char)))
                (len content)))
    :rule-classes :linear))

(define parse-char-list
  ((content integer-listp)
   (char-list character-listp))
  :returns (mv (success symbolp)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom char-list))
        (mv t content))
       ((if (atom content))
        (mv nil content))
       ((if (equal (car content) (char-code (car char-list))))
        (parse-char-list (cdr content) (cdr char-list))))
      (mv nil content))

  ///
  (defthm parse-char-list-success-implies-smaller-content
    (implies (and (consp char-list)
                  (mv-nth 0 (parse-char-list content char-list)))
             (< (len (mv-nth 1 (parse-char-list content char-list)))
                (len content)))
    :rule-classes :linear))


(define parse-string
  ((content integer-listp)
   (string stringp))
  :returns (mv (success symbolp)
               (new-content integer-listp :hyp :guard))
  (parse-char-list content (explode string))

  ///
  (defthm parse-string-success-implies-smaller-content
    (implies (and (stringp string)
                  (not (equal string ""))
                  (mv-nth 0 (parse-string content string)))
             (< (len (mv-nth 1 (parse-string content string)))
                (len content)))
    :rule-classes :linear))


(define parse-comments
  ((content integer-listp))
  :returns (mv (err stringp)
               (new-content integer-listp :hyp :guard))
  :measure (len content)
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            nil))
       ((mv success new-content)
        (parse-string content "c "))
       ((if (not success))
        (mv *empty-string* content))
       ((mv err new-content)
        (skip-line new-content))
       ((if (not (empty-stringp err)))
        (mv err new-content)))
      (parse-comments new-content)))

(define parse-spaces
  ((content integer-listp))
  :returns (mv (success symbolp)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv nil content))
       ((unless (member (car content) (list (char-code #\Space)
                                            (char-code #\Tab))))
        (mv nil content))
       ((mv ?success new-content)
        (parse-spaces (cdr content))))
      (mv t new-content))

  ///
  (defthm parse-spaces-at-least-as-small-content
    (<= (len (mv-nth 1 (parse-spaces content)))
        (len content))
    :rule-classes :linear)

  (defthm parse-spaces-success-implies-smaller-content
    (implies (mv-nth 0 (parse-spaces content))
             (< (len (mv-nth 1 (parse-spaces content)))
                (len content)))
    :rule-classes :linear))


(define parse-whitespace
  ((content integer-listp))
  :returns (mv (success symbolp)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv nil content))
       ((unless (member (car content) (list (char-code #\Space)
                                            (char-code #\Tab)
                                            (char-code #\Newline))))
        (mv nil content))
       ((mv ?success new-content)
        (parse-whitespace (cdr content))))
      (mv t new-content))

  ///
  (defthm parse-whitespace-at-least-as-small-content
    (<= (len (mv-nth 1 (parse-whitespace content)))
        (len content))
    :rule-classes :linear)

  (defthm parse-whitespace-success-implies-smaller-content
    (implies (mv-nth 0 (parse-whitespace content))
             (< (len (mv-nth 1 (parse-whitespace content)))
                (len content)))
    :rule-classes :linear))




(define parse-problem-line
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-vars natp :hyp :guard)
               (num-clauses natp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            0 0 nil))
       ((mv success new-content)
        (parse-string content "p cnf"))
       ((if (not success))
        (mv (error-string-define "No problem statement.")
            0 0 content))
       ((mv success new-content)
        (parse-spaces new-content))
       ((if (not success))
        (mv (error-string-define "Malformed statement.")
            0 0 content))
       ((mv err num-vars new-content)
        (parse-natural-my new-content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Bad vars." err)
            0 0 content))
       ((mv success new-content)
        (parse-spaces new-content))
       ((if (not success))
        (mv (error-string-define "Malformed statement.")
            0 0 content))
       ((mv err num-clauses new-content)
        (parse-natural-my new-content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Bad clauses." err)
            0 0 content))
       ((mv err new-content)
        (skip-line new-content))
       ((if (not (empty-stringp err)))
        (mv err 0 0 content)))
      (mv *empty-string* num-vars num-clauses new-content)))

(define parse-preamble
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-vars natp :hyp :guard)
               (num-clauses natp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  (b* (((mv err new-content)
        (parse-comments content))
       ((if (not (empty-stringp err)))
        (mv err 0 0 new-content)))
      (parse-problem-line new-content)))

(define encode
  ((n integerp))
  :returns (val integerp :hyp :guard)
  (if (< n 0)
      (1+ (* -2 n))
    (* 2 n)))

(define parse-clause
  ((content integer-listp))
  :returns (mv (err stringp)
               (clause integer-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  :measure (len content)
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            nil nil))
       ((mv err num new-content1)
        (parse-integer-my content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Bad integer." err)
            nil content))
       ((if (equal num 0))
        (mv *empty-string* nil new-content1))
       ((mv success new-content2)
        (parse-whitespace new-content1))
       ((if (not success))
        (mv (error-string-define "No separator.")
            nil new-content1))
       ((mv err clause new-content3)
        (parse-clause new-content2)))
      (mv err (cons (encode num) clause) new-content3))

  ///
  (defthm parse-clause-at-least-as-small-content
    (<= (len (mv-nth 2 (parse-clause content)))
        (len content))
    :rule-classes :linear)

  (defthm parse-clause-no-error-implies-smaller-content
    (implies (empty-stringp (mv-nth 0 (parse-clause content)))
             (< (len (mv-nth 2 (parse-clause content)))
                (len content)))
    :rule-classes :linear))


(define integer-list-listp
  ((x))
  :returns (result booleanp)
  (if (atom x)
      (null x)
    (and (integer-listp (car x))
         (integer-list-listp (cdr x))))

  ///
  (defthm integer-list-listp-cons
    (implies (and (integer-listp x)
                  (integer-list-listp y))
             (integer-list-listp (cons x y)))))


(define parse-formula
  ((content integer-listp))
  :returns (mv (err stringp)
               (formula integer-list-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  :measure (len content)
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            nil nil))
       (new-content (trim-whitespace content))
       ((mv err clause new-content)
        (parse-clause new-content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Bad clause." err)
            nil content))
       ((if (atom new-content))
        (mv *empty-string* (list clause) new-content))
       ((mv success new-content)
        (parse-whitespace new-content))
       ((if (not success))
        (mv (error-string-define "No trailing whitespace after clause.")
            nil new-content))
       ((if (atom new-content))
        (mv *empty-string* (list clause) new-content))
       ((mv err formula new-content)
        (parse-formula new-content)))
      (mv err (cons clause formula) new-content)))




(define parse-dimacs
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-vars natp :hyp :guard)
               (num-clauses natp :hyp :guard)
               (formula integer-list-listp :hyp :guard))
  (b* (((mv err num-vars num-clauses new-content)
        (parse-preamble content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Preamble error.")
            num-vars num-clauses nil))
       ((mv err formula ?new-content)
        (parse-formula new-content)))
      (mv err num-vars num-clauses formula)))




(set-state-ok t)

(define read-dimacs
  ((filename stringp)
   (state state-p))
  :returns (mv (contents integer-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (read-file-bytes filename state))
       ((if (stringp contents))
        (mv nil state)))
      (mv contents state)))

(define read-and-parse-with-state
  ((filename stringp)
   (state state-p))
  :returns (mv (err stringp)
               (num-vars natp)
               (num-clauses natp)
               (formula integer-list-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (read-dimacs filename state))
       ((mv err num-vars num-clauses formula)
        (parse-dimacs contents)))
      (mv err num-vars num-clauses formula state)))

(defmacro read-and-parse (filename)
  `(read-and-parse-with-state ,filename state))

(set-state-ok nil)


; Post processing for unique, tautology, etc


;; ===================================================================


(defthm integer-listp-append
  (implies (and (integer-listp x)
                (integer-listp y))
           (integer-listp (append x y))))

(defthm integer-listp-rev
  (implies (integer-listp x)
           (integer-listp (rev x))))

;; (defthm integer-listp-append
;;   (equal (integer-listp (append x y))
;;          (and (integer-listp x)
;;               (integer-listp y))))

(define parse-formula-flat
  ((content integer-listp))
  :returns (mv (err stringp)
               (formula integer-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  :measure (len content)
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            nil nil))
       (new-content (trim-whitespace content))
       ((mv err clause new-content)
        (parse-clause new-content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Bad clause." err)
            nil content))
       ((if (atom new-content))
        (mv *empty-string* (append clause (list 0)) new-content))
       ((mv success new-content)
        (parse-whitespace new-content))
       ((if (not success))
        (mv (error-string-define "No trailing whitespace after clause.")
            nil new-content))
       ((if (atom new-content))
        (mv *empty-string* (append clause (list 0)) new-content))
       ((mv err formula new-content)
        (parse-formula-flat new-content)))
      (mv err (append (append clause (list 0)) formula) new-content)))




(define parse-dimacs-flat
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-vars natp :hyp :guard)
               (num-clauses natp :hyp :guard)
               (formula integer-listp :hyp :guard))
  (b* (((mv err num-vars num-clauses new-content)
        (parse-preamble content))
       ((if (not (empty-stringp err)))
        (mv (error-string-define "Preamble error.")
            num-vars num-clauses nil))
       ((mv err formula ?new-content)
        (parse-formula-flat new-content)))
      (mv err num-vars num-clauses formula)))


(set-state-ok t)  ;; The use of STATE is OK.

(define read-dimacs-flat
  ((filename stringp)
   (state state-p))
  :returns (mv (contents integer-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (read-file-bytes filename state))
       ((if (stringp contents))
        (mv nil state)))
      (mv contents state)))

(define read-and-parse-with-state-flat
  ((filename stringp)
   (state state-p))
  :returns (mv (err stringp)
               (num-vars natp)
               (num-clauses natp)
               (formula integer-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (read-dimacs-flat filename state))
       ((mv err num-vars num-clauses formula)
        (parse-dimacs-flat contents)))
      (mv err num-vars num-clauses formula state)))

(defmacro read-and-parse-flat (filename)
  `(read-and-parse-with-state-flat ,filename state))

(set-state-ok nil)


;; ===================================================================

(defthm integer-list-listp-rev
  (implies (integer-list-listp x)
           (integer-list-listp (rev x)))
  :hints (("Goal" :in-theory (enable integer-list-listp))))


(define parse-formula-count-clauses1
  ((err stringp)
   (num-clauses natp)
   (formula integer-list-listp)
   (content integer-listp))
  :returns (mv (err stringp)
               (num-clauses natp :hyp :guard)
               (formula integer-list-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  :measure (len content)
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            num-clauses formula nil))
       (new-content (trim-whitespace content))
       ((mv clause-err clause new-content)
        (parse-clause new-content))
       ((if (not (empty-stringp clause-err)))
        (mv (error-string-define "Bad clause." clause-err)
            num-clauses formula content))
       ((if (atom new-content))
        (mv *empty-string*
            (1+ num-clauses)
            (cons clause formula)
            new-content))
       ((mv success new-content)
        (parse-whitespace new-content))
       ((if (not success))
        (mv (error-string-define "No trailing whitespace after clause.")
            num-clauses formula new-content))
       ((if (atom new-content))
        (mv *empty-string*
            (1+ num-clauses)
            (cons clause formula)
            new-content)))
      (parse-formula-count-clauses1 err
                                    (1+ num-clauses)
                                    (cons clause formula)
                                    new-content)))

(define parse-formula-count-clauses
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-clauses natp :hyp :guard)
               (formula integer-list-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  (b* (((mv err num-clauses formula new-content)
        (parse-formula-count-clauses1 *empty-string* 0 nil content)))
      (mv err num-clauses (rev formula) new-content)))

(define parse-proof
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-clauses natp :hyp :guard)
               (formula integer-list-listp :hyp :guard))
  (b* (((mv err num-clauses formula ?new-content)
        (parse-formula-count-clauses content)))
      (mv err num-clauses formula)))


(set-state-ok t)  ;; The use of STATE is OK.

(define read-proof
  ((filename stringp)
   (state state-p))
  :returns (mv (contents integer-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (read-file-bytes filename state))
       ((if (stringp contents))
        (mv nil state)))
      (mv contents state)))

(define read-and-parse-proof-with-state
  ((filename stringp)
   (state state-p))
  :returns (mv (err stringp)
               (num-clauses natp)
               (formula integer-list-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (time$ (read-proof filename state)))
       ((mv err num-clauses formula)
        (time$ (parse-proof contents))))
      (mv err num-clauses formula state)))

(defmacro read-and-parse-proof (filename)
  `(read-and-parse-proof-with-state ,filename state))

(set-state-ok nil)


;; ===================================================================

(define parse-formula-flat-count-clauses1
  ((err stringp)
   (num-clauses natp)
   (formula integer-listp)
   (content integer-listp))
  :returns (mv (err stringp)
               (num-clauses natp :hyp :guard)
               (formula integer-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  :measure (len content)
  (b* (((if (atom content))
        (mv (error-string-define "End of list.")
            num-clauses formula nil))
       (new-content (trim-whitespace content))
       ((mv clause-err clause new-content)
        (parse-clause new-content))
       ((if (not (empty-stringp clause-err)))
        (mv (error-string-define "Bad clause." clause-err)
            num-clauses formula content))
       ((if (atom new-content))
        (mv *empty-string*
            (1+ num-clauses)
            (append (append (list 0) (rev clause))
                    formula)
            new-content))
       ((mv success new-content)
        (parse-whitespace new-content))
       ((if (not success))
        (mv (error-string-define "No trailing whitespace after clause.")
            num-clauses formula new-content))
       ((if (atom new-content))
        (mv *empty-string*
            (1+ num-clauses)
            (append (append (list 0) (rev clause))
                    formula)
            new-content)))
      (parse-formula-flat-count-clauses1 err
                                         (1+ num-clauses)
                                         (append (append (list 0) (rev clause))
                                                 formula)
                                         new-content)))

(define parse-formula-flat-count-clauses
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-clauses natp :hyp :guard)
               (formula integer-listp :hyp :guard)
               (new-content integer-listp :hyp :guard))
  (b* (((mv err num-clauses formula new-content)
        (parse-formula-flat-count-clauses1 *empty-string* 0 nil content)))
      (mv err num-clauses (rev formula) new-content)))

(define parse-proof-flat
  ((content integer-listp))
  :returns (mv (err stringp)
               (num-clauses natp :hyp :guard)
               (formula integer-listp :hyp :guard))
  (b* (((mv err num-clauses formula ?new-content)
        (parse-formula-flat-count-clauses content)))
      (mv err num-clauses formula)))


(set-state-ok t)  ;; The use of STATE is OK.

(define read-proof-flat
  ((filename stringp)
   (state state-p))
  :returns (mv (contents integer-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (read-file-bytes filename state))
       ((if (stringp contents))
        (mv nil state)))
      (mv contents state)))

(define read-and-parse-proof-with-state-flat
  ((filename stringp)
   (state state-p))
  :returns (mv (err stringp)
               (num-clauses natp)
               (formula integer-listp)
               (state state-p :hyp :guard))
  (b* (((mv contents state)
        (time$ (read-proof-flat filename state)))
       ((mv err num-clauses formula)
        (time$ (parse-proof-flat contents))))
      (mv err num-clauses formula state)))

(defmacro read-and-parse-proof-flat (filename)
  `(read-and-parse-proof-with-state-flat ,filename state))

(set-state-ok nil)

;; ===================================================================

;; (defttag t)

;; (remove-untouchable acl2::create-state t)

;; (defun read-and-parse-local-state (filename)
;;   ;; This function requires the STATE, so we use the next form
;;   (with-local-state
;;    ;; "Powerful" macro that provides access to the state, but
;;    ;; requires the two events above.
;;    (mv-let (err num-vars num-clauses formula state)
;;            (read-and-parse filename)
;;            (mv err num-vars num-clauses formula))))

;; Example Read -- filename can be changed.

;; (defconst *f*
;;   (mv-let (err num-vars num-clauses formula)
;;           (read-and-parse-local-state "rbcl_xits_07_UNSAT.cnf")
;;           (list err num-vars num-clauses formula)))
