(in-package "SAT")

(include-book "supplemental/ternary")
(include-book "supplemental/literal")
(include-book "supplemental/sets")
(include-book "supplemental/assignment")
(include-book "supplemental/clause")
(include-book "supplemental/evaluator")
(include-book "supplemental/unit-propagation")



;; ========================== CLAUSE-LISTP ===========================

(defun clause-listp (clause-list)
  (declare (xargs :guard t))
  (if (atom clause-list)
      (null clause-list)
    (and (clausep (car clause-list))
         (clause-listp (cdr clause-list)))))


(defthm clause-listp-implies-true-listp
  (implies (clause-listp clause-list)
           (true-listp clause-list)))


;; ====================== CLAUSE-TO-ASSIGNMENT =======================
;; ====================== ASSIGNMENT-TO-CLAUSE =======================

(defun clause-to-assignment (clause)
  (declare (xargs :guard (clausep clause)))
  (if (atom clause)
      nil
    (cons (negate (car clause))
          (clause-to-assignment (cdr clause)))))

(defun assignment-to-clause (assignment)
  (if (atom assignment)
      nil
    (cons (negate (car assignment))
          (assignment-to-clause (cdr assignment)))))

(defthm clause-to-assignment-assignment-to-clause
  (implies (literal-listp x)
           (equal (clause-to-assignment (assignment-to-clause x))
                  x)))

(defthm assignment-to-clause-clause-to-assignment
  (implies (literal-listp x)
           (equal (assignment-to-clause (clause-to-assignment x))
                  x)))

(defthm member-negate-clause-to-assignment
  (implies (and (literalp e)
                (literal-listp x))
           (iff (member e (clause-to-assignment x))
                (member (negate e) x))))

(defthm member-negate-assignment-to-clause
  (implies (and (literalp e)
                (literal-listp x))
           (iff (member e (assignment-to-clause x))
                (member (negate e) x))))

(defthm assignmentp-clause-to-assignment
  (implies (clausep clause)
           (assignmentp (clause-to-assignment clause))))

(defthm assignmentp-assignment-to-clause
  (implies (assignmentp clause)
           (clausep (assignment-to-clause clause)))
  :hints (("Goal" :in-theory (enable assignmentp))))


;; ========================= EVALUATOR FACTS =========================

(defthm iff-truep-evaluate-clause-cons
  (iff (truep (evaluate-clause clause (cons literal assignment)))
       (or (member literal clause)
           (truep (evaluate-clause clause assignment)))))

(defthm truep-evaluate-formula-cons
  (equal (truep (evaluate-formula (cons clause formula) assignment))
         (and (truep (evaluate-clause clause assignment))
              (truep (evaluate-formula formula assignment)))))

; needed?
(defthm not-undefp-evaluate-clause-cons
  (implies (not (undefp (evaluate-clause clause assignment)))
           (not (undefp (evaluate-clause clause (cons literal assignment))))))



(defthm is-unit-clause-implies-undefp-evaluate-clause
  (implies (is-unit-clause clause assignment)
           (undefp (evaluate-clause clause assignment))))


;; ===================== POTENTIAL-UNIT-CLAUSES ======================

(defun potential-unit-clauses (formula assignment)
  (declare (xargs :guard (and (formulap formula)
                              (assignmentp assignment))))
  (if (atom formula)
      0
    (if (undefp (evaluate-clause (car formula) assignment))
        (1+ (potential-unit-clauses (cdr formula) assignment))
      (potential-unit-clauses (cdr formula) assignment))))


(defthm natp-potential-unit-clauses
  (natp (potential-unit-clauses formula assignment))
  :rule-classes :type-prescription)
           
(defthm potential-unit-clauses-cons
  (<= (potential-unit-clauses formula (cons literal assignment))
      (potential-unit-clauses formula assignment))
  :rule-classes :linear)

(defthm potential-unit-clauses-cons-2
  (implies (mv-nth 0 (find-unit-clause formula assignment))
           (< (potential-unit-clauses formula
                                      (cons (mv-nth 0 (find-unit-clause
                                                       formula
                                                       assignment))
                                            assignment))
              (potential-unit-clauses formula assignment))))


;; ======================== UNIT-PROPAGATION =========================

(defun unit-propagation (formula assignment)
  (declare (xargs :guard (and (formulap formula)
                              (assignmentp assignment))
                  :measure (potential-unit-clauses formula assignment)))
  (mv-let (unit-literal unit-clause)
          (find-unit-clause formula assignment)
          (declare (ignorable unit-clause))
          (if (not unit-literal)
              assignment
            (unit-propagation formula (cons unit-literal assignment)))))


(defthm assignmentp-unit-propagation
  (implies (formulap formula)
           (equal (assignmentp (unit-propagation formula assignment))
                  (assignmentp assignment))))
           

;; ========================= REMOVE-LITERAL ==========================

(defun remove-literal (literal clause)
  (declare (xargs :guard (and (literalp literal)
                              (clausep clause))))
  (if (atom clause)
      nil
    (if (equal (car clause) literal)
        (remove-literal literal (cdr clause))
      (cons (car clause)
            (remove-literal literal (cdr clause))))))


(defthm not-member-remove-literal
  (not (member lit (remove-literal lit clause))))

(defthm member-remove-literal
  (iff (member lit1 (remove-literal lit2 clause))
       (if (equal lit1 lit2)
           nil
         (member lit1 clause))))


(defthm literal-listp-remove-literal
  (implies (and (literalp literal)
                (clausep clause))
           (literal-listp (remove-literal literal clause))))

(defthm unique-literalsp-remove-literal
  (implies (and (literalp literal)
                (clausep clause))
           (unique-literalsp (remove-literal literal clause))))

(defthm no-conflicting-literalsp-remove-literal
  (implies (and (literalp literal)
                (clausep clause))
           (no-conflicting-literalsp (remove-literal literal clause))))


(defthm clausep-remove-literal
  (implies (and (literalp literal)
                (clausep clause))
           (clausep (remove-literal literal clause))))


;; =========================== RESOLUTION ============================

(defun resolution (lit A B)
  (declare (xargs :guard (and (literalp lit)
                              (clausep A)
                              (clausep B))))
  (union (remove-literal lit A)
         (remove-literal (negate lit) B)))


(defthm member-union
  (iff (member e (union A B))
       (or (member e A)
           (member e B))))


(defthm literal-listp-union
  (implies (and (clausep A)
                (clausep B))
           (literal-listp (union A B))))

(defthm unique-literalsp-union
  (implies (and (clausep A)
                (clausep B))
           (unique-literalsp (union A B))))


(defthm literal-listp-resolution
  (implies (and (literalp lit)
                (clausep A)
                (clausep B))
           (literal-listp (resolution lit A B))))

(defthm unique-literalsp-resolution
  (implies (and (literalp lit)
                (clausep A)
                (clausep B))
           (unique-literalsp (resolution lit A B))))


;; ===================================================================

(defun ALA (formula clause)
  (declare (xargs :guard (and (formulap formula)
                              (clausep clause))))
  (unit-propagation formula (clause-to-assignment clause)))

(defun ATp (formula clause)
  (declare (xargs :guard (and (formulap formula)
                              (clausep clause))))
  (falsep (evaluate-formula formula
                            (ALA formula clause))))

(defun RATp1 (clause-list formula clause literal)
  (declare (xargs :guard (and (clause-listp clause-list)
                              (formulap formula)
                              (clausep clause)
                              (literalp literal))))
  (b* (((when (atom clause-list)) t)
       ((when (not (member (negate literal) (car clause-list))))
        (RATp1 (cdr clause-list) formula clause literal))
       (resolvent (resolution literal clause (car clause-list)))
       ((when (not (no-conflicting-literalsp resolvent)))
        (RATp1 (cdr clause-list) formula clause literal)))
      (and (ATp formula resolvent)
           (RATp1 (cdr clause-list) formula clause literal))))

(defun RATp (formula clause literal)
  (declare (xargs :guard (and (formulap formula)
                              (clausep clause)
                              (literalp literal))))
  (RATp1 formula formula clause literal))
        
    
(defun verify-clause (clause formula)
  (declare (xargs :guard (and (clausep clause)
                              (formulap formula))))
  (or (ATp formula clause)
      (and (not (atom clause))
           (RATp formula clause (car clause)))))


;; ======================= VERIFY-UNSAT-PROOF ========================


(defun verify-proof (clause-list formula)
  (declare (xargs :guard (and (formulap formula)
                              (clause-listp clause-list))))
  (if (atom clause-list)
      t
    (if (verify-clause (car clause-list) formula)
        (verify-proof (cdr clause-list) (cons (car clause-list) formula))
      nil)))


(defun proofp (proof formula)
  (declare (xargs :guard (formulap formula)))
  (and (clause-listp proof)
       (verify-proof proof formula)))

(defconst *empty-clause* nil)

(defun refutationp (proof formula)
  (declare (xargs :guard (formulap formula)
                  :verify-guards nil))
  (and (proofp proof formula)
       (member *empty-clause* proof)))



(defun solutionp (solution formula)
  (and (assignmentp solution)
       (truep (evaluate-formula formula solution))))

(defun-sk exists-solution (formula)
  (exists assignment (solutionp assignment formula)))
(in-theory (disable exists-solution
                    exists-solution-suff))


;; ===================================================================
;; ===================================================================
;; ===================================================================

;; ============================= ATP NIL =============================

(defthm truep-evaluate-clause-subsetp
  (implies (and (subsetp sub super)
                (truep (evaluate-clause clause sub)))
           (truep (evaluate-clause clause super))))

;; (defthm truep-evaluate-formula-subsetp-lemma
;;   (implies (and (not (falsep (evaluate-clause clause sub)))
;;                 (not (undefp (evaluate-clause clause sub)))
;;                 (subsetp sub super))
;;            (truep (evaluate-clause clause super))))

; yuck hints
(defthm truep-evaluate-formula-subsetp
  (implies (and (subsetp sub super)
                (truep (evaluate-formula formula sub)))
           (truep (evaluate-formula formula super))))
  ;; :hints (("Subgoal *1/6" 
  ;;          :in-theory (disable ternaryp-not-false-not-undef)
  ;;          :use ((:instance ternaryp-not-false-not-undef
  ;;                           (x (evaluate-clause (car formula) sub)))))
  ;;         ("Subgoal *1/2" 
  ;;          :in-theory (disable ternaryp-not-false-not-undef)
  ;;          :use ((:instance ternaryp-not-false-not-undef
  ;;                           (x (evaluate-clause (car formula) sub)))))))



(defthm assignmentp-and-member-implies-not-member-negate
  (implies (and (assignmentp assignment)
                (member literal assignment))
           (not (member (negate literal) assignment)))
  :hints (("Goal"
           :in-theory (enable assignmentp))))

; slow
(defthm falsep-evaluate-clause-subsetp
  (implies (and (subsetp sub super)
                (falsep (evaluate-clause clause sub))
                (assignmentp super))
           (falsep (evaluate-clause clause super))))
                            
(defthm falsep-evaluate-formula-subsetp
  (implies (and (subsetp sub super)
                (falsep (evaluate-formula formula sub))
                (assignmentp super))
           (falsep (evaluate-formula formula super))))


; also slow
(defthm truep-evaluate-clause-is-unit-clause-subsetp
  (implies (and (subsetp sub super)
                (truep (evaluate-clause clause super))
                (is-unit-clause clause sub)
                (assignmentp super))
           (member (is-unit-clause clause sub)
                   super)))


(in-theory (disable assignmentp-and-member-implies-not-member-negate))



(defthm truep-evaluate-formula-find-unit-clause-subsetp
  (implies (and (subsetp sub super)
                (truep (evaluate-formula formula super))
                (mv-nth 0 (find-unit-clause formula sub))
                (assignmentp super))
           (member (mv-nth 0 (find-unit-clause formula sub))
                   super)))

(defthm truep-evaluate-formula-unit-propagation-subsetp
  (implies (and (subsetp sub super)
                (truep (evaluate-formula formula super))
                (assignmentp super))
           (subsetp (unit-propagation formula sub) super)))


(defthm evaluate-formula-unit-propagation-nil
  (implies (and (assignmentp solution)
                (truep (evaluate-formula formula solution)))
           (not (falsep (evaluate-formula formula
                                          (unit-propagation formula nil)))))
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable falsep-evaluate-formula-subsetp
                               truep-evaluate-formula-unit-propagation-subsetp)
           :use ((:instance truep-evaluate-formula-unit-propagation-subsetp
                            (sub nil)
                            (super solution))
                 (:instance falsep-evaluate-formula-subsetp
                            (sub (unit-propagation formula nil))
                            (super solution))))))



(defthm solutionp-implies-not-ATp-*empty-clause*
  (implies (solutionp solution formula)
           (not (ATp formula *empty-clause*))))




;; ======================= EXISTS-TRUE-LITERAL =======================

(defun find-true-literal-in-clause (clause assignment)
  (if (atom clause)
      nil
    (if (member (car clause) assignment)
        (car clause)
      (find-true-literal-in-clause (cdr clause) assignment))))


;; (defthm truep-evaluate-clause-implies-find-true-literal-in-clause
;;   (implies (and (clausep clause)
;;                 (truep (evaluate-clause clause assignment)))
;;            (find-true-literal-in-clause clause assignment)))
                  
(defthm truep-evaluate-clause-implies-find-true-literal-in-clause
  (implies (clausep clause)
           (iff (find-true-literal-in-clause clause assignment)
                (truep (evaluate-clause clause assignment)))))

(defthm find-true-literal-in-clause-implies-member-clause
  (implies (find-true-literal-in-clause clause assignment)
           (member (find-true-literal-in-clause clause assignment)
                   clause)))

(defthm find-true-literal-in-clause-implies-member-assignment
  (implies (find-true-literal-in-clause clause assignment)
           (member (find-true-literal-in-clause clause assignment)
                   assignment)))

(defthm find-true-literal-in-clause-implies-member-assignment
  (implies (find-true-literal-in-clause clause assignment)
           (member (find-true-literal-in-clause clause assignment)
                   assignment)))


(defthm find-true-literal-in-clause-implies-literalp
  (implies (and (clausep clause)
                (find-true-literal-in-clause clause assignment))
           (literalp (find-true-literal-in-clause clause assignment))))

(defun-sk exists-true-literal (clause assignment)
  (exists literal (and (literalp literal)
                       (member literal clause)
                       (member literal assignment))))
(in-theory (disable exists-true-literal
                    exists-true-literal-suff))


(defthm truep-evaluate-clause-implies-exists-true-literal
  (implies (and (clausep clause)
                (truep (evaluate-clause clause assignment)))
           (exists-true-literal clause assignment))
  :hints (("Goal"
           :use ((:instance exists-true-literal-suff
                            (literal (find-true-literal-in-clause
                                      clause
                                      assignment)))))))

(defthm member-both-implies-truep-evaluate-clause
  (implies (and (member literal clause)
                (member literal assignment))
           (truep (evaluate-clause clause assignment))))

(defthm exists-true-literal-implies-truep-evaluate-clause
  (implies (exists-true-literal clause assignment)
           (truep (evaluate-clause clause assignment)))
  :hints (("Goal"
           :use ((:instance (:definition exists-true-literal))))))


;; ===================================================================


;(i-am-here)


;; (in-theory (disable ATp resolution clausep evaluate-clause))


;; (defun-sk exists-false-clause (formula assignment)
;;   (exists clause (and (clausep clause)
;;                       (member clause formula)
;;                       (falsep (evaluate-clause clause assignment)))))
;; (in-theory (disable exists-false-clause
;;                     exists-false-clause-suff))

;; (skip-proofs
;;  (defthm falsep-evaluate-formula-implies-exists-false-clause
;;    (implies (and (formulap formula)
;;                  (falsep (evaluate-formula formula assignment)))
;;             (exists-false-clause formula assignment))))


;; (defthm truep-evaluate-formula-and-member-implies-truep-evaluate-clause
;;   (implies (and (truep (evaluate-formula formula solution))
;;                 (member clause formula))
;;            (truep (evaluate-clause clause solution))))


;; (implies (and (member literal (unit-propagation formula assignment))
;;               (not (member literal assignment))
;;               (subsetp assignment mid-propagation-assignment))
              

;; (defthm solutionp-and-ATp-implies-truep-evaluate-clause
;;   (implies (and (falsep (evaluate-clause clause
;;                                           (unit-propagation formula
;;                                                             assignment)))
;;                 (truep (evaluate-clause clause solution))
;;                 (truep (evaluate-formula formula solution))
;;                 (formulap formula)
;;                 (clausep at-clause)
;;                 (member clause formula)
;;                 (hide (equal assignment (invert-clause at-clause)))
;;                 (assignmentp solution))
;;            (exists-true-literal at-clause solution))
;;   :hints (("Goal"
;;            :in-theory (disable exists-true-literal-implies-truep-evaluate-clause)
;;            :use ((:instance exists-true-literal-implies-truep-evaluate-clause
;;                             (clause at-clause)
;;                             (assignment solution))))))
;;           ("Subgoal *1/3'"
;;            :expand ((hide (equal assignment (invert-clause at-clause)))
;;                     (hide (equal (cons (mv-nth 0 (find-unit-clause
;;                                                   formula
;;                                                   assignment))
;;                                        assignment)
;;                                  (invert-clause at-clause)))))
;;           ("Subgoal *1/1'"
;;            :expand (hide (equal assignment (invert-clause at-clause)))
;;            :use ((:instance truep-evaluate-clause-implies-exists-true-literal
;;                             (clause (mv-nth 1 (find-unit-clause formula assignment)))
;;                             (assignment solution))
;;                  (:instance (:definition exists-true-literal)
;;                             (clause (mv-nth 1 (find-unit-clause formula assignment)))
;;                             (assignment solution))))))


;; (defthm solutionp-and-ATp-implies-truep-evaluate-clause
;;   (implies (and (falsep (evaluate-formula formula
;;                                           (unit-propagation formula
;;                                                             (invert-clause
;;                                                              clause))))
;;                 (truep (evaluate-formula formula solution))
;;                 (formulap formula)
;;                 (clausep clause)
;;                 (assignmentp solution))
;;            (exists-true-literal clause solution))
;;   :hints (("Goal"
;;            :in-theory (disable unit-propagation invert-clause formulap clausep assignmentp))))
;;   :hints (("Goal"
;;            :use ((:instance falsep-evaluate-formula-implies-exists-false-clause
;;                             (assignment (unit-propagation formula
;;                                                           (invert-clause
;;                                                            clause))))
;;                  (:instance (:definition exists-false-clause)
;;                             (assignment (unit-propagation formula
;;                                                           (invert-clause
;;                                                            clause))))))))

;; (defthm solutionp-and-ATp-implies-truep-evaluate-clause
;;   (implies (and (ATp formula clause)
;;                 (truep (evaluate-formula formula solution))
;;                 (formulap formula)
;;                 (clausep clause)
;;                 (assignmentp solution))
;;            (exists-true-literal clause solution)))


(defthm truep-evaluate-clause-assignment-to-clause
  (implies (and (literalp literal)
                (member literal solution)
                (member (negate literal) assignment))
           (truep (evaluate-clause (assignment-to-clause assignment)
                                   solution))))

(defthm evaluate-clause-implies-truep-evaluate-clause-assignment-to-clause
  (implies (and (clausep clause)
                (truep (evaluate-clause clause solution))
                (falsep (evaluate-clause clause assignment)))
           (truep (evaluate-clause (assignment-to-clause assignment)
                                   solution))))

(defthm evaluate-formula-implies-truep-evaluate-clause-assignment-to-clause
  (implies (and (formulap formula)
                (truep (evaluate-formula formula solution))
                (falsep (evaluate-formula formula assignment)))
           (truep (evaluate-clause (assignment-to-clause assignment)
                                   solution)))
  :hints (("Goal"
           :in-theory (disable clausep))))


(defthm assignmentp-and-member-implies-not-member-negate
  (implies (and (assignmentp assignment)
                (member literal assignment))
           (not (member (negate literal) assignment)))
  :hints (("Goal"
           :in-theory (enable assignmentp))))

(defthm not-is-unit-clause-literal-implies-member-negate
  (implies (and (is-unit-clause clause assignment)
                (member literal clause)
                (not (equal literal (is-unit-clause clause assignment))))
           (member (negate literal) assignment)))

(defthm not-equal-unit-literal-implies-member-negate
  (implies (and (mv-nth 0 (find-unit-clause formula assignment))
                (member literal (mv-nth 1 (find-unit-clause
                                           formula
                                           assignment)))
                (not (equal literal (mv-nth 0 (find-unit-clause
                                               formula
                                               assignment)))))
           (member (negate literal) assignment)))


(defthm not-equal-unit-literal-implies-truep-assignment-to-clause
  (implies (and (literalp literal)
                (mv-nth 0 (find-unit-clause formula assignment))
                (member literal (mv-nth 1 (find-unit-clause
                                           formula
                                           assignment)))
                (not (equal literal (mv-nth 0 (find-unit-clause
                                               formula
                                               assignment))))
                (member literal solution))
           (truep (evaluate-clause (assignment-to-clause assignment)
                                    solution))))

(defthm truep-evaluate-formula-implies-exists-true-literal
  (implies (and (formulap formula)
                (truep (evaluate-formula formula solution))
                (mv-nth 0 (find-unit-clause formula assignment)))
           (exists-true-literal (mv-nth 1 (find-unit-clause
                                           formula
                                           assignment))
                                solution)))


(defthm find-unit-clause-and-member-negate-implies-truep-assignment-to-clause
  (implies (and (formulap formula)
                (assignmentp solution)
                (truep (evaluate-formula formula solution))
                (mv-nth 0 (find-unit-clause formula assignment))
                (member (negate (mv-nth 0 (find-unit-clause
                                           formula
                                           assignment)))
                        solution))
           (truep (evaluate-clause (assignment-to-clause assignment)
                                   solution)))
  :otf-flg t
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable
                       truep-evaluate-formula-implies-exists-true-literal
                       exists-true-literal-implies-truep-evaluate-clause)
                       ;find-true-literal-in-clause-implies-member-assignment)
           :use ((:instance truep-evaluate-formula-implies-exists-true-literal)
                 (:instance (:definition exists-true-literal)
                            (clause (mv-nth 1 (find-unit-clause
                                               formula
                                               assignment)))
                            (assignment solution))))
          ("Goal'4'"
           :cases ((equal (mv-nth 0 (find-unit-clause formula
                                                      assignment))
;(find-true-literal-in-clause
                          (exists-true-literal-witness
                           (mv-nth 1 (find-unit-clause formula
                                                       assignment))
                           solution))))
          ("Subgoal 1"
           :use ((:instance assignmentp-and-member-implies-not-member-negate
                            (literal (mv-nth 0 (find-unit-clause formula assignment)))
                            (assignment solution))))
          ("Subgoal 2"
           :use ((:instance
                  not-equal-unit-literal-implies-truep-assignment-to-clause
                  (literal (exists-true-literal-witness
                            (mv-nth 1 (find-unit-clause formula
                                                        assignment))
                            solution)))))
          ))


; redundant
(defthm find-unit-clause-and-member-negate-implies-truep-assignment-to-clause
  (implies (and (formulap formula)
                (assignmentp solution)
                (truep (evaluate-formula formula solution))
                (mv-nth 0 (find-unit-clause formula assignment))
                (member (negate (mv-nth 0 (find-unit-clause
                                           formula
                                           assignment)))
                        solution))
           (truep (evaluate-clause (assignment-to-clause assignment)
                                   solution))))



(defthm falsep-EF-UP-implies-truep-EC-a2c
  (implies (and (falsep (evaluate-formula formula
                                          (unit-propagation formula
                                                            assignment)))
                (truep (evaluate-formula formula solution))
                (formulap formula)
                (assignmentp assignment)
                (assignmentp solution))
           (truep (evaluate-clause (assignment-to-clause assignment) solution))))

(defthm falsep-EF-UP-c2a-implies-truep-EC
  (implies (and (falsep (evaluate-formula formula
                                          (unit-propagation formula
                                                            (clause-to-assignment
                                                             clause))))
                (truep (evaluate-formula formula solution))
                (formulap formula)
                (clausep clause)
                (assignmentp solution))
           (truep (evaluate-clause clause solution)))
  :hints (("Goal"
           :use ((:instance falsep-EF-UP-implies-truep-EC-a2c
                            (assignment (clause-to-assignment clause)))))))


(defthm solutionp-and-ATp-implies-truep-evaluate-clause
  (implies (and (ATp formula clause)
                (truep (evaluate-formula formula solution))
                (formulap formula)
                (clausep clause)
                (assignmentp solution))
           (truep (evaluate-clause clause solution))))



; AT-checker proof
(defthm solutionp-and-ATp-implies-solutionp-cons
  (implies (and (ATp formula clause)
                (solutionp solution formula)
                (formulap formula)
                (clausep clause))
           (solutionp solution (cons clause formula)))
  :hints (("Goal" :in-theory (disable ATp clausep))))


(defthm exists-solution-implies-not-ATp-*empty-clause*
  (implies (exists-solution formula)
           (not (ATp formula *empty-clause*)))
  :hints (("Goal"
           :use ((:instance (:definition exists-solution))))))

(defthm exists-solution-and-ATp-implies-exists-solution
  (implies (and (ATp formula clause)
                (exists-solution formula)
                (formulap formula)
                (clausep clause))
           (exists-solution (cons clause formula)))
  :hints (("Goal"
           :in-theory (disable solutionp)
           :use ((:instance (:definition exists-solution))
                 (:instance exists-solution-suff
                            (assignment (exists-solution-witness formula))
                            (formula (cons clause formula)))))))

;; ===================================================================





(defun modify-solution (solution literal)
  (cons literal
        (remove-literal literal
                        (remove-literal (negate literal)
                                        solution))))

(defthm assignmentp-modify-solution
  (implies (and (assignmentp solution)
                (literalp literal))
           (assignmentp (modify-solution solution literal)))
  :hints (("Goal"
           :in-theory (enable assignmentp))))





;; ===================================================================


;(i-am-here)

;done
;***ANDTHISGUY***
(defthm member-implies-truep-evaluate-clause-modify-solution
  (implies (and (clausep clause)
                ;(literalp literal)
                (assignmentp solution)
                (member literal clause))
           (truep (evaluate-clause clause
                                   (modify-solution solution literal)))))




;; ===================================================================


(defthm truep-evaluate-clause-and-not-member-negate-implies-truep-evaluate-clause-modify-solution
  (implies (and (not (member (negate literal) clause))
                (truep (evaluate-clause clause solution)))
           (truep (evaluate-clause clause
                                   (modify-solution solution literal)))))





;; ===================================================================

(defun find-conflicting-literal (pseudo-clause)
  (if (atom pseudo-clause)
      nil
    (if (member (negate (car pseudo-clause)) (cdr pseudo-clause))
        (car pseudo-clause)
      (find-conflicting-literal (cdr pseudo-clause)))))

(defthm not-no-conflicting-literalsp-implies-find-conflicting-literal
  (implies (and (literal-listp pseudo-clause)
                (not (no-conflicting-literalsp pseudo-clause)))
           (find-conflicting-literal pseudo-clause)))

(defthm find-conflicting-literal-implies-member
  (implies (find-conflicting-literal pseudo-clause)
           (member (find-conflicting-literal pseudo-clause)
                   pseudo-clause)))

(defthm find-conflicting-literal-implies-member-negate
  (implies (find-conflicting-literal pseudo-clause)
           (member (negate (find-conflicting-literal pseudo-clause))
                   pseudo-clause)))
  
(defun-sk exists-conflicting-literal (pseudo-clause)
  (exists literal (and (member literal pseudo-clause)
                       (member (negate literal) pseudo-clause))))
(in-theory (disable exists-conflicting-literal
                    exists-conflicting-literal-suff))


(defthm not-no-conflicting-literalsp-implies-exists-conflicting-literal
  (implies (and (literal-listp pseudo-clause)
                (not (no-conflicting-literalsp pseudo-clause)))
           (exists-conflicting-literal pseudo-clause))
  :hints (("Goal"
           :use ((:instance exists-conflicting-literal-suff
                            (literal (find-conflicting-literal
                                      pseudo-clause)))))))

(defthm member-and-member-negate-implies-not-no-conflicting-literalsp
  (implies (and (literalp literal)
                (member literal pseudo-clause)
                (member (negate literal) pseudo-clause))
           (not (no-conflicting-literalsp pseudo-clause))))

(defthm exists-conflicting-literal-implies-not-no-conflicting-literalsp
  (implies (and (literal-listp pseudo-clause)
                (exists-conflicting-literal pseudo-clause))
           (not (no-conflicting-literalsp pseudo-clause)))
  :hints (("Goal"
           :use ((:instance (:definition exists-conflicting-literal))))))

;; ===================================================================



;; (implies (not (no-conflicting-literalsp psuedo-clause))
;;          (exists-conflict-literal pseudo-clause)
         

;; (implies (and (clausep clause)
;;               (clausep rat-clause)
;;               (member literal rat-clause)
;;               (member (negate literal) clause)
;;               (member witness-literal (resolution literal rat-clause clause))
;;               (member (negate witness-literal) (resolution literal rat-clause
;;                                                            clause)))
;;          (

;; (implies (and (no-conflicting-literalsp rat-clause)
;;               (no-conflicting-literalsp clause)
;;               (not (no-conflicting-literalsp (union rat-clause clause))))
;;          (exists-conflicting-literal rat-clause clause))

;; (implies (and (member literal rat-clause)
;;               (member (negate literal) clause)
;;               (member (negate literal) solution))
         


;; (defthm bb
;;   (implies (and (clausep clause)
;;                 (clausep rat-clause)
;;                 (member literal rat-clause)
;;                 (member (negate literal) clause)
;;                 (not (no-conflicting-literalsp (resolution literal rat-clause clause)))
;;                 (falsep (evaluate-clause rat-clause solution))
;;                 (truep (evaluate-clause clause solution)))
;;            (truep (evaluate-clause clause (modify-solution solution
;;                                                            literal))))

;; (defthm member-both-and-not-equal-implies-truep-EC-modify-solution
;;   (implies (and (member true-literal clause)
;;                 (member true-literal solution)
;;                 (not (equal true-literal (negate literal)))
;;                 (member (negate literal) clause))
;;            (truep (evaluate-clause clause (modify-solution solution literal)))))

;; (thm
;;  (implies (and (clausep rat-clause)
;;                (member literal-witness rat-clause))
;;           (iff (member (negate literal-witness) (resolution literal rat-clause
;;                                                             clause))
;;                (member (negate literal-witness) clause))))

;; (thm
;;  (implies (and (clausep rat-clause) (literalp literal) (not (equal
;;  literal-witness literal))
;;                (not (equal literal-witness (negate literal))))
;;           (iff (member literal-witness (resolution literal rat-clause clause))
;;                  (or (member literal-witness rat-clause)
;;                      (member literal-witness clause)))) )
;;  :hints (("Goal"
;;           :do-not-induct t
;;           :in-theory (disable
;;                       member-and-member-negate-implies-not-no-conflicting-literalsp
;;                       exists-conflicting-literal-implies-not-no-conflicting-literalsp
;;                       not-no-conflicting-literalsp-implies-exists-conflicting-literal))))






(defthm bb2
  (implies (and (clausep clause)
                (clausep rat-clause)
                ;(member literal rat-clause)
                ;(member (negate literal) clause)
                (member conflicting-literal rat-clause)
                (member (negate conflicting-literal) clause)
                ;(not (equal literal conflicting-literal))
                (falsep (evaluate-clause rat-clause solution))
                );(truep (evaluate-clause clause solution)))
           (member (negate conflicting-literal) solution)) )
           ;(truep (evaluate-clause clause (modify-solution solution
           ;                                                literal))))
  ;; :hints (("Goal"
  ;;          :in-theory (disable modify-solution))))

(defthm bb
  (implies (and (clausep clause)
                (clausep rat-clause)
                (member literal rat-clause)
                (member (negate literal) clause)
                (not (no-conflicting-literalsp (resolution literal rat-clause clause)))
                (falsep (evaluate-clause rat-clause solution))
                (truep (evaluate-clause clause solution)))
           (truep (evaluate-clause clause (modify-solution solution
                                                           literal))))
  :otf-flg t
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable evaluate-clause
                               not-no-conflicting-literalsp
                               modify-solution
                               ;not-no-conflicting-literalsp-implies-exists-conflicting-literal
                               ;exists-conflicting-literal-implies-not-no-conflicting-literalsp
                               );member-remove-literal
                               ;member-union)
           :use ((:instance
                  not-no-conflicting-literalsp-implies-exists-conflicting-literal
                  (pseudo-clause (resolution literal rat-clause clause)))
                 (:instance (:definition exists-conflicting-literal)
                            (pseudo-clause (resolution literal
                                                       rat-clause
                                                       clause)))))
          ("Subgoal 2"
           :do-not-induct t
           :in-theory (disable bb2 member-union)
           :use ((:instance bb2
                            (conflicting-literal
                             (exists-conflicting-literal-witness
                              (resolution literal rat-clause clause))))
                 (:instance member-union
                            (e (NEGATE (EXISTS-CONFLICTING-LITERAL-WITNESS
                                  (UNION (REMOVE-LITERAL LITERAL RAT-CLAUSE)
                                         (REMOVE-LITERAL (NEGATE LITERAL)
                                                         CLAUSE)))))
                            (a (REMOVE-LITERAL LITERAL RAT-CLAUSE))
                            (b (REMOVE-LITERAL (NEGATE LITERAL)
                                            CLAUSE)))))
          ("Subgoal 1"
           :do-not-induct t
           :in-theory (disable bb2 member-union)
           :use ((:instance bb2
                            (conflicting-literal
                             (negate (exists-conflicting-literal-witness
                              (resolution literal rat-clause clause)))))
                 (:instance member-union
                            (e (NEGATE (EXISTS-CONFLICTING-LITERAL-WITNESS
                                  (UNION (REMOVE-LITERAL LITERAL RAT-CLAUSE)
                                         (REMOVE-LITERAL (NEGATE LITERAL)
                                                         CLAUSE)))))
                            (a (REMOVE-LITERAL LITERAL RAT-CLAUSE))
                            (b (REMOVE-LITERAL (NEGATE LITERAL)
                                            CLAUSE)))))))






           
           ;; :in-theory (disable member-union)
           ;; :use ((:instance member-union
           ;;                  (e (NEGATE (EXISTS-CONFLICTING-LITERAL-WITNESS
           ;;                        (UNION (REMOVE-LITERAL LITERAL RAT-CLAUSE)
           ;;                               (REMOVE-LITERAL (NEGATE LITERAL)
           ;;                                               CLAUSE)))))
           ;;                  (a (REMOVE-LITERAL LITERAL RAT-CLAUSE))
           ;;                  (b (REMOVE-LITERAL (NEGATE LITERAL)
           ;;                                  CLAUSE)))))))



           ;; :in-theory (disable resolution
           ;;                     evaluate-clause
           ;;                     no-conflicting-literalsp
           ;;                     clausep
           ;;                     modify-solution
           ;;                     truep-evaluate-clause-implies-exists-true-literal)
           ;; :use ((:instance truep-evaluate-clause-implies-exists-true-literal
           ;;                  (assignment solution))
           ;;       (:instance (:definition exists-true-literal)
           ;;                  (assignment solution)))
           ;; :cases ((equal (exists-true-literal-witness clause solution)
           ;;                (negate literal))))))



; HERE

(defthm truep-evaluate-formula-and-member-implies-truep-evaluate-clause
  (implies (and (truep (evaluate-formula formula solution))
                (member clause formula))
           (truep (evaluate-clause clause solution))))


;(i-am-here)


;; ===================================================================
;; ===================================================================
;; (defthm bb
;;   (implies (and (clausep clause)
;;                 (literalp literal)
;;                 (assignmentp assignment)
;;                 (not (member (negate literal) clause))
;;                 (truep (evaluate-clause clause solution)))
;;            (truep (evaluate-clause clause (modify-solution solution literal)))))


;; ;; (defthm truep-evaluate-formula-and-RATp-implies-truep-evaluate-formula-modify-solution
;; ;;   (implies (and ;(formulap formula)
;; ;;                 (clausep clause)
;; ;;                 (literalp literal)
;; ;;                 (assignmentp solution)
;; ;;                 ;(RATp formula clause literal)
;; ;;                 (ATp formula (resolution literal clause formula-clause))
;; ;;                 (truep (evaluate-clause formula-clause solution))
;; ;;                 (member literal clause)
;; ;;                 (member formula-clause formula))
;; ;;            (truep (evaluate-clause formula-clause
;; ;;                                     (modify-solution solution literal))))
;; ;;   :otf-flg t
;; ;;   :hints (("Goal"
;; ;;            :in-theory (disable ;ATp
;; ;;                                resolution modify-solution clausep
;; ;;                                RATp
;; ;;                                ;formulap
;; ;;                                ;member-equal
;; ;;                                ;evaluate-formula
;; ;;                                evaluate-clause
;; ;;                                ))))


;; (defthm truep-evaluate-formula-and-RATp-implies-truep-evaluate-formula-modify-solution
;;   (implies (and (formulap formula)
;;                 (clausep clause)
;;                 (literalp literal)
;;                 (assignmentp solution)
;;                 ;(RATp1 clause-list formula clause literal)
;;                 (truep (evaluate-formula formula solution))
;;                 (member literal clause)
;;                 ;(subsetp clause-list formula)
;;                 );(subsetp formula clause-list))
;;            (truep (evaluate-formula formula
;;                                     (modify-solution solution literal))))
;;   :otf-flg t
;;   :hints (("Goal"
;;            :in-theory (disable ATp resolution modify-solution clausep
;;                                ;RATp
;;                                RATp1
;;                                ;formulap
;;                                member-equal
;;                                ;evaluate-formula
;;                                evaluate-clause
;;                                ))))


;; (defthm truep-evaluate-formula-and-RATp-implies-truep-evaluate-formula-modify-solution
;;   (implies (and (formulap formula)
;;                 (clausep clause)
;;                 (literalp literal)
;;                 (assignmentp solution)
;;                 (RATp1 clause-list formula clause literal)
;;                 (truep (evaluate-formula formula solution))
;;                 (member literal clause)
;;                 (subsetp clause-list formula)
;;                 (subsetp formula clause-list))
;;            (truep (evaluate-formula formula
;;                                     (modify-solution solution literal))))
;;   :otf-flg t
;;   :hints (("Goal"
;;            :in-theory (disable ATp resolution modify-solution clausep
;;                                ;RATp
;;                                ;RATp1
;;                                ;formulap
;;                                member-equal
;;                                ;evaluate-formula
;;                                evaluate-clause
;;                                subsetp
;;                                ))))
;; ===================================================================
;; ===================================================================







(defthm solution-literal-and-not-equal-implies-true-evaluate-clause-modify-solution
  (implies (and (not (equal solution-literal (negate literal)))
                (member solution-literal clause)
                (member solution-literal solution))
           (truep (evaluate-clause clause
                                   (modify-solution solution literal)))))
  
;; (defthm conflicting-literals-implies-ATp
;;   (implies (not (no-conflicting-literalsp psuedo-clause))
;;            (ATp formula psuedo-clause)))

(defthm clausep-resolution
  (implies (and (clausep rat-clause)
                (clausep clause)
                (literalp literal)
                (no-conflicting-literalsp (resolution literal rat-clause
                                                      clause)))
           (clausep (resolution literal rat-clause clause))))

(defthm falsep-evaluate-clause-and-member-implies-not-member
  (implies (and (falsep (evaluate-clause clause solution))
                (member literal solution))
           (not (member literal clause))))

(defthm not-member-and-member-resolution-implies-member
  (implies (and (not (member witness-literal rat-clause))
                (member witness-literal (resolution literal rat-clause clause)))
           (member witness-literal clause)))

(defthm member-resolution-implies-not-equal-negate
 (implies (and ;(literalp literal)
               ;(literalp witness-literal)
               ;(clausep clause)
               (clausep rat-clause)
               (member literal rat-clause)
               ;(member (negate literal) clause)
               (member witness-literal (resolution literal rat-clause clause)))
          (not (equal witness-literal (negate literal)))))

(defthm bbb
  (implies (and (clausep rat-clause)
                (clausep clause)
                (literalp literal)
                (no-conflicting-literalsp (resolution literal rat-clause
                                                      clause))
                (assignmentp solution)
                (member literal rat-clause)
                (member (negate literal) clause)
                (truep (evaluate-clause clause solution))
                (falsep (evaluate-clause rat-clause solution))
                (truep (evaluate-clause (resolution literal rat-clause clause)
                                        solution)))
           (truep (evaluate-clause clause (modify-solution solution literal))))
  :otf-flg t
  :hints (("Goal"
           :in-theory (disable resolution modify-solution clausep
                               evaluate-clause member
                               EXISTS-TRUE-LITERAL-IMPLIES-TRUEP-EVALUATE-CLAUSE
                               TRUEP-EVALUATE-CLAUSE-IMPLIES-EXISTS-TRUE-LITERAL
                               falsep-evaluate-clause-and-member-implies-not-member
                               not-member-and-member-resolution-implies-member
                               member-resolution-implies-not-equal-negate)
           :use ((:instance truep-evaluate-clause-implies-exists-true-literal
                            (assignment solution))
                 (:instance (:definition exists-true-literal)
                            (assignment solution))))
          ("Goal'4'"
           :cases ((equal (exists-true-literal-witness clause solution)
                          (negate literal))))
          ("Subgoal 1'"
           :use ((:instance truep-evaluate-clause-implies-exists-true-literal
                            (assignment solution)
                            (clause (resolution literal rat-clause clause)))
                 (:instance (:definition exists-true-literal)
                            (assignment solution)
                            (clause (resolution literal rat-clause clause)))))
          ("Subgoal 1'4'"
           :use ((:instance
                  falsep-evaluate-clause-and-member-implies-not-member
                  (clause rat-clause)
                  (literal (exists-true-literal-witness (resolution literal
  rat-clause clause) solution)))
                 (:instance
                  not-member-and-member-resolution-implies-member
                  (witness-literal (exists-true-literal-witness (resolution literal
                                                                            rat-clause
  clause) solution)))
                 (:instance member-resolution-implies-not-equal-negate
                  (witness-literal (exists-true-literal-witness (resolution literal
                                                                            rat-clause
  clause) solution))) ))))
                            
                 ;; (:instance member-both-implies-truep-evaluate-clause
                 ;;            (literal (exists-true-literal-witness (resolution literal
                 ;;                                                            rat-clause
                 ;;                                                            clause) solution))
                 ;;            (assignment solution))))))


                            
           

(defthm ATp-and-truep-evaluate-clause-implies-truep-evaluate-clause-modify-solution
  (implies (and (formulap formula)
                (clausep clause)
                (assignmentp solution)
                (clausep rat-clause)
                (literalp literal)
                (member literal rat-clause)
                (member (negate literal) clause)
                (ATp formula (resolution literal rat-clause clause))
                (truep (evaluate-formula formula solution))
                (truep (evaluate-clause clause solution))
                (falsep (evaluate-clause rat-clause solution))
                (no-conflicting-literalsp (resolution literal rat-clause clause))
                ;; from ATp proof
                ;; (truep (evaluate-clause (resolution literal rat-clause clause)
                ;;                         solution))
                )
           (truep (evaluate-clause clause (modify-solution solution literal))))
  :hints (("Goal"
           :in-theory (disable ATp resolution modify-solution clausep
                               solutionp-and-ATp-implies-truep-evaluate-clause)
           :use ((:instance solutionp-and-ATp-implies-truep-evaluate-clause
                            (clause (resolution literal rat-clause clause)))))))












;; ===================================================================
;; ===================================================================
;; ===================================================================



(defthm truep-evaluate-clause-and-RATp1-implies-truep-evaluate-clause-modify-solution
  (implies (and (formulap formula)
                (clausep clause)
                (assignmentp solution)
                (clausep rat-clause)
                ;(literalp literal)
                (RATp1 clause-list formula RAT-clause literal)
                (member clause clause-list)
                (member literal RAT-clause)
                (truep (evaluate-clause clause solution))
                (subsetp clause-list formula)
                (truep (evaluate-formula formula solution))
                (falsep (evaluate-clause RAT-clause solution)))
           (truep (evaluate-clause clause (modify-solution solution literal))))
  :hints (("Goal"
           :induct (RATp1 clause-list formula RAT-clause literal)
           :in-theory (disable ATp resolution modify-solution clausep
                               evaluate-clause
                               ))));TRUEP-EVALUATE-CLAUSE-AND-NOT-MEMBER-NEGATE-IMPLIES-TRUEP-EVALUATE-CLAUSE-MODIFY-SOLUTION))))

(defthm subsetp-reflexive
  (subsetp x x))

(defthm truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
  (implies (and (formulap formula)
                (clausep clause)
                (assignmentp solution)
                (clausep rat-clause)
                ;(literalp literal)
                (RATp formula RAT-clause literal)
                (member clause formula)
                (member literal RAT-clause)
                (truep (evaluate-clause clause solution))
                (truep (evaluate-formula formula solution))
                (falsep (evaluate-clause RAT-clause solution)))
           (truep (evaluate-clause clause (modify-solution solution literal))))
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable ATp resolution modify-solution clausep
                               evaluate-clause
                               truep-evaluate-clause-and-RATp1-implies-truep-evaluate-clause-modify-solution
                               )
           :use ((:instance
                  truep-evaluate-clause-and-RATp1-implies-truep-evaluate-clause-modify-solution
                  (clause-list formula))))))
;TRUEP-EVALUATE-CLAUSE-AND-NOT-MEMBER-NEGATE-IMPLIES-TRUEP-EVALUATE-CLAUSE-MODIFY-SOLUTION))))

; lemma 2

;; (thm
;;   (implies (RATp1 formula formula clause literal)
;;            (RATp1 (cdr formula) formula clause literal)))

;; (thm
;;  (implies (RATp formula clause literal)
;;           (RATp (cdr formula) clause literal)))

;; (thm
;;  (implies (RATp formula formula clause literal)
;;           (RATp (cdr formula) (cdr formula) clause literal)))

;; (defthm subsetp-cdr-something-else
;;   (implies (subsetp a b)
;;            (subsetp (cdr a) b)))


;; (AND
;;  (IMPLIES
;;     (AND (NOT (ATOM FORMULA))
;;          (NOT (FALSEP (EVALUATE-CLAUSE (CAR FORMULA)
;;                                        (MODIFY-SOLUTION SOLUTION LITERAL))))
;;          (NOT (FALSEP (EVALUATE-FORMULA (CDR FORMULA)
;;                                         (MODIFY-SOLUTION SOLUTION LITERAL))))
;;          (NOT (UNDEFP (EVALUATE-CLAUSE (CAR FORMULA)
;;                                        (MODIFY-SOLUTION SOLUTION LITERAL))))
;;          (:P CLAUSE (CDR FORMULA)
;;              LITERAL SOLUTION))
;;     (:P CLAUSE FORMULA LITERAL SOLUTION))
;;  (IMPLIES
;;     (AND (NOT (ATOM FORMULA))
;;          (NOT (FALSEP (EVALUATE-CLAUSE (CAR FORMULA)
;;                                        (MODIFY-SOLUTION SOLUTION LITERAL))))
;;          (NOT (FALSEP (EVALUATE-FORMULA (CDR FORMULA)
;;                                         (MODIFY-SOLUTION SOLUTION LITERAL))))
;;          (UNDEFP (EVALUATE-CLAUSE (CAR FORMULA)
;;                                   (MODIFY-SOLUTION SOLUTION LITERAL)))
;;          (:P CLAUSE (CDR FORMULA)
;;              LITERAL SOLUTION))
;;     (:P CLAUSE FORMULA LITERAL SOLUTION))
;;  (IMPLIES
;;      (AND (NOT (ATOM FORMULA))
;;           (NOT (FALSEP (EVALUATE-CLAUSE (CAR FORMULA)
;;                                         (MODIFY-SOLUTION SOLUTION LITERAL))))
;;           (FALSEP (EVALUATE-FORMULA (CDR FORMULA)
;;                                     (MODIFY-SOLUTION SOLUTION LITERAL)))
;;           (:P CLAUSE (CDR FORMULA)
;;               LITERAL SOLUTION))
;;      (:P CLAUSE FORMULA LITERAL SOLUTION))
;;  (IMPLIES (AND (NOT (ATOM FORMULA))
;;                (FALSEP (EVALUATE-CLAUSE (CAR FORMULA)
;;                                         (MODIFY-SOLUTION SOLUTION LITERAL))))
;;           (:P CLAUSE FORMULA LITERAL SOLUTION))
;;  (IMPLIES (ATOM FORMULA)
;;           (:P CLAUSE FORMULA LITERAL SOLUTION))).


;; (thm
;;  (implies (and (clausep clause)
;;                (literalp literal)
;;                (assignmentp solution)
;;                (truep (evaluate-clause clause solution)))
;;           (not (undefp (evaluate-clause clause (modify-solution solution
;;                                                                 literal))))))

;; (implies (truep (evaluate-formula formula solution))
;;          (not (undefp (evaluate-formula formula (modify-solution solution
;;          literal)))))



(defun find-false-clause (formula assignment)
  (if (atom formula)
      (mv nil nil)
    (if (falsep (evaluate-clause (car formula) assignment))
        (mv t (car formula))
      (find-false-clause (cdr formula) assignment))))

(defthm falsep-evaluate-formula-implies-find-false-clause
  (implies (and (formulap formula)
                (falsep (evaluate-formula formula assignment)))
           (mv-nth 0 (find-false-clause formula assignment))))

(defthm find-false-clause-implies-clausep
  (implies (and (formulap formula)
                (mv-nth 0 (find-false-clause formula assignment)))
           (clausep (mv-nth 1 (find-false-clause formula assignment)))))

(defthm find-false-clause-implies-member
  (implies (mv-nth 0 (find-false-clause formula assignment))
           (member (mv-nth 1 (find-false-clause formula assignment))
                   formula)))

(defthm find-false-clause-implies-falsep-evaluate-clause
  (implies (mv-nth 0 (find-false-clause formula assignment))
           (falsep (evaluate-clause (mv-nth 1 (find-false-clause
                                               formula
                                               assignment))
                                    assignment))))

(defun-sk exists-false-clause (formula assignment)
  (exists clause (and (clausep clause)
                      (member clause formula)
                      (falsep (evaluate-clause clause assignment)))))
(in-theory (disable exists-false-clause
                    exists-false-clause-suff))

(defthm falsep-evaluate-formula-implies-exists-false-clause
  (implies (and (formulap formula)
                (falsep (evaluate-formula formula assignment)))
           (exists-false-clause formula assignment))
  :hints (("Goal"
           :use ((:instance exists-false-clause-suff
                            (clause (mv-nth 1 (find-false-clause
                                               formula
                                               assignment))))))))





(defun find-undef-clause (formula assignment)
  (if (atom formula)
      (mv nil nil)
    (if (undefp (evaluate-clause (car formula) assignment))
        (mv t (car formula))
      (find-undef-clause (cdr formula) assignment))))

(defthm undefp-evaluate-formula-implies-find-undef-clause
  (implies (and (formulap formula)
                (undefp (evaluate-formula formula assignment)))
           (mv-nth 0 (find-undef-clause formula assignment))))

(defthm find-undef-clause-implies-clausep
  (implies (and (formulap formula)
                (mv-nth 0 (find-undef-clause formula assignment)))
           (clausep (mv-nth 1 (find-undef-clause formula assignment)))))

(defthm find-undef-clause-implies-member
  (implies (mv-nth 0 (find-undef-clause formula assignment))
           (member (mv-nth 1 (find-undef-clause formula assignment))
                   formula)))

(defthm find-undef-clause-implies-undefp-evaluate-clause
  (implies (mv-nth 0 (find-undef-clause formula assignment))
           (undefp (evaluate-clause (mv-nth 1 (find-undef-clause
                                               formula
                                               assignment))
                                    assignment))))

(defun-sk exists-undef-clause (formula assignment)
  (exists clause (and (clausep clause)
                      (member clause formula)
                      (undefp (evaluate-clause clause assignment)))))
(in-theory (disable exists-undef-clause
                    exists-undef-clause-suff))

(defthm undefp-evaluate-formula-implies-exists-undef-clause
  (implies (and (formulap formula)
                (undefp (evaluate-formula formula assignment)))
           (exists-undef-clause formula assignment))
  :hints (("Goal"
           :use ((:instance exists-undef-clause-suff
                            (clause (mv-nth 1 (find-undef-clause
                                               formula
                                               assignment))))))))


(defthm member-cdr
  (implies (member e (cdr x))
           (member e x)))

;****THIS GUY****
(defthm truep-evaluate-formula-and-RATp-implies-truep-evaluate-formula-modify-solution
  (implies (and (formulap formula)
                (clausep clause)
                ;(literalp literal)
                (assignmentp solution)
                (RATp formula clause literal)
                (truep (evaluate-formula formula solution))
                (member literal clause)
                (falsep (evaluate-clause clause solution)))
           (truep (evaluate-formula formula
                                    (modify-solution solution literal))))
  :otf-flg t
  :hints (("Goal"
           :in-theory (disable ATp resolution modify-solution clausep
                               RATp
                               ;formulap
                               member-equal
                               ;evaluate-formula
                               evaluate-clause
                               subsetp
                               ))
          ("Subgoal *1/12.2'"
           :in-theory (disable ATp resolution modify-solution clausep
                               RATp
                               ;formulap
                               member-equal
                               ;evaluate-formula
                               evaluate-clause
                               subsetp
                               TRUEP-EVALUATE-FORMULA-AND-MEMBER-IMPLIES-TRUEP-EVALUATE-CLAUSE
                               truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                               )
           :use ((:instance undefp-evaluate-formula-implies-exists-undef-clause
                            (formula (cdr formula))
                            (assignment (modify-solution solution literal)))
                 (:instance (:definition exists-undef-clause)
                            (formula (cdr formula))
                            (assignment (modify-solution solution literal)))
                 (:instance
                  TRUEP-EVALUATE-FORMULA-AND-MEMBER-IMPLIES-TRUEP-EVALUATE-CLAUSE
                  (clause (exists-undef-clause-witness (cdr formula)
                                                       (modify-solution
                                                        solution literal))))
                 (:instance
                  truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                  (rat-clause clause)
                  (clause (exists-undef-clause-witness (cdr formula)
                                                       (modify-solution
                                                        solution literal))))))

          ("Subgoal *1/10.2'"
           :in-theory (enable ATp resolution modify-solution clausep
                              RATp evaluate-clause subsetp)
           :use ((:instance
                  truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                  (rat-clause clause)
                  (clause (car formula)))))
          ("Subgoal *1/8.2'"
           :in-theory (enable ATp resolution modify-solution clausep
                              RATp evaluate-clause subsetp)
           :use ((:instance
                  truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                  (rat-clause clause)
                  (clause (car formula)))))
          ("Subgoal *1/4.1'"
           :in-theory (disable ATp resolution modify-solution clausep
                               RATp
                               ;formulap
                               member-equal
                               ;evaluate-formula
                               evaluate-clause
                               subsetp
                               TRUEP-EVALUATE-FORMULA-AND-MEMBER-IMPLIES-TRUEP-EVALUATE-CLAUSE
                               truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                               )
           :use ((:instance falsep-evaluate-formula-implies-exists-false-clause
                            (formula (cdr formula))
                            (assignment (modify-solution solution literal)))
                 (:instance (:definition exists-false-clause)
                            (formula (cdr formula))
                            (assignment (modify-solution solution literal)))
                 (:instance
                  TRUEP-EVALUATE-FORMULA-AND-MEMBER-IMPLIES-TRUEP-EVALUATE-CLAUSE
                  (clause (exists-false-clause-witness (cdr formula)
                                                       (modify-solution
                                                        solution literal))))
                 (:instance
                  truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                  (rat-clause clause)
                  (clause (exists-false-clause-witness (cdr formula)
                                                       (modify-solution
                                                        solution literal))))))
          ("Subgoal *1/2.1'"
           :in-theory (enable ATp resolution modify-solution clausep
                              RATp evaluate-clause subsetp)
           :use ((:instance
                  truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
                  (rat-clause clause)
                  (clause (car formula)))))))











           ;; :cases ((truep (evaluate-formula (cdr formula) (modify-solution
           ;;                                                 solution literal)))
           ;;         (undefp (evaluate-formula (cdr formula) (modify-solution
           ;;                                                 solution literal)))
           ;;         (falsep (evaluate-formula (cdr formula) (modify-solution
           ;;                                                 solution literal)))))))


           ;; :in-theory (disable ratp1 resolution modify-solution clausep evaluate-clause))))
           ;; :do-not-induct t
           ;; :in-theory (disable
           ;;             truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
           ;;             RATp1 RATp resolution modify-solution clausep
           ;;             evaluate-clause)
           ;; :use ((:instance
           ;;        truep-evaluate-clause-and-RATp-implies-truep-evaluate-clause-modify-solution
           ;;        (rat-clause clause)
           ;;        (clause (car formula)))))))
                  


;; ; to do
;; (defthm truep-evaluate-formula-and-RATp-implies-truep-evaluate-formula-cons-modify-solution
;;   (implies (and (formulap formula)
;;                 (clausep clause)
;;                 (literalp literal)
;;                 (assignmentp solution)
;;                 (RATp formula clause literal)
;;                 (truep (evaluate-formula formula solution))
;;                 (member literal clause)
;;                 (not (truep (evaluate-clause clause solution))))
;;            (truep (evaluate-formula (cons clause formula)
;;                                     (modify-solution solution literal))))
;;   :otf-flg t
;;   :hints (("Goal"
;;            :in-theory (disable ATp resolution modify-solution clausep
;;                                RATp
;;                                formulap
;;                                member-equal
;;                                ;evaluate-formula
;;                                ))))







;; (defthm solutionp-and-RATp-implies-solutionp-modify-solution
;;   (implies (and (formulap formula)
;;                 (clausep clause)
;;                 (literalp literal)
;;                 (solutionp solution formula)
;;                 (RATp formula clause literal)
;;                 (member literal clause)
;;                 (not (truep (evaluate-clause clause solution))))
;;            (solutionp (modify-solution solution literal)
;;                       (cons clause formula)))
;;   :hints (("Goal"
;;            :in-theory (disable ATp resolution modify-solution clausep
;;                                evaluate-formula))))

;; (skip-proofs
;;  (defthm solutionp-and-RATp-implies-solutionp-modify-solution
;;    (implies (and (formulap formula)
;;                  (clausep clause)
;;                  (literalp literal)
;;                  (solutionp solution formula)
;;                  (RATp formula clause literal)
;;                  (member literal clause)
;;                  ;(not (solutionp solution (cons clause formula))))
;;                  (not (truep (evaluate-clause clause solution))))
;;             (solutionp (modify-solution solution literal)
;;                        (cons clause formula)))))

;; (defthm not-solutionp-cons-and-solutionp-implies-not-truep-evaluate-clause
;;   (implies (and (not (solutionp solution (cons clause formula)))
;;                 (solutionp solution formula))
;;            (not (truep (evaluate-clause clause solution)))))



;; (defthm truep-EF-and-falsep-EF-cons-implies-falsep-EC
;;   (implies (and (truep (evaluate-formula formula solution))
;;                 (falsep (evaluate-formula (cons clause formula) solution)))
;;            (falsep (evaluate-clause clause solution))))


(defthm equal-falsep-evaluate-formula-cons
  (implies (truep (evaluate-formula formula assignment))
           (equal (falsep (evaluate-formula (cons clause formula) assignment))
                  (falsep (evaluate-clause clause assignment)))))


(defthm exists-solution-and-RATp-and-truep-implies-exists-solution
  (implies (and (formulap formula)
                (clausep clause)
                ;(literalp literal)
                (assignmentp solution)
                (truep (evaluate-formula formula solution))
                (RATp formula clause literal)
                (member literal clause)
                (falsep (evaluate-clause clause solution)))
           (exists-solution (cons clause formula)))
  :otf-flg t
  :hints (("Goal"
           :in-theory (disable clausep RATp evaluate-formula evaluate-clause modify-solution)
           :use ((:instance exists-solution-suff
                            (assignment (modify-solution solution literal))
                            (formula (cons clause formula)))))))



;; ===================================================================
;; ============================ UNDEF-EC =============================

;; ====================== EXISTS-UNDEF-LITERAL =======================

(defun find-undef-literal (clause assignment)
  (if (atom clause)
      nil
    (if (and (not (member (car clause) assignment))
             (not (member (negate (car clause)) assignment)))
        (car clause)
      (find-undef-literal (cdr clause) assignment))))


(defthm undefp-EC-implies-find-undef-literal
  (implies (and (clausep clause)
                (undefp (evaluate-clause clause assignment)))
           (find-undef-literal clause assignment)))

(defthm find-undef-literal-implies-member-clause
  (implies (find-undef-literal clause assignment)
           (member (find-undef-literal clause assignment)
                   clause)))

(defthm find-undef-literal-implies-not-member-assignment
  (implies (find-undef-literal clause assignment)
           (not (member (find-undef-literal clause assignment)
                        assignment))))

(defthm find-undef-literal-implies-not-member-negate-assignment
  (implies (find-undef-literal clause assignment)
           (not (member (negate (find-undef-literal clause assignment))
                        assignment))))

;; (defthm find-undef-literal-implies-literalp
;;   (implies (and ;(clausep clause)
;;                 (find-undef-literal clause assignment))
;;            (literalp (find-undef-literal clause assignment))))

(defun-sk exists-undef-literal (clause assignment)
  (exists literal (and ;(literalp literal)
                       (member literal clause)
                       (not (member literal assignment))
                       (not (member (negate literal) assignment)))))
(in-theory (disable exists-undef-literal
                    exists-undef-literal-suff))


(defthm undefp-EC-implies-exists-undef-literal
  (implies (and (clausep clause)
                (undefp (evaluate-clause clause assignment)))
           (exists-undef-literal clause assignment))
  :hints (("Goal"
           :use ((:instance exists-undef-literal-suff
                            (literal (find-undef-literal
                                      clause
                                      assignment)))))))

;; (defthm member-clause-and-not-member-not-member-negate-implies-undefp-EC
;;   (implies (and (not (truep (evaluate-clause clause assignment)))
;;                 (member literal clause)
;;                 (not (member literal assignment))
;;                 (not (member (negate literal) assignment)))
;;            (undefp (evaluate-clause clause assignment))))

;; (defthm exists-undef-literal-implies-undefp-EC
;;   (implies (and (not (truep (evaluate-clause clause assignment)))
;;                 (exists-undef-literal clause assignment))
;;            (undefp (evaluate-clause clause assignment)))
;;   :hints (("Goal"
;;            :use ((:instance (:definition exists-undef-literal))))))


;; ===================================================================

; redundant
;; (defthm truep-EF-and-undefp-EF-cons-implies-undefp-EC
;;   (implies (and (truep (evaluate-formula formula solution))
;;                 (undefp (evaluate-formula (cons clause formula) solution)))
;;            (undefp (evaluate-clause clause solution))))

;; (defthm truep-EC-and-not-member-not-member-negate-implies-truep-EC-cons
;;   (implies (and ;(clausep clause)
;;                 ;(assignmentp solution)
;;                 ;(literalp literal)
;;                 (not (member literal solution))
;;                 (not (member (negate literal) solution))
;;                 (truep (evaluate-clause clause solution)))
;;            (truep (evaluate-clause clause (cons literal solution)))))

(defthm truep-EF-implies-truep-EF-cons
  (implies (truep (evaluate-formula formula solution))
           (truep (evaluate-formula formula (cons literal solution))))
  :hints (("Goal"
           :in-theory (disable evaluate-clause))))


;; (defthm undefp-EC-and-not-member-not-member-negate-implies-truep-EC-cons
;;   (implies (and (clausep clause)
;;                 (assignmentp solution)
;;                 (literalp literal)
;;                 (member literal clause)
;;                 (not (member literal solution))
;;                 (not (member (negate literal) solution))
;;                 (undefp (evaluate-clause clause solution)))
;;            (truep (evaluate-clause clause (cons literal solution))))
;;   :hints (("Goal"
;;            :in-theory (disable evaluate-clause))))

;; (defthm truep-EF-and-undefp-EC-and-not-member-not-member-negate-implies-truep-EF-cons-cons
;;   (implies (and ;(formulap formula)
;;                 ;(clausep clause)
;;                 ;(assignmentp solution)
;;                 ;(literalp literal)
;;                 (member literal clause)
;;                 ;(not (member literal solution))
;;                 ;(not (member (negate literal) solution))
;;                 (truep (evaluate-formula formula solution))
;;                 (undefp (evaluate-clause clause solution)))
;;            (truep (evaluate-formula (cons clause formula)
;;                                     (cons literal solution))))
;;   :hints (("Goal"
;;            :in-theory (disable evaluate-clause))))


(defthm equal-undefp-evaluate-formula-cons
  (implies (truep (evaluate-formula formula assignment))
           (equal (undefp (evaluate-formula (cons clause formula) assignment))
                  (undefp (evaluate-clause clause assignment)))))

(defthm truep-EF-and-undefp-EF-cons-implies-exists-solution
  (implies (and (formulap formula)
                (clausep clause)
                (assignmentp solution)
                (truep (evaluate-formula formula solution))
                (undefp (evaluate-formula (cons clause formula) solution)))
           (exists-solution (cons clause formula)))
  :hints (("Goal"
           :in-theory (disable evaluate-clause evaluate-formula)
           :use ((:instance undefp-EC-implies-exists-undef-literal
                            (assignment solution))
                 (:instance (:definition exists-undef-literal)
                            (assignment solution))
                 (:instance exists-solution-suff
                            (assignment (cons (exists-undef-literal-witness
                                               clause
                                               solution)
                                              solution))
                            (formula (cons clause formula)))))))

;; (defthm solutionp-and-undefp-EF-cons-implies-exists-solution
;;   (implies (and (formulap formula)
;;                 (clausep clause)
;;                 (solutionp solution formula)
;;                 (undefp (evaluate-formula (cons clause formula) solution)))
;;            (exists-solution (cons clause formula))))




;; ===================================================================
;; ============================ TRUEP-EC =============================

;; (defthm solutionp-and-truep-EF-cons-implies-exists-solution
;;   (implies (and (formulap formula)
;;                 (clausep clause)
;;                 (solutionp solution formula)
;;                 (truep (evaluate-formula (cons clause formula) solution)))
;;            (exists-solution (cons clause formula)))
;;   :hints (("Goal"
;;            :use ((:instance exists-solution-suff
;;                             (assignment solution)
;;                             (formula (cons clause formula)))))))


(defthm solutionp-and-truep-EF-cons-implies-exists-solution
  (implies (and (formulap formula)
                (clausep clause)
                (assignmentp solution)
                (truep (evaluate-formula formula solution))
                (truep (evaluate-clause clause solution)))
           (exists-solution (cons clause formula)))
  :hints (("Goal"
           :use ((:instance exists-solution-suff
                            (assignment solution)
                            (formula (cons clause formula)))))))



;; =========================== CASE-SPLIT ============================               

; case split on original solution
(defthm solutionp-and-RATp-implies-exists-solution-cons
  (implies (and (formulap formula)
                (clausep clause)
                ;(literalp literal)
                (solutionp solution formula)
                (RATp formula clause literal)
                (member literal clause))
           (exists-solution (cons clause formula)))
  :otf-flg t
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable RATp evaluate-formula clausep)
           :cases ((truep (evaluate-formula (cons clause formula) solution))
                   (undefp (evaluate-formula (cons clause formula) solution))
                   (falsep (evaluate-formula (cons clause formula) solution)))) ))
          

(defthm exists-solution-and-RATp-implies-exists-solution-cons
  (implies (and (formulap formula)
                (clausep clause)
                ;(literalp literal)
                (exists-solution formula)
                (RATp formula clause literal)
                (member literal clause))
           (exists-solution (cons clause formula)))
  :otf-flg t
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable RATp solutionp clausep)
           :use ((:instance (:definition exists-solution))))))


; verify-proof induction
(defthm solution-and-member-nil-implies-not-verify-proof
  (implies (and (clause-listp clause-list)
                (formulap formula)
                (exists-solution formula)
                (member *empty-clause* clause-list))
           (not (verify-proof clause-list formula)))
  :hints (("Goal"
           :in-theory (disable RATp ATp clausep))))




;; =========================== MAIN PROOF ============================

;; (defun-sk exists-solution (formula)
;;   (exists assignment (solutionp assignment formula)))
;; (in-theory (disable exists-solution
;;                     exists-solution-suff))


(defthm exists-solution-implies-not-refutationp
  (implies (and (clause-listp clause-list)
                (formulap formula)
                (exists-solution formula))
           (not (refutationp clause-list formula)))
  :hints (("Goal"
           :in-theory (disable solutionp)
           :use ((:instance exists-solution)))))

(defthm refutationp-implies-not-exists-solution
  (implies (and (clause-listp clause-list)
                (formulap formula)
                (refutationp clause-list formula))
           (not (exists-solution formula)))
  :hints (("Goal"
           :use ((:instance exists-solution-implies-not-refutationp)))))


;; ===================================================================
;; ===================================================================
;; ===================================================================

;; (i-am-here)

;; (ld "rat-parser.lisp")

;; (in-package "SAT")


;; (defun get-formula (clause-list num-clauses)
;;   (cond
;;    ((atom clause-list) (mv nil nil))
;;    ((<= num-clauses 0) (mv nil clause-list))
;;    (t
;;     (mv-let (formula learned)
;;             (get-formula (cdr clause-list) (1- num-clauses))
;;             (mv (cons (car clause-list) formula) learned)))))


;; (defun verify-file (filename)
;;   (declare (xargs :mode :program))
;;   (b* (((mv fail ?num-vars num-clauses clause-list)
;;         (acl2::parse-unsat-proof filename))
;;        ((when fail) 'parse-fail)
;;        ((mv formula proof) (get-formula clause-list num-clauses)))
;;       (verify-proof proof formula)))

;; (i-am-here)

;; (defconst *input* (mv-let (fail num-vars num-clauses clause-list)
;;                           (acl2::parse-unsat-proof "rat-1")
;;                           (list fail num-vars num-clauses clause-list)))



;; (defconst *separated-input* (mv-let (formula proof)
;;                                     (get-formula (nth 3 *input*) (nth 2 *input*))
;;                                     (list formula proof)))

;; (defconst *formula* (nth 0 *separated-input*))
;; (defconst *proof* (nth 1 *separated-input*))



;; (defthm unsat-proof
;;   (not (exists-solution *formula*))
;;   :hints (("Goal"
;;            :use ((:instance refutationp-implies-not-exists-solution
;;                             (clause-list *proof*)
;;                             (formula *formula*))))))
