(in-package "SAT")

(include-book "ternary")
(include-book "literal")
(include-book "unique")
(include-book "sets")
(include-book "assignment")
(include-book "clause")
(include-book "all-literals")


;; ===================================================================
;; ======================= LITERAL EVALUATION ========================

(defun evaluate-literal (literal assignment)
  (declare (xargs :guard (and (literalp literal)
                              (assignmentp assignment))))
  (cond
   ((member literal assignment) (true))
   ((member (negate literal) assignment) (false))
   (t (undef))))


;; ===================================================================
;; ========================= EVALUATE-CLAUSE =========================

(defun evaluate-clause (clause assignment)
  (declare (xargs :guard (and (clausep clause)
                              (assignmentp assignment))))
  (if (atom clause)
      (false)
    (let* ((literal (car clause))
           (literal-value (evaluate-literal literal assignment)))
      (if (truep literal-value)
          (true)
        (let* ((remaining-clause (cdr clause))
               (remaining-clause-value (evaluate-clause remaining-clause
                                                        assignment)))
          (cond
           ((truep remaining-clause-value) (true))
           ((undefp literal-value) (undef))
           (t remaining-clause-value)))))))
       
(defthm ternaryp-evaluate-clause
  (ternaryp (evaluate-clause clause assignment)))


;; ===================================================================
;; ======================== EVALUATE-FORMULA =========================

(defun evaluate-formula (formula assignment)
  (declare (xargs :guard (and (formulap formula)
                              (assignmentp assignment))))
  (if (atom formula)
      (true)
    (let* ((clause (car formula))
           (clause-value (evaluate-clause clause assignment)))
      (if (falsep clause-value)
          (false)
        (let* ((remaining-formula (cdr formula))
               (remaining-formula-value (evaluate-formula remaining-formula
                                                          assignment)))
          (cond
           ((falsep remaining-formula-value) (false))
           ((undefp clause-value) (undef))
           (t remaining-formula-value)))))))

(defthm ternaryp-evaluate-formula
  (ternaryp (evaluate-formula formula assignment)))


;; ===================================================================
;; =================== SET EQUALITY AND EVALUATION ===================

; We can reduce any of these to truep implies truep.  They might be stronger
; than needed.

(defthm set-equalp-implies-equal-evaluate-literal
  (implies (and (set-equalp x y)
                (literalp literal)
                (assignmentp x)
                (assignmentp y))
           (equal (evaluate-literal literal y)
                  (evaluate-literal literal x))))


(defthm set-equalp-implies-equal-evaluate-clause
  (implies (and (set-equalp x y)
                (clausep clause)
                (assignmentp x)
                (assignmentp y))
           (equal (evaluate-clause clause y)
                  (evaluate-clause clause x))))
  ;; :hints (("Goal" :in-theory (disable
  ;; NOT-SET-DIFFERENCE-VARIABLES-IMPLIES-SUBSET-VARIABLESP
  ;; SUBSETP-AND-SET-EQUAL-VARIABLESP-IMPLIES-SUBSETP
  ;; SUBSET-VARIABLESP-AND-SUBSETP-IMPLIES-SUBSETP))))


(defthm set-equalp-implies-equal-evaluate-formula
  (implies (and (set-equalp x y)
                (formulap formula)
                (assignmentp x)
                (assignmentp y))
           (equal (evaluate-formula formula y)
                  (evaluate-formula formula x)))
  :hints (("Goal" :in-theory (disable set-equalp)))) ; want to remove this



;; ===================================================================
;; ===================== SHORTEN-LONG-ASSIGNMENT =====================

(defun shorten-long-assignment (assignment all-literals)
  (declare (xargs :guard (and (literal-listp assignment)
                              (literal-listp all-literals))))
  (union-variables assignment all-literals))


;; =============== ASSIGNMENTP-SHORTEN-LONG-ASSIGNMENT ===============

(encapsulate
 ()

 (local
  (defthm literal-listp-shorten-long-assignment
    (implies (assignmentp assignment)
             (literal-listp (shorten-long-assignment assignment all-literals)))))
 
 (local
  (defthm unique-literalsp-shorten-long-assignment
    (implies (assignmentp x)
             (unique-literalsp (shorten-long-assignment x y)))))
 
 (local
  (defthm no-conflicting-literalsp-shorten-long-assignment
    (implies (assignmentp x)
             (no-conflicting-literalsp (shorten-long-assignment x y)))))

 (defthm assignmentp-shorten-long-assignment
   (implies (assignmentp x)
            (assignmentp (shorten-long-assignment x y)))
   :hints (("Goal"
            :in-theory (enable assignmentp))))
 )


;; ============ SUBSET-VARIABLESP-SHORTEN-LONG-ASSIGNMENT ============

(defthm subsetp-shorten-long-assignment
  (subsetp (shorten-long-assignment x y) x))

(defthm subset-variablesp-shorten-long-assignment
  (subset-variablesp (shorten-long-assignment x y) y))


;; ========= TRUEP-EVALUATE-FORMULA-SHORTEN-LONG-ASSIGNMENT ==========

(defthm truep-evaluate-literal-shorten-long-assignment
  (implies (and (truep (evaluate-literal literal assignment))
                (member literal all-literals))
           (truep (evaluate-literal literal (shorten-long-assignment
                                             assignment
                                             all-literals)))))

(defthm truep-evaluate-clause-shorten-long-assignment
  (implies (and (truep (evaluate-clause clause assignment))
                (subsetp clause all-literals))
           (truep (evaluate-clause clause (shorten-long-assignment
                                           assignment
                                           all-literals)))))


(defthm subsetp-list-truep-evaluate-formula-shorten-long-assignment
  (implies (and (truep (evaluate-formula formula assignment))
                (subsetp-list formula all-literals))
           (truep (evaluate-formula formula
                                    (shorten-long-assignment
                                     assignment
                                     all-literals))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment))))

(defthm truep-evaluate-formula-shorten-long-assignment
  (implies (truep (evaluate-formula formula assignment))
           (truep (evaluate-formula formula
                                    (shorten-long-assignment
                                     assignment
                                     (all-literals formula)))))
  :hints (("Goal" 
           :in-theory (disable shorten-long-assignment))))


;; ============ STRONG-ASSIGNMENT-SHORTEN-LONG-ASSIGNMENT ============

(defthm strong-assignment-shorten-long-assignment
  (implies (and (assignmentp assignment)
                (truep (evaluate-formula formula assignment)))
           (and (assignmentp (shorten-long-assignment
                              assignment
                              (all-literals formula)))
                (truep (evaluate-formula formula 
                                         (shorten-long-assignment
                                          assignment
                                          (all-literals formula))))
                (subset-variablesp (shorten-long-assignment
                                    assignment
                                    (all-literals formula))
                                   (all-literals formula))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment))))
                 

;; ===================================================================
;; ===================== EXTEND-SHORT-ASSIGNMENT =====================

; consider switching append order?
(defun extend-short-assignment (assignment all-literals)
  (declare (xargs :guard (and (literal-listp assignment)
                              (literal-listp all-literals))))
  (append (set-difference-variables (remove-conflicting-literals all-literals)
                                    assignment)
          assignment))


;; =============== ASSIGNMENTP-EXTEND-SHORT-ASSIGNMENT ===============

(encapsulate
 ()
 
 (local
  (defthm literal-listp-extend-short-assignment
    (implies (and (assignmentp assignment)
                  (literal-listp all-literals))
             (literal-listp (extend-short-assignment assignment
                                                     all-literals)))))

 (local
  (defthm unique-literalsp-extend-short-assignment
    (implies (and (assignmentp assignment)
                  (unique-literalsp all-literals))
             (unique-literalsp (extend-short-assignment assignment
                                                        all-literals)))))

 (local
  (defthm no-conflicting-literalsp-extend-short-assignment
    (implies (assignmentp assignment)
             (no-conflicting-literalsp (extend-short-assignment
                                        assignment
                                        all-literals)))))

 (defthm assignmentp-extend-short-assignment
   (implies (and (assignmentp assignment)
                 (literal-listp all-literals)
                 (unique-literalsp all-literals))
            (assignmentp (extend-short-assignment assignment all-literals))))
 )


;; ============ SUBSET-VARIABLESP-EXTEND-SHORT-ASSIGNMENT ============

(defthm subset-variablesp-extend-short-assignment
  (implies (literal-listp all-literals)
           (subset-variablesp all-literals
                              (extend-short-assignment assignment
                                                       all-literals))))       
       
(defthm subset-variablesp-extend-short-assignment2
  (implies (subset-variablesp assignment all-literals)
           (subset-variablesp (extend-short-assignment assignment
                                                       all-literals)
                              all-literals)))


;; ========= TRUEP-EVALUATE-FORMULA-EXTEND-SHORT-ASSIGNMENT ==========

(defthm truep-evaluate-literal-extend-long-assignment
  (implies (and (truep (evaluate-literal literal assignment))
                (member literal all-literals))
           (truep (evaluate-literal literal (extend-short-assignment
                                             assignment
                                             all-literals)))))

(defthm truep-evaluate-clause-extend-short-assignment
  (implies (and (truep (evaluate-clause clause assignment))
                (subsetp clause all-literals))
           (truep (evaluate-clause clause (extend-short-assignment
                                           assignment
                                           all-literals)))))

(defthm subsetp-list-truep-evaluate-formula-extend-short-assignment
  (implies (and (truep (evaluate-formula formula assignment))
                (subsetp-list formula all-literals))
           (truep (evaluate-formula formula (extend-short-assignment
                                             assignment
                                             all-literals))))
  :hints (("Goal"
           :in-theory (disable extend-short-assignment))))

(defthm truep-evaluate-formula-extend-short-assignment
  (implies (and (assignmentp assignment)
                (truep (evaluate-formula formula assignment)))
           (truep (evaluate-formula formula (extend-short-assignment
                                             assignment
                                             (all-literals formula)))))
  :hints (("Goal"
           :in-theory (disable extend-short-assignment))))


;; ============ STRONG-ASSIGNMENT-EXTEND-SHORT-ASSIGNMENT ============

(defthm strong-assignment-extend-short-assignment
  (implies (and (formulap formula)
                (assignmentp assignment)
                (truep (evaluate-formula formula assignment)))
           (and (assignmentp (extend-short-assignment
                              assignment
                              (all-literals formula)))
                (truep (evaluate-formula formula (extend-short-assignment
                                                  assignment
                                                  (all-literals formula))))
                (subset-variablesp (all-literals formula)
                                   (extend-short-assignment
                                    assignment
                                    (all-literals formula)))))
  :hints (("Goal"
           :in-theory (disable extend-short-assignment))))


;; ===================================================================
;; ============== SATISFYING-IMPLIES-STRONG-SATISFYING ===============

(defun-sk exists-strong-satisfying-assignment (formula)
  (exists assignment (and (assignmentp assignment)
                          (truep (evaluate-formula formula assignment))
                          (set-equal-variablesp (all-literals formula)
                                                assignment))))
(in-theory (disable exists-strong-satisfying-assignment
                    exists-strong-satisfying-assignment-suff))


(defthm satisfying-assignment-implies-exists-strong-satisfying-assignment
  (implies (and (formulap formula)
                (assignmentp assignment)
                (truep (evaluate-formula formula assignment)))
           (exists-strong-satisfying-assignment formula))
  :hints (("Goal"
           :in-theory (disable extend-short-assignment
                               shorten-long-assignment
                               )
           :use ((:instance exists-strong-satisfying-assignment-suff
                            (assignment (extend-short-assignment
                                         (shorten-long-assignment
                                          assignment
                                          (all-literals formula))
                                         (all-literals formula))))))))













































;; ===================================================================
;; ===================================================================

#||

(defun find-true-literal-in-clause (clause assignment)
  (declare (xargs :guard (and (clausep clause)
                              (assignmentp assignment))))
  (if (atom clause)
      nil
    (if (truep (evaluate-literal (car clause) assignment))
        (car clause)
      (find-true-literal-in-clause (cdr clause) assignment))))


(defun-sk exists-true-literal-in-clause (clause assignment)
  (exists literal (and (member literal clause)
                       (truep (evaluate-literal literal assignment)))))
(in-theory (disable exists-true-literal-in-clause
                    exists-true-literal-in-clause-suff))

(defthm member-find-true-literal-in-clause-assignment
  (implies (truep (evaluate-clause clause assignment))
           (member (find-true-literal-in-clause clause assignment)
                   assignment)))

(defthm truep-evaluate-literal-find-true-literal-in-clause
  (implies (truep (evaluate-clause clause assignment))
           (truep (evaluate-literal (find-true-literal-in-clause clause
                                                                 assignment)
                                    assignment))))

(defthm member-find-true-literal-in-clause-clause
  (implies (truep (evaluate-clause clause assignment))
           (member (find-true-literal-in-clause clause assignment)
                   clause)))

(defthm truep-evaluate-literal-and-member-implies-truep-evaluate-clause
  (implies (and (truep (evaluate-literal literal assignment))
                (member literal clause))
           (truep (evaluate-clause clause assignment))))



(defthm truep-evaluate-clause-implies-exists-true-literal-in-clause
  (implies (truep (evaluate-clause clause assignment))
           (exists-true-literal-in-clause clause assignment))
  :hints (("Goal"
           :use ((:instance exists-true-literal-in-clause-suff
                            (literal (find-true-literal-in-clause
                                      clause
                                      assignment)))))))


(defthm exists-true-literal-in-clause-implies-truep-evaluate-clause
  (implies (exists-true-literal-in-clause clause assignment)
           (truep (evaluate-clause clause assignment)))
  :hints (("Goal"
           :in-theory (disable evaluate-literal)
           :use ((:instance (:definition exists-true-literal-in-clause))))))

||#



;; ===================================================================
;; ===================================================================


#||

(defthm member-shorten-long-assignment
  (implies (and (member literal assignment)
                (member literal all-literals))
           (member literal
                   (shorten-long-assignment
                    assignment
                    all-literals))))

(defthm truep-evaluate-literal-shorten-long-assignment
  (implies (and (truep (evaluate-literal literal assignment))
                (member literal all-literals))
           (truep (evaluate-literal literal
                                    (shorten-long-assignment
                                     assignment
                                     all-literals)))))
;; (defthm bleep-bloop
;;   (implies (member e (remove-duplicate-literals x))
;;            (member e x)))

;; (defthm bleep-bloop2
;;   (implies (member e x)
;;            (member e (remove-duplicate-literals x))))

;; (defthm bleep-bloop3
;;   (iff (member e (remove-duplicate-literals x))
;;        (member e x)))

;; (defthm bleep-bloop4
;;   (iff (member e (append x y))
;;        (or (member e x)
;;            (member e y))))

;; (defthm bleep-bloop5
;;   (implies (and (member literal clause)
;;                 (member clause formula)
;;                 (clausep clause)
;;                 (formulap formula))
;;            (member literal (all-literals formula))))

;; (defthm bb
;;   (implies (and (member literal clause)
;;                 (member clause formula)
;;                 (literalp literal)
;;                 (clausep clause)
;;                 (formulap formula))
;;            (member literal (all-literals formula))))

(skip-proofs
(defthm bleep-bloop6
  (implies (and (clausep clause)
                (formulap formula)
                (member clause formula)
                )
           (subsetp clause (all-literals formula))) ))

(defthm exists-true-literal-implies-exists-true-literal-fill-assignment
  (implies (and (clausep clause)
                (subsetp clause all-literals)
                (exists-true-literal-in-clause clause assignment))
           (exists-true-literal-in-clause clause (shorten-long-assignment
                                                  assignment
                                                  all-literals)))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment)
           :use ((:instance (:definition exists-true-literal-in-clause))
                 (:instance exists-true-literal-in-clause-suff
                            (literal (exists-true-literal-in-clause-witness
                                      clause
                                      assignment))
                            (assignment (shorten-long-assignment assignment
                                                                 all-literals)))))))

(defthm truep-evaluate-clause-shorten-long-assignment
  (implies (and (clausep clause)
                (subsetp clause all-literals)
                (truep (evaluate-clause clause assignment)))
           (truep (evaluate-clause clause (shorten-long-assignment assignment
                                                                   all-literals))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment))))

(skip-proofs
(defthm boo
  (implies (and (consp x)
                )
           (subsetp (all-literals (cdr x)) (all-literals x)))))

(skip-proofs
(defthm boo3
  (implies (and (consp x)
                (subsetp formula formulaplus)
                )
           (subsetp (shorten-long-assignment assignment (all-literals formula))
                    (shorten-long-assignment assignment (all-literals formulaplus))))))

(defthm boo2
  (implies (and (subsetp shorter-assignment short-assignment)
                (truep (evaluate-clause clause shorter-assignment)))
           (truep (evaluate-clause clause short-assignment))))

(defthm boo22
  (implies (and ;(subsetp sub-all-lits super-all-lits)
            (subsetp shorter-assignment short-assignment)
                (truep (evaluate-formula formula shorter-assignment)))
                                         ;(shorten-long-assignment assignment
                                         ;                            sub-all-lits))))
           (truep (evaluate-formula formula short-assignment))))
                                    ;(shorten-long-assignment assignment
                                    ;                                 super-all-lits)))))

(defthm truep-evaluate-formula-shorten-long-assignment
  (implies (and (formulap formula)
                (truep (evaluate-formula formula assignment)))
           (truep (evaluate-formula formula (shorten-long-assignment assignment
                                                                   (all-literals
                                                                    formula)))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment all-literals))))

||#

;; ===================================================================
;; ===================================================================
;; ===================================================================
;; ===================================================================


#||
(defthm member-shorten-long-assignment
  (implies (and (member literal assignment)
                (member literal (all-literals formula)))
           (member literal
                   (shorten-long-assignment
                    assignment
                    (all-literals formula)))))

(defthm truep-evaluate-literal-shorten-long-assignment
  (implies (and (truep (evaluate-literal literal assignment))
                (member literal (all-literals formula)))
           (truep (evaluate-literal literal
                                    (shorten-long-assignment
                                     assignment
                                     (all-literals formula))))))
(defthm bleep-bloop
  (implies (member e (remove-duplicate-literals x))
           (member e x)))

(defthm bleep-bloop2
  (implies (member e x)
           (member e (remove-duplicate-literals x))))

(defthm bleep-bloop3
  (iff (member e (remove-duplicate-literals x))
       (member e x)))

(defthm bleep-bloop4
  (iff (member e (append x y))
       (or (member e x)
           (member e y))))

(defthm bleep-bloop5
  (implies (and (member literal clause)
                (member clause formula)
                (clausep clause)
                (formulap formula))
           (member literal (all-literals formula))))

(defthm bb
  (implies (and (member literal clause)
                (member clause formula)
                (literalp literal)
                (clausep clause)
                (formulap formula))
           (member literal (all-literals formula))))

(defthm exists-true-literal-implies-exists-true-literal-fill-assignment
  (implies (and (clausep clause)
                (formulap formula)
                (member clause formula)
                (exists-true-literal-in-clause clause assignment))
           (exists-true-literal-in-clause clause (shorten-long-assignment
                                                  assignment
                                                  (all-literals formula))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment all-literals)
           :use ((:instance (:definition exists-true-literal-in-clause))
                 (:instance exists-true-literal-in-clause-suff
                            (literal (exists-true-literal-in-clause-witness
                                      clause
                                      assignment))
                            (assignment (shorten-long-assignment assignment
  (all-literals formula))))))))

(defthm truep-evaluate-clause-shorten-long-assignment
  (implies (and (clausep clause)
                (formulap formula)
                (member clause formula)
                (truep (evaluate-clause clause assignment)))
           (truep (evaluate-clause clause (shorten-long-assignment assignment
                                                                   (all-literals
  formula)))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment all-literals))))

(defthm truep-evaluate-formula-shorten-long-assignment
  (implies (and (formulap formula)
                (truep (evaluate-formula formula assignment)))
           (truep (evaluate-formula formula (shorten-long-assignment assignment
                                                                   (all-literals
                                                                    formula)))))
  :hints (("Goal"
           :in-theory (disable shorten-long-assignment)))) all-literals))))



||#

;; ===================================================================
;; ===================================================================

#||

(defthm truep-evaluate-literal-shorten-long-assignment
  (implies (and (literalp literal)
                (member literal (all-literals formula))
                (assignmentp assignment)
                (formulap formula)
                (truep (evaluate-literal literal assignment)))
           (truep (evaluate-literal literal
                                    (shorten-long-assignment
                                     assignment
                                     (all-literals formula))))) )
  ;; :hints (("Goal"
  ;;          :in-theory (e/d (evaluate-literal) ())) ))


; stronger than needed
(defthm equal-evaluate-literal-shorten-long-assignment
  (implies (and (literalp literal)
                (member literal (all-literals formula))
                (assignmentp assignment)
                (formulap formula))
           (equal (evaluate-literal literal
                                    (shorten-long-assignment
                                     assignment
                                     (all-literals formula)))
                  (evaluate-literal literal assignment))) )
  ;; :hints (("Goal"
  ;;          :in-theory (enable evaluate-literal))))

;(i-am-here)

(defthm truep-evaluate-clause-shorten-long-assignment
  (implies (and (clausep clause)
                (subsetp clause (all-literals formula))
                (assignmentp assignment)
                (formulap formula)
                (truep (evaluate-clause clause assignment)))
           (truep (evaluate-clause clause
                                    (shorten-long-assignment
                                     assignment
                                     (all-literals formula)))))
  :hints (("Goal" 
           :in-theory (e/d ()
                           (all-literals
                            shorten-long-assignment
                            evaluate-literal)) )))

(defthm bleep-bloop
  (implies (member e (remove-duplicate-literals x))
           (member e x)))

(defthm bleep-bloop2
  (implies (member e x)
           (member e (remove-duplicate-literals x))))

(defthm bleep-bloop3
  (iff (member e (remove-duplicate-literals x))
       (member e x)))

(defthm bleep-bloop4
  (iff (member e (append x y))
       (or (member e x)
           (member e y))))

(defthm bleep-bloop5
  (implies (and (member literal clause)
                (member clause formula)
                (clausep clause)
                (formulap formula))
           (member literal (all-literals formula))))


(defthm bleep-bloop6
  (implies (and (clausep clause)
                (formulap formula)
                (member clause formula)
                )
           (subsetp clause (all-literals formula))) )
  ;; :hints (("Goal"
  ;;          :in-theory (disable all-literals clausep formulap-member subsetp-cdr)
  ;;          :induct (subsetp clause (all-literals formula)))
  ;;         ("Subgoal *1/3''"
  ;;          :use ((:instance bleep-bloop5
  ;;                           (literal (car clause)))))))

;; (thm
;;  (implies (and (formulap f)
;;                (assignmentp a)
;;                (literalp x)
;;           (truep (evaluate-formula f
;;                                    (shorten-long-assignment a f))))
;;           (truep (evaluate-formula f
;;                                    (shorten-long-assignment a (cons x f)))))
;;           :hints (("Goal"
;;                    :in-theory (e/d () (all-literals
;;                                        shorten-long-assignment
;;                                        evaluate-clause)) )))
;; (thm
;;  (implies (and (formulap f)
;;                (assignmentp a)
;;                (clausep x)
;;           (truep (evaluate-formula f
;;                                    (shorten-long-assignment a f))))
;;           (truep (evaluate-formula f
;;                                    (shorten-long-assignment a (append x f)))))
;;           :hints (("Goal"
;;                    :in-theory (e/d () (all-literals
;;                                        shorten-long-assignment
;;                                        evaluate-clause)) )))

(defthm evaluate-clause-super
 (implies (and (subsetp sub super)
               (clausep clause)
               (assignmentp sub)
               (assignmentp super)
               (truep (evaluate-clause clause sub)))
          (truep (evaluate-clause clause super))))

(defthm evaluate-formula-super
 (implies (and (subsetp sub super)
               (formulap formula)
               (assignmentp sub)
               (assignmentp super)
               (truep (evaluate-formula formula sub)))
          (truep (evaluate-formula formula super))) )
  ;; :hints (("Goal" :in-theory (disable clausep))

  ;;         ("Subgoal *1/8.2"
  ;;          :in-theory (e/d (ternaryp)
  ;;                          (ternaryp-evaluate-clause
  ;;                           evaluate-clause-super))
  ;;          :use ((:instance evaluate-clause-super
  ;;                           (clause (car formula)))
  ;;                (:instance ternaryp-evaluate-clause
  ;;                           (clause (car formula))
  ;;                           (assignment sub))))))
  ;;          :in-theory (e/d (ternaryp truep falsep undefp) (ternaryp-evaluate-clause))
  ;;          :use ((:instance ternaryp-evaluate-clause
  ;;                           (clause (car formula))
  ;;                           (assignment sub))
  ;;                (:instance ternaryp-evaluate-clause
  ;;                           (clause (car formula))
  ;;                           (assignment super))))))

(defthm truep-evaluate-formula-shorten-long-assignment
  (implies (and (formulap formula)
                (assignmentp assignment)
                (truep (evaluate-formula formula assignment)))
           (truep (evaluate-formula formula
                                    (shorten-long-assignment
                                     assignment
                                     (all-literals formula))))) )
  :hints (("Goal"
           :in-theory (e/d () (all-literals 
                               shorten-long-assignment
                               evaluate-clause)) )))
           :induct (formulap formula))))


;; ===================================================================

(defun extend-short-assignment (assignment all-literals)
  (declare (xargs :guard (and (literal-listp assignment)
                              (literal-listp all-literals))))
  (append (set-difference-literals all-literals assignment)
          assignment))

(defthm subset-literals-
  (implies (and (literal-listp x)
                )
           (subset-literals x (append (set-difference-literals x y) y))))
; extends short assignments
; x be all-lits, y be assignment

;; (defthm 
;;   (implies (and (assignmentp assignment)
;;                 (truep (evaluate-formula formula assignment))


;; (defthm satisfying-assignment-implies-exists-strong-satisfying-assignment
;;   (implies (and (assignmentp assignment)
;;                 (truep (evaluate-formula formula assignment)))
;;            (exists-strong-satisfying-assignment formula))
;;   :otf-flg t
;;   :hints (("Goal"
;;            :do-not-induct t
;;            :use ((:instance exists-strong-satisfying-assignment-suff
;;                             (assignment ...))))))

||#

































;; ===================================================================
;; ======================= EXISTS TRUE LITERAL =======================

#||

(defun find-true-literal-in-clause (clause assignment)
  (declare (xargs :guard (and (clausep clause)
                              (assignmentp assignment))))
  (if (atom clause)
      nil
    (if (truep (evaluate-literal (car clause) assignment))
        (car clause)
      (find-true-literal-in-clause (cdr clause) assignment))))


(defun-sk exists-true-literal-in-clause (clause assignment)
  (exists literal (and (member literal clause)
                       (truep (evaluate-literal literal assignment)))))
(in-theory (disable exists-true-literal-in-clause
                    exists-true-literal-in-clause-suff))


(defthm truep-evaluate-literal-find-true-literal-in-clause
  (implies (truep (clause-checker clause assignment))
           (truep (evaluate-literal (find-true-literal-in-clause clause
                                                                 assignment)
                                    assignment))))

(defthm member-find-true-literal-in-clause
  (implies (truep (clause-checker clause assignment))
           (member (find-true-literal-in-clause clause assignment)
                   clause)))

(defthm truep-evaluate-literal-and-member-implies-truep-clause-checker
  (implies (and (truep (evaluate-literal literal assignment))
                (member literal clause))
           (truep (clause-checker clause assignment))))


(defthm truep-clause-checker-implies-exists-true-literal-in-clause
  (implies (truep (clause-checker clause assignment))
           (exists-true-literal-in-clause clause assignment))
  :hints (("Goal"
           :use ((:instance exists-true-literal-in-clause-suff
                            (literal (find-true-literal-in-clause
                                      clause
                                      assignment)))))))

(defthm exists-true-literal-in-clause-implies-truep-clause-checker
  (implies (exists-true-literal-in-clause clause assignment)
           (truep (clause-checker clause assignment)))
  :hints (("Goal"
           :use ((:instance (:definition exists-true-literal-in-clause))))))



;; ===================================================================
;; ========================= FILL-ASSIGNMENT =========================


(defun fill-assignment (assignment)
  (declare (xargs :guard (assignmentp assignment)))
  (cond
   ((atom assignment) assignment)
   ((undefp (car assignment))
    (cons (true) (fill-assignment (cdr assignment))))
   (t
    (cons (car assignment) (fill-assignment (cdr assignment))))))


(defthm assignmentp-fill-assignment
  (implies (assignmentp assignment)
           (assignmentp (fill-assignment assignment))))

(defthm undef-count-cons-0
  (implies (and (assignmentp assignment)
                (equal (undef-count assignment) 0)
                (not (undefp x)))
           (equal (undef-count (cons x assignment)) 0)))

(defthm undef-count-fill-assignment
  (equal (undef-count (fill-assignment assignment)) 0))

(defthm len-fill-assignment
  (equal (len (fill-assignment assignment))
         (len assignment)))




(defthm assignment-get-fill-assignment
  (implies (not (undefp (assignment-get assignment variable)))
           (equal (assignment-get (fill-assignment assignment) variable)
                  (assignment-get assignment variable)))
  :hints (("Goal" 
           :in-theory (enable assignment-get))))

(defthm assignment-get-variable-fill-assignment
  (implies (not (undefp (assignment-get-variable assignment variable)))
           (equal (assignment-get-variable (fill-assignment assignment)
                                           variable)
                  (assignment-get-variable assignment
                                           variable)))
  :hints (("Goal"
           :in-theory (enable assignment-get-variable))))

(defthm truep-evaluate-literal-fill-assignment
  (implies (truep (evaluate-literal literal assignment))
           (truep (evaluate-literal literal (fill-assignment assignment))))
  :hints (("Goal"
           :in-theory (enable evaluate-literal literal-value))))


(defthm exists-true-literal-implies-exists-true-literal-fill-assignment
  (implies (exists-true-literal-in-clause clause assignment)
           (exists-true-literal-in-clause clause (fill-assignment assignment)))
  :hints (("Goal"
           :use ((:instance (:definition exists-true-literal-in-clause))
                 (:instance exists-true-literal-in-clause-suff
                            (literal (exists-true-literal-in-clause-witness
                                      clause
                                      assignment))
                            (assignment (fill-assignment assignment)))))))


(defthm truep-clause-checker-fill-assignment
  (implies (truep (clause-checker clause assignment))
           (truep (clause-checker clause (fill-assignment assignment)))))


(defthm SAT-checker-fill-assignment
  (implies (and (assignmentp assignment)
                (truep (SAT-checker clause-list assignment)))
           (truep (SAT-checker clause-list (fill-assignment assignment)))))


(defun-sk exists-full-satisfying-assignment (clause-list)
  (exists assignment (and (assignmentp assignment)
                          (truep (SAT-checker clause-list assignment))
                          (equal (undef-count assignment) 0))))
(in-theory (disable exists-full-satisfying-assignment
                    exists-full-satisfying-assignment-suff))

(defthm sat-assignment-implies-exists-full-sat-assignment
  (implies (and (assignmentp assignment)
                (truep (SAT-checker clause-list assignment)))
           (exists-full-satisfying-assignment clause-list))
  :hints (("Goal"
           :use ((:instance exists-full-satisfying-assignment-suff
                            (assignment (fill-assignment assignment)))))))


;; ===================================================================
;; =========================== FIX LENGTH ============================

;; (defun assignment-shorten (assignment length)
;;   (declare (xargs :guard (and (assignmentp assignment)
;;                               (natp length))))
;;   (cond
;;    ((not (natp length)) nil)
;;    ((equal length 0) nil)
;;    ((atom assignment) nil)
;;    (t
;;     (cons (car assignment)
;;           (assignment-shorten (cdr assignment) (1- length))))))

;; (defun assignment-extend (assignment length)
;;   (declare (xargs :guard (and (assignmentp assignment)
;;                               (natp length))))
;;   (cond
;;    ((not (natp length)) nil)
;;    ((equal length 0) nil)
;;    ((atom assignment)
;;     (cons (undef)
;;           (assignment-extend nil (1- length))))
;;    (t
;;     (cons (car assignment)
;;           (assignment-extend (cdr assignment) (1- length))))))

;; (defun fix-length (assignment length)
;;   (declare (xargs :guard (and (assignmentp assignment)
;;                               (natp length))))
;;   (if (< (len assignment) length)
;;       (assignment-extend assignment length)
;;     (assignment-shorten assignment length)))



(defun fix-length (assignment length)
  (declare (xargs :guard (and (assignmentp assignment)
                              (natp length))))
  (cond
   ((not (natp length)) nil)
   ((and (atom assignment)
         (equal length 0))
    nil)
   ((equal length 0) nil)
   ((atom assignment)
    (cons (undef)
          (fix-length nil (1- length))))
   (t
    (cons (car assignment)
          (fix-length (cdr assignment) (1- length))))))




(defthm len-fix-length
  (implies (and (assignmentp assignment)
                (natp length))
           (equal (len (fix-length assignment length))
                  length)))

(defthm assignmentp-fix-length
  (implies (and (assignmentp assignment)
                (natp length))
           (assignmentp (fix-length assignment length))))




(defthm truep-assignment-get-fix-length
  (implies (and (natp length)
                (truep (assignment-get assignment x))
                (< x length))
           (truep (assignment-get (fix-length assignment length) x)))
  :hints (("Goal" 
           :in-theory (enable assignment-get))))

(defthm truep-assignment-get-variable-fix-length
  (implies (and (natp length)
                (< (+ -1 (var-to-nat variable)) length)
                (truep (assignment-get-variable assignment variable)))
           (truep (assignment-get-variable (fix-length assignment length)
                                           variable)))
  :hints (("Goal"
           :in-theory (e/d (assignment-get-variable) (fix-length)))))

(defthm falsep-assignment-get-fix-length
  (implies (and (natp length)
                (falsep (assignment-get assignment x))
                (< x length))
           (falsep (assignment-get (fix-length assignment length) x)))
  :hints (("Goal" 
           :in-theory (enable assignment-get)) ))

(defthm falsep-assignment-get-variable-fix-length
  (implies (and (natp length)
                (< (+ -1 (var-to-nat variable)) length)
                (falsep (assignment-get-variable assignment variable)))
           (falsep (assignment-get-variable (fix-length assignment length)
                                           variable)))
  :hints (("Goal"
           :in-theory (enable assignment-get-variable))))



(defthm truep-evaluate-literal-fix-length
   (implies (and (assignmentp assignment)
                 (literalp literal)
                 (natp length)
                 (< (+ -1 (var-to-nat (lit-to-var literal))) length)
                 (truep (evaluate-literal literal assignment)))
            (truep (evaluate-literal literal (fix-length assignment length))))
   :otf-flg t
   :hints (("Goal"
            :in-theory (enable evaluate-literal literal-value))
           ("Subgoal 5''"
            :in-theory (e/d (ternaryp)
                            (ternaryp-assignment-get-variable
                             truep-assignment-get-variable-fix-length))
            :use ((:instance truep-assignment-get-variable-fix-length
                             (variable (lit-to-var literal)))
                  (:instance ternaryp-assignment-get-variable
                             (variable (lit-to-var literal)))))
           ("Subgoal 4''"
            :in-theory (e/d (ternaryp)
                            (ternaryp-assignment-get-variable
                             truep-assignment-get-variable-fix-length))
            :use ((:instance truep-assignment-get-variable-fix-length
                             (variable (lit-to-var literal)))
                  (:instance ternaryp-assignment-get-variable
                             (variable (lit-to-var literal)))))
           ))





(defthm exists-true-literal-in-clause-fix-length
  (implies (and (assignmentp assignment)
                (clausep clause)
                (natp length)
                (<= (max-var-clause clause) length)
                (exists-true-literal-in-clause clause assignment))
           (exists-true-literal-in-clause clause (fix-length assignment
                                                             length)))
  :hints (("Goal"
           :in-theory (disable truep-evaluate-literal-fix-length
                               var-to-nat-lit-to-var-<=-max-var-clause)
            :use ((:instance (:definition exists-true-literal-in-clause))
                  (:instance exists-true-literal-in-clause-suff
                             (literal (exists-true-literal-in-clause-witness
                                       clause
                                       assignment))
                             (assignment (fix-length assignment length)))
                  (:instance truep-evaluate-literal-fix-length
                             (literal (exists-true-literal-in-clause-witness
                                       clause assignment)))
                  (:instance var-to-nat-lit-to-var-<=-max-var-clause
                             (literal (exists-true-literal-in-clause-witness
                                       clause assignment)))
                  ))))



                        
;; (defthm truep-clause-checker-fix-length
;;   (implies (and (natp length)
;;                 (<= (max-var-clause clause) length)
;;                 (clausep clause)
;;                 (assignmentp assignment)
;;                 (truep (clause-checker clause assignment)))
;;            (truep (clause-checker clause (fix-length assignment length)))))





;==============================

(defun <=-list (x n)
  (if (atom x)
      t
    (and (<= (var-to-nat (lit-to-var (car x))) n)
         (<=-list (cdr x) n))))

(defthm <=-list-lemma
  (implies (and (<=-list x n1)
                (<= n1 n2))
           (<=-list x n2)))

(defthm <=-list-max-var-clause
  (implies (clausep clause)
           (<=-list clause (max-var-clause clause)))
  :hints (("Goal"
           :in-theory (enable max-var-clause))))

(defun <=-list-list (x n)
  (if (atom x)
      t
    (and (<=-list (car x) n)
         (<=-list-list (cdr x) n))))

(defthm <=-list-list-implies-<=-list
  (implies (and (clause-listp cl)
                (<=-list-list cl n)
                (member c cl))
           (<=-list c n)))

(defthm <=-list-list-lemma
  (implies (and (<=-list-list x n1)
                (<= n1 n2))
           (<=-list-list x n2)))

(defthm <=-list-list-max-var-clause-list
  (implies (clause-listp cl)
           (<=-list-list cl (max-var-clause-list cl)))
  :hints (("Goal"
           :in-theory (enable max-var-clause-list))
          ("Subgoal *1/4.1"
           :in-theory (disable <=-list-lemma)
           :use ((:instance <=-list-lemma
                            (x (car cl))
                            (n1 (max-var-clause (car cl)))
                            (n2 (max-var-clause-list (cdr cl))))))))
                            
;==============================


(defthm truep-clause-checker-fix-length
  (implies (and (natp length)
                (<=-list clause length)
                (clausep clause)
                (assignmentp assignment)
                (truep (clause-checker clause assignment)))
           (truep (clause-checker clause (fix-length assignment length)))))


(defthm truep-sat-checker-fix-length
  (implies (and (natp length)
                (clause-listp clause-list)
                (assignmentp assignment)
                (<=-list-list clause-list length)
                (truep (SAT-checker clause-list assignment)))
           (truep (SAT-checker clause-list (fix-length assignment length)))))



;; ===================================================================



(defthm truep-SAT-checker-fix-length-max-var-clause-list
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (SAT-checker clause-list assignment)))
           (truep (SAT-checker clause-list
                               (fix-length assignment
                                           (max-var-clause-list
                                            clause-list))))))



(defun-sk exists-max-var-length-satisfying-assignment (clause-list)
  (exists assignment (and (assignmentp assignment)
                          (truep (SAT-checker clause-list assignment))
                          (equal (len assignment)
                                 (max-var-clause-list clause-list)))))
(in-theory (disable exists-max-var-length-satisfying-assignment
                    exists-max-var-length-satisfying-assignment-suff))

; clause-listp comes from natp-max-var-clause-list
(defthm sat-assignment-implies-exists-max-var-length-sat-assignment
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (SAT-checker clause-list assignment)))
           (exists-max-var-length-satisfying-assignment clause-list))
  :otf-flg t
  :hints (("Goal"
           :use ((:instance exists-max-var-length-satisfying-assignment-suff
                            (assignment (fix-length assignment
                                                    (max-var-clause-list
                                                     clause-list))))))))



(defun-sk exists-nice-satisfying-assignment (clause-list)
  (exists assignment (and (assignmentp assignment)
                          (truep (SAT-checker clause-list assignment))
                          (equal (undef-count assignment) 0)
                          (equal (len assignment)
                                 (max-var-clause-list clause-list)))))
(in-theory (disable exists-nice-satisfying-assignment
                    exists-nice-satisfying-assignment-suff))




; clause-listp comes from natp-max-var-clause-list
(defthm sat-assignment-implies-exists-nice-sat-assignment
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (SAT-checker clause-list assignment)))
           (exists-nice-satisfying-assignment clause-list))
  :otf-flg t
  :hints (("Goal"
           :use ((:instance
                  exists-nice-satisfying-assignment-suff
                  (assignment (fill-assignment (fix-length assignment
                                                           (max-var-clause-list
                                                            clause-list)))))))))











||#










































#||

(defthm asdfd
  (implies (and (assignmentp assignment)
                (natp n)
                (<= (len assignment) n))
           (equal (assignment-get assignment n)
                  (undef)))
  :hints (("Goal" :in-theory (enable assignment-get))))


(defthm asdfdd
  (implies (and (assignmentp assignment)
                (natp n)
                (natp length)
                (< (len assignment) length))
           (equal (assignment-get (fix-length assignment length) n)
                  (undef)))
  :hints (("Goal"
           :in-theory (enable assignment-get))))

(defthm lem-1
  (implies (and (assignmentp assignment)
                (natp n)
                (natp length)
                (< (len assignment) length))
           (equal (assignment-get (fix-length assignment length) n)
                  (assignment-get assignment n)))
  :hints (("Goal"
           ;:induct (fix-length assignment length)
           :in-theory (enable assignment-get))))


(defthm lem-1
  (implies (and (assignmentp assignment)
                (natp n)
                (natp length)
                (< n length))
           (equal (assignment-get (fix-length assignment length)
                                  n)
                  (assignment-get assignment
                                  n)))
  :hints (("Goal"
           :induct (ind-hint assignment length)
           :in-theory (enable assignment-get))))


(defthm lem-1
  (implies (and (assignmentp assignment)
                (varp variable)
                (natp length)
                (<= (var-to-nat variable)
                    length))
           (equal (assignment-get-variable (fix-length assignment length)
                                           variable)
                  (assignment-get-variable assignment
                                           variable))))
  :hints (("Goal"
           :in-theory (enable evaluate-literal))))

(defthm lem-1
  (implies (and (assignmentp assignment)
                (literalp literal)
                (natp length)
                (<= (var-to-nat (lit-to-var literal))
                    length))
           (equal (evaluate-literal literal (fix-length assignment length))
                  (evaluate-literal literal assignment)))
  :hints (("Goal"
           :in-theory (enable evaluate-literal))))

              

(defthm evaluate-literal-fix-length
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (evaluate-literal literal assignment)))
           (truep (evaluate-literal literal (fix-length assignment
                                                        (max-var-clause-list
                                                         clause-list)))))
  :hints (("Goal"
           :in-theory (enable evaluate-literal max-var-clause-list))))




(defthm clause-checker-fix-length
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (clause-checker clause assignment)))
           (truep (clause-checker clause (fix-length assignment
                                                     (max-var-clause-list
                                                      clause-list))))))

(defthm SAT-checker-fix-length
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (SAT-checker clause-list assignment)))
           (truep (SAT-checker clause-list (fix-length assignment
                                                       (max-var-clause-list
                                                        clause-list))))))




(defun-sk exists-max-var-length-satisfying-assignment (clause-list)
  (exists assignment (and (assignmentp assignment)
                          (truep (SAT-checker clause-list assignment))
                          (equal (len assignment)
                                 (max-var-clause-list clause-list)))))
(in-theory (disable exists-max-var-length-satisfying-assignment
                    exists-max-var-length-satisfying-assignment-suff))

; clause-listp comes from natp-max-var-clause-list
(defthm sat-assignmnet-implies-exists-max-var-length-sat-assignment
  (implies (and (assignmentp assignment)
                (clause-listp clause-list)
                (truep (SAT-checker clause-list assignment)))
           (exists-max-var-length-satisfying-assignment clause-list))
  :otf-flg t
  :hints (("Goal"
           :use ((:instance exists-max-var-length-satisfying-assignment-suff
                            (assignment (fix-length assignment
                                                    (max-var-clause-list
                                                     clause-list))))))))





||#


;; ========================== DO NOT DELETE ==========================

;; (defconst *clause-list1*
;;   '((1 2 -3) (3 -4) (-1) (-2)))

;; (defconst *assignment1*
;;   '(f f t t))


#||

; [Integer] -> [bool] -> bool
(defun Clause-checker (clause assignment)
  (declare (xargs :guard (and (clausep clause)
                              (assignmentp assignment)
                              (assignment-covers-clause assignment clause))
                  :guard-hints (("Goal"
                                 :in-theory (disable ternaryp)))))
  (if (atom clause)
      'f
    (let* ((literal (car clause))
           (variable (lit-to-var literal))
           (assigned-value (assignment-get-variable assignment variable))
           (literal-value (literal-value literal assigned-value)))
      (cond
       ((equal literal-value 'undef)
        (if (equal (Clause-checker (cdr clause) assignment) 't)
            't
          'undef))
       ((equal literal-value 't) 't)
       (t ; 'f
        (Clause-checker (cdr clause) assignment))))))


; [[Integer]] -> [bool] -> bool
(defun SAT-checker (clause-list assignment)
  (declare (xargs :guard (and (clause-listp clause-list)
                              (assignmentp assignment)
                              (assignment-covers-clause-list assignment
                                                             clause-list))))
  (cond 
   ((atom clause-list) 't)
   ((equal (Clause-checker (car clause-list) assignment) 'f) 'f)
   ((equal (Clause-checker (car clause-list) assignment) 'undef)
    (if (equal (Clause-checker (cdr clause-list) assignment) 'f)
        'f
      'undef))
   (t
    (SAT-checker (cdr clause-list) assignment))))

||#






























































#||

(defthm nlem3-1-2
  (implies (and ;(assignmentp assignment)
                ;(natp x)
                (natp length)
                (falsep (assignment-get assignment x))
                (< x length))
           (falsep (assignment-get (fix-length assignment length) x)))
  :hints (("Goal" 
           :in-theory (enable assignment-get)) ))

(defthm nlem3-2-2
  (implies (and ;(assignmentp assignment)
                ;(varp variable)
                (natp length)
                (< (+ -1 (var-to-nat variable)) length)
                (falsep (assignment-get-variable assignment variable)))
           (falsep (assignment-get-variable (fix-length assignment length)
                                           variable)))
  :hints (("Goal"
           :in-theory (enable assignment-get-variable))))
           
(defun ind-hint--1 (x)
  (if (or (not (natp x))
          (equal x 0))
      nil
    (ind-hint--1 (1- x))))

(defthm asdfd
  ;(implies ;(and ;(natp length)
            ;    (natp x))
           (undefp (assignment-get (fix-length nil length) x));)
  :hints (("Goal"
           :induct (ind-hint--1 length) )))
           :in-theory (enable assignment-get))))

(defthm nlem3-1-3
  (implies (and ;(assignmentp assignment)
                ;(natp x)
                (natp length)
                (undefp (assignment-get assignment x))
                (< x length))
           (undefp (assignment-get (fix-length assignment length) x)))
  :hints (("Goal" 
           :in-theory (enable assignment-get)) ))

(defthm nlem3-2-3
  (implies (and ;(assignmentp assignment)
                ;(varp variable)
                (natp length)
                (< (+ -1 (var-to-nat variable)) length)
                (undefp (assignment-get-variable assignment variable)))
           (undefp (assignment-get-variable (fix-length assignment length)
                                           variable)))
  :hints (("Goal"
           :in-theory (enable assignment-get-variable))))
           

(defthm nlem3-3-1
  (implies (and ;(assignmentp assignment)
                ;(varp variable)
                (natp length)
                (< (+ -1 (var-to-nat variable)) length)
                (truep (literal-value literal
                                      (assignment-get-variable assignment
                                                               variable))))
           (truep (literal-value literal
                                 (assignment-get-variable (fix-length
                                                           assignment length)
                                                          variable))))
  :otf-flg t
  :hints (("Goal"
           :in-theory (enable literal-value))))

||#