#ifndef learnpnp_ExploitWithHysteresis_h__guard
#define learnpnp_ExploitWithHysteresis_h__guard

#include <pnp/learning_plan/ExpPolicy.h>

namespace learnpnp {

	/**
	 * @brief Exploration policy that exploits with an hysteresis.
	 *
	 * This class remembers the policy of the last episode. In each choice point, it switches to
	 * the new best policy only if it is \p hysteresis times better. If hysteresis == 0 it is equivalent
	 * to Exploit.
	 *
	 * With hysteresis = std::numeric_limits<double>::max() it may be used to execute the last policy
	 * exactly as it was (no switch can take place).
	 *
	 * It maintains the last policy in memory, so it is not robust to restarts (as opposed to Learner that
	 * stores the value function in a file). Such a memory is indicized by the plan name, therefore it
	 * must be provided to the constructor.
	 *
	 * \author Matteo Leonetti
	 **/
	class ExploitWithHysteresis : public learnpnp::ExpPolicy {

public:

	/**
	 * @brief Ctor
	 *
	 * @param planName the name of the plan, used to retrieve the last policy for that plan
	 * @param hysteresis the number of time the new value must be better than the old one for a switch. 0 is neutral.
	 * @param remember whether or not it must remember the choice during the same episode (rather than making a
	 * new choice every time a choice point is encountered). Defaults to true.
	 **/
	 ExploitWithHysteresis(const std::string& planName, double hysteresis, bool remember = true);
	
	virtual ~ExploitWithHysteresis();
	
	MemoryType *getPolicy();

	/**
	 * \brief Deletes the previous policies of all plans.
	 * 
	 * ATTENTION: the current implementation invalidates all instances of this
	 * class that would hold a pointer to the deleted memory.
	 */
	static void clearMemory();

	/**
	 * \brief Deletes the previous policy of a specific plan
	 */
	static void forget(const std::string& planName);
	
	private:
		MemoryType *oldMemory;
		double hysteresis;
		std::string planName;
		
		int makeChoice(Learner *learner,const Marking &current, const std::vector<Marking> &states);

		static std::map<std::string, MemoryType> memories;
};

}

#endif
