package ir.classifiers;

import java.io.*;
import java.util.*;
import ir.vsr.*;
import ir.utilities.*;

/**
 * A perceptron classifier that trains a perceptron to recognize each
 * category.
 *
 * @author   Ray Mooney
 */


public class Perceptron extends Classifier
{
    /** Stores one PerceptronUnit for each category
     * which is trained to fire only for examples in this category. */
    PerceptronUnit[] categoryPerceptrons;

    /** Name of classifier */
    public static final String name = "Perceptron";

    /** Number of categories */
    protected int numCategories; 

    /** Flag for debug print statements */
    protected boolean debug = false;

    /** Function to indicate that this class does not use an inverted index */
    public boolean usesInvertedIndex() {
      return false;
    }

    /** Since Perceptron does not use an inverted Index, this function 
     *  does nothing */
    public void setInvertedIndex(InvertedIndex index){};


    /** Create an Perceptron classifier with these attributes
     *
     * @param categories  The array of Strings containing the category names
     * @param debug  Flag to turn on detailed output
     */
    public Perceptron(String [] categories, boolean debug) {
	this.categories = categories;
	this.debug = debug;
	numCategories = categories.length;
	// Intialize a perceptron for each category
	categoryPerceptrons = new PerceptronUnit[numCategories];
	for(int i=0; i < numCategories; i++) {
	    categoryPerceptrons[i] = new PerceptronUnit(categories, debug);
	}
    }


    /** Sets the debug flag */
    public void setDebug(boolean bool){
	debug = bool;
    }
    

    /** Returns the name */
    public String getName() {
      return name;
    }
    

    /** Trains the perceptron by training a PerceptronUnit for each
     *  category.
     *
     *   @param trainExamples  The list of training examples
     */
    public void train(List trainExamples)
    {
	// Train a perceptron for each category
	for(int i=0; i < numCategories; i++) {
	    // Re-initialize the PerceptronUnit for each category
	    categoryPerceptrons[i].clear();
	    // Then retrain it on the current training examples
	    categoryPerceptrons[i].trainCategory(trainExamples, i);
	}
    }

		    
    /** Categorizes the test example using the trained Perceptron classifier, returning true if
     *   the predicted category is same as the actual category
     *   
     *   @param testExample  The test example to be categorized
     */
    public boolean test(Example testExample)
    {
	// Compute the confidence in the perceptron for each category firing
	// for this example...
	double[] confidences = new double[numCategories];
	for(int i=0; i < numCategories; i++) {		
	    confidences[i] = categoryPerceptrons[i].classify(testExample);	    
	}
	// The predicted class is the category with the highest confidence, i.e.
	// the category whose net input most exceeds its threshold
	int predictedClass = argMax(confidences);
	if (debug) {
	    System.out.print("Document: " + testExample.name + "\nResults: ");
	    for (int j=0; j<numCategories; j++) {
		System.out.print(categories[j] + "(" + confidences[j] + ")\t");	
	    }
	    System.out.println("\nCorrect class: " + testExample.getCategory() + 
			       ", Predicted class: " + predictedClass  + "\n");
	}
	return (predictedClass == testExample.getCategory());
    }

}
