; ACL2 Version 3.5 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2009  University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic
; NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

; The original version of this file was contributed by Bob Boyer and
; Warren A. Hunt, Jr.  The design of this system of Hash CONS,
; function memoization, and fast association lists (applicative hash
; tables) was initially implemented by Boyer and Hunt.

(in-package "ACL2")

(defmacro defn (f a &rest r)

  ":Doc-Section Programming
   definition with ~il[guard] ~c[t]~/

   ~c[Defn] is ~ilc[defun] with ~il[guard] ~c[t].~/~/"

  `(defun ,f ,a (declare (xargs :guard t)) ,@r))

(defdoc hons-and-memoization
  ":Doc-Section Hons-and-Memoization

  hash cons, function memoization, and applicative hash tables~/

  Bob Boyer and Warren Hunt have developed a canonical representation
  for ACL2 data objects and a function memoization mechanism to
  facilitate reuse of previously computed results.  This facility
  includes procedures to read and print ACL2 expressions in such a way
  that repetition of some ACL2 objects is eliminated, thereby
  permitting a kind of on-the-fly file compression.  The
  implementation does not alter the semantics of ACL2 except to add a
  handful of definitions.

  Much of the documentation for this topic is taken from the paper
  ``Function Memoization and Unique Object Representation for ACL2
  Functions'' by Robert S. Boyer and Warren A. Hunt, Jr., which has
  appeared in the Sixth International Workshop on the ACL2 Theorem
  Prover and Its Applications, ACM Digital Library, 2006.

  You can build an experimental version of ACL2 that includes hash
  cons, function memoization, and fast association lists (applicative
  hash tables).  An easy way is to include the following with a
  ~c[make] command:
  ~bv[]
  ACL2_HONS=h
  ~ev[]
  So for example, to make an executable image and also documentation
  (which will appear in subdirectories ~c[doc/EMACS] and
  ~c[doc/HTML]):
  ~bv[]
  make large DOC ACL2_HONS=h
  ~ev[]~/

  In the implementation of the ACL2 logic, ACL2 data objects are
  represented by Common Lisp objects of the same type, and the ACL2
  pairing operation is internally implemented by the Common Lisp
  ~ilc[cons] function.  In Common Lisp, ~c[cons] is guaranteed to
  provide a new pair, distinct from any previously created pair.  We
  have defined a new ACL2 function ~ilc[HONS] that is logically
  identical to the ACL2 ~c[cons] function, but whose implementation
  usually reuses an existing pair if its components are identical to
  the components of an existing pair.  A record of ACL2 HONS objects
  is kept, and when an ACL2 function calls ~c[hons] ACL2 searches for
  an existing identical pair before allocating a new pair; this
  operation been called ``hash consing''.

  It appears that hash consing was first conceived by A. P. Ershov in
  1957, to speed up the recognition of common subexpressions.  Ershov
  showed how to collapse trees to minimal DAGs by traversing trees
  bottom up, and he used hashing to eliminate the re-evaluation of
  common subexpressions.  Later, Eiichi Goto implemented a Lisp system
  with a built-in hash consing operation: his h-CONS cells were
  rewrite protected and free of duplicate copies, and Goto used this
  hash consing operation to facilitate the implementation of a
  symbolic algebra system he developed.

  Memoizing functions also has a long history.  In 1967, Donald Michie
  proposed using memoized functions to improve the performance of
  machine learning.  Rote learning was improved by a learning function
  not forgetting what it had previously learned; this information was
  stored as memoized function values.

  The use of hash consing has appeared many times.  For instance,
  Henry Baker using hash consing to improve the performance of the
  well-known Boyer rewriting benchmark.  Baker used both hash consing
  and function memoization to improve the speed of the Takeuchi
  function, exactly in the spirit of our implementation, but without
  the automated, system-wide integration we report here.

  The ACL2 implementation permits memoization of user-defined
  functions.  During execution a user may enable or disable function
  memoization on an individual function basis, may clear memoization
  tables, and may even keep a stack of memoization tables.  This
  facility takes advantage of our implementation where we keep one
  copy of each distinct ACL2 data object.  Due to the functional
  nature of ACL2, it is sufficient to have at most one copy of any
  data structure; thus, a user may arrange to keep data canonicalized.
  This implementation extends to the entire ACL2 system the benefits
  enjoyed by BDDs: canonicalization, memoization, and fast equality
  check.

  We have defined various algorithms using these features, and we have
  observed, in some cases, substantial performance increases.  For
  instance, we have implemented unordered set intersection and union
  operations that operate in time roughly linear in the size of the
  sets.  Without using arrays, we defined a canonical representation
  for Boolean functions using ACL2 objects.  We have investigated the
  performance of rewriting and tree consensus algorithms to good
  effect, and we believe function memoization offers interesting
  opportunities to simplify algorithm definition while simultaneously
  providing performance improvements.

  We recommend that users focus at first on the logical definitions of
  ~ilc[hons] and other primitives rather than their underlying Common
  Lisp implementations.  Integrated with this memoization system is a
  performance monitoring system, which can provide real-time feedback
  on the operation and usefulness of ~ilc[hons] and function
  memoization.  For a more detailed description of these tools, please
  see the ACL2 2006 workshop paper mentioned above.

  The Fibonacci function is a small example that demonstrates the
  utility of function memoization.  The following definition exhibits
  a runtime that is exponential in its input argument.
  ~bv[]
  (defun fib (x)
    (declare (xargs :guard (natp x)))
    (mbe
     :logic 
     (cond ((zp x) 0)
           ((= x 1) 1)
           (t (+ (fib (- x 1)) (fib (- x 2)))))
     :exec
     (if (< x 2)
         x
       (+ (fib (- x 1)) (fib (- x 2))))))
  ~ev[]

  Below we show how the ACL2 ~ilc[time$] utility can measure the time
  to execute a call to the ~c[fib] function (with some editing to
  avoid noise from the underlying Common Lisp implementation).
  ~ilc[Memoize] is actually an ACL2 macro that expands to a call of
  the ACL2 function used to identify a function for memoization;
  ~pl[memoize].  This function also accepts a well-formed term that
  must be true in order for the system to memoize a call of the
  memoized function; to ensure that an instance of the term is safe
  for evaluation in Common Lisp, a check is performed that if the
  ~il[guard] of the memoized function is satisfied, then this instance
  will execute without error.  ~bv[] ACL2 !>(time$ (fib 40))

  ... took 2,641 milliseconds (2.641 seconds) to run with 8 available
  CPU cores.  During that period, 2,644 milliseconds (2.644 seconds)
  were spent in user mode 0 milliseconds (0.000 seconds) were spent in
  system mode 16 bytes of memory allocated.  102334155 ACL2 !>(memoize
  'fib)

  Summary
  Form:  ( TABLE MEMOIZE-TABLE ...)
  Rules: NIL
  Warnings:  None
  Time:  0.04 seconds (prove: 0.00, print: 0.00, other: 0.04)

  Summary
  Form:  ( PROGN (TABLE MEMOIZE-TABLE ...) ...)
  Rules: NIL
  Warnings:  None
  Time:  0.04 seconds (prove: 0.00, print: 0.00, other: 0.04)
   FIB
  ACL2 !>(time$ (fib 40))

  ... took 19 milliseconds (0.019 seconds) to run with 8 available CPU
  cores.  During that period, 20 milliseconds (0.020 seconds) were
  spent in user mode 0 milliseconds (0.000 seconds) were spent in
  system mode
   539,088 bytes of memory allocated.
   193 minor page faults, 0 major page faults, 0 swaps.
  102334155
  ACL2 !>(time$ (fib 100))

  ... took 0 milliseconds (0.000 seconds) to run with 8 available CPU
  cores.  During that period, 0 milliseconds (0.000 seconds) were
  spent in user mode 0 milliseconds (0.000 seconds) were spent in
  system mode
   5,760 bytes of memory allocated.
   3 minor page faults, 0 major page faults, 0 swaps.
  354224848179261915075
  ACL2 !>(unmemoize 'fib)
  ~ev[]

  We see that once the function ~c[fib] is identified as a function
  for which function calls should be memoized, the execution times are
  substantially reduced.  Finally, we can prevent ~c[fib] from being
  further memoized; in fact, ~ilc[unmemoize] erases the previously
  memoized values.

  The implementation of hash consing, memoization, and applicative
  hash tables involves several facets: canonical representation of
  ACL2 data, function memoization, and the use of Lisp hash tables to
  improve the performance of association list operations.  We discuss
  each of these in turn, and we mention some subtle
  interrelationships.  Although it is not necessary to understand the
  discussion in this section, it may permit some users to better use
  this implementation.  This discussion may be confusing for some ACL2
  users as it makes references to Lisp implementation features.

  The ACL2 system is actually implemented as a Lisp program that is
  layered on top of a Common Lisp system implementation.  ACL2 data
  constants are implemented with their corresponding counterparts in
  Common Lisp; that is, ACL2 cons pairs, strings, characters, numbers,
  and symbols are implemented with their specific Common Lisp
  counterparts.  This choice permits a number of ACL2 primitive
  functions to be implemented with their corresponding Common Lisp
  functions, but there are indeed differences.  ACL2 is a logic, and
  as such, it does not specify anything to do with physical storage or
  execution performance.  When ACL2 is used, the knowledgeable user
  can write functions to facilitate the reuse of some previously
  computed values.

  Recall the three mechanisms under discussion: hash consing, function
  memoization, and fast association list operations.  The function
  memoization mechanism takes advantage of the canonical
  representation of data objects provided by the ~ilc[hons] operation
  as does the fast association list operation mechanism.  Each time
  ~c[hons] is invoked, its arguments are themselves converted, if
  necessary, to uniquely represented objects.

  The ACL2 universe is recursively closed under the ~c[cons] pairing
  operation and the atoms.  Hash consing (~ilc[hons]) is logically
  identical to ~c[cons], but a set of tables is used to record each
  ~c[hons] pair.  When a ~c[hons] pair is requested, the
  implementation checks, in the current set of tables, whether the
  requested pair already exists.  If not, a new pair is created and a
  record of that pair is made; otherwise, a reference to the
  previously created pair is returned.  Thus, any data object created
  with ~c[hons] has a unique representation, as does every
  subcomponent.  We also arrange for strings to have a unique
  representation ~-[] only one copy of each different string is kept
  ~-[] and when any previously unseen string is an argument to
  ~c[hons], we add the string to a unique table of strings.  A
  system-wide benefit of having a canonical representation for data is
  that equality comparisons between any two data objects can be done
  in constant time.

  The definition of ~ilc[hons] in no way changes the operation of
  ~c[cons] ~-[] ~c[hons] creates a ~c[cons] pair.  When asked to
  create a ~c[hons], the implementation checks to see if there is a
  ~c[cons] with the same ~ilc[car] and ~ilc[cdr] as the ~c[hons] being
  requested.  Thus, the only difference between the results of a
  ~c[hons] call and a corresponding ~c[cons] call is a notation in
  some invisible (to the ACL2 logic) tables where we record which
  ~c[cons] elements are also ~c[hons] elements.  Since a ~c[hons] is
  nothing but a ~c[cons], the operation of ~c[car] and ~c[cdr] is
  unchanged.  In fact, the implementation is designed so that at any
  time it is safe to clear the table identifying which ~c[cons]
  elements are also considered ~c[hons] elements.

  User-defined functions with defined and verified guards can be
  memoized.  When a function is memoized, a user-supplied condition
  restricts the domain when memoization is attempted.  When the
  condition is satisfied, memoization is attempted (assuming that
  memoization for the function is presently enabled); otherwise, the
  function is just evaluated.  Each memoized function has a hash table
  that is used to keep track of a unique list of function arguments
  paired with their values.  If appropriate, for each function the
  corresponding table is checked to see if a previous call with
  exactly the same arguments already exists in the table: if so, then
  the associated value is returned; if not, then the function is
  evaluated and a new key-value pair is added to the table of memoized
  values so that some future call will benefit from the memoization.
  With ACL2 user functions memoization can be dynamically enabled and
  disabled.  There is an ACL2 function that clears a specific
  memoization table.  And finally, just as with the ~c[hons] table, a
  stack of these function memoization tables is maintained; that is,
  it is possible to memoize a function, use it a bit, set the memoized
  values aside, start a new table, use it, and then return to the
  original table.

  We next discuss the fast lookup operation for association lists.
  When a pair is added to an association list using the functions
  ~c[hons-acons] or ~c[hons-acons!], the system also records the
  key-value pair in an associated hash table.  As long as a user only
  used these two functions when placing key-value pairs on an
  association list, the key-value pairs in the corresponding hash
  table will be synchronized with the key-value pairs in the
  association list.  Later, if ~c[hons-get] is used to look up a key,
  then instead of performing a linear search of the association list
  we consult the associated hash table.  If a user does not strictly
  follow this discipline, then a linear search may be required.  In
  some sense, these association lists are much like ACL2 arrays, but
  without the burden of explicitly naming the arrays.  The ACL2 array
  ~ilc[compress1] function is analogous to the functions
  ~c[hons-shrink-alist] and ~c[hons-shrink-alist!].  There are
  user-level ACL2 functions that allow the associated hash tables to
  be cleared and removed.

  Finally, we would advise anyone who is using CCL in a research
  environment to stay plugged into the ``trunk'' or ``bleeding edge''
  of CCL development.  This is very easy to do by typing a few
  commands to a shell, for example standing above the target directory
  as follows, provided one has ~c[svn] working.
  ~bv[]

   For linux:

     rm -rf ccl
     svn co http://svn.clozure.com/publicsvn/openmcl/trunk/linuxx8664/ccl

   For an x86 Macintosh running the Darwin OS:

     svn co http://svn.clozure.com/publicsvn/openmcl/trunk/darwinx8664/ccl

   To keep up to date, you may find it sufficient to do:

     cd ccl
     svn update

  ~ev[]

  ~sc[References]

  Baker, Henry G., The Boyer Benchmark at Warp Speed. ACM Lisp
  Pointers V,3 (Jul-Sep 1992), pages 13-14.

  Baker, Henry G., A Tachy 'TAK'.  ACM Lisp Pointers Volume 3,
  July-September, 1992, pages 22-23.

  Robert S. Boyer and Warren A. Hunt, Jr., Function Memoization
  and Unique Object Representation for ACL2 Functions, in the Sixth
  International Workshop on the ACL2 Theorem Prover and Its
  Applications, ACM Digital Library, 2006.

  A. P. Ershov.  On Programming of Arithmetic  Operations.  In
  the Communications of the ACM, Volume 118, Number 3, August,
  1958, pages 427-430.

  Eiichi Goto, Monocopy and Associative Algorithms in Extended Lisp, 
  TR-74-03, University of Toyko, 1974.

  Richard P. Gabriel.  Performance and Evaluation of Lisp Systems.
  MIT Press, 1985.

  Donald Michie.  Memo functions: a Language Feature with Rote
  Learning Properties.  Technical Report MIP-R-29, Department of
  Artificial Intelligence, University of Edinburgh, Scotland, 1967.

  Donald Michie.  Memo Functions and Machine Learning.  In Nature,
  Volumne 218, 1968, pages 19-22.
  ~/")

; The next few functions require special "under the hood"
; implementations so that we always use a previously allocated CONS
; object if a pair EQUAL to (CONS x y) is requested and currently
; exists.

#+(or acl2-loop-only (not hons))
(defn clear-hash-tables ()

  ;; A logical no-op.  In some Lisps, this may reduces the number of
  ;; honses to those that must be retained.

  nil)

#+(or acl2-loop-only (not hons))
(defn hons (x y)

  ;; Has an "under the hood" implementation.

  ":Doc-Section Hons-and-Memoization

  hash cons~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt.
  ~l[hons-and-memoization].

  Logically, ~c[hons] is merely another name for ~c[cons], i.e., the
  following is an ACL2 theorem.
  ~bv[]
  (equal (hons x y) (cons x y))
  ~ev[]
  However, ~c[hons] creates a special kind of cons pair, which we call
  a hons pair or, for short, hons.

  ~c[Hons] generally runs slower than ~c[cons] because in creating a
  hons, an attempt is made to see whether a hons already exists with
  the same ~c[car] and ~c[cdr].  This involves search and the use of
  hash-tables.  In the worst case, honsing can be arbitrarily slow,
  but our experience in CCL is that honsing is generally about 20
  times as slow as consing.

  In our implementation, every hons really is a cons pair, but not
  every cons pair is a hons, as can be determined by using ~ilc[eq] in
  raw Common Lisp.  For example, consider this interaction in raw
  Common Lisp.
  ~bv[]
  ? (setq x (cons 1 2))
  (1 . 2)
  ? (setq y (hons 1 2))
  (1 . 2)
  ? (setq z (hons 1 2))
  (1 . 2)
  ? (eq y z)
  T
  ? (eq x y)
  NIL
  ~ev[]~/~/"

  (cons x y))

#+(or acl2-loop-only (not hons))
(defn hons-equal (x y)
  ":Doc-Section Hons

  equality for hons pairs~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt.
  ~l[hons-and-memoization].

  Logically, ~c[hons-equal] is merely another name for ~ilc[equal],
  i.e., the following is a theorem.
  ~bv[]
  (equal (hons-equal x y) (equal x y))
  ~ev[]

  But ~c[hons-equal] may run much faster than ~ilc[equal] in some
  cases.  If both arguments are determined to be honses, then
  ~c[hons-equal] becomes a mere ~ilc[eq].~/~/"

  ;; Has an "under the hood" implementation.
  (equal x y))

#+(or acl2-loop-only (not hons))
(defn hons-copy (x)
  ":Doc-Section Hons

  identity function that creates hons pairs~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt.
  ~l[hons-and-memoization].

  Logically, ~c[hons-copy] is merely another name for the
  ~ilc[identity] function, i.e., the following is a theorem.
  ~bv[]
  (equal (hons-copy x) x)
  ~ev[]

  ~c[Hons-copy] returns a value in which all conses are ~il[hons]es.
  You cannot directly tell this from within ACL2, where both ~c[x] and
  ~c[(hons-copy x)] are definitely ~ilc[equal].  But ACL2 will not
  permit you to call ~ilc[eq] on them because the guard for ~c[eq] is
  ~c[(or (symbolp x) (symbolp y))].~/~/"
  ;; Has an "under the hood" implementation.
  x)

(defn hons-assoc-equal (x y)
  ":Doc-Section Hons

   like assoc-equal but does not cause an error.~/

   Hash-Cons'ed association lists used with HONS-GET, HONS-ACONS, and
   HONS-ACONS!, may have any symbol at the end instead of just NIL, as
   ALISTP requires.  The symbol allows an association list to be
   'named'.  Such naming may be important to keep fast hash alists from
   colliding with one another, resulting in inefficient performance.~/~/"
  
  (cond ((atom y) nil)
        ((and (consp (car y))
              (hons-equal x (car (car y))))
         (car y))
        (t (hons-assoc-equal x (cdr y)))))

#+(or acl2-loop-only (not hons))
(defn hons-get-fn-do-hopy (x l)

; Has an "under the hood" implementation.

 (hons-assoc-equal x l))

#+(or acl2-loop-only (not hons))
(defn hons-get-fn-do-not-hopy (x l)

; Has an "under the hood" implementation.

  (hons-assoc-equal x l))

(defmacro hons-get (x l)
  ":Doc-Section Hons-and-Memoization

  assoc-equal for hons~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt.
  ~l[hons-and-memoization].

  Logically, ~c[HONS-GET] is merely another name for the
  ~ilc[ASSOC-EQUAL] function when the guard of ~ilc[ASSOC-EQUAL]
  is satisfied, i.e., the following is a theorem.
  ~bv[]
    (implies (alistp alist)
             (equal (hons-get key alist) 
                    (assoc-equal key alist)))
  ~ev[]

  If alist has been formed via calls to HONS-ACONS, HONS-GET
  should operate at hash-table lookup speed.

  ~c[HONS-GET] has an 'under the hood' implementation.~/~/"

  (list 'hons-get-fn-do-hopy x l))

(add-macro-alias hons-get hons-get-fn-do-hopy)

#+(or acl2-loop-only (not hons))
(defn hons-acons (key value l)
  ":Doc-Section Hons-and-Memoization

  acons with hons~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt.
  ~l[hons-and-memoization].

  Logically, ~c[hons-acons] is merely another name for the
  ~ilc[acons] function, i.e., the following is a theorem.
  ~bv[]
  (equal (hons-acons key value alist) 
         (cons (cons key value) alist))
  ~ev[]

  Practically speaking, HONS-COPY is called on the KEY
  so that fast access with HONS-GET is a possibility.

  Each time that HONS-ACONS is called when alist has an
  underlying hash table for fast access, that hash table
  is 'stolen' and associated with the result of the
  HONS-ACONS.

  Has an 'under the hood' implementation.~/~/"

  (cons (cons (hons-copy key) value) l))

#+(or acl2-loop-only (not hons))
(defn hons-acons! (key value l)

  ":Doc-Section Hons-and-Memoization

  acons with hons~/

  Logically, ~c[hons-acons!] is merely another name for the
  ~ilc[acons] function, i.e., the following is a theorem.
  ~bv[]
  (equal (hons-acons! key value alist) 
         (cons (cons key value) alist))
  ~ev[]

  Practically speaking, HONS-COPY is called on the KEY, VALUE, and
  ALIST, so that fast access with HONS-GET is a possibility.

  Each time that HONS-ACONS is called when alist has an
  underlying hash table for fast access, that hash table
  is 'stolen' and associated with the result of the
  HONS-ACONS.

  Has an 'under the hood' implementation.  The (HONS KEY VALUE)
  below will cause VALUE to have a unique representation, which,
  for large structures, may require a substantial amount of work.~/~/"

  (hons (hons (hons-copy key) value) l))

(defn fast-alist-len-acc (al acc)
  (cond ((atom al) (len acc))
        ((consp (car al))
         (cond ((hons-assoc-equal (caar al) acc)
                (fast-alist-len-acc (cdr al) acc))
               (t (fast-alist-len-acc (cdr al) (cons (car al) acc)))))
        (t (fast-alist-len-acc (cdr al) acc))))

#+(or acl2-loop-only (not hons))
(defn fast-alist-len (al)

  "The length of the alist al, not counting entries that are masked by
  previous entries.  Fast if al is a fast alist."

  (fast-alist-len-acc al nil))

; Someday it would be good to call COMPACT-READ and
; COMPACT-PRINT from within ACL2 code.  This is currently
; obscure because in ACL2, READ-OBJECT does not yet have a
; serious specification, i.e., the user is left to browse
; his copy of the ANSI specification for Common Lisp to
; figure out what the Lisp reader does.  A correct
; definition of something like COMPACT-READ would have to be
; based upon a not yet existent specification of
; READ-OBJECT.

; Here we provide ACL2 functions that are designed to effect
; the underlying implementation of our HONS procedure, our
; fast association list mechanism, and our function
; memoization feature.

#+(or acl2-loop-only (not hons))
(defn flush-hons-get-hash-table-link (x)
  ;; A logical no-op.  Removes X from the *HONS-ACONS-HT*.  Good to
  ;; call when one is done with an object built with HONS-ACONS or
  ;; HONS-ACONS! because it may release unnecessary hash tables for
  ;; garbage collection.
   x)

#+(or acl2-loop-only (not hons))
(defn clear-memoize-table (fn)

  ":Doc-Section Hons-and-Memoization

  Forget values remembered for the function FN~/

  A logical no-op.  The values memoized for fn are forgotten.~/~/"
  
  fn)


#+(or acl2-loop-only (not hons))
(defn clear-memoize-tables ()

  ":Doc-Section Hons-and-Memoization

  Forget values remembered for all the memoized functions.~/

  A logical no-op.  All memoized values are forgotten.

  Returns NIL.  Invokes CLEAR-MEMOIZE-TABLE for
  each memoized function.~/~/"

  nil)

; The macros MEMOIZE-LET, MEMOIZE-ON, and MEMOIZE-OFF have the utterly
; bizarre property that if evaluated at the top level of the ACL2
; loop, they really, really do nothing but evaluate form, as their
; semantics suggest.  However, if they are used within an ACL2
; function and either (a) that function is executed in program mode or
; (b) that function is known to be Common Lisp compliant and is
; executed in any mode, then there may be "under the hood" effects
; that, though not changing the semantics of what ACL2 returns, may
; affect the speed and/or space utilization of the computation.

#+(or acl2-loop-only (not hons))
(defmacro memoize-let (fn form)
  (declare (ignore fn))

; MEMOIZE-LET evaluates form.  At the beginning of that evaluation, no
; old values are remembered of calls of the symbol fn.  Afterwards,
; those old values will be restored if no stobjs have been altered,
; but all newer memoized values are forgotten.  The symbol fn must be
; memoized before MEMOIZE-LET is called.

  form)

; The functions memoize and unmemoize have rather innocent looking
; semantics.  But under the hood, they enable and disable memoization.
; The function memoize might cause errors due to compilation problems.

(defconst *hons-primitive-fns*
  '(hons hons-equal hons-copy hons-get-fn-do-hopy
         clear-hash-tables
         hons-get-fn-do-not-hopy
         hons-acons hons-acons!
         fast-alist-len fast-alist-len-acc
         cons-subtrees number-subtrees
         flush-hons-get-hash-table-link clear-memoize-table
         clear-memoize-tables
         hons-shrink-alist! hons-shrink-alist))

(defconst *hons-primitives* ; hons-related macros and primitive fns
  (append '(memoize-let
            hons-get
            memoize unmemoize
            memoize-on memoize-off)
          *hons-primitive-fns*))

(defun hons-enabledp (state)
  (declare (xargs :guard (and (state-p state)
                              (boundp-global 'hons-enabled state))))
  (f-get-global 'hons-enabled state))

(defmacro memoize (fn &key
                      (condition 't condition-p)
                      condition-fn hints otf-flg
                      (inline 't)
                      trace)

; If condition and condition-fn are both non-nil, then the intent is
; that we do exactly what we would normally do for condition except
; that we use the name condition-fn.

  ":Doc-Section Events

  turn on memoization for one function~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt.
  ~l[hons-and-memoization] for a general discussion of memoization and
  the related features of hash consing and applicative hash tables.

  ~bv[]
  Examples:
  (memoize 'foo)                      ; remember the values of calls
                                      ; of foo
  (memoize 'foo :condition t)         ; same as above
  (memoize 'foo :condition '(test x)) ; memoize for args satisfying
                                      ; condition
  (memoize 'foo :condition 'test-fn)  ; memoize for args satisfying
                                      ; test-fn
  (memoize 'foo :inline nil)          ; do not inline the definition
                                      ; of foo~/

  General Form:
  (memoize fn                         ; memoizes fn and returns fn
           :condition    condition    ; optional (default t)
           :condition-fn condition-fn ; optional
           :hints        hints        ; optional
           :otf-flg      otf-flg      ; optional
           :inline       inline       ; optional (default t)
           )
  ~ev[]
  where ~c[fn] evaluates to a user-defined function symbol;
  ~c[condition] is either ~c[t] (the default)
  or ~c['t] or else evaluates to an expression whose free variables
  are among the formal parameters of ~c[fn]; ~c[condition-fn] is
  either ~c[nil] (the default) or else evaluates to a legal function
  symbol.  Further restrictions are discussed below.

  Generally ~c[fn] must evaluate to a defined function symbol whose
  ~ilc[guard]s have been verified.  However, this value can be the
  name of a macro that is associated with such a function symbol;
  ~pl[macro-aliases-table].  That associated function symbol is the
  one called ``memoized'' in the discussion below, but we make no more
  mention of this subtlety.

  It is illegal to memoize a function that is already memoized.
  To turn off memoization, ~pl[unmemoize].

  In the most common case, ~c[memoize] takes a single argument, which
  evaluates to a function symbol.  We call this function symbol the
  ``memoized function'' because ``memos'' are saved and re-used, in
  the following sense.  When a call of the memoized function is
  evaluated, the result is ``memoized'' by associating the call's
  arguments with that result, in a suitable table.  But first an
  attempt is made to avoid such evaluation, by doing a lookup in that
  table on the given arguments for the result, as stored for a
  previous call on those arguments.  If such a result is found, then
  it is returned without further computation.  This paragraph also
  applies if ~c[:condition] is supplied but is ~c[t] or ~c['t].

  If in addition ~c[:condition-fn] is supplied, but ~c[:condition] is
  not, then the result of evaluating ~c[:condition-fn] must be a
  defined function symbol whose ~il[guard]s have been verified and
  whose formal parameter list is the same as for the function being
  memoized.  Such a ``condition function'' will be run whenever the
  memoized function is called, on the same parameters, and the lookup
  or table store described above are only performed if the result from
  the condition function call is non-~c[nil].

  If however ~c[:condition] is supplied, then an attempt will be made
  to define a condition function whose ~il[guard] and formal
  parameters list are the same as those of the memoized function, and
  whose body is the result, ~c[r], of evaluating the given
  ~c[condition].  The name of that condition function is the result of
  evaluating ~c[:condition-fn] if supplied, else is the result of
  concatenating the string ~c[\"-MEMOIZE-CONDITION\"] to the end of
  the name of the memoized function.  The condition function will be
  defined with ~il[guard] verification turned off, but that definition
  will be followed immediately by a ~ilc[verify-guards] event; and
  this is where the optional ~c[:hints] and ~c[:otf-flg] are attached.
  At evaluation time the condition function is used as described in
  the preceding paragraph; so in effect, the condition (~c[r], above)
  is evaluated, with its variables bound to the corresponding actuals
  of the memoized function call, and the memoized function attempts a
  lookup or table store if and only if the result of that evaluation
  is non-~c[nil].

  Calls of this macro generate events of the form
  ~c[(table memoize-table fn (list condition-fn inline))].  When
  successful, the returned value is of the form
  ~c[(mv nil function-symbol state)].

  When ~c[:inline] has value ~c[nil], then ~c[memoize] does not use
  the definitional body of ~c[fn] in the body of the new, memoized
  definition of ~c[fn].  Instead, ~c[memoize] lays down a call to the
  ~c[symbol-function] for ~c[fn] that was in effect prior to
  memoization.  Use value ~c[t] for ~c[:inline] to avoid memoizing
  recursive calls to ~c[fn] directly from within ~c[fn].

  If ~c[:trace] has a non-~c[nil] value, then ~c[memoize] also traces
  in a traditional Lisp style.  If ~c[:trace] has value ~c[notinline]
  or ~c[notinline], then a corresponding declaration is added at the
  beginning of the new definition of ~c[fn].

  ~c[Memoize] works for functions that have ~il[stobj] arguments or
  return ~il[stobj] values.  However, in some cases the memoized
  version of the function may execute much less inefficiently than the
  unmemoized version.

  ~c[Memoize] causes an error if ~c[fn] uses ~ilc[state] as an
  explicit parameter.~/

  :cite hons-and-memoization"

  (declare (xargs :guard t)
           (ignorable condition-p condition condition-fn hints otf-flg inline
                      trace))
  #-acl2-loop-only
  `(progn (when (eql *ld-level* 0)

; We are not inside the ACL2 loop (hence not in certify-book, for
; example).

            (let ((state *the-live-state*))
              (warning$ 'memoize nil
                        "No change for function ~x0: Memoization ~
                         requests are ignored in raw Lisp.  In raw ~
                         Lisp, use memoize-fn."
                        ',fn)))
          (value-triple nil))
  #+acl2-loop-only
  (cond
   ((and condition-fn (null condition-p))
    `(with-output :off summary
                  (progn (table memoize-table
                                (deref-macro-name ,fn (macro-aliases world))
                                (list ,condition-fn ,inline ,trace))
                         (value-triple (deref-macro-name
                                        ,fn
                                        (macro-aliases (w state)))))))
   ((and condition-p
         (not (eq condition t))
         (not (equal condition ''t)))
    `(with-output
      :off summary
      (make-event
       (let* ((wrld (w state))
              (fn (deref-macro-name ,fn (macro-aliases wrld)))
              (condition ,condition)
              (formals
               (and (symbolp fn) ; guard for getprop
                    (getprop fn 'formals t
                             'current-acl2-world wrld)))
              (condition-fn (or ,condition-fn
                                (intern-in-package-of-symbol
                                 (concatenate
                                  'string
                                  (symbol-name fn)
                                  "-MEMOIZE-CONDITION")
                                 fn)))
              (hints ,hints)
              (otf-flg ,otf-flg)
              (inline ,inline)
              (trace ,trace))
         (cond ((not (and
                      (symbolp fn)
                      (not (eq t formals))
                      (not (eq t (getprop
                                  fn 'stobjs-in t
                                  'current-acl2-world wrld)))
                      (not (eq t (getprop
                                  fn 'stobjs-out t
                                  'current-acl2-world wrld)))
                      (cltl-def-from-name fn nil wrld)))
                (er hard 'memoize
                    "The symbol ~x0 is not a known function symbol, and thus ~
                     it cannot be memoized."
                    fn))
               ((not (eq :common-lisp-compliant
                         (symbol-class fn wrld)))
                (er hard 'memoize
                    "~x0 is not Common Lisp compliant, so is best memoized ~
                     and called from raw Lisp (but raw Lisp should be ~
                     avoiding unless you are hacking)."
                    fn))

; Certify-book seems to do things twice, so the following is commented out.

;                 ((cdr (assoc-eq fn (table-alist 'memoize-table wrld)))
;                  (er hard 'memoize "~x0 is already memoized." fn))

               ((not (member-eq inline '(t nil)))
                (er hard 'memoize
                    "The value ~x0 for inline is illegal (must be ~x1 or ~x2)."
                    inline t nil))
               ((not (member-eq trace '(t nil)))
                (er hard 'memoize
                    "The value ~x0 for trace is illegal (must be ~x1 or ~x2)."
                    trace t nil))
               (t
                `(progn
                   (defun ,condition-fn ,formals
                     (declare
                      (ignorable ,@formals)
                      (xargs :guard
                             ,(getprop fn 'guard *t*
                                       'current-acl2-world wrld)
                             :verify-guards nil))
                     ,condition)
                   (verify-guards ,condition-fn
                                  ,@(and hints `(:hints ,hints))
                                  ,@(and otf-flg
                                         `(:otf-flg ,otf-flg)))
                   (table memoize-table
                          ',fn
                          (list ',condition-fn ',inline ',trace))
                   (value-triple ',fn))))))))
   (t `(with-output
        :off summary
        (progn (table memoize-table
                      (deref-macro-name ,fn (macro-aliases world))
                      (list t ,inline ,trace))
               (value-triple (deref-macro-name
                              ,fn
                              (macro-aliases (w state)))))))))

(defmacro unmemoize (fn)

  ":Doc-Section Events

  turn off memoization for the specified function~/
  ~bv[]
  Example:
  (unmemoize 'foo) ; turn off memoization of foo~/

  General Form:
  (unmemoize fn)
  ~ev[]

  where ~c[fn] evaluates to a function symbol that is currently
  memoized; ~pl[memoize].  An exception is that as with ~ilc[memoize],
  ~c[fn] may evaluate to the name of a macro that is associated with
  such a function symbol; ~pl[macro-aliases-table].

  Calls of this macro generate events of the form
  ~c[(table memoize-table fn nil)].  When successful, the returned
  value is of the form ~c[(mv nil function-symbol state)]."

  (declare (xargs :guard t))
  #-acl2-loop-only
  `(progn (when (eql *ld-level* 0)

; We are not inside the ACL2 loop (hence not in certify-book, for
; example).

            (warning$ 'unmemoize nil
                      "No change for function ~x0: Unmemoization ~
                       requests are ignored in raw Lisp.  In raw ~
                       Lisp, use unmemoize-fn."
                      ',fn))
          (value-triple nil))
  #+acl2-loop-only
  `(with-output
    :off summary
    (progn (table memoize-table
                  (deref-macro-name ,fn (macro-aliases world)) nil)
           (value-triple
            (deref-macro-name ,fn (macro-aliases (w state)))))))

#+(or acl2-loop-only (not hons))
(defn memoize-on (fn x)

; MEMOIZE-ON evaluates x.  During the evaluation the symbol fn has as
; its symbol-function what it had immediately AFTER the memoization of
; fn.  Hence, the values of calls of fn may be remembered during the
; evaluation and later.  Warning: to use MEMOIZE-ON, fn must already
; be memoized.

  (declare (ignore fn))
  x)

#+(or acl2-loop-only (not hons))
(defn memoize-off (fn x)

; MEMOIZE-OFF evaluates x.  During the evaluation the symbol fn has as
; its symbol-function what it had immediately BEFORE the memoization
; of fn.  Hence the values of calls of fn may not be remembered during
; the evaluation.  Warning: to use MEMOIZE-OFF, fn must already be
; memoized."

  (declare (ignore fn))
  x)

(defmacro memoizedp (fn)
  (declare (xargs :guard t))
  `(cond
    ((not (hons-enabledp state))
     (er hard 'memoizedp
         "Memoizedp cannot be called in this ACL2 image, as it requires a ~
          hons-aware ACL2.  See :DOC hons-and-memoization."))
    (t
     (cdr (assoc-eq ,fn (table-alist 'memoize-table (w state)))))))

;;; hons-shrink-alist

; HONS-SHRINK-ALIST, when called with an atomic second
; argument, produces an alist that is alist-equivalent
; to the first argument, but with all irrelevant entries in
; the first argument deleted.  Informal remark: the alist
; returned is a hons when the initial ANS is not an atom.

#|| Comment about the last clause above.  Or really? 
Counterexamples?

mbu> stp
? (honsp (hons-shrink-alist '((a . b) (a . b2)) (hons-acons 1 2 3)))
NIL

mbu> stp
? (honsp (hons-shrink-alist '((a . b) (a . b2)) nil))
NIL
? 

||#


#+(or acl2-loop-only (not hons))
(defn hons-shrink-alist! (alst ans)
  (cond
   ((atom alst) ans)
   ((atom (car alst)) (hons-shrink-alist! (cdr alst) ans))
   (t (let ((p (hons-get (car (car alst)) ans)))
        (cond (p (hons-shrink-alist! (cdr alst) ans))
              (t (hons-shrink-alist! (cdr alst)
                                     (hons-acons! (car (car alst))
                                                  (cdr (car alst))
                                                  ans))))))))

#+(or acl2-loop-only (not hons))
(defn hons-shrink-alist (alst ans)
  (cond ((atom alst) ans)
        ((atom (car alst)) (hons-shrink-alist (cdr alst) ans))
        (t (let ((p (hons-get (car (car alst)) ans)))
             (cond (p (hons-shrink-alist (cdr alst) ans))
                   (t (hons-shrink-alist (cdr alst)
                                         (hons-acons (car (car alst))
                                                     (cdr (car alst))
                                                     ans))))))))

(defconst *untroublesome-characters*

  (coerce "~/:{}=+-_.,!?@#%^%<>|&$*`\\/\";()" 'list)

  ":Doc-Section Hons-and-Memoization
  Some not so weird characters.~/

  *UNTROUBLESOME-CHARACTERS* should not include single quote,
  whitespace characters, control characters, or anything else weird
  for a file name.  Such characters should only be used in file names
  by experts at quoting.~/~/")

(defn cons-subtrees (x al)

  ":Doc-Section Hons-and-Memoization

  Computes the hons graph of a term~/

  (cons-subtrees x NIL) is an alist that associates each subtree of
  x with t, without duplication.  It may run slower than
  FASTER-CONS-SUBTREES.~/~/"

  (cond ((atom x) al)
        ((hons-assoc-equal x al) al)
        (t (cons-subtrees
            (car x)
            (cons-subtrees (cdr x)
                           (hons-acons x t al))))))

(defn faster-cons-subtrees (x al)

  ;; FASTER-CONS-SUBTREES x returns the same value as CONS-SUBTREES

  (cond ((atom x) al)
        ((hons-get x al) al)
        (t (faster-cons-subtrees
            (car x)
            (faster-cons-subtrees (cdr x)
                                  (hons-acons x t al))))))

#+(or acl2-loop-only (not hons))
(defn number-subtrees (x)

  ;; Has an "under the hood" implementation.
  (len (cons-subtrees x 'number-subtrees)))


#+(or acl2-loop-only (not hons))
(defn honsp-check (x)

; Logically, this function is (or (consp x) (stringp x)).  However, it
; causes an error if x a cons but not a hons, or if x is a string that
; isn't hashed.

  ;; Has an "under the hood" implementation.
  (or (consp x) (stringp x)))

; For some additional helper functions and lemmas, see the files
; books/misc/hons-help.lisp and books/misc/hons-help2.lisp.

(deftheory definition-minimal-theory
  (definition-runes
    *definition-minimal-theory*
    nil
    world))

(deftheory executable-counterpart-minimal-theory
  (definition-runes
    *built-in-executable-counterparts*
    t
    world))

(deftheory minimal-theory

; Warning: The resulting value must be a runic-theoryp.  See
; theory-fn-callp.

; Keep this definition in sync with translate-in-theory-hint.

  (union-theories (theory 'definition-minimal-theory)
                  (union-theories

; Without the :executable-counterpart of force, the use of (theory
; 'minimal-theory) will produce the warning "Forcing has transitioned
; from enabled to disabled", at least if forcing is enabled (as is the
; default).

                   '((:executable-counterpart force))
                   (theory 'executable-counterpart-minimal-theory)))
  :doc
  ":Doc-Section Theories

  a minimal theory to enable~/~/

  This ~ilc[theory] (~pl[theories]) enables only a few built-in
  functions and executable counterparts.  It can be useful when you
  want to formulate lemmas that rather immediately imply the theorem
  to be proved, by way of a ~c[:use] hint (~pl[hints]), for example as
  follows.
  ~bv[]
  :use (lemma-1 lemma-2 lemma-3)
  :in-theory (union-theories '(f1 f2) (theory 'minimal-theory))
  ~ev[]
  In this example, we expect the current goal to follow from lemmas
  ~c[lemma-1], ~c[lemma-2], and ~c[lemma-3] together with rules ~c[f1]
  and ~c[f2] and some obvious facts about built-in functions (such as
  the ~il[definition] of ~ilc[implies] and the
  ~c[:]~ilc[executable-counterpart] of ~ilc[car]).  The
  ~c[:]~ilc[in-theory] hint above is intended to speed up the proof by
  turning off all inessential rules.~/

  :cited-by theory-functions")

(deftheory ground-zero (current-theory :here)

; WARNING: Keep this near the end of *acl2-pass-2-files* in order for
; the ground-zero theory to be as expected.

  :doc
  ":Doc-Section Theories

  ~il[enable]d rules in the ~il[startup] theory~/

  ACL2 concludes its initialization ~c[(boot-strapping)] procedure by
  defining the theory ~c[ground-zero]; ~pl[theories].  In fact, this
  theory is just the theory defined by ~c[(current-theory :here)] at
  the conclusion of initialization; ~pl[current-theory].~/

  Note that by evaluating the event
  ~bv[]
  (in-theory (current-theory 'ground-zero))
  ~ev[]
  you can restore the current theory to its value at the time you
  started up ACL2.~/

  :cited-by theory-functions")

(deflabel

; WARNING: Put this at the end of the last file in
; *acl2-pass-2-files*.

  end-of-pass-2)
