;; Mechanization of Jade Alglave's ``Herding cats'' paper

(in-package "ACL2")

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; CONSTRAINED FUNCTIONS ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Here, we introduce the constrained functions corresponding to our basic
; concepts of events, program order, and the rf and co relations.

(encapsulate
 (((writep *) => *)
  ((readp *) => *)

  ((addr *) => *)
  ((proc *) => *)

  ((po * *) => *)

  ((rf * *) => *)
  ((co * *) => *)
  
  ((rf-inv-fn *) => *))

 (local (defun writep (x) 
          ;; everything is a write
          (declare (ignore x))
          t))
 (local (defun readp (x)
          (declare (ignore x))
          nil))

 (local (defun addr (x)
          (declare (ignore x))
          0))
 (local (defun proc (x)
          (declare (ignore x))
          0))

 (defthm writep-booleanp (booleanp (writep x))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription writep-booleanp)))

 (defthm readp-booleanp (booleanp (readp x))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription readp-booleanp))) 

 ;; things can't be both reads and writes

 ; disable?
 (defthm read-write-exclusive
   (implies (readp x)
            (not (writep x)))
   :rule-classes :forward-chaining)

 (local (defun po (x y)
          (and (lexorder x y)
               (equal (proc x) (proc y))
               (not (equal x y)))))

 (local (defun rf (x y)
          (and (writep x)
               (readp y)
               (equal (addr x) (addr y))
               (lexorder x y))))
 (local (defun co (x y) 
          (and (lexorder x y)
               (equal (addr x) (addr y))
               (not (equal x y)))))

 (defthm po-booleanp (booleanp (po x y))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription po-booleanp))) 

 (defthm rf-booleanp (booleanp (rf x y))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription rf-booleanp))) 

 (defthm co-booleanp (booleanp (co x y))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription co-booleanp))) 

 ;; rf is invertible, so we need a function that takes a read and finds the
 ;; write it reads from
 (local (defun rf-inv-fn (x) 
          ;; everything is a write, so this function doesn't
          ;; need to do anything
          (not x)))

 ;; po rules
 (defthmd po-proc
   (implies (po x y)
            (equal (proc x) (proc y))))
 (defthm po-read-or-write-1
   (implies (and (po x y)
                 (not (readp x)))
            (writep x)))
 (defthm po-read-or-write-2
   (implies (and (po x y)
                 (not (readp y)))
            (writep y)))
 (defthm po-irreflexive
   (not (po x x)))
 (defthm po-transitive
   (implies (and (po x y)
                 (po y z))
            (po x z))
   :rule-classes ((:rewrite :match-free :all)))
 (defthm po-total
   (implies (and (equal (proc x) (proc y))
                 (not (equal x y))
                 (not (po x y)))
            (po y x)))

 ;; rf-inv-fn rules
 (defthm rf-inv-fn-irreflexive
   (not (equal (rf-inv-fn x) x)))
 (defthm rf-inv-fn-read-write
   (implies (readp r) (writep (rf-inv-fn r))))

 ;; rf rules
 
 ; We make this a forward chaining rule since it's needed so often; could be a
 ; bad idea but seems harmless for now
 ; disable?
 (defthm rf-write-read
   (implies (rf w r)
            (and (writep w)
                 (readp r)))
   :rule-classes :forward-chaining)
 (defthmd rf-addr
   (implies (rf w r)
            (equal (addr w) (addr r)))
   :rule-classes :forward-chaining)
 ; Make the following a forward-chaining rule?
 (defthmd rf-rf-inv-fn
   (equal (readp r)
          (rf (rf-inv-fn r) r)))
 (defthm rf-write-unique
   (implies (syntaxp (not (equal w `(rf-inv-fn ,x))))
            (equal (rf w x)
                   (and (readp x)
                        (equal w (rf-inv-fn x))))))

 ;; co rules
 ; We make this a forward chaining rule since it's needed so often; could be a
 ; bad idea but seems harmless for now
 ; disable?
 (defthm co-write-write
   (implies (co x y)
            (and (writep x)
                 (writep y)))
   :rule-classes :forward-chaining)
 (defthmd co-addr
   (implies (co x y)
            (equal (addr x) (addr y)))
   :rule-classes :forward-chaining)
 ;; what rewrite rule will this generate? (co x x) -> nil is intended
 (defthm co-irreflexive
   (not (co x x)))
 (defthm co-transitive
   (implies (and (co x y)
                 (co y z))
            (co x z)))
;   :rule-classes ((:rewrite :match-free :all)))
 (defthm co-total
   (implies (and (writep x)
                 (writep y)
                 (equal (addr x) (addr y))
                 (not (equal x y))
                 (not (co x y)))
            (co y x))))

;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT PO ;;
;;;;;;;;;;;;;;;;;;;;

; Is there a way to prove this without a hint?
(defthm po-asymmetric
  (implies (po x y)
           (not (po y x)))
  :hints (("Goal" 
           :use ((:instance po-transitive
                            (x x)
                            (y y)
                            (z x))))))

;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT RF ;;
;;;;;;;;;;;;;;;;;;;;

; Note that (rf x y) always gets rewritten to (and (readp y) (equal x
; (rf-inv-fn y), so there's no real reason to prove facts directly about rf.

(defthm rf-inv-fn-addr
  (implies (readp x)
           (equal (addr (rf-inv-fn x)) (addr x)))
  :hints (("Goal"
           :in-theory (enable rf-rf-inv-fn rf-addr)))
  :rule-classes :forward-chaining)

;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT CO ;;
;;;;;;;;;;;;;;;;;;;;

; Is there a way to prove this without a hint?
(defthm co-asymmetric
  (implies (co x y)
           (not (co y x)))
  :hints (("Goal" :use ((:instance co-transitive
                                   (x x)
                                   (y y)
                                   (z x))))))

;; <talk>
;;;;;;;;;;;;;;;;;;;;;;
;; DEFINITION OF FR ;;
;;;;;;;;;;;;;;;;;;;;;;

(defun-sk fr (x z)
  (exists y
          (and (rf y x) (co y z))))

; The following theorem guarantees that ACL2 never reasons about fr directly.
(defthm fr-rewrite
  (equal (fr x z)
         (and (rf (rf-inv-fn x) x)
              (co (rf-inv-fn x) z)))
  :hints (("Goal" :use ((:instance fr-suff
                                   (y (rf-inv-fn x)))))))

(in-theory (disable fr))

;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT FR ;;
;;;;;;;;;;;;;;;;;;;;

; disable?
(defthm fr-read-write
  (implies (fr x y)
           (and (readp x)
                (writep y)))
  :rule-classes :forward-chaining)

;;;;;;;;;;;;;;;;;;;;;;;
;; DEFINITION OF COM ;;
;;;;;;;;;;;;;;;;;;;;;;;

(defun com (x y)
  (or (rf x y)
      (co x y)
      (fr x y)))

;;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT COM ;;
;;;;;;;;;;;;;;;;;;;;;

(defthm com-irreflexive
  (not (com x x)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; RELATIONSHIP BETWEEN RF, CO, FR ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Necessary?
(defthm rf-implies-not-fr
  (implies (rf x y)
           (not (fr y x))))

(defthm rf-fr
  (implies (and (rf x z)
                (fr z y))
           (co x y)))

(defthm fr-co
  (implies (and (fr x z)
                (co z y))
           (fr x y)))

;; <talk>
;;;;;;;;;;;;;;;;;;;;;;;;;;
;; DEFINITION OF CO->RF ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

;; co sequenced with rf
(defun-sk co->rf (x z)
  (exists y
    (and (co x y) (rf y z))))

;;;;;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT CO->RF ;;
;;;;;;;;;;;;;;;;;;;;;;;;

(defthm co->rf-booleanp
  (booleanp (co->rf x y))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription co->rf-booleanp)))

(defthm co->rf-write-read
  (implies (co->rf x y)
           (and (writep x)
                (readp y)))
  :rule-classes :forward-chaining)

;; <talk>
;;;;;;;;;;;;;;;;;;;;;;;;;;
;; DEFINITION OF FR->RF ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

;; fr sequenced with rf
(defun-sk fr->rf (x z)
  (exists y
    (and (fr x y) (rf y z))))

;;;;;;;;;;;;;;;;;;;;;;;;
;; FACTS ABOUT FR->RF ;;
;;;;;;;;;;;;;;;;;;;;;;;;

(defthm fr->rf-booleanp
  (booleanp (fr->rf x y))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription fr->rf-booleanp)))

(defthm fr->rf-irreflexive
  (not (fr->rf x x)))

; disable?
(defthm fr->rf-read-read
  (implies (fr->rf x y)
           (and (readp x)
                (readp y)))
  :rule-classes :forward-chaining)

(defthm fr->rf-different-writes
  (implies (and (fr->rf x y)
                (rf w x))
           (not (rf w y))))

(defthm fr->rf-transitive
  (implies (and (fr->rf x y)
                (fr->rf y z))
           (fr->rf x z))
  :hints (("Goal"
           :use ((:instance fr->rf-suff
                            (x x)
                            (y (rf-inv-fn z))
                            (z z))))))

(defthm fr->rf-asymmetric
  (implies (fr->rf x y)
           (not (fr->rf y x))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; REWRITE FR->RF (DISABLED) ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We prove a useful rewrite rule about fr->rf, but leave it disabled since we
;; generally want to keep fr->rf around

(local
 (defthmd fr->rf-rewrite-lemma1
   (implies (fr->rf x y)
            (and (readp x)
                 (readp y)
                 (co (rf-inv-fn x) (rf-inv-fn y))))))

(local
 (defthmd fr->rf-rewrite-lemma2
   (implies (and (readp x)
                 (readp y)
                 (co (rf-inv-fn x) (rf-inv-fn y)))
            (fr->rf x y))
   :hints (("Goal" 
            :in-theory (enable rf-rf-inv-fn)
            :use ((:instance fr->rf-suff
                             (x x)
                             (y (rf-inv-fn y))
                             (z y)))))))

(in-theory (enable (:type-prescription co-booleanp)
                   (:type-prescription fr->rf-booleanp)))

(local
 (defthmd fr->rf-rewrite-iff
   (iff (fr->rf x y)
        (and (readp x)
             (readp y)
             (co (rf-inv-fn x) (rf-inv-fn y))))
   :hints (("Goal" 
            :in-theory (e/d (fr->rf-rewrite-lemma1
                             fr->rf-rewrite-lemma2)
                            (fr->rf))))))

(in-theory (disable (:type-prescription co-booleanp)
                    (:type-prescription fr->rf-booleanp)))

(defthmd fr->rf-rewrite
  (equal (fr->rf x y)
         (and (readp x)
              (readp y)
              (co (rf-inv-fn x) (rf-inv-fn y))))
  :hints (("Goal"
           :in-theory (e/d (fr->rf-rewrite-iff) (fr->rf)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; COMPOSING RF, CO, FR, CO->RF, FR->RF ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm co-co->rf
  (implies (and (co x y)
                (co->rf y z))
           (co->rf x z))
  :hints (("Goal"
           :use ((:instance co->rf-suff
                            (x x)
                            (y (rf-inv-fn z))
                            (z z))))))

(defthm rf-fr->rf
  (implies (and (rf x y)
                (fr->rf y z))
           (co->rf x z))
  :hints (("Goal"
           :use ((:instance co->rf-suff
                            (x x)
                            (y (rf-inv-fn z))
                            (z z))))))

(defthm fr->rf-fr
  (implies (and (fr->rf x y)
                (fr y z))
           (fr x z)))

(defthm fr-co->rf
  (implies (and (fr x y)
                (co->rf y z))
           (fr->rf x z))
  :hints (("Goal"
           :use ((:instance fr->rf-suff
                            (x x)
                            (y (rf-inv-fn z))
                            (z z))))))

(defthm co->rf-fr->rf
  (implies (and (co->rf x y)
                (fr->rf y z))
           (co->rf x z))
  :hints (("Goal"
           :use ((:instance co->rf-suff
                            (x x)
                            (y (rf-inv-fn z))
                            (z z))))))

(defthm co->rf-fr
  (implies (and (co->rf x y)
                (fr y z))
           (co x z)))

(defthm fr->rf-and-rf
  (implies (and (rf x y)
                (fr->rf y z))
           (co->rf x z))
  :hints (("Goal"
           :use ((:instance co->rf-suff
                            (x (rf-inv-fn y))
                            (y (rf-inv-fn z))
                            (z z))))))

(defthm co->rf-and-rf
  (implies (and (rf x y)
                (co->rf x z))
           (fr->rf y z))
  :hints (("Goal"
           :use ((:instance fr->rf-suff
                            (x y)
                            (y (rf-inv-fn z))
                            (z z))
                 (:instance rf-rf-inv-fn
                            (r y))))))

; Necessary?
(defthm fr-implies-not-co->rf
  (implies (fr x y)
           (not (co->rf y x))))

;;;;;;;;;;;;;;;;;;;;;;;
;; TOTALITY THEOREMS ;;
;;;;;;;;;;;;;;;;;;;;;;;

;; The first totality theorem is transitivity of the co relation, which is
;; assumed; the other two follow from this.

(defthm write-read-totality
  (implies (and (readp r)
                (writep w)
                (equal (addr r) (addr w))
                (not (co->rf w r))
                (not (rf w r)))
           (fr r w))
  :hints (("Goal" 
           :in-theory (enable rf-rf-inv-fn rf-addr)
           :use ((:instance fr-suff
                            (x r)
                            (y (rf-inv-fn r))
                            (z w))
                 (:instance co->rf-suff
                            (x w)
                            (y (rf-inv-fn r))
                            (z r))))))

(defthm read-read-totality
  (implies (and (readp r1)
                (readp r2)
                (equal (addr r1) (addr r2))
                (not (fr->rf r1 r2))
                (not (equal (rf-inv-fn r1) (rf-inv-fn r2))))
           (fr->rf r2 r1))
  :hints (("Goal"
           :in-theory (enable rf-rf-inv-fn)
           :use ((:instance fr-suff
                            (x r1)
                            (y (rf-inv-fn r1))
                            (z (rf-inv-fn r2)))
                 (:instance fr->rf-suff
                            (x r1)
                            (y (rf-inv-fn r2))
                            (z r2))
                 (:instance fr-suff
                            (x r2)
                            (y (rf-inv-fn r2))
                            (z (rf-inv-fn r1)))
                 (:instance fr->rf-suff
                            (x r2)
                            (y (rf-inv-fn r1))
                            (z r1))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;; DEFINITION OF COM+ ;;
;;;;;;;;;;;;;;;;;;;;;;;;

(defun com-pathp (path x y)
  (cond ((endp path) (com x y))
        (t (and (com x (car path))
                (com-pathp (cdr path) (car path) y)))))

(defthm com-pathp-booleanp
  (booleanp (com-pathp path x y))
   :rule-classes (:rewrite :type-prescription))
 (in-theory (disable (:type-prescription com-pathp-booleanp)))

;; <talk>
;; And now define the irreflexive-transitive closure by quantifying over path
(defun-sk com+ (x y)
  (exists path (com-pathp path x y)))

(defthm com+-booleanp
  (booleanp (com+ x y))
  :rule-classes (:rewrite :type-prescription))
(in-theory (disable (:type-prescription com+-booleanp)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ALL RELATIONS IMPLY COM-PATHP ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm com-implies-com-pathp
  (implies (com x y)
           (com-pathp nil x y)))

(defthm co->rf-implies-com-pathp
  (implies (co->rf x y)
           (com-pathp (list (rf-inv-fn y)) x y)))

(defthm fr->rf-implies-com-pathp
  (implies (fr->rf x y)
           (com-pathp (list (rf-inv-fn y)) x y)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ALL RELATIONS IMPLY COM+ ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm com-implies-com+
  (implies (com x y)
           (com+ x y))
  :hints (("Goal"
           :use ((:instance com+-suff
                            (path nil))))))

(defthm co->rf-implies-com+
  (implies (co->rf x y)
           (com+ x y))
  :hints (("Goal"
           :use ((:instance com+-suff
                            (path (list (rf-inv-fn y))))))))

(defthm fr->rf-implies-com+
  (implies (fr->rf x y)
           (com+ x y))
  :hints (("Goal"
           :use ((:instance com+-suff
                            (path (list (rf-inv-fn y))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ALTERNATE DEFINITION OF COM+ ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Now we can describe com+ as the union of three relations: com, co->rf (co
;; sequenced with rf), and fr->rf.

(defun com+-alt (x y)
  (or (com x y)
      (co->rf x y)
      (fr->rf x y)))

(defthm com+-alt-booleanp
  (booleanp (com+-alt x y))
  :hints (("Goal" :in-theory (enable com+-alt)))
  :rule-classes (:rewrite :type-prescription))

(in-theory (disable (:type-prescription com+-alt-booleanp)))

(in-theory (disable co->rf fr->rf fr-rewrite rf-write-unique))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; PROPERTIES OF COM+-ALT ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm com+-alt-irreflexive
  (not (com+-alt x x)))

(defthm com+-alt-transitive
  (implies (and (com+-alt x y)
                (com+-alt y z))
           (com+-alt x z)))

(defthm com+-alt-asymmetric
  (implies (com+-alt x y)
           (not (com+-alt y x))))

(defthm com+-alt-read-or-write
  (implies (com+-alt x y)
           (and (or (readp x) (writep x))
                (or (readp y) (writep y))))
  :rule-classes :forward-chaining)

(defthm com+-alt-addr
  (implies (com+-alt x y)
           (equal (addr x) (addr y)))
  :hints (("Goal" 
           :in-theory (enable fr->rf-rewrite 
                              fr-rewrite
                              co->rf
                              rf-addr
                              co-addr
                              rf-inv-fn-addr)))
  :rule-classes :forward-chaining)

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; TOTALITY OF COM+-ALT ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm com+-alt-totality
  (implies (and (or (readp x) (writep x))
                (or (readp y) (writep y))
                (equal (addr x) (addr y))
                (not (com+-alt x y))
                (not (and (writep x)
                          (writep y)
                          (equal x y)))
                (not (and (readp x)
                          (readp y)
                          (equal (rf-inv-fn x) (rf-inv-fn y)))))
           (com+-alt y x)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; REWRITE COM+ TO COM+-ALT ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm com-pathp-characterization
  (implies (com-pathp path x y)
           (com+-alt x y)))

(defthm com+-characterization
  (implies (com+ x y)
           (com+-alt x y)))

(in-theory (disable com-pathp com+))

;; TODO: WTF is going on here? Why does rewrite-com+-iff mean something
;; different depending on which type-prescription rules are enabled vs. disabled?

(in-theory (enable (:type-prescription com+-booleanp)
                   (:type-prescription com+-alt-booleanp)))
 
(defthmd rewrite-com+-iff
  (iff (com+ x y)
       (com+-alt x y)))

(in-theory (disable (:type-prescription com+-booleanp)
                    (:type-prescription com+-alt-booleanp)))

(defthm rewrite-com+
  (equal (com+ x y)
         (com+-alt x y))
  :hints (("Goal"
           :in-theory (enable rewrite-com+-iff))))
