; In this book we give three definitions of sequential consistency per location
; and prove they are all equivalent.

(in-package "ACL2")

(include-book "executions")

;;;;;;;;;;;;;;;;;;;;;
;; SC Per Location ;;
;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;
;; POL DEFINITION ;;
;;;;;;;;;;;;;;;;;;;;;;;

(defun pol (x y)
  (and (po x y)
       (equal (addr x) (addr y))))

(defthm pol-read-or-write-1
  (implies (and (pol x y)
                (not (readp x)))
           (writep x)))

(defthm pol-read-or-write-2
  (implies (and (pol x y)
                (not (readp y)))
           (writep y)))

(defthm pol-read-or-write-fc
  (implies (pol x y)
           (and (or (readp x) (writep x))
                (or (readp y) (writep y))))
  :rule-classes :forward-chaining)

(defthm pol-addr
  (implies (pol x y)
           (equal (addr x) (addr y)))
  :rule-classes :forward-chaining)

; The union of the pol and com relations
(defun pol-com (x y)
  (or (pol x y)
      (com x y)))

(defun pol-com-pathp (path x y)
  (cond ((endp path) (pol-com x y))
        (t (and (pol-com x (car path))
                (pol-com-pathp (cdr path) (car path) y)))))
 
(defun pol-com-cyclep (cycle x)
  (pol-com-pathp cycle x x))

; The transitive closure of pol-com
(defun pol-com+ (x y)
  (or (pol x y)
      (com+ x y)))

; We also define cycles in pol-com+, which will be easier to work with in
; our proof that Def2 => Def1. We can't necessarily shorten two adjacent coms
; into one com, but we can shorten to adjacent com+'s to one com+.
(defun pol-com+-pathp (path x y)
  (cond ((endp path) (pol-com+ x y))
        (t (and (pol-com+ x (car path))
                (pol-com+-pathp (cdr path) (car path) y)))))
 
(defun pol-com+-cyclep (cycle x)
  (pol-com+-pathp cycle x x))

; We also need to know that a cycle in pol-com is a cycle in
; pol-com+. Seems obvious but need to know it as fact so we can prove Def2
; => Def1.
(defthm pol-com-implies-pol-com+
  (implies (pol-com x y)
           (pol-com+ x y))
  :hints (("Goal" :in-theory (enable com+-alt))))

(defthm pol-com-pathp-implies-pol-com+-pathp
  (implies (pol-com-pathp cycle x y)
           (pol-com+-pathp cycle x y))
  :hints (("Goal"
           :in-theory (disable pol-com
                               pol-com+))))

(defthm pol-com-cyclep-implies-pol-com+-cyclep
  (implies (pol-com-cyclep cycle x)
           (pol-com+-cyclep cycle x))
  :hints (("Goal"
           :do-not-induct t
           :in-theory (disable pol-com-pathp
                               pol-com+-pathp))))

;;;;;;;;;;;;;;;;;;
;; POL FACTS ;;
;;;;;;;;;;;;;;;;;;

(defthm pol-irreflexive
  (not (pol x x)))

(defthm pol-transitive
  (implies (and (pol x y)
                (pol y z))
           (pol x z))
  :rule-classes ((:rewrite :match-free :all)))

(defthm pol-asymmetric
  (implies (pol x y)
           (not (pol y x))))

(defthm pol-total
  (implies (and (equal (proc x) (proc y))
                (equal (addr x) (addr y))
                (not (equal x y))
                (not (pol x y)))
           (pol y x)))

;;;;;;;;;;;;;;;;;;;;
;; HAT DEFINITION ;;
;;;;;;;;;;;;;;;;;;;;

(defun-sk hat (r1 r2)
  (exists w
          (and (rf w r1) (rf w r2))))

(defthm hat-booleanp
  (booleanp (hat r1 r2))
  :rule-classes (:rewrite :type-prescription))
(in-theory (disable (:type-prescription hat-booleanp)))

(defthm hat-reflexive
  (implies (readp r)
           (hat r r))
  :hints (("Goal" :in-theory (enable rf-rf-inv-fn))))

(defthm hat-symmetric
  (implies (hat x y) (hat y x)))

(defthm hat-transitive
  (implies (and (hat x y)
                (hat y z))
           (hat x z))
  :hints (("Goal" 
           :in-theory (enable rf-write-unique)
           :use ((:instance hat-suff
                            (r1 x)
                            (r2 z)
                            (w (rf-inv-fn x))))))
  :rule-classes ((:rewrite :match-free :all)))

(local
 (defthmd hat-rewrite-1
   (implies (and (readp x)
                 (readp y)
                 (equal (rf-inv-fn x)
                        (rf-inv-fn y)))
            (hat x y))
   :hints (("Goal" 
            :in-theory (enable rf-rf-inv-fn)))))

(local 
 (defthmd hat-rewrite-2
   (implies (hat x y)
            (and (readp x)
                 (readp y)
                 (equal (rf-inv-fn x)
                        (rf-inv-fn y))))
   :hints (("Goal" :in-theory (enable rf-write-unique)))))

(in-theory (enable (:type-prescription hat-booleanp)))

(local
 (defthmd hat-rewrite-iff
   (iff (hat x y)
        (and (readp x)
             (readp y)
             (equal (rf-inv-fn x)
                    (rf-inv-fn y))))
 :hints (("Goal" :in-theory (e/d (hat-rewrite-1
                                  hat-rewrite-2)
                                 (hat))))))

(in-theory (disable (:type-prescription hat-booleanp)))

(defthm hat-rewrite
  (equal (hat x y)
         (and (readp x)
              (readp y)
              (equal (rf-inv-fn x)
                     (rf-inv-fn y))))
  :hints (("Goal"
           :in-theory (e/d (hat-rewrite-iff)
                           (hat)))))

(in-theory (disable hat))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; READ-READ TOTALITY THEOREM FOR HAT ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm fr->rf-total-wrt-hat
  (implies (and (readp x)
                (readp y)
                (equal (addr x) (addr y))
                (not (fr->rf x y))
                (not (hat x y)))
           (fr->rf y x)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;; COM+-HAT DEFINITION ;;
;;;;;;;;;;;;;;;;;;;;;;;;;

(defun com+-hat (x y)
  (or (com+ x y) (hat x y)))

;;;;;;;;;;;;;;;;;;;
;; SOME THEOREMS ;;
;;;;;;;;;;;;;;;;;;;

(defthm hat-implies-not-com+
  (implies (hat x y)
           (not (com+ y x)))
  :hints (("Goal"
           :in-theory (enable fr->rf-rewrite))))

;; <talk>
;;;;;;;;;;;;;;;;;;
;; DEFINITION 1 ;;
;;;;;;;;;;;;;;;;;;

(defun-sk sc-per-location-1 ()
  (forall (x potential-cycle)
          (not (pol-com-cyclep potential-cycle x))))

;;;;;;;;;;;;;;;;;;
;; DEFINITION 2 ;;
;;;;;;;;;;;;;;;;;;

(defun-sk sc-per-location-2 ()
  (forall (x y)
          (implies (pol x y)
                   (not (com+ y x)))))

;;;;;;;;;;;;;;;;;;
;; DEFINITION 3 ;;
;;;;;;;;;;;;;;;;;;

(defun-sk sc-per-location-3 ()
  (forall (x y)
          (implies (pol x y)
                   (com+-hat x y))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; EQUIVALENCE OF THREE DEFINITIONS OF SC-PER-LOCATION ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;
; Def3 => Def2 ;;
;;;;;;;;;;;;;;;;;

(defthm sc-per-location-3-unquantified
  (implies (and (sc-per-location-3)
                (pol x y))
           (com+-hat x y))
  :hints (("Goal" 
           :use ((:instance sc-per-location-3-necc)))))

(defthmd sc-per-location-3-lemma1
  (implies (and (com+-hat x y)
                (pol x y))
           (not (com+ y x)))
  :hints (("Goal"
           :in-theory (enable fr->rf-rewrite))))

(defthm sc-per-location-3-lemma2
  (implies (and (sc-per-location-3)
                (pol x y))
           (not (com+ y x)))
  :hints (("Goal" 
           :in-theory (disable sc-per-location-3
                               pol
                               com+)
           :use ((:instance sc-per-location-3-lemma1)))))

(defthm sc-per-location-3-implies-2-unquantified
  (implies (and (sc-per-location-3)
                (pol x y))
           (not (com+ y x)))
  :hints (("Goal" :in-theory (disable rewrite-com+
                                      com+-hat))))

(defthm sc-per-location-3-implies-2
  (implies (sc-per-location-3)
           (sc-per-location-2))
  :hints (("Goal" 
           :use ((:instance sc-per-location-3-implies-2-unquantified
                            (x (mv-let (x y)
                                       (sc-per-location-2-witness)
                                       (declare (ignore y))
                                       x))
                            (y (mv-let (x y)
                                       (sc-per-location-2-witness)
                                       (declare (ignore x))
                                       y)))))))

;;;;;;;;;;;;;;;;;;
;; Def2 => Def3 ;;
;;;;;;;;;;;;;;;;;;

(defthm sc-per-location-2-unquantified
  (implies (and (sc-per-location-2)
                (pol x y))
           (not (com+ y x)))
  :hints (("Goal" 
           :use ((:instance sc-per-location-2-necc)))))

(defthmd sc-per-location-2-lemma1
  (implies (and (or (readp x) (writep x))
                (or (readp y) (writep y))
                (not (com+ y x))
                (equal (addr x) (addr y))
                (not (equal x y)))
           (com+-hat x y))
  :hints (("Goal"
           :in-theory (disable hat-rewrite))))

(defthm sc-per-location-2-lemma2
  (implies (and (not (com+ y x))
                (pol x y))
           (com+-hat x y))
  :hints (("Goal" 
           :in-theory (disable hat-rewrite com+-hat rewrite-com+)
           :use ((:instance sc-per-location-2-lemma1)))))

(defthm sc-per-location-2-implies-3-unquantified
  (implies (and (sc-per-location-2)
                (pol x y))
           (com+-hat x y))
  :hints (("Goal"
           :in-theory (disable com+-hat rewrite-com+ pol sc-per-location-2))))

(defthm sc-per-location-2-implies-3
  (implies (sc-per-location-2)
           (sc-per-location-3))
  :hints (("Goal"
           :use ((:instance sc-per-location-2-implies-3-unquantified
                            (x (mv-let (x y)
                                       (sc-per-location-3-witness)
                                       (declare (ignore y))
                                       x))
                            (y (mv-let (x y)
                                       (sc-per-location-3-witness)
                                       (declare (ignore x))
                                       y)))))))

;;;;;;;;;;;;;;;;;;
;; Def1 => Def2 ;;
;;;;;;;;;;;;;;;;;;

(defthm pol-com-not-sc-per-location-1
  (implies (and (sc-per-location-1)
                (pol x y))
           (not (com y x)))
  :hints (("Goal"
           :use ((:instance sc-per-location-1-necc
                            (x x)
                            (potential-cycle (list y)))))))

(defthm pol-co->rf-cycle
  (implies (and (pol x y)
                (co->rf y x))
           (pol-com-cyclep (list y (co->rf-witness y x)) x))
  :hints (("Goal" :in-theory (enable co->rf))))

(defthm pol-co->rf-not-sc-per-location-1
  (implies (and (sc-per-location-1)
                (pol x y))
           (not (co->rf y x)))
  :hints (("Goal"
           :in-theory (disable sc-per-location-1
                               pol)
           :use ((:instance sc-per-location-1-necc
                            (x x)
                            (potential-cycle (list y (co->rf-witness y x))))))))

(defthm pol-fr->rf-cycle
  (implies (and (pol x y)
                (fr->rf y x))
           (pol-com-cyclep (list y (fr->rf-witness y x)) x))
  :hints (("Goal" :in-theory (enable fr->rf))))

;; For the paper: combine pol-com-not-sc-per-location-1,
;; pol-co->rf-not-sc-per-location-1, and pol-fr->rf-cycle into one

(defthm sc-per-location-1-implies-2-unquantified-conc
  (implies (and (sc-per-location-1)
                (pol x y))
           (not (fr->rf y x)))
  :hints (("Goal"
           :in-theory (disable sc-per-location-1
                               pol)
           :use ((:instance sc-per-location-1-necc
                            (x x)
                            (potential-cycle (list y (fr->rf-witness y x))))))))

(defthm sc-per-location-1-implies-2
  (implies (sc-per-location-1)
           (sc-per-location-2))
  :hints (("Goal" :in-theory (disable sc-per-location-1 pol))))

;;;;;;;;;;;;;;;;;;
;; Def2 => Def1 ;;
;;;;;;;;;;;;;;;;;;

; This is the most difficult part of the equivalence proof. The general
; approach is to take a pol-com cycle and shorten it until it has length 2
; (i.e. the non-inclusive intermediate path has length 1). Instead of writing
; an explicit shorten function first, we first prove a theorem that, given a
; cycle of length 3 or more, shows that there is a shorter cycle. Then we
; define the shorten function in terms of this theorem, in the sense that the
; definition of the function follows the case-splitting of the theorem
; exactly. It is then straightforward to show that the result of calling the
; shorten function on a cycle results in a pair (x y) such that (pol x y)
; and (com+ y x), thus essentially proving our theorem after we add the
; quantifiers back in.

; The thing that makes this proof more difficult than the others is that the
; existence of a 3+-cycle on a single event x does *not* guarantee the
; existence of a shorter cycle on that same event x - rather more generally, it
; guarantees the exists of another event y, on which there exists a shorter
; cycle. So our ``shortening'' theorem has to reflect this.

; A key lemma for this proof is the ``totality'' of the com+ relation on
; writes and reads on the same location, which appears in rel.lisp. This
; allows us to state that any two such events are in some way related by
; com+. We use this fact to prove that we can shorten cycles of length 3 or
; greater. It is the usage of this that distinguishes our proof from the one
; given in [Alglave 2014].

(in-theory (disable pol com+-alt hat-rewrite))

;; Every pol-com-cyclep is a pol-com+-cyclep
(defthm pol-com-cyclep-implies-pol-com+-cyclep
  (implies (pol-com-cyclep cycle x)
           (pol-com+-cyclep cycle x))
  :hints (("Goal" :in-theory (disable rewrite-com+))))

; There are no cycles of length 1
(defthm cycle-0
  (implies (endp cycle)
           (not (pol-com+-cyclep cycle x))))

; Cycles of length two directly violate sc-per-location-2
(defthm cycle-1
  (implies (and (pol-com+-cyclep cycle x) 
                (endp (cdr cycle))
                (not (and (pol x (car cycle))
                          (com+ (car cycle) x))))
           (and (pol (car cycle) x)
                (com+ x (car cycle)))))

(defthm cycle-n-lemma1
  (implies (pol-com+-cyclep (list* p1 p2 rst) x)
           (and (or (readp x) (writep x))
                (or (readp p1) (writep p1))
                (or (readp p2) (writep p2))
                (equal (addr x) (addr p1))
                (equal (addr x) (addr p2))))
  :rule-classes :forward-chaining)

(defthm cycle-n-lemma2
  (implies (and (readp x)
                (readp p2)
                (equal (rf-inv-fn x) (rf-inv-fn p2))
                (com+-alt p1 p2))
           (com+-alt p1 x))
  :hints (("Goal"
           :in-theory (enable com+-alt 
                              rf-write-unique
                              fr->rf-rewrite
                              co->rf))))

(defthm cycle-n-lemma3
  (implies (and (readp x)
                (readp p2)
                (equal (rf-inv-fn x) (rf-inv-fn p2))
                (com+-alt x p1))
           (com+-alt p2 p1))
  :hints (("Goal"
           :in-theory (enable com+-alt
                              fr->rf-rewrite
                              fr-rewrite
                              rf-write-unique
                              co->rf))))

; If we have a cycle of length 3 or greater, it can be shortened
; We use this theorem implicitly in the collapse-cycle function, which case
; splits on all the different cases in this theorem. This theorem uses the
; ``totality'' of com+.
(defthm cycle-n
  (implies (and (pol-com+-cyclep (list* p1 p2 rst) x)
                (not (pol-com+-cyclep (list* p2 rst) x))
                (not (pol-com+-cyclep rst x))
                (not (pol-com+-cyclep (list p2) p1)))
           (pol-com+-cyclep (list p1) x))
  :hints (("Goal"
           :cases ((com+-alt x p2)
                   (and (writep x)
                        (writep p2)
                        (equal x p2))
                   (and (readp x) 
                        (readp p2)
                        (equal (rf-inv-fn x) (rf-inv-fn p2)))
                   (com+-alt p2 x)))))

(in-theory (disable pol-com+-cyclep))

; Given a cycle in pol-com+, return a pair (x y) such that (pol x y) and
; (com+ y x).
(defun collapse-cycle (cycle x)
  (let* ((p1 (car cycle))
         (p2 (cadr cycle))
         (rst (cddr cycle)))
    (cond ((endp cycle) (mv nil x))
          ((endp (cdr cycle))
           (if (pol x (car cycle))
               (mv x (car cycle))
             (mv (car cycle) x)))
          ((pol-com+-cyclep (list* p2 rst) x)
           (collapse-cycle (list* p2 rst) x))
          ((pol-com+-cyclep rst x)
           (collapse-cycle rst x))
          ((pol-com+-cyclep (list p2) p1)
           (collapse-cycle (list p2) p1))
          (t (collapse-cycle (list p1) x)))))

(defthm collapse-cycle-pol-com+
  (implies (pol-com+-cyclep cycle x)
           (mv-let (new-x new-y)
                   (collapse-cycle cycle x)
                   (and (pol new-x new-y)
                        (com+ new-y new-x)))))

(in-theory (disable pol-com-cyclep
                    collapse-cycle
                    pol
                    rewrite-com+))

; The result of collapse-cycle on a cycle gives us a pair that violates
; sc-per-location-2. 
(defthm collapse-cycle-pol-com
  (implies (pol-com-cyclep cycle x)
           (mv-let (new-x new-y)
                   (collapse-cycle cycle x)
                   (and (pol new-x new-y)
                        (com+ new-y new-x))))
  :hints (("Goal" :use ((:instance collapse-cycle-pol-com+)))))

; Lifts the previous result to put sc-per-location-2 in the hypothesis.
(defthm sc-per-location-2-implies-1-unquantified
  (implies (sc-per-location-2)
           (not (pol-com-cyclep cycle a)))
  :hints (("Goal"
           :in-theory (disable sc-per-location-2 mv-nth)
           :use ((:instance sc-per-location-2-necc
                            (x (mv-let (new-x new-y)
                                       (collapse-cycle cycle a)
                                       (declare (ignore new-y))
                                       new-x))
                            (y (mv-let (new-x new-y)
                                       (collapse-cycle cycle a)
                                       (declare (ignore new-x))
                                       new-y)))))))

; Lifts the previous theorem to put sc-per-location-1 in the conclusion.
(defthm sc-per-location-2-implies-1
  (implies (sc-per-location-2)
           (sc-per-location-1)))

(in-theory (disable sc-per-location-1
                    sc-per-location-2
                    sc-per-location-3))

;;;;;;;;;;;;;;;;;;;;;;
;; The final result ;;
;;;;;;;;;;;;;;;;;;;;;;

; For good measure...

(defthm sc-per-location-1-iff-2
  (iff (sc-per-location-1)
       (sc-per-location-2)))

(defthm sc-per-location-1-iff-3
  (iff (sc-per-location-1)
       (sc-per-location-3)))

(defthm sc-per-location-2-iff-3
  (iff (sc-per-location-2)
       (sc-per-location-3)))
