(in-package "ACL2")

#|

  total-order.lisp
  ~~~~~~~~~~~~~~~~

In this book, we show that there is a total order of all ACL2 objects. The
total order is derived from the ACL2 axioms that provide a total order on all
bad atoms. This book is modified from an original book written by Pete
Manolios. We keep the function <<= enabled, and also prove some theorems
connecting <<= and <<. The reason for this is that we need not a << order on
the elements of the array that we sort, but rather, an order in which we can
say that either x << y or y <<= x.

In principle, so far as the quicksort goes, we can prove the sort with respect
to any arbitrary total order given that we constrain the elements of the array
to be in the domain where the total order is inforced. So I just decided to
leave such hypotheses about domains of elements out, and prove with respect to
ACL2 total order.

|#

(local
(defthm bad-atom<=-reflexive
  (implies (bad-atom x)
           (bad-atom<= x x))
  :hints (("goal"
           :by (:instance bad-atom<=-total (y x)))))
)

(defun bool (x)
  (if x t nil))

(defun atom-order (x y)
  (cond ((rationalp x)
         (if (rationalp y)
             (<= x y)
           t))
        ((rationalp y) nil)
        ((complex-rationalp x)
         (if (complex-rationalp y)
             (or (< (realpart x) (realpart y))
                 (and (= (realpart x) (realpart y))
                      (<= (imagpart x) (imagpart y))))
           t))
        ((complex-rationalp y)
         nil)
        ((characterp x)
         (if (characterp y)
             (<= (char-code x)
                 (char-code y))
           t))
        ((characterp y) nil)
        ((stringp x)
         (if (stringp y)
             (bool (string<= x y))
           t))
        ((stringp y) nil)
        ((symbolp x)
         (if (symbolp y)
             (not (symbol-< y x))
           t))
        ((symbolp y) nil)
        (t (bool (bad-atom<= x y)))))

(local
(defthm symbol-equality-rewrite
  (implies (and (symbolp s1)
                (symbolp s2)
                (equal (symbol-name s1)
                       (symbol-name s2))
                (equal (symbol-package-name s1)
                       (symbol-package-name s2)))
           (equal (equal s1 s2) t))
  :hints (("Goal" :use symbol-equality)))
)

(local
(defthm equal-coerce
  (implies (and (stringp x)
                (stringp y))
           (equal (equal (coerce x 'list)
                         (coerce y 'list))
                  (equal x y)))
  :hints (("Goal" :use
           ((:instance coerce-inverse-2 (x x))
            (:instance coerce-inverse-2 (x y)))
           :in-theory (disable coerce-inverse-2))))
)

(local
(defthm equal-char-code-rewrite
  (implies (and (characterp x)
                (characterp y))
           (implies (equal (char-code x) (char-code y))
                    (equal (equal x y) t)))
  :hints (("Goal" :use equal-char-code)))
)

(local
(in-theory (enable string< symbol-<))
)

(local
(defthm atom-order-reflexive
  (implies (atom x)
           (atom-order x x))
  :hints (("Subgoal 2" 
           :use (:instance bad-atom<=-reflexive))))
)

(local
(defthm string<-l-not
  (implies (and (not (string<-l (coerce y 'list)
                                (coerce x 'list)
                                0))
                (not (string<-l (coerce z 'list)
                                (coerce y 'list)
                                0)))
           (not (string<-l (coerce z 'list)
                           (coerce x 'list)
                           0)))
  :hints (("goal"
           :use ((:instance string<-l-transitive 
                            (x (coerce x 'list))
                            (y (coerce y 'list))
                            (z (coerce z 'list))
                            (i 0) (j 0) (k 0))))))
)

(local
(defthm string<-l-not2
  (implies (and (not (string<-l (coerce y 'list)
                                x
                                0))
                (not (string<-l (coerce z 'list)
                                (coerce y 'list)
                                0))
                (character-listp x))
           (not (string<-l (coerce z 'list)
                           x
                           0)))
  :hints (("goal"
           :use ((:instance string<-l-transitive 
                            (y (coerce y 'list))
                            (z (coerce z 'list))
                            (i 0) (j 0) (k 0))))))
)

(local
(defthm string<-l-not2-1
  (implies (and (not (string<-l x 
                                (coerce y 'list)
                                0))
                (not (string<-l (coerce z 'list)
                                x
                                0))
                (character-listp x))
           (not (string<-l (coerce z 'list)
                           (coerce y 'list)
                           0)))
  :hints (("goal"
           :in-theory (disable string<-l-not2)
           :use ((:instance string<-l-transitive 
                            (y x)
                            (x (coerce y 'list))
                            (z (coerce z 'list))
                            (i 0) (j 0) (k 0))))))
)

(local
(defthm string<-l-not2-2
  (implies (and (not (string<-l x 
                                (coerce y 'list)
                                0))
                (not (string<-l (coerce y 'list)
                                (coerce z 'list)
                                0))
                (character-listp x))
           (not (string<-l x
                           (coerce z 'list)
                           0)))
  :hints (("goal"
           :in-theory (disable string<-l-not2-1)
           :use ((:instance string<-l-transitive 
                            (z x)
                            (x (coerce z 'list))
                            (y (coerce y 'list))
                            (i 0) (j 0) (k 0))))))
)

(local
(defthm string<-l-not3
  (implies (and (not (string<-l (coerce (symbol-name y) 'list)
                                x
                                0))
                (not (string<-l x
                                (coerce (symbol-name y) 'list)
                                0))
                (character-listp x)
                (symbolp y))
           (equal (symbol-name y) (coerce x 'string))))
)

(local
(defthm string<-l-not4
  (implies (and (equal x (coerce (symbol-name y) 'list))
                (equal z (coerce x 'string))
                (character-listp x)
                (symbolp y))
           (equal (equal (symbol-name y) z) t)))
)

(local
(defthm atom-order-transitive
  (implies (and (atom-order x y) 
                (atom-order y z)
                (atom x)
                (atom y)
                (atom z))
           (atom-order x z))
  :hints (("subgoal 1" :use (:instance bad-atom<=-transitive))))
)

(local
(defthm atom-order-anti-symmetric
  (implies (and (not (consp x))
                (not (consp y))
                (atom-order x y))
           (equal (atom-order y x)
                  (equal x y)))
  :hints (("goal"
           :in-theory (disable code-char-char-code-is-identity)
           :use ((:instance symbol-equality (s1 x) (s2 y))
                 (:instance bad-atom<=-antisymmetric)
                 (:instance code-char-char-code-is-identity (c y))
                 (:instance code-char-char-code-is-identity (c x))))))
)

(local
(defthm atom-order-total
  (implies (and (not (consp x))
                (not (consp y))
                (not (atom-order x y)))
           (atom-order y x))
  :hints (("goal"
           :use (:instance bad-atom<=-total))))
)

(defun <<= (x y)
  (cond ((atom x)
         (if (atom y)
             (atom-order x y)
           t))
        ((atom y) nil)
        ((equal (car x) (car y))
         (<<= (cdr x) (cdr y)))
        (t (<<= (car x) (car y)))))

(in-theory (disable atom-order))

(defthm <<=-reflexive
  (<<= x x))

(defthm <<=-anti-symmetric
  (implies (and (<<= x y) 
                (<<= y x))  
           (equal x y))
  :rule-classes :forward-chaining)

(defthm <<=-transitive
  (implies (and (<<= x y) 
                (<<= y z))
           (<<= x z)))


(defthm <<=-total
  (implies (not (<<= x y))
           (<<= y x))
  :rule-classes :forward-chaining)

(in-theory (disable <<=))

(defun << (x y)
  (and (<<= x y)
       (not (equal x y))))

(defthm <<-irreflexive
  (not (<< x x)))

(defthm <<-transitive
  (implies (and (<< x y)
                (<< y z))
           (<< x z))
  :hints (("Goal"
           :do-not-induct t)
          ("Subgoal 2"
           :use <<=-transitive)
          ("Subgoal 1"
           :use <<=-anti-symmetric)))

(defthm <<-asymmetric
  (implies (<< x y)
           (not (<< y x)))
  :hints (("Goal"
           :use <<=-anti-symmetric)))

(defthm <<-trichotomy
  (implies (not (<< y x))
           (equal (<< x y)
		  (not (equal x y))))
  :hints (("Goal"
           :use <<=-total))
  :rule-classes :forward-chaining)

(defthm <<-anti-symmetric
  (implies (<< x y)
           (not (<<= y x)))
  :hints (("Goal"
           :use <<=-anti-symmetric))
  :rule-classes :forward-chaining)

(defthm <<-total
  (implies (not (<<= x y))
           (<< y x))
  :hints (("Goal"
           :use (<<=-total <<=-anti-symmetric)))
  :rule-classes :forward-chaining)


(in-theory (disable <<))

