; As explained in the top-level README file of our entry, we do all the work in
; two_way_sort-proof.lisp.  The present file is a self-contained theory, in
; which the theorems follow from the definitions without the clutter of
; auxiliary definitions and lemmas, which aren't logically necessary (but were
; very useful for the proof).

(in-package "ACL2")

(local (include-book "two_way_sort-proof"))

; Optional: Check that everything here is in the above file.
(set-enforce-redundancy t)

;;; 1. Safety. Verify that every array access is made within bounds. 

; For safety (every array access is made within bounds), we rely on ACL2's
; guard mechanism.  The guards of get-val and set-val guarantee that every
; access is in bounds, because the index is a natural number less than the
; length of the array.

; As noted in the README, the defun command of ACL2 verifies termination and,
; if a :guard is specified, it proves that the guard implies the guards of all
; subexpressions in the defun.  By imposing the :guards below on get-val and
; set-val we insure that when swap is defined, the array indices are verified
; to be in bounds.

(defun get-val (a i)
  (declare (xargs :guard (and (true-listp a)
                              (natp i)
                              (< i (len a)))))
  (nth i a))

(defun set-val (a i v)
  (declare (xargs :guard (and (true-listp a)
                              (natp i)
                              (< i (len a)))))
  (update-nth i v a))

(defun swap (a i j)
  (declare (xargs :guard (and (true-listp a)
                              (natp i)
                              (natp j)
                              (< i (len a))
                              (< j (len a)))))
  (let* ((temp (get-val a i))
         (a (set-val a i (get-val a j)))
         (a (set-val a j temp)))
    a))

;;; 2. Termination. Prove that function two way sort always terminates. 

; Termination is automatically proved by ACL2 using the :measure provided
; below.

(defun two_way_sort-rec (a i j)
  (declare (xargs :guard (and (true-listp a)
                              (natp i)
                              (integerp j)
                              (<= -1 j)
                              (<= i (len a))
                              (< j (len a)))
                  :measure (nfix (- (+ 1 j) i))))

; The mbe ("must be equal") just below contains two expressions, one labeled
; :logic and one labeled :exec.  The guards of mbe require ACL2 to prove that
; these two expressions are equal under the :guard of two_way_sort-rec.  The
; mbe causes the axiomatic definition of two_way_sort-rec to test (and (natp i)
; (natp j) (<= i j)) but the Common Lisp counterpart to test (<= i j), which
; avoids the natp runtime ``type'' tests.  Guard verification insures that the
; two are equivalent when the Common Lisp program is run on data satisfying the
; :guard of two_way_sort-rec above.  This makes the Common Lisp program faster
; while provably preserving the axiomatic semantics of the function.  This is a
; common technique in ACL2.

  (cond ((mbe :logic (and (natp i)
                          (natp j)
                          (<= i j))
              :exec (<= i j))
         (cond ((not (get-val a i))
                (two_way_sort-rec a (+ i 1) j))
               ((get-val a j)
                (two_way_sort-rec a i (- j 1)))
               (t (let ((a (swap a i j)))
                    (two_way_sort-rec a (+ i 1) (- j 1))))))
        (t a)))

(defun two_way_sort (a)
  (declare (xargs :guard (true-listp a)))
  (two_way_sort-rec a 0 (- (len a) 1)))

;;; 3. Behavior. Verify that after execution of function two way sort,the
;;; following properties hold. (a) Array a is sorted in increasing order.

; Defun-sk is an ACL2 built-in that allows the user to introduce quantification
; into function definitions.  As used below it defines (sortedp a) to be
; equivalent to the quantified statement: for all naturals i and j such that i
; < j < (len a), (get-val a i) implies (get-val a j), i.e., all earlier
; elements of a are ``weakly smaller'' than all subsequent elements.

(defun-sk sortedp (a)
  (forall (i j)
          (implies (and (natp i)
                        (natp j)
                        (< i j)
                        (< j (len a))
                        (get-val a i))
                   (get-val a j))))

(defthm sortedp-two_way_sort
  (sortedp (two_way_sort a)))

;;; (b) Array a is a permutation of its initial contents.

; We prove that the initial and final array each contain the same number of
; occurrences of each object, which can be viewed as the definition of
; permutation.

(defun how-many (e x)
  (cond
   ((endp x)
    0)
   ((equal e (car x))
    (1+ (how-many e (cdr x))))
   (t
    (how-many e (cdr x)))))

(defun-sk permp (a1 a2)
  (forall e
          (equal (how-many e a1)
                 (how-many e a2))))

; (permp a1 a2) means that for all e, the number of times e occurs as an
; element of a1 is equal to the number of times it occurs as an element of a2.
; We define permp that way.  One of the standard books distributed with ACL2,
; namely books/sorting/convert-perm-to-how-many.lisp shows that this
; definition is equivalent to a recursive definition on lists a1 and a2.  That
; book renders perm executable (while the quantified definition above is not).
; However, we see no requirement here that the spec be executable so we did not
; use the recursive definition of perm.

(defthm permp-two_way_sort
  (permp (two_way_sort a) a))
