
(in-package "ACL2")

;; library for defining sum-of-products data structures, pattern matching
(include-book "tools/defsum" :dir :system)
;; macro library for destructuring binders, etc
(include-book "tools/bstar" :dir :system)
;; library for reasoning about functions that return multivalues
(include-book "tools/mv-nth" :dir :system)
;;library for basic arithmetic
(include-book "arithmetic/top" :dir :system)

;; Needed for defsum, tell ACL2 it's ok to provide a termination argument for a
;; function that doesn't need it.
(set-bogus-defun-hints-ok t)


;;-------------------------------------------------------------------------
;; Data type definitions.
;;-------------------------------------------------------------------------

;; Definition of a term: either S, K, or an APP of two terms l and r.
(defsum term
  (S)
  (K)
  (app (term-p l)
       (term-p r)))

(in-theory (enable term-executable-counterparts))

;; Definition of values as a predicate on terms -- something that matches one
;; of the listed patterns.  (The patterns are mutually exclusive so we don't
;; need to check others once we've found one that matches.)
(defun value-p (x)
  (declare (xargs :guard (term-p x)))
  (pattern-match x
    ((K)                   t)
    ((S)                   t)
    ((app (K) v)           (value-p v))
    ((app (S) v)           (value-p v))
    ((app (app (S) v1) v2) (and (value-p v1)
                                (value-p v2)))))


;; Definition of CBV contexts --
;;  ctx-box    []
;;  ctx-left   (C t)
;;  ctx-right  (v C).
;; This defines the basic syntax, which doesn't check whether ctx-left has a
;; value in the left place.
(defsum ctx
  (ctx-box)
  (ctx-left (ctx-p l) (term-p r))
  (ctx-right (term-p l) (ctx-p r)))

(in-theory (enable ctx-executable-counterparts))

;; This checks that a context is well-formed, in that the term argument of each
;; ctx-right form is a value, as required.
(defun good-ctx-p (x)
  (declare (Xargs :guard (ctx-p x)))
  (pattern-match x
    ((ctx-box) t)
    ((ctx-left l &) (good-ctx-p l))
    ((ctx-right l r) (and (value-p l)
                          (good-ctx-p r)))))


;;-------------------------------------------------------------------------
;; Definition of the reduction relation.
;;-------------------------------------------------------------------------

;; Determines whether a term is base reducible, i.e. it matches
;; either ((K v1) v2) or (((S v1) v2) v3).
(defun base-reducible (x)
  (declare (Xargs :guard (term-p x)))
  (pattern-match x
    ((app (app (k) v1) v2)              (and (value-p v1)
                                             (value-p v2)))
    ((app (app (app (s) v1) v2) v3)     (and (value-p v1)
                                             (value-p v2)
                                             (value-p v3)))))

;; Assuming x is base reducible, return its reduction, i.e.
;;   ((K v1) v2)       --> v1
;;   (((S v1) v2) v3)  --> ((v1 v3) (v2 v3)).
(defun base-reduce (x)
  (declare (Xargs :guard (and (term-p x) (base-reducible x))))
  (pattern-match x
    ((app (app (k) v1) &)               v1)
    ((app (app (app (s) v1) v2) v3)     (app (app v1 v3)
                                             (app v2 v3)))))

;; if x is base-reducible, (base-reduce x) produces a term
(defthm term-p-base-reduce
  (implies (and (base-reducible x)
                (term-p x))
           (term-p (base-reduce x))))

(in-theory (disable base-reduce base-reducible))

;; Defines the [] operator, i.e. returns ctx[term].
(defun ctx-subst (ctx term)
  (declare (xargs :guard (and (ctx-p ctx)
                              (term-p term))))
  (pattern-match ctx
    ((ctx-left l r)      (app (ctx-subst l term) r))
    ((ctx-right l r)     (app l (ctx-subst r term)))
    (& ;; (ctx-box) or ill-formed
     term)))

;; When the inputs are well-formed, ctx-subst produces a term.
(defthm term-p-ctx-subst
  (implies (and (term-p term)
                (ctx-p ctx))
           (term-p (ctx-subst ctx term))))

;; Checks a valid reduction step t1 -> t2.  This takes terms t1 and t2 and a
;; certificate that this is a valid reduction, consisting of a context C and an
;; base-reducible term R, for which C[R] equals T1 and C[R'] is T2.
(defun arrow-validp (t1 t2 c r)
  (declare (xargs :guard (and (term-p t1)
                              (term-p t2)
                              (ctx-p c)
                              (term-p r))))
  ;; First check that R is base-reducible and get its reduction.
  (and (base-reducible r)
       (good-ctx-p c) ;; C is a valid ctx
       (equal t1 (ctx-subst c r)) ;; T1 matches C[R]
       (equal t2 (ctx-subst c (base-reduce r))) ;; T2 matches C[R'].
       ))

(local
 ;; dumb lemma needed for the proof below
 (defthm term-p-of-app
   (iff (term-p (app x y))
        (and (term-p x)
             (term-p y)))))

(defthm term-p-ctx-subst-when-other-subst-is-term-p
  (implies (and (term-p (ctx-subst c r))
                (term-p r2))
           (term-p (ctx-subst c r2))))

;; ARROW-VALIDP does not recognize any transition from a well-formed term t1 to
;; an ill-formed term t2.
(defthm arrow-validp-does-not-allow-bad-syntax
  (implies (and (term-p t1)
                (not (term-p t2)))
           (not (arrow-validp t1 t2 c r))))
           


;; T1 -> T2 if there is a derivation satisfying arrow-validp.
(defun-sk arrow (t1 t2)
  (exists (c r)
          (arrow-validp t1 t2 c r)))

(defthm arrow-does-not-allow-bad-syntax
  (implies (and (term-p t1) (not (term-p t2)))
           (not (arrow t1 t2))))

;; Quantifies over possible single-step transitions to define the "no-arrow"
;; predicate, i.e. returns T if t1 has no possible transitions.
(defun-sk no-arrow (t1)
  (forall (t2)
          (not (arrow t1 t2))))


;; A reduction (->*) derivation will be a list of triples (t2 c r), each
;; defining one step (as checked by arrow-validp, above.)  This simply checks that
;; it has valid syntax, not that it is a valid derivation.
(defun deriv-listp (x)
  (declare (xargs :guard t))
  (or (atom x)
      (let ((triple (first x)))
        (and (true-listp triple)
             (term-p (first triple))
             (ctx-p (second triple))
             (term-p (third triple))
             (deriv-listp (rest x))))))
      

;; Checks a valid reduction of ti to tf, i.e. multiple reduction steps.
;; Here a deriv is a list of triples (t2 c r), each of which gives one
;; reduction step.
(defun arrow*-validp (ti tf deriv)
  (declare (xargs :guard (and (term-p ti)
                              (term-p tf)
                              (deriv-listp deriv))))
  (if (atom deriv)
      ;; 0-step derivation: ti must equal tf.
      (equal ti tf)
    (b* ((triple            (first deriv))
         ((list t2 c r)     triple))
      (and (arrow-validp ti t2 c r)           ;; ti -> t2 is a valid reduction
           (arrow*-validp t2 tf (rest deriv)) ;; t2 ->* tf is a valid reduction.
           ))))

(defthm arrow*-validp-does-not-allow-bad-syntax
  (implies (and (term-p ti)
                (not (term-p tf)))
           (not (arrow*-validp ti tf deriv))))

;; T1 ->* T2 if there exists a derivation showing that the steps are all valid.
(defun-sk arrow* (ti tf)
  (exists deriv
          (arrow*-validp ti tf deriv)))

(defthm arrow*-does-not-allow-bad-syntax
  (implies (and (term-p ti)
                (not (term-p tf)))
           (not (arrow* ti tf))))


;;-------------------------------------------------------------------------
;; Definition of the single-step reduction function.
;;-------------------------------------------------------------------------

;; Given a term, determines whether it's reducible, returning a t/nil flag
;; saying so, and if so, a context and base term defining its reduction.
(defun find-reduction (x)
  (declare (Xargs :guard (term-p x)))
  (b* (((when (base-reducible x))       (mv t (ctx-box) x))) ;; box[x] = x.
    (pattern-match x
      ((S)           (mv nil nil nil))    ;; not reducible
      ((K)           (mv nil nil nil))    ;; not reducible
      ((app l r)
       (if (value-p l)
           ;; reduce on the right
           (b* (((mv ok right-ctx base)  (find-reduction r))
                ((unless ok)             (mv nil nil nil))) ;; not reducible
             (mv t
                 (ctx-right l right-ctx)
                 base))
         ;; reduce on the left
         (b* (((mv ok left-ctx base)   (find-reduction l))
              ((unless ok)             (mv nil nil nil))) ;; not reducible
           (mv t
               (ctx-left left-ctx r)
               base))))
      (& (mv nil nil nil)))))

;; FIND-REDUCTION, if it succeeds, provides a ctx and a reducible base term
;; such that ctx[base] = x.
(defthm find-reduction-correct
  (b* (((mv find-ok ctx base)  (find-reduction x)))
    (implies find-ok
             (and (equal (ctx-subst ctx base) x)
                  (base-reducible base)
                  (good-ctx-p ctx))))) ;; ctx has values in the right places

;; Furthermore, if the input term is well-formed and it succeeds, then
;; find-reduction produces a well-formed context and term as its results.
(defthm find-reduction-well-formed
  (b* (((mv find-ok ctx base)  (find-reduction x)))
    (implies (and find-ok
                  (term-p x))
             (and (ctx-p ctx)
                  (term-p base)))))

;; These two lemmas are important for our completeness proofs below
(defthm not-base-reducible-when-value-p
  (implies (value-p x)
           (not (base-reducible x)))
  :hints(("Goal" :in-theory (enable base-reducible))))

(defthm not-value-subst-of-base-reducible
  (implies (base-reducible x)
           (not (value-p (ctx-subst ctx x))))
  :hints(("Goal" :in-theory (enable value-p))))

(in-theory (disable value-p))

;; If find-reduction does not succeed, then there is no transition possible
;; from x:
(defthm find-reduction-complete1
  (b* (((mv ok & &) (find-reduction x)))
    (implies (and (not ok)
                  (base-reducible base)
                  (good-ctx-p ctx))
             ;; If we don't find a reduction, then there is no reduction, i.e.
             ;; there is no ctx/base pair such that base is base-reducible
             ;; and ctx[base] = x.
             (not (equal (ctx-subst ctx base) x)))))


(defthm find-reduction-complete
  ;; if we don't find a reduction from X, then there is no reduction from X.
  (b* (((mv ok & &) (find-reduction x)))
    (implies (not ok)
             (no-arrow x)))
  :hints (("goal" :in-theory (enable arrow-validp))))

(in-theory (disable find-reduction))




;; Reduce1 is the single-step reduction function.  It finds a term that is a
;; 1-step reduction from X, if one exists.  Returns a flag (exists or not) and
;; the new term.
(defun reduce1 (x)
  (declare (xargs :guard (term-p x)))
  (b* (((mv ok1 ctx base) (find-reduction x))
       ((unless ok1)      (mv nil nil)) ;; no reduction
       )
    ;; success -- the new term is ctx[base'].
    (mv t (ctx-subst ctx (base-reduce base)))))

(defthm reduce1-correct1
  (b* (((mv reduc-ok new-x)   (reduce1 x))
       ((mv & ctx base) (find-reduction x)))
    (implies reduc-ok
             (arrow-validp x new-x ctx base))))

;; Reduce1 produces a well-formed new term
(defthm reduce1-well-formed
  (b* (((mv reduc-ok new-x)   (reduce1 x)))
    (implies (and reduc-ok
                  (term-p x))
             (term-p new-x))))

(in-theory (disable reduce1 arrow-validp))

;; Reduce1, when it succeeds, finds a new term for which there is a valid -> derivation.
(defthm reduce1-correct
  (b* (((mv reduc-ok new-x)   (reduce1 x)))
    (implies (and reduc-ok (term-p x))
             (arrow x new-x)))
  :hints (("goal" :use reduce1-correct1
           :in-theory (disable reduce1-correct1 arrow))))

       
;; If reduce1 fails, then there is no valid transition from x.
(defthm reduce1-complete
  (b* (((mv ok &) (reduce1 x)))
    (implies (not ok)
             (no-arrow x)))
  :hints (("goal" :in-theory (e/d (reduce1) (no-arrow)))))





;;-------------------------------------------------------------------------
;; Definition of the full reduction function.
;;-------------------------------------------------------------------------

;; We first define reduce-n, which runs N reduction steps, then returns a flag
;; saying whether it finished or not and the final term.
(defun reduce-n (n x)
  (declare (xargs :guard (and (natp n)
                              (term-p x))))
  (b* (((when (zp n)) ;; Ran out of our step limit.
        (mv nil x))
       ((mv reduc-ok new-x) (reduce1 x)))
    (if reduc-ok
        (reduce-n (- n 1) new-x)
      ;; Finished.
      (mv t x))))

(defthm term-p-reduce-n
  (b* (((mv & new-x) (reduce-n n x)))
    (implies (term-p x)
             (term-p new-x))))


;; Whatever number of steps you allow, if reduce-n finishes, the result is the same.
(defthm reduce-n-same-result-when-completed
  (b* (((mv ok1 res1) (reduce-n n x))
       ((mv ok2 res2) (reduce-n m x)))
    (implies (and ok1 ok2)
             (equal res1 res2)))
  :rule-classes nil)

;; Defines a Skolem function that chooses N big enough to completely reduce X.
;; If X is not reducible, we don't know what this returns.
(defchoose steps-required (n) (x)
  (b* (((mv terminates &) (reduce-n n x)))
    terminates))

;; if reduce-n succeeds for any n, then it succeeds for (steps-required x).
(defthm reduce-n-succeeds-on-steps-required
  (b* (((mv ok-n &) (reduce-n n x))
       ((mv ok   &) (reduce-n (steps-required x) x)))
    (implies ok-n ok))
  :hints (("goal" :use ((:instance steps-required)))))

;; Normalize reduce-n calls to n = steps-required.
(defthm reduce-n-use-steps-required
  (b* (((mv ok-n new-x-n) (reduce-n n x))
       ((mv & new-x) (reduce-n (steps-required x) x)))
    (implies (and ;; prevent this rewrite rule from firing on terms already in
                  ;; the normal form
                  (syntaxp (not (equal n `(steps-required ,x))))
                  ok-n)
             (equal new-x-n new-x)))
  :hints (("goal" :use ((:instance reduce-n-same-result-when-completed
                         (m (steps-required x)))))))


;; If reduce1(x) = y, then y ->* terminates iff x ->* terminates,
;; and they get the same results.  This is the lemma needed to prove the
;; logical and executable definitions of reduction, below, to be equivalent.
(defthm reduce-n-step
  (b* ((steps-x (steps-required x))
       ((mv x-terminates x-final) (reduce-n steps-x x))
       ((mv step-ok y) (reduce1 x))
       (steps-y (steps-required y))
       ((mv y-terminates y-final) (reduce-n steps-y y)))
    (implies step-ok
             (and (iff y-terminates x-terminates)
                  (implies x-terminates
                           (equal y-final x-final)))))
  :hints ('(:expand ((reduce-n (steps-required x) x)))
          (and stable-under-simplificationp
               '(:expand ((reduce-n (steps-required (mv-nth 1 (reduce1 x)))
                                    (mv-nth 1 (reduce1 x))))
                 :use ((:instance steps-required
                        (n (+ 1 (steps-required (mv-nth 1 (reduce1 x))))))))))
  :otf-flg t)

;; REDUCTION -- logically, runs reduce-n the required number of steps.  In
;; actual execution, it keeps stepping until it finishes, and perhaps doesn't
;; terminate.
(defun reduction (x)
  (declare (xargs :guard (term-p x)
                  :guard-hints (("goal" :use ((:instance steps-required
                                               (n 1)))))))
  (mbe :logic
       ;; Logical definition: run (steps-required x) steps, and return that
       ;; result if done by that point; otherwise don't return anything (in
       ;; actuality, this means we didn't terminate)
       (b* ((steps (steps-required x))
                   ((mv done new-x) (reduce-n steps x)))
                (if done
                    (mv t new-x)
                  (mv nil nil))) ;; doesn't terminate

       ;; Executable definition: just run step after step until we're done.  If
       ;; we don't terminate, we don't terminate.
       ;; Guard verification proves these two definitions equivalent.
       :exec (b* (((mv reduc-ok new-x) (reduce1 x)))
               (if reduc-ok
                   (reduction new-x)
                 ;; Finished.
                 (mv t x)))))

;; reduction must produce a well-formed term if its input term is well-formed
;; and it terminates.
(defthm reduction-produces-good-term
  (b* (((mv terminates result) (reduction x)))
    (implies (and (term-p x)
                  terminates)
             (term-p result))))

;;-------------------------------------------------------------------------
;; Verification 1: if reduction(t) returns t', then t ->* t' and t' -/>.
;;-------------------------------------------------------------------------

;; Reduce-n runs N single-step reductions.  Reduce-n-derivation produces a
;; derivation for those N reductions.
(defun reduce-n-derivation (n x)
  (declare (Xargs :guard (and (natp n) (term-p x))))
  (b* (((when (zp n)) ;; Ran out of our step limit.
        nil)
       ((mv reduc-ok new-x) (reduce1 x))
       ((unless reduc-ok) nil)
       ((mv & ctx base) (find-reduction x)))
    (cons (list new-x ctx base)
          (reduce-n-derivation (- n 1) new-x))))

;; Reduce-n always produces a reduction that has a valid derivation.
(defthm reduce-n-correct1
  (b* (((mv & new-x) (reduce-n n x))
       (deriv (reduce-n-derivation n x)))
    (arrow*-validp x new-x deriv)))

(defthm reduce-n-correct
  (b* (((mv & new-x) (reduce-n n x)))
    (arrow* x new-x))
  :hints (("goal" :use reduce-n-correct1
           :in-theory (disable reduce-n-correct1))))

;; Whenever reduce-n says it has finished, it produces a result that has no transitions.
(defthm reduce-n-complete
  (b* (((mv done new-x) (reduce-n n x)))
    (implies done
             (no-arrow new-x)))
  :hints(("Goal" :in-theory (disable no-arrow))))



;;-------------------------------------------------------------------------
;; Main theorem for Verification Task 1.
;;-------------------------------------------------------------------------

;; If reduction(t) terminates and returns t', then t ->* t' and t' -/>.
(defthm reduction-correct-and-complete
  (b* (((mv terminates result) (reduction x)))
    (implies terminates
             (and (arrow* x result)
                  (no-arrow result)))))





;;-------------------------------------------------------------------------
;; Verification 2: reduction terminates on any term that does not contain S.
;;-------------------------------------------------------------------------

;; The key here is that if the term does not contain S combinators, then
;; base-reduce always reduces the size of the term, in a well-founded sense.

;; The defsum macro already has defined for us a function called term-measure
;; that is suitable for describing the size of such terms.

(defun s-free-term-p (x)
  (declare (xargs :guard (term-p x)))
  (pattern-match x
    ((s) nil)
    ((k) t)
    ((app l r) (and (s-free-term-p l)
                    (s-free-term-p r)))))

;; Base-reduction decreases the size of an S-free term.
(defthm base-reduce-decreases-term-measure-when-s-free
  (implies (and (s-free-term-p x)
                (base-reducible x))
           (< (term-measure (base-reduce x))
              (term-measure x)))
  :hints(("Goal" :in-theory (enable base-reducible base-reduce)))
  :rule-classes :linear)

;; Base reduction of an s-free term produces an s-free term.
(defthm base-reduce-preserves-s-free
  (implies (and (s-free-term-p x)
                (base-reducible x))
           (s-free-term-p (base-reduce x)))
  :hints(("Goal" :in-theory (enable base-reduce base-reducible))))

;; Substitution of a smaller term into a context produces a smaller term.
(defthm term-measure-of-ctx-subst
  (implies (and (< (term-measure base1)
                   (term-measure base2)))
           (< (term-measure (ctx-subst ctx base1))
              (term-measure (ctx-subst ctx base2))))
  :rule-classes :linear)

;; Defines a context which has no S values.
(defun s-free-ctx-p (ctx)
  (declare (xargs :guard (ctx-p ctx)))
  (pattern-match ctx
    ((ctx-left l r) (and (s-free-ctx-p l)
                         (s-free-term-p r)))
    ((ctx-right l r) (and (s-free-term-p l)
                          (s-free-ctx-p r)))
    (& t)))

;; Find-reduction, when given an s-free term, produces an s-free context and base.
(defthm s-free-find-reduction
  (b* (((mv ok ctx base) (find-reduction x)))
    (implies (and ok (s-free-term-p x))
             (and (s-free-term-p base)
                  (s-free-ctx-p ctx))))
  :hints(("Goal" :in-theory (enable find-reduction))))

;; Substitution of an s-free term into an s-free context produces an s-free term.
(defthm s-free-term-p-ctx-subst
  (implies (and (s-free-term-p base)
                (s-free-ctx-p ctx))
           (s-free-term-p (ctx-subst ctx base))))

;; When the input term is S-free, reduce1 produces a smaller term if it succeeds.
(defthm term-measure-of-reduce1-when-s-free
  (b* (((mv ok new-x) (reduce1 x)))
    (implies (and ok (s-free-term-p x))
             (< (term-measure new-x)
                (term-measure x))))
  :hints(("Goal" :in-theory (enable reduce1)
          :use ((:instance term-measure-of-ctx-subst
                 (ctx (mv-nth 1 (find-reduction x)))
                 (base1 (base-reduce (mv-nth 2 (find-reduction x))))
                 (base2 (mv-nth 2 (find-reduction x)))))))
  :rule-classes :linear)

;; Reduce1 produces an S-free term when given an S-free term.
(defthm reduce1-preserves-s-free
  (b* (((mv ok new-x) (reduce1 x)))
    (implies (and ok (s-free-term-p x))
             (s-free-term-p new-x)))
  :hints(("Goal" :in-theory (enable reduce1))))

;;  Therefore, the term measure (or any greater integer) is usable as a step
;;  limit for reduce-n.
(defthm reduce-n-terminates-when-s-free1
  (b* (((mv done &) (reduce-n n x)))
    (implies (and (<= (term-measure x) n)
                  (integerp n)
                  (s-free-term-p x))
             done)))


;; And therefore, the reduction function terminates when X is s-free.
(defthm reduction-terminates-when-s-free
  (b* (((mv terminates &) (reduction x)))
    (implies (s-free-term-p x)
             terminates))
  :hints (("goal" :use ((:instance steps-required
                         (n (term-measure x)))))))


  
             
;;-------------------------------------------------------------------------
;; Verification 3: Prove that (reduction (ks n)) returns K when n is even
;; and (k k) when n is odd.
;;-------------------------------------------------------------------------

(defun ks (n)
  (declare (xargs :guard (natp n)))
  (if (zp n) ;; n is zero or not a natural number
      (K)
    (app (ks (- n 1)) (K))))

;; Terms produced by KS are s-free, which gives us termination of REDUCTION for
;; free
(defthm s-free-term-p-of-ks
  (s-free-term-p (ks n)))

(defthm value-p-of-ks
  (equal (value-p (ks n))
         (or (not (integerp n))
             (<= n 1)))
  :hints(("Goal" :in-theory (e/d (value-p)))))

;; Apparently we don't need to define what kind of context we get from
;; find-reduction -- this goes through without that extra reasoning.
(defthm reduce1-of-ks
  (equal (reduce1 (ks n))
         (if (and (integerp n) (< 1 n))
             (mv t (ks (- n 2)))
           (mv nil nil)))
  :hints(("Goal" :in-theory (e/d (reduce1)
                                 ((:definition ks)))
          :induct (ks n)
          :expand ((:free (a) (base-reducible (app (app a '(k))'(k))))
                   (:free (a) (find-reduction (app a'(k))))
                   (ks n)
                   (ks (+ -1 n))
                   (ks (+ -2 n))))))

;; this is just used as an induction scheme for the theorem below.
(defun down-by-2s (n)
  (declare (Xargs :guard (natp n)))
  (cond ((zp n) 0)
        ((eql n 1) 1)
        (t (down-by-2s (- n 2)))))


;; Reduce-n on (ks n) produces 
(defthm reduce-n-of-ks-result
  (b* (((mv & res)
        (reduce-n (steps-required (ks n))
                  (ks n))))
    (equal res (ks (if (and (natp n) (oddp n)) 1 0))))
  :hints (("goal" :induct (down-by-2s n)
           :expand ((reduce-n (steps-required (ks n))
                              (ks n))))
          (and stable-under-simplificationp
               '(:use ((:instance steps-required
                        (x (ks n))
                        (n (term-measure (ks n))))))))
  :otf-flg t)


(defthm reduce-n-of-ks-terminates
  (b* (((mv ok &)
        (reduce-n (steps-required (ks n))
                  (ks n))))
    ok)
  :hints(("Goal" :use ((:instance steps-required
                        (x (ks n))
                        (n (term-measure (ks n))))))))

;; Theorem for Verification Task 3: (reduction (ks n)) returns K when n is
;; even, and (K K) when n is odd.
(defthm reduction-of-ks
  (b* (((mv terminates result) (reduction (ks n))))
    (and terminates     ;; it finishes
         (equal result
                (if (and (natp n) (oddp n))
                    (app (k) (k)) ;; (ks 1)
                  (k)  ;; (ks 0)
                  )))))

