
(in-package "ACL2")

;; We start by loading several of the books distributed with ACL2:

;; -- library for defining sum-of-products data structures, pattern matching
(include-book "tools/defsum" :dir :system)
;; -- macro library for destructuring binders, etc
(include-book "tools/bstar" :dir :system)

;; Now we load the actual proof script, as per our convention explained in the
;; top-level README file.

(local (include-book "combinators-proof"))

;; The events below are all redundant copies of ones defined in
;; combinators-proof.lisp.
(set-enforce-redundancy t)

;; Definition of a term: either S, K, or an APP of two terms l and r.
(defsum term
  (S)
  (K)
  (app (term-p l)
       (term-p r)))


;; Definition of values as a predicate on terms -- something that matches one
;; of the listed patterns.  (The patterns are mutually exclusive so we don't
;; need to check others once we've found one that matches.)
(defun value-p (x)
  (declare (xargs :guard (term-p x)))
  (pattern-match x
    ((K)                   t)
    ((S)                   t)
    ((app (K) v)           (value-p v))
    ((app (S) v)           (value-p v))
    ((app (app (S) v1) v2) (and (value-p v1)
                                (value-p v2)))))

;; Definition of CBV contexts --
;;  ctx-box    []
;;  ctx-left   (C t)
;;  ctx-right  (v C).
;; This defines the basic syntax, which doesn't check whether ctx-left has a
;; value in the left place.
(defsum ctx
  (ctx-box)
  (ctx-left (ctx-p l) (term-p r))
  (ctx-right (term-p l) (ctx-p r)))

;; This checks that a context is well-formed, in that the term argument of each
;; ctx-right form is a value, as required.
(defun good-ctx-p (x)
  (declare (Xargs :guard (ctx-p x)))
  (pattern-match x
    ((ctx-box) t)
    ((ctx-left l &) (good-ctx-p l))
    ((ctx-right l r) (and (value-p l)
                          (good-ctx-p r)))))


;;-------------------------------------------------------------------------
;; Definition of the reduction relation.
;;-------------------------------------------------------------------------

;; Determines whether a term is base reducible, i.e. it matches
;; either ((K v1) v2) or (((S v1) v2) v3).
(defun base-reducible (x)
  (declare (Xargs :guard (term-p x)))
  (pattern-match x
    ((app (app (K) v1) v2)              (and (value-p v1)
                                             (value-p v2)))
    ((app (app (app (S) v1) v2) v3)     (and (value-p v1)
                                             (value-p v2)
                                             (value-p v3)))))

;; Assuming x is base reducible, return its reduction, i.e.
;;   ((K v1) v2)       --> v1
;;   (((S v1) v2) v3)  --> ((v1 v3) (v2 v3)).
(defun base-reduce (x)
  (declare (Xargs :guard (and (term-p x) (base-reducible x))))
  (pattern-match x
    ((app (app (K) v1) &)               v1)
    ((app (app (app (S) v1) v2) v3)     (app (app v1 v3)
                                             (app v2 v3)))))


;; Defines the [] operator, i.e. returns ctx[term].
(defun ctx-subst (ctx term)
  (declare (xargs :guard (and (ctx-p ctx)
                              (term-p term))))
  (pattern-match ctx
    ((ctx-left l r)      (app (ctx-subst l term) r))
    ((ctx-right l r)     (app l (ctx-subst r term)))
    (& ;; (ctx-box) or ill-formed
     term)))

;; Checks a valid reduction step t1 -> t2.  This takes terms t1 and t2 and a
;; certificate that this is a valid reduction, consisting of a context C and an
;; base-reducible term R, for which C[R] equals T1 and C[R'] is T2.
(defun arrow-validp (t1 t2 c r)
  (declare (xargs :guard (and (term-p t1)
                              (term-p t2)
                              (ctx-p c)
                              (term-p r))))
  ;; First check that R is base-reducible and get its reduction.
  (and (base-reducible r)
       (good-ctx-p c) ;; C is a valid ctx
       (equal t1 (ctx-subst c r)) ;; T1 matches C[R]
       (equal t2 (ctx-subst c (base-reduce r))) ;; T2 matches C[R'].
       ))

;; Defun-sk below is an ACL2 built-in that allows us to define functions
;; containing quantifiers in their bodies.  As used below, defun-sk introduces
;; (arrow t1 t2) so that it is equivalent to:  exists c and r such that
;; (arrow-validp t1 t2 c r).  

;; T1 -> T2 if there is a derivation satisfying arrow-validp.
(defun-sk arrow (t1 t2)
  (exists (c r)
          (arrow-validp t1 t2 c r)))

;; Quantifies over possible single-step transitions to define the "no-arrow"
;; predicate, i.e. returns T if t1 has no possible transitions.
(defun-sk no-arrow (t1)
  (forall (t2)
          (not (arrow t1 t2))))



;; A reduction (->*) derivation will be a list of triples (t2 c r), each
;; defining one step (as checked by arrow-validp, above.)  This simply checks that
;; it has valid syntax, not that it is a valid derivation.
(defun deriv-listp (x)
  (declare (xargs :guard t))
  (or (atom x)
      (let ((triple (first x)))
        (and (true-listp triple)
             (term-p (first triple))
             (ctx-p (second triple))
             (term-p (third triple))
             (deriv-listp (rest x))))))


;; Checks a valid reduction of ti to tf, i.e. multiple reduction steps.
;; Here a deriv is a list of triples (t2 c r), each of which gives one
;; reduction step.
(defun arrow*-validp (ti tf deriv)
  (declare (xargs :guard (and (term-p ti)
                              (term-p tf)
                              (deriv-listp deriv))))
  (if (atom deriv)
      ;; 0-step derivation: ti must equal tf.
      (equal ti tf)
    (b* ((triple            (first deriv))
         ((list t2 c r)     triple))
      (and (arrow-validp ti t2 c r)           ;; ti -> t2 is a valid reduction
           (arrow*-validp t2 tf (rest deriv)) ;; t2 ->* tf is a valid reduction.
           ))))


;; T1 ->* T2 if there exists a derivation showing that the steps are all valid.
(defun-sk arrow* (ti tf)
  (exists deriv
          (arrow*-validp ti tf deriv)))

;; The arrow* relation never allows for a well-formed term to transition to an
;; ill-formed term.
(defthm arrow*-does-not-allow-bad-syntax
  (implies (and (term-p ti)
                (not (term-p tf)))
           (not (arrow* ti tf))))



;;-------------------------------------------------------------------------
;; Definition of the single-step reduction function.
;;-------------------------------------------------------------------------

;; Given a term, determines whether it's reducible, returning a t/nil flag
;; saying so, and if so, a context and base term defining its reduction.

;; The way the ACL2 user returns a vector of three values is via the
;; idiom (mv a b c).  If a function call like (find-reduction x) returns three
;; values, the standard way to call it and field the values is:

;; (mv-let (a b c) 
;;         (find-reduction x)
;;         <body>)

;; which means ``evaluate (find-reduction x), bind a, b, and c respectively to
;; the three values returned, and evaluate and return <body> as the result.''
;; The expression <body> may refer to the variables a, b, c, x and whatever
;; other variables are in scope of the (mv-let ...) expression.

;; However, we have previously loaded the standard ACL2 book bstar.lisp, which
;; defines the macro b*.  This macro is like Lisp's let* in that it
;; sequentially binds variables and then evaluates and returns the final form.
;; However, b* extends let*'s behavior in several ways, including allowing us
;; to write
;; (b* (((mv a b c) (find-reduction x))
;;      ...)
;;     <body>)
;; which means ``bind a, b, and c to the values produced by (find-reduction x),
;; then do the other bindings described in the ..., and finally evaluate and
;; return <body>.''

(defun find-reduction (x)
  (declare (Xargs :guard (term-p x)))
  (b* (((when (base-reducible x))       (mv t (ctx-box) x))) ;; box[x] = x.
    (pattern-match x
      ((S)           (mv nil nil nil))    ;; not reducible
      ((K)           (mv nil nil nil))    ;; not reducible
      ((app l r)
       (if (value-p l)
           ;; reduce on the right
           (b* (((mv ok right-ctx base)  (find-reduction r))
                ((unless ok)             (mv nil nil nil))) ;; not reducible
             (mv t
                 (ctx-right l right-ctx)
                 base))
         ;; reduce on the left
         (b* (((mv ok left-ctx base)   (find-reduction l))
              ((unless ok)             (mv nil nil nil))) ;; not reducible
           (mv t
               (ctx-left left-ctx r)
               base))))
      (& (mv nil nil nil)))))


;; Reduce1 is the single-step reduction function.  It finds a term that is a
;; 1-step reduction from X, if one exists.  Returns a flag (exists or not) and
;; the new term.
(defun reduce1 (x)
  (declare (xargs :guard (term-p x)))
  (b* (((mv ok1 ctx base) (find-reduction x))
       ((unless ok1)      (mv nil nil)) ;; no reduction
       )
    ;; success -- the new term is ctx[base'].
    (mv t (ctx-subst ctx (base-reduce base)))))

;;-------------------------------------------------------------------------
;; Definition of the full reduction function.
;;-------------------------------------------------------------------------

;; We first define reduce-n, which runs N reduction steps, then returns a flag
;; saying whether it finished or not and the final term.
(defun reduce-n (n x)
  (declare (xargs :guard (and (natp n)
                              (term-p x))))
  (b* (((when (zp n)) ;; Ran out of our step limit.
        (mv nil x))
       ((mv reduc-ok new-x) (reduce1 x)))
    (if reduc-ok
        (reduce-n (- n 1) new-x)
      ;; Finished.
      (mv t x))))



;; Defines a Skolem function that chooses N big enough to completely reduce X.
;; If X is not reducible, we don't know what this returns.  More precisely,
;; ACL2 is logic of total functions, so (steps-required n) returns SOMETHING,
;; but the axioms introduced by the defchoose below do not specify what.

(defchoose steps-required (n) (x)
  (b* (((mv terminates &) (reduce-n n x)))
    terminates))


;; REDUCTION -- logically, runs reduce-n the required number of steps.  In
;; actual execution, it keeps stepping until it finishes, and perhaps doesn't
;; terminate.
(defun reduction (x)
  (declare (xargs :guard (term-p x)
                  :guard-hints (("goal" :use ((:instance steps-required
                                                         (n 1)))))))
  (mbe :logic
       ;; Logical definition: run (steps-required x) steps, and return that
       ;; result if done by that point; otherwise don't return anything (in
       ;; actuality, this means we didn't terminate)
       (b* ((steps (steps-required x))
            ((mv done new-x) (reduce-n steps x)))
           (if done
               (mv t new-x)
             (mv nil nil))) ;; doesn't terminate

       ;; Executable definition: just run step after step until we're done.  If
       ;; we don't terminate, we don't terminate.
       ;; Guard verification proves these two definitions equivalent.
       :exec (b* (((mv reduc-ok new-x) (reduce1 x)))
                 (if reduc-ok
                     (reduction new-x)
                   ;; Finished.
                   (mv t x)))))



;; reduction must produce a well-formed term if its input term is well-formed
;; and it terminates.
(defthm reduction-produces-good-term
  (b* (((mv terminates result) (reduction x)))
    (implies (and (term-p x)
                  terminates)
             (term-p result))))



;;-------------------------------------------------------------------------
;; Verification Task 1.
;;-------------------------------------------------------------------------

;; If reduction(t) terminates and returns t', then t ->* t' and t' -/>.
(defthm reduction-correct-and-complete
  (b* (((mv terminates result) (reduction x)))
    (implies terminates
             (and (arrow* x result)
                  (no-arrow result)))))



;;-------------------------------------------------------------------------
;; Verification 2: reduction terminates on any term that does not contain S.
;;-------------------------------------------------------------------------

;; Recognizes a term that has no S combinators.
(defun s-free-term-p (x)
  (declare (xargs :guard (term-p x)))
  (pattern-match x
    ((s) nil)
    ((k) t)
    ((app l r) (and (s-free-term-p l)
                    (s-free-term-p r)))))

;; The reduction function terminates when X is s-free.
(defthm reduction-terminates-when-s-free
  (b* (((mv terminates &) (reduction x)))
    (implies (s-free-term-p x)
             terminates)))


;;-------------------------------------------------------------------------
;; Verification 3: Prove that (reduction (ks n)) returns K when n is even
;; and (k k) when n is odd.
;;-------------------------------------------------------------------------


(defun ks (n)
  (declare (xargs :guard (natp n)))
  (if (zp n) ;; n is zero or not a natural number
      (K)
    (app (ks (- n 1)) (K))))


;; Theorem for Verification Task 3: (reduction (ks n)) returns K when n is
;; even, and (K K) when n is odd.
(defthm reduction-of-ks
  (b* (((mv terminates result) (reduction (ks n))))
    (and terminates     ;; it finishes
         (equal result
                (if (and (natp n) (oddp n))
                    (app (k) (k))  ;; N is odd -- (K K).
                  (k)  ;; N is even -- K
                  )))))


;; Tests...
(assert-event (b* (((mv ok result) (reduction (ks 1000))))
                  (and ok (equal result (k)))))

(assert-event (b* (((mv ok result) (reduction (ks 1001))))
                  (and ok (equal result (app (k) (k))))))
