(in-package "ACL2")

(include-book "ordinals/lexicographic-ordering" :dir :system)
(include-book "tools/bstar" :dir :system)

(defun node (x y)
  (declare (xargs :guard t))
  (cons x y))

(defun leaf ()
  (declare (xargs :guard t))
  :LEAF)

(defun treep (x)
  (if (consp x)
      (and (treep (car x))
           (treep (cdr x)))
      (eq x :LEAF)))

(defun depths_rec (d x)
  (cond ((atom x)
         (list d))
        (t (append (depths_rec (+ d 1) (car x))
                   (depths_rec (+ d 1) (cdr x))))))

(defun depths (x)
  (depths_rec 0 x))

(defun build_rec-measure (s d)
  (llist (len s) (- (ifix (car s)) (ifix d))))

(defun build_rec (d s)
  (declare (xargs :measure (build_rec-measure s d)
                  :well-founded-relation l<
                  :guard (and (integerp d)
                              (integer-listp s))
                  :verify-guards nil))
  (cond
   ((endp s)
    (mv t nil s))
   (t (let ((h (ifix (car s)))
	    (d (ifix d)))
	(cond ((< h d)
	       (mv t nil s))
	      ((equal h d)
	       (let ((s (cdr s)))
		 (mv nil (leaf) s)))
	      (t (mv-let (erp1 val1 s1)
			 (build_rec (+ d 1) s)
			 (cond
                          (erp1 (mv erp1 nil s1))
                          ((mbt (or (equal s1 s)
                                    (< (len s1) (len s))))
                           (mv-let (erp2 val2 s2)
                                   (build_rec (+ d 1) s1)
                                   (if erp2
                                       (mv erp2 nil s2)
                                       (mv nil (node val1 val2) s2))))
                          (t (mv :impossible nil s))))))))))

; The next two lemmas establish facts we need to prove that the guards of
; build_rec are always satisfied.  Because of the way MBT is defined, verifying
; the guards on build_rec will establish that the test marked MBT is
; is always true.

(defthm mbt-is-true
  (implies (not (equal (mv-nth 2 (build_rec d s)) s))
           (< (len (mv-nth 2 (build_rec d s))) (len s)))
  :rule-classes :linear)

(defthm integer-listp-preserved
  (implies (integer-listp s)
           (integer-listp (mv-nth 2 (build_rec d s))))
  :rule-classes
  (:rewrite
   (:type-prescription :corollary (implies (integer-listp s)
                                           (true-listp (mv-nth 2 (build_rec d s)))))))

(verify-guards build_rec)

; We now demonstrate that our build_rec is exactly yours.  We do it by:
; (1) lifting the runtime ifix tests out of the definition and relying on
;     the guards to insure that they are unnecessary,
; (2) defining Lisp macros to manipulate triples so as to treat the
;     first returned value as an error flag and the last as the new value of
;     s, and
; (3) exploiting the fact that the MBT test used in build_rec always
;     returns T.

; Here are the macros we mentioned in (2):

(defmacro fail () `(mv t nil s))

(defmacro succeed (val) `(mv nil ,val s))

(defmacro let! (bindings body)
  (cond ((endp bindings) body)
        (t `(mv-let (erp ,(car (car bindings)) s)
                    ,(cadr (car bindings))
                    (cond
                     (erp (fail))
                     (t (let! ,(cdr bindings) ,body)))))))

(def-b*-binder !
   `(let! ((,(car args) ,(car forms)))
          ,rest-expr))

(defthm booleanp-build_rec-flag
   (booleanp (car (build_rec d s)))
   :rule-classes :type-prescription)

; And here is the ``pretty'' version of our build_rec that more obviously
; implements the algorithm shown in the problem statement.
; This theorem shows that our build_rec satisfies the equation
; suggested in the problem statement, with appropriate handling of failure.

(defthm pretty-def-of-build_rec
  (implies (and (integerp d)
                (integer-listp s))
           (equal (build_rec d s)
                  (b* (((when (endp s))      (fail))
                       (h (first s))
                       ((when (< h d))       (fail))
                       ((when (equal h d))   (let ((s (rest s)))
                                               (succeed (leaf))))
                       ((! val1)   (build_rec (+ d 1) s))
                       ((! val2)   (build_rec (+ d 1) s)))
                      (succeed (node val1 val2)))))
  :rule-classes nil)

; Because of the work done by verify-guards, as explained in our comment about
; MBT in the definition of build_rec, the runtime (Common Lisp) version of our
; build_rec actually computes EXACTLY as suggested by the theorem above:
; provided d is an integer and s is a list of integers, no unnecessary runtime
; tests are performed.

; We now define build in terms of build_rec.

(defun build (s)
  (declare (xargs :guard (integer-listp s)))
  (mv-let (erp tree s)
          (build_rec 0 s)
          (cond
           (erp :FAIL)
           ((not (endp s)) :FAIL)
           (t tree))))

; -----------------------------------------------------------------
; Verification Task 1 (after 2 lemmas)

(defthm assoc-of-append
  (equal (append (append a b) c)
         (append a (append b c))))

(defthm soundness-lemma
  (implies (and (integer-listp s)
                (natp d)
                (not (mv-nth 0 (build_rec d s))))
           (equal (append (depths_rec d (mv-nth 1 (build_rec d s)))
                          (mv-nth 2 (build_rec d s)))
                  s))
  :rule-classes nil)

(defthm verification-task-1
  (implies (and (integer-listp s)
                (not (equal (build s) :fail)))
           (equal (depths (build s)) s))
  :hints (("Goal" :use (:instance soundness-lemma
                                  (d 0)))))

; -----------------------------------------------------------------
; Verification Task 2 (after 8 lemmas and one definition):

(defthm len-append
  (equal (len (append a b))
         (+ (len a) (len b))))

(defthm build_rec-append
  (implies (and (not (car (build_rec d a)))
                (integer-listp a)
                (integerp d))
           (and (not (car (build_rec d (append a b))))
                (equal (mv-nth 1 (build_rec d (append a b)))
                       (mv-nth 1 (build_rec d a)))
                (equal (mv-nth 2 (build_rec d (append a b)))
                       (append (mv-nth 2 (build_rec d a)) b)))))

(defthm integer-listp-depths_rec
  (implies (integerp d)
           (integer-listp (depths_rec d x))))

(defthm car-append
  (equal (car (append a b))
         (if (consp a) (car a) (car b))))

(defthm cdr-append
  (equal (cdr (append a b))
         (if (consp a) (append (cdr a) b) (cdr b))))

(defthm car-depths_rec
  (<= d (car (depths_rec d x)))
  :rule-classes :linear)

(defun tree-fix (x)
  (if (atom x)
      (leaf)
      (node (tree-fix (car x))
            (tree-fix (cdr x)))))

(defthm completeness-lemma
  (implies (integerp d)
           (and (not (mv-nth 0 (build_rec d (depths_rec d x))))
                (equal (mv-nth 1 (build_rec d (depths_rec d x)))
                       (tree-fix x))
                (equal (mv-nth 2 (build_rec d (depths_rec d x)))
                       nil))))

(defthm verification-task-2
  (implies (equal (build s) :fail)
           (not (equal (depths x) s))))
           
; -----------------------------------------------------------------
; Verification Task 3:

(defthm verification-task-3
  (and (lexp (build_rec-measure s d))
       (implies (and (not (endp s))
                     (<= (ifix d) (ifix (car s)))
                     (not (equal (ifix (car s)) (ifix d))))
                (l< (build_rec-measure s (+ (ifix d) 1))
                    (build_rec-measure s d)))
       (implies (and (not (endp s))
                     (<= (ifix d) (ifix (car s)))
                     (not (equal (ifix (car s)) (ifix d)))
                     (not (mv-nth 0 (build_rec (+ (ifix d) 1) s))))
                (l< (build_rec-measure (mv-nth 2 (build_rec (+ (ifix d) 1) s))
                                       (+ (ifix d) 1))
                    (build_rec-measure s d))))
  :hints (("Goal" :cases ((equal (mv-nth 2 (build_rec (+ (ifix d) 1) s)) s))))
  :rule-classes nil)
                
; -----------------------------------------------------------------
; Verification Task 4:

(defthm verification-task-4
  (and (equal (build '(1 3 3 2))
              (node (leaf) (node (node (leaf) (leaf)) (leaf))))
       (equal (build '(1 3 2 2))
              :FAIL))
  :rule-classes nil)

; To demonstrate that our build is reasonably efficient, we note that every
; Lisp s-expression is isomorphic to some tree and that by using depths and
; build we can build those isomorphic trees.  We can build quite large ones
; reasonably quickly.  For example, the following theorem just takes the defun
; of build_rec itself -- which is an s-expression -- and builds

(defthm verification-task-4-redux
  (let ((tree (tree-fix
               '(defun build_rec (d s)
                  (declare (xargs :measure (llist (len s)
                                                  (nfix (- (ifix (car s)) (ifix d))))
                                  :well-founded-relation l<
                                  :guard (and (integerp d)
                                              (integer-listp s))
                                  :verify-guards nil))
                  (cond
                   ((endp s)
                    (mv t nil s))
                   (t (let ((h (ifix (car s)))
                            (d (ifix d)))
                        (cond ((< h d)
                               (mv t nil s))
                              ((equal h d)
                               (let ((s (cdr s)))
                                 (mv nil (leaf) s)))
                              (t (mv-let (erp1 val1 s1)
                                         (build_rec (+ d 1) s)
                                         (cond
                                          (erp1 (mv erp1 nil s1))
                                          ((mbt (or (equal s1 s)
                                                    (< (len s1) (len s))))
                                           (mv-let (erp2 val2 s2)
                                                   (build_rec (+ d 1) s1)
                                                   (if erp2
                                                       (mv erp2 nil s2)
                                                       (mv nil (node val1 val2) s2))))
                                          (t (mv :impossible nil s)))))))))))))
    (and (equal (depths tree)
                '(1 2 4 5 5 5 7 8 10 12 13 13 13 15 17 19 20 20 18 18 19 19 17 14 12
                  10 11 12 14 16 17 17 17 18 18 16 14 15 15 6 6 9 10 10 10 11 12 13
                  13 9 9 11 14 16 18 19 19 17 15 15 17 18 18 16 14 14 17 18 19 19 18
                  19 20 21 21 17 18 19 20 20 19 22 24 25 25 23 21 22 23 25 25 25 25 21
                  18 18 20 22 23 24 24 23 25 26 27 27 25 25 24 26 28 29 30 31 31 27 28
                  30 32 33 34 34 33 35 36 36 36 37 37 35 32 29 29 31 32 33 33 32 34 35
                  36 36 34 34 33 34 36 37 38 39 39 37 38 40 41 42 42 40 40 36 32 28 28
                  30 31 32 33 33 29 27 23 19 17 13 10 8 5))
         (equal (build (depths tree))
                tree)))
  :rule-classes nil)
