; VSSTE 2012 Problem 5

; -----------------------------------------------------------------------------
;
;                              PRELIMINARIES
;
; -----------------------------------------------------------------------------

(in-package "ACL2")

;; Basic arithmetic, sets, and lists libraries
(include-book "finite-set-theory/osets/sets" :dir :system)
(include-book "arithmetic/top-with-meta" :dir :system)
(include-book "data-structures/list-defthms" :dir :system)
(include-book "unicode/take" :dir :system)

;; Miscellaneous tools
(include-book "tools/do-not" :dir :system)
(include-book "tools/bstar" :dir :system)
(include-book "tools/mv-nth" :dir :system)
(include-book "unicode/two-nats-measure" :dir :system)
(include-book "cutil/defsection" :dir :system)
(include-book "partial")

;; Tell ACL2 not to automatically generalize or fertilize
(do-not generalize fertilize)

;; A couple of missing lemmas.

(defthm car-of-append
  (equal (car (append x y))
         (if (consp x)
             (car x)
           (car y))))

(defthm last-of-append-singleton
  (equal (last (append x (list b)))
         (list b)))

(defthm len-positive-when-consp
  (implies (consp x)
           (posp (len x)))
  :rule-classes ((:type-prescription)))

(defthm len-zero-when-atom
  (implies (atom x)
           (equal (len x) 0))
  :rule-classes ((:type-prescription)))



; -----------------------------------------------------------------------------
;
;                         MODEL OF THE BFS ALGORITHM
;
; -----------------------------------------------------------------------------

; We'll just allow any object to be a vertex.

; We'll just use ordinary sets of objects to represent vertex sets.

; The following encapsulate introduces (SUCC V) as an abstract function that we
; assume returns the set of successors for a particular vertex.

(encapsulate
  (((succ *) => *))
  (local (defun succ (v)
           (declare (ignore v))
           nil))
  (defthm setp-of-succ
    (sets::setp (succ v))))



; For the algorithm, we're going to keep everything mostly the same as in the
; pseudocode.
;
; However, we changed the names of V, N, and C to Vs, Ns, and Cs.  This helps
; us remember that they are sets, and also avoids a name clash between v and V,
; since ACL2 is case insensitive.

(defun bfs-inner-loop (succs Vs Ns)

; This corresponds to the inner loop, whose pseudocode is:
;
;    for each w in succ(v) do
;      if w is not in Vs then
;        add w to Vs
;        add w to Ns
;      endif
;    endfor
;
; The idea is that succs should initially be set to succ(v), and we'll just
; recur down succs, doing the set-insertions into Vs and Ns as needed.
;
; We return the updated sets of Vs and Ns, as an mv-vector (mv Vs Ns).
;
; A subtle tweak is that, in the base case, we set-fix Vs and Ns.  This fixing
; is the identity on sets, but it gives us unconditional setp theorems.

  (declare (xargs :guard (and (sets::setp succs)
                              (sets::setp Vs)
                              (sets::setp Ns))))
  (b* (((when (sets::empty succs))
        (mv (sets::sfix Vs) (sets::sfix Ns)))
       (w (sets::head succs))
       ((unless (sets::in w Vs))
        (bfs-inner-loop (sets::tail succs)
                        (sets::insert w Vs)
                        (sets::insert w Ns))))
      (bfs-inner-loop (sets::tail succs) Vs Ns)))

(defun bfs-main-step (dest Vs Cs Ns d)

; This is one step of through the "while" statement of the BFS algorithm.
; It returns multiple values (mv Vs' Cs' Ns' d' flg), where:
; - Vs', Cs', Ns', and d' are the next values of Vs, Cs, Ns, and d;
; - flg is true if we are to go through the loop again, else nil.

  (declare (xargs :guard (and (sets::setp Vs)
                              (sets::setp Cs)
                              (sets::setp Ns)
                              (natp d))))
  (b* (((when (sets::empty Cs))
        ;; Cs has become empty, so we fail "no path."
        (mv Vs Cs Ns :NO-PATH nil))

       ;; Remove one vertex, v, from Cs.  I'll just choose to take its head.
       (v  (sets::head Cs))
       (Cs (sets::tail Cs))

       ;; if v = dest then return d.
       ((when (equal v dest)) ; return
        (mv Vs Cs Ns d nil))

       ;; for each w in succ(v) do ... endfor
       ((mv Vs Ns) (bfs-inner-loop (succ v) Vs Ns))

       ;; the if statement...         ; if Cs is empty then
       ((mv Cs Ns d)                  ;   Cs <- Ns
        (if (sets::empty Cs)          ;   Ns <- {}
            (mv Ns nil (+ d 1))       ;   d <- d + 1
          (mv Cs Ns d))))             ; endif

      (mv Vs Cs Ns d t)))

(defun bfs-main-loop (dest Vs Cs Ns d clk)

; This is basically the BFS algorithm from the ``while'' statement onwards,
; that is, it's just missing the preliminary setup.
;
;   - Dest is the dest vertex we've been given
;   - Vs, Cs, Ns are the sets we're manipulating
;   - d is the depth we've explored to, so far.
;
; What is clk?
;
; The BFS algorithm doesn't necessarily terminate, so we add a ``clock''
; argument to let us model when it diverges.  Basically, clk says how many more
; times we are willing to continue increasing D.
;
; We had originally used a simpler clock that always decreased on every
; recursion, but in the course of the verification tasks, we decided to change
; how the clock works so that it only decreases when Cs is empty.  This fits
; better with the simplifications we do when introdcing BFS-MAIN-LOOP-ALT,
; below.
;
; To show ACL2 that this function terminates, we have to use a two-part measure
; with the CLK as the major component and the size of Cs as the minor
; component.  (Perhaps the ACL2 Sedan's Calling Context Graph (CCG) analysis
; could get this automatically, but we have not tried.)

  (declare (xargs :measure (two-nats-measure clk
                                             (sets::cardinality Cs))))


  (b* (((when (zp clk))
         ;; Ran out of clock, there is no path within our artificial time bound.
         ;; (Later we'll consider the existence of a sufficiently large clk).
         ;; We return the keyword constant :DIVERGE to signal divergence.
         :DIVERGE)

        ((mv Vs Cs Ns d1 flg)
         (bfs-main-step dest Vs Cs Ns d)))

       (if flg
           ;; Continue the while loop.
           (bfs-main-loop dest Vs Cs Ns d1
                                  (if (equal d d1) clk (- clk 1)))
         d1)))

; Note: Defun-sk allows us to introduce a function containing a top-level
; quantifier in its body.  (bfs-finishes source dest Vs Cs Ns d) means ``there
; exists clk such that (bfs-main-loop source dest Vs Cs Ns d clk) is not
; :DIVERGE.''

(defun-sk bfs-finishes (dest Vs Cs Ns d)

; We will say that BFS-FINISHES if there exists some clock for which the main
; loop does not diverge.  Quantifiers are formalized with (not necessarily
; uncomputable) Skolem-like witness functions.  This defun-sk introduces
; BFS-FINISHES-WITNESS, which is such a clock, if one exists.

  (exists (clk)
          (not (equal (bfs-main-loop dest Vs Cs Ns d clk)
                      :DIVERGE))))

(defun bfs-main-loop-exec (dest Vs Cs Ns d)

; This variant of bfs-main-loop eliminates the clock.  The mbe form below
; specifies a logical definition in its :logic component.  That logical
; definition uses a skolem function, bfs-finishes-witness, produced by the
; defun-sk form above, to provide a sufficiently large clock (if such exists)
; to avoid returning :DIVERGE.  The :exec part avoids the clock and may not
; terminate, but is provably equal to the :logic part, as is guaranteed by the
; guard verification.  Our guard verification also guarantees that
; bfs-main-loop-exec is truly an executable program (except for the use of the
; abstract function, succ), not appealing to an existential quantifier (from
; bfs-finishes) during evaluation of its calls.  See the final section of this
; file for a demonstration.

  (declare (xargs :guard (and (sets::setp Vs)
                              (sets::setp Cs)
                              (sets::setp Ns)
                              (natp d))
                  ;; Delay the guard verification until after we prove some lemmas.
                  :verify-guards nil))
  (mbe :logic
       (if (bfs-finishes dest Vs Cs Ns d)
           ;; The main loop is going to finish if we give it enough clock, so
           ;; just go ahead and run it for enough steps.
           (let ((long-enough (bfs-finishes-witness dest Vs Cs Ns d)))
             (bfs-main-loop dest Vs Cs Ns d long-enough))
         :DIVERGE)
       :exec
       (b* (((mv Vs Cs Ns d1 flg)
             (bfs-main-step dest Vs Cs Ns d)))

           (if flg
               ;; Continue the while loop.
               (bfs-main-loop-exec dest Vs Cs Ns d1)
             d1))))

; We now proceed with the main verifications, using the logical definition of
; bfs-main-loop-exec.  We have not yet proved that the :logic and :exec
; versions, above, are equivalent.  Because this is somewhat tangential to the
; main verification, we do this proof (which is part of the guard verification
; of this function) at the end of this file.

(defun bfs (source dest)

; Finally, here is the whole BFS algorithm.  We do the setup stuff, and then
; (via quantification) check whether there exists any depth that will cause the
; main loop to terminate.
;
; You might object to this use of quantification.  After all, this is supposed
; to be a program, and it hardly seems fair to use universal quantification in
; its definition.  In a way this seems fair enough, but really the algorithm
; that has been proposed is only a partial function anyway, and all we're doing
; with this quantification is making explicit whether or not it indeed
; terminates.

  (declare (xargs :guard t
                  ;; put off verifying the guards until we've verified the
                  ;; guards for bfs-main-loop-exec
                  :verify-guards nil))
  (b* (                                ;; Preliminary setup stuff.
       (Vs (sets::insert source nil))  ;; Vs <- {source}
       (Cs (sets::insert source nil))  ;; Cs <- {source}
       (Ns nil)                        ;; Ns <- {}
       (d  0))                         ;; d  <- 0
      (bfs-main-loop-exec dest Vs Cs Ns d)))

; -----------------------------------------------------------------------------
;
;                        SIMPLIFYING BFS-MAIN-LOOP
;
; -----------------------------------------------------------------------------

; Before we try to do any proofs about BFS, we do a bit of work to show that the
; main loop can be thought of in simpler terms.

(defsection bfs-inner-loop

; It's easy to see that the BFS inner loop produces sets, and in fact these
; sets are just the same as:
;
;    Vs := succ(v) \union Vs
;    Ns := (succ(v) - Vs) \union Ns
;
; The proofs are typical pick-a-point proofs in the style of the ACL2 ordered
; sets library.


  (defthm setp-of-vs-after-bfs-inner-loop
    (sets::setp (mv-nth 0 (bfs-inner-loop succs Vs Ns))))

  (local (defthm l1
           (equal (sets::in a (mv-nth 0 (bfs-inner-loop succs Vs Ns)))
                  (or (sets::in a succs)
                      (sets::in a Vs)))))

  (defthm vs-after-bfs-inner-loop
    (equal (mv-nth 0 (bfs-inner-loop succs Vs Ns))
           (sets::union succs Vs)))

  (defthm setp-of-ns-after-bfs-inner-loop
    (sets::setp (mv-nth 1 (bfs-inner-loop succs Vs Ns))))

  (local (defthm l2
           (equal (sets::in a (mv-nth 1 (bfs-inner-loop succs Vs Ns)))
                  (or (sets::in a Ns)
                      (and (sets::in a succs)
                           (not (sets::in a Vs)))))))

  (defthm ns-after-bfs-inner-loop
    (equal (mv-nth 1 (bfs-inner-loop succs Vs Ns))
           (let ((missing (sets::difference succs Vs)))
             (sets::union missing Ns)))))


(defsection bfs-main-loop-redefinition

; The definition of bfs-main-loop is really ugly.  We really don't like the way
; it recurs on CS so strangely, and how it does the destination checking inline
; as it recurs.  We think these things will make it hard to reason about
; directly.  So, instead, below, we prove that bfs-main-loop is equal to
; bfs-main-loop-alt, where we've separated out the two kinds of recursion.  This
; leaves us with a pretty nice core, along with an ugly wrapper around it.

  (defun bfs-core (dest Vs Cs Ns d)
    (b* (((when (sets::empty Cs))
          (mv (sets::sfix Vs) (sets::sfix Ns)))
         (v  (sets::head Cs))
         (Cs (sets::tail Cs))
         ((mv Vs Ns) (bfs-inner-loop (succ v) Vs Ns)))
      (bfs-core dest Vs Cs Ns d)))

  (defun bfs-main-loop-alt (dest Vs Cs Ns d clk)
    (declare (xargs :measure (nfix clk)))
    (b* (((when (zp clk))
          :DIVERGE)
         ((when (sets::empty Cs))
          :NO-PATH)
         ((when (sets::in dest Cs))
          d)
         ((mv Vs Ns) (bfs-core dest Vs Cs Ns d))
         ((mv Cs Ns d clk)
          (mv Ns nil (+ d 1) (- clk 1))))
      (bfs-main-loop-alt dest Vs Cs Ns d clk)))

; It's a basic inductive proof to show bfs-main-loop is the same as the
; alternate definition, above.  The only lemma we need is to justify bringing
; the membership check out before the core.

  (local (defthm l1
           (implies (and (not (zp clk))
                         (not (sets::empty Cs))
                         (sets::in dest Cs))
                    (equal (bfs-main-loop dest Vs Cs Ns d clk)
                           d))
           :hints(("Goal"
                   :induct (bfs-main-loop dest Vs Cs Ns d clk)
                   :expand ((bfs-main-loop dest Vs Cs Ns d clk))))))

  (defthm bfs-main-loop-redefinition
    (equal (bfs-main-loop dest Vs Cs Ns d clk)
           (bfs-main-loop-alt dest Vs Cs Ns d clk))
    :hints(("Goal"
            :induct (bfs-main-loop dest Vs Cs Ns d clk)
            :do-not '(generalize fertilize)
            :expand ((bfs-main-loop dest Vs Cs Ns d clk)
                     (bfs-main-loop-alt dest Vs Cs Ns d clk))))))



(defsection union-succs

; We want to express the operation of BFS-CORE as a set operation.  For this, we
; want to be able to union together all of the successors of all of the
; vertices in Cs.  We write a new function to do this.

  (defun union-succs (Cs)
    (if (sets::empty Cs)
        nil
      (sets::union (succ (sets::head Cs))
                   (union-succs (sets::tail Cs)))))

  (defthm setp-of-union-succs
    (sets::setp (union-succs Cs)))

  (defthm successors-are-in-union-succs
    (implies (and (sets::in P CS)
                  (sets::in A (succ P)))
             (sets::in A (union-succs CS)))
    :rule-classes ((:rewrite)
                   (:rewrite :corollary
                             (implies (and (sets::in A (succ P))
                                           (sets::in P CS))
                                      (sets::in A (union-succs CS))))))


; It's also sometimes useful to find a predecessor of a particular vertex among
; the vertices of Cs.  Here's a function to do that.

  (defun find-predecessor (a Cs)
    (if (sets::empty Cs)
        (mv nil nil)
      (if (sets::in a (succ (sets::head Cs)))
          (mv t (sets::head Cs))
        (find-predecessor a (sets::tail Cs)))))

  (defthm find-predecessor-works-when-in-union-succs
    (implies (sets::in a (union-succs Cs))
             (mv-nth 0 (find-predecessor a Cs))))

  (defthm find-predecessor-is-a-predecessor
    (implies (mv-nth 0 (find-predecessor a Cs))
             (let ((p (mv-nth 1 (find-predecessor a Cs))))
               (and (sets::in p Cs)
                    (sets::in a (succ p)))))))



(defsection effect-of-bfs-core

; Now we want to abstract away from BFS-CORE and just think about set
; operations.  This is again just ordinary pick-a-point proofs in the style of
; the set theory library.

  (defthm setp-of-Vs-after-bfs-core
    (sets::setp (mv-nth 0 (bfs-core dest Vs Cs Ns d))))

  (local (defthm l1
           (equal (sets::in a (mv-nth 0 (bfs-core dest Vs Cs Ns d)))
                  (or (sets::in a Vs)
                      (sets::in a (union-succs Cs))))))

  (defthm Vs-after-bfs-core
    (equal (mv-nth 0 (bfs-core dest Vs Cs Ns d))
           (sets::union Vs (union-succs Cs))))


  (defthm setp-of-Ns-after-bfs-core
    (sets::setp (mv-nth 1 (bfs-core dest Vs Cs Ns d))))

  (local (defthm l2
           (equal (sets::in a (mv-nth 1 (bfs-core dest Vs Cs Ns d)))
                  (or (sets::in a Ns)
                      (and (not (sets::in a Vs))
                           (sets::in a (union-succs Cs)))))))

  (defthm Ns-after-bfs-core
    (equal (mv-nth 1 (bfs-core dest Vs Cs Ns d))
           (sets::union Ns (sets::difference (union-succs Cs) Vs)))))



(defsection even-simpler-definition

; Now we just take BFS-MAIN-LOOP-ALT and replace the call of BFS-CORE
; with the set-theory descriptions of the new Vs and Ns.  Then, just by some
; basic simplifications, we noticed that Ns doesn't even do anything now, and we
; can basically get all the way down to the following definition, which is by
; now very simple.

  (defun bfs-main-loop-alt2 (dest Vs Cs d clk)
    (declare (xargs :measure (nfix clk)))
    (b* (((when (zp clk))           :DIVERGE)
         ((when (sets::empty Cs))   :NO-PATH)
         ((when (sets::in dest Cs)) d)
         (Succs (union-succs Cs)))
      (bfs-main-loop-alt2 dest
                          (sets::union Vs Succs)
                          (sets::difference Succs Vs)
                          (+ 1 d)
                          (- clk 1))))

  (defthm bfs-main-loop-redefinition-2
    (equal (bfs-main-loop dest Vs Cs nil d clk)
           (bfs-main-loop-alt2 dest Vs Cs d clk))))





; -----------------------------------------------------------------------------
;
;            DEFINITIONS OF PATHS, REACHABILITY, AND INVARIANTS
;
; -----------------------------------------------------------------------------

; For our verification tasks, we need to talk about paths, shortest paths, the
; existence of paths, etc.  We now introduce some basic definitions related to
; paths and reachability.

(defsection path-defs

  (defun path-p (x)
    (cond ((atom x) nil)      ;; the empty list is NOT a valid path
          ((atom (cdr x)) t)  ;; a singleton list IS a valid path (length 0)
          (t
           (let ((v1 (first x))
                 (v2 (second x)))
             (and (sets::in v2 (succ v1))  ;; v2 must be a successor of v1, and
                  (path-p (cdr x)))))))    ;; (v2 ...) must be a path

  (defun path-length (x)
    ;; The length of a path is one less than the actual length of the list.
    (- (len x) 1))

  (defun path-from (x)
    ;; The vertex where a path starts.
    (car x))

  (defun path-to (x)
    ;; The vertex where a path ends.
    (car (last x)))

  ;; Just a few examples to show that our definitions of paths seem to be
  ;; basically reasonable.

  (assert-event (equal (path-length '(a b c)) 2))
  (assert-event (equal (path-from   '(a b c)) 'a))
  (assert-event (equal (path-to     '(a b c)) 'c)))



(defsection reachability-defs

  (defun-sk reachable (from to)
    ;; Does there exist a path from FROM to TO?
    (exists (path)
            (and (path-p path)
                 (equal (path-from path) from)
                 (equal (path-to path) to))))

  (defun-sk reachable-at-n (from to n)
    ;; Does there exist a path from FROM to TO of length N?
    (exists (path)
            (and (path-p path)
                 (equal (path-from path) from)
                 (equal (path-to path) to)
                 (equal (path-length path) n))))

  (defun-sk reachable-before-n (from to n)
    ;; Does there exist a path from FROM to TO with length less than N?
    (exists (path)
            (and (path-p path)
                 (equal (path-from path) from)
                 (equal (path-to path) to)
                 (< (path-length path) n))))

  (in-theory (disable reachable reachable-at-n reachable-before-n)))



(defsection invariant-defs

; Just reading the algorithm and thinking about it, it seems like the idea is
; for Vs to contain all the vertices we can reach within D steps, and for Cs to
; contain all the vertices we can reach at D steps (but not fewer steps).  We
; now express these intuitions as correctness predicates about the sets Vs and
; Cs.

  (defun-sk vs-correct (Vs from d)
    ;; Vs should contain all vertices reachable within D steps.
    (forall (to)
            (iff (sets::in to Vs)
                 (or (reachable-at-n from to d)
                     (reachable-before-n from to d)))))

  (defun-sk cs-correct (Cs from d)
    ;; Cs should contain all vertices reachable in D steps, but not reachable
    ;; in fewer than D steps.
    (forall (to)
            (iff (sets::in to Cs)
                 (and (reachable-at-n from to d)
                      (not (reachable-before-n from to d))))))

  (in-theory (disable vs-correct cs-correct)))




; -----------------------------------------------------------------------------
;
;                    GROUNDWORK AND PROOFS OF INVARIANTS
;
; -----------------------------------------------------------------------------

(defsection path-lemmas

  (defthm path-p-when-length-zero
    (implies (equal (len x) 0)
             (not (path-p x))))

  (defthm path-p-of-append-singleton
    (implies (path-p x)
             (equal (path-p (append x (list b)))
                    (sets::in b (succ (path-to x))))))

  (defthm path-p-of-simpler-take
    (implies (and (path-p x)
                  (posp n)
                  (< n (len x)))
             (path-p (simpler-take n x)))
    :hints(("Goal" :in-theory (enable simpler-take))))

  (defthm path-length-when-different-terminals
    (implies (and (not (equal (path-from p) (path-to p)))
                  (path-p p))
             (< 1 (len p)))
    :rule-classes ((:rewrite) (:linear)))

  (defthm normalize-path-from-when-length-zero
    (implies (and (equal (path-length p) 0)
                  (path-p p))
             (equal (path-from p)
                    (path-to p)))))



(defsection basic-reachability-facts

  (defthm natp-when-reachable-at-n
    (implies (reachable-at-n source dest n)
             (natp n))
    :rule-classes :forward-chaining
    :hints(("Goal" :in-theory (enable reachable-at-n))))

  (defthm not-reachable-before-zero
    (not (reachable-before-n from to 0))
    :hints(("Goal"
            :in-theory (enable reachable-before-n)
            :use ((:instance path-p
                             (x (reachable-before-n-witness from to 0)))))))

  (defthm self-reachable-at-zero
    (reachable-at-n from from 0)
    :hints(("Goal"
            :in-theory (disable reachable-at-n-suff)
            :use ((:instance reachable-at-n-suff
                             (path (list from))
                             (from from)
                             (to   from)
                             (n    0))))))

  (defthm nonself-not-reachable-at-zero
    (implies (not (equal from to))
             (not (reachable-at-n from to 0)))
    :hints(("Goal"
            :in-theory (enable reachable-at-n)
            :use ((:instance path-length-when-different-terminals
                             (p (reachable-at-n-witness from to 0)))))))

  (defthm reachable-before-n-of-increment
    (implies (reachable-before-n source a d)
             (reachable-before-n source a (+ 1 d)))
    :hints(("Goal"
            :in-theory (e/d (reachable-before-n)
                            (reachable-before-n-suff))
            :use ((:instance reachable-before-n-suff
                             (path (reachable-before-n-witness source a d))
                             (from source)
                             (to   a)
                             (n    (+ 1 d)))))))

  (defthm reachable-before-n-of-succ-and-increment
    (implies (and (reachable-before-n source p d)
                  (sets::in a (succ p))
                  (natp d))
             (reachable-before-n source a (+ 1 d)))
    :hints(("Goal"
            :in-theory (enable reachable-before-n)
            :use ((:instance reachable-before-n-suff
                             (path (append (reachable-before-n-witness source p d)
                                           (list a)))
                             (from source)
                             (to   a)
                             (n    (+ 1 d)))))))

  (defthm reachable-at-n-of-succ-and-increment
    (implies (and (reachable-at-n source p d)
                  (sets::in a (succ p)))
             (reachable-at-n source a (+ 1 d)))
    :hints(("Goal"
            :in-theory (e/d (reachable-at-n)
                            (reachable-at-n-suff))
            :use ((:instance reachable-at-n-suff
                             (path (append (reachable-at-n-witness source p d) (list a)))
                             (from source)
                             (to   a)
                             (n    (+ 1 d)))))))

  (defthm reachable-at-n-then-reachable-before-increment
    (implies (and (reachable-before-n source a (+ 1 d))
                  (not (reachable-at-n source a d))
                  (natp d))
             (reachable-before-n source a d))
    :hints(("Goal"
            :in-theory (enable reachable-before-n)
            :use ((:instance reachable-before-n-suff
                             (path (reachable-before-n-witness source a (+ 1 d)))
                             (from source)
                             (to   a)
                             (n    d))
                  (:instance reachable-at-n-suff
                             (path (reachable-before-n-witness source a (+ 1 d)))
                             (from source)
                             (to   a)
                             (n    d))))))

  (defthm reachable-at-zero-when-reachable-before-one
    (implies (reachable-before-n source dest 1)
             (reachable-at-n source dest 0))
    :hints(("Goal"
            :in-theory (enable reachable-before-n)
            :use ((:instance reachable-at-n-suff
                             (path (reachable-before-n-witness source dest 1))
                             (from source)
                             (to   dest)
                             (n    0))))))

  (defthm reachable-before-increment-when-reachable-at
    (implies (and (reachable-at-n source dest n)
                  (natp n))
             (reachable-before-n source dest (+ 1 n)))
    :hints(("Goal"
            :in-theory (enable reachable-at-n)
            :use ((:instance reachable-before-n-suff
                             (path (reachable-at-n-witness source dest n))
                             (from source)
                             (to   dest)
                             (n    (+ 1 n)))))))

  (defthm unreachable-at-n-when-unreachable-before-greater
    (implies (and (not (reachable-before-n source dest d))
                  (not (<= d n))
                  (natp n)
                  (natp d))
             (not (reachable-at-n source dest n)))
    :hints(("Goal" :expand ((reachable-at-n source dest n))))))




(defsection invariants-hold-at-the-start

; As a basis, we need to show that our invariants are satisfied initially.  Well,
; the initial set of Vs is just {source}, and the initial set of Cs is just
; {source}, and the initial D is just 0.  This all works out basically because
; the only paths of length 0 are from a source to itself.

  (defthm vs-initially-correct
    (vs-correct (sets::insert source nil) source 0)
    :hints(("Goal" :in-theory (enable vs-correct))))

  (defthm cs-initially-correct
    (cs-correct (sets::insert source nil) source 0)
    :hints(("Goal" :in-theory (enable cs-correct)))))



(defsection pre-path

; To show the invariants hold in the next step, we found the following
; construction to be useful.
;
; Suppose that A is reachable in D+1 steps.  Then, it is possible to construct
; a path to a predecessor of A that takes D steps.  To call this PRE-PATH for
; "predecessor path."
;
; The actual construction is very easy.  If we know that A is reachable in D+1
; steps, then let P be a path to A with length D+1.  Now,
;
;   - The path length of P is D+1
;   - The Lisp length of the list is D+2.
;
; If we just drop the final element of P, i.e., if we "take" the first D+1
; elements of the Lisp list, we get a new path to a predecessor of A whose path
; length is D.

  (defund pre-path (source a d)
    (simpler-take (+ 1 d) (reachable-at-n-witness source a (+ 1 d))))

; We now prove that the PRE-PATH is indeed a valid path of length D that does,
; in fact, lead to a predecessor of A.

  (local (in-theory (enable pre-path)))

  (local (defthm successor-of-simple-take
           (implies (and (path-p x)
                         (posp n)
                         (equal n (- (len x) 1)))
                    (sets::in (path-to x)
                              (succ (path-to (simpler-take n x)))))
           :hints(("Goal"
                   :induct (simpler-take n x)
                   :in-theory (enable simpler-take path-p)))))

  (defthm path-p-of-pre-path
    (implies (and (reachable-at-n source a (+ 1 d))
                  (natp d))
             (path-p (pre-path source a d)))
    :hints(("Goal" :in-theory (enable reachable-at-n))))

  (local (defthm len-of-pre-path
           (implies (and (reachable-at-n source a (+ 1 d))
                         (natp d))
                    (equal (len (pre-path source a d))
                           (+ 1 d)))
           :hints(("Goal" :in-theory (enable reachable-at-n)))))

  (local (defthm car-of-pre-path
           (implies (and (reachable-at-n source a (+ 1 d))
                         (natp d))
                    (equal (car (pre-path source a d))
                           source))
           :hints(("Goal" :in-theory (enable reachable-at-n)))))

  (defthm pre-path-leads-to-predecessor
    (implies (and (reachable-at-n source a (+ 1 d))
                  (natp d))
             (sets::in a (succ (path-to (pre-path source a d)))))
    :hints(("Goal"
            :in-theory (e/d (reachable-at-n)
                            (successor-of-simple-take))
            :use ((:instance successor-of-simple-take
                             (x (reachable-at-n-witness source a (+ 1 d)))
                             (n (+ 1 d)))))))

  (defthm pre-path-is-reachable-at-d
    (implies (and (reachable-at-n source a (+ 1 d))
                  (natp d))
             (reachable-at-n source (path-to (pre-path source a d)) d))
    :hints(("Goal"
            :in-theory (disable pre-path reachable-at-n-suff)
            :use ((:instance reachable-at-n-suff
                             (path (pre-path source a d))
                             (from source)
                             (to   (path-to (pre-path source a d)))
                             (n    d)))))))



(defsection invariants-hold-after-step

; Now we want to show that, assuming the Vs and Cs are correct for some depth d,
; then the next Vs are correct for depth d+1.

; Forward direction sketch.
;
; WTS: Every element A in Vs \union (union-succs Cs) is reachable at or before
; D+1 steps.
;
; Well, if A was a member of Vs, then it was already reachable at or before D
; steps, so it will still be reachable in D+1 steps.
;
; Otherwise, A was a member of (union-succs Cs).  So let P be the predecessor
; of A, i.e., P \in Cs and A \in succ(P).  Then we know P is reachable at depth
; D, so A is reachable in D+1.

  (defthm vs-correct-forward-part-1
    (implies (and (vs-correct Vs source d)
                  (sets::in a Vs))
             (reachable-before-n source a (+ 1 d)))
    :hints(("Goal"
            :in-theory (enable reachable-at-n)
            :use ((:instance vs-correct-necc
                             (to   a)
                             (vs   vs)
                             (d    d)
                             (from source))))))

  (defthm vs-correct-forward-part-2
    (implies (and (cs-correct Cs source d)
                  (sets::in b (union-succs Cs)))
             (reachable-at-n source b (+ 1 d)))
    :hints(("Goal"
            :use ((:instance cs-correct-necc
                             (to   (mv-nth 1 (find-predecessor b Cs)))
                             (cs   cs)
                             (from source)
                             (d    d))))))

  (defthm vs-correct-forward
    (implies (and (vs-correct Vs source d)
                  (cs-correct Cs source d)
                  (sets::in a (sets::union Vs (union-succs Cs)))
                  (not (reachable-at-n source a (+ 1 d))))
             (reachable-before-n source a (+ 1 d))))


; Backward direction sketch.
;
; WTS: Every vertex A that is reachable at or before D+1 steps is in Vs \union
; (union-succs Cs).
;
; Case 1.  A is reachable before depth D+1.  Then A was reachable at or before
; depth D, and hence A was already in Vs.
;
; Case 2.  A is reachable only at depth D+1.  Then, the PRE-PATH of A is a path
; to a predecessor of A, say P, that is reachable at only at depth D.  Hence,
; since the Cs are correct, P is among the Cs, and hence A must be among
; (union-succs Cs).

  (defthm vs-correct-backward-part-1
    (implies (and (vs-correct Vs source d)
                  (reachable-before-n source a (+ 1 d))
                  (natp d))
             (sets::in a Vs))
    :hints(("Goal"
            :use ((:instance vs-correct-necc
                             (to   a)
                             (vs   vs)
                             (from source)
                             (d    d))))))

  (defthm vs-correct-backward-part-2
    (implies (and (cs-correct Cs source d)
                  (reachable-at-n source a (+ 1 d))
                  (not (reachable-before-n source a (+ 1 d)))
                  (natp d))
             (sets::in a (union-succs Cs)))
    :hints(("Goal"
            :in-theory (disable successors-are-in-union-succs
                                path-to)
            :use ((:instance successors-are-in-union-succs
                             (p (path-to (pre-path source a d))))
                  (:instance cs-correct-necc
                             (from source)
                             (to   (path-to (pre-path source a d))))))))

  (defthm vs-correct-backward
    (implies (and (vs-correct Vs source d)
                  (cs-correct Cs source d)
                  (or (reachable-at-n source a (+ 1 d))
                      (reachable-before-n source a (+ 1 d)))
                  (natp d))
             (sets::in a (sets::union Vs (union-succs Cs))))
    :hints(("Goal"
            :in-theory (disable vs-correct-backward-part-1
                                vs-correct-backward-part-2)
            :use ((:instance vs-correct-backward-part-1)
                  (:instance vs-correct-backward-part-2)))))


; The proof that the VS invariant holds after the step is now just the
; combination of the above directions.

  (defthm vs-invariant-step
    (implies (and (vs-correct Vs source d)
                  (cs-correct Cs source d)
                  (natp d))
             (vs-correct (sets::union Vs (union-succs Cs)) source (+ 1 d)))
    :hints(("Goal"
            :expand ((vs-correct (sets::union Vs (union-succs Cs)) source (+ 1 d))))))


; The proof that the Cs invariant holds after the next step is actually very
; similar, and essentially involves the same reasoning about Vs and
; (union-succs CS).  So, we actually don't need any more lemmas beyond what
; we've already proven.

  (defthm cs-invariant-step
    (implies (and (vs-correct Vs source d)
                  (cs-correct Cs source d)
                  (natp d))
             (cs-correct (sets::difference (union-succs Cs) Vs) source (+ 1 d)))
    :hints(("Goal"
            :expand ((cs-correct (sets::difference (union-succs Cs) Vs) source (+ 1 d)))))))






; -----------------------------------------------------------------------------
;
;                                SOUNDNESS
;
; -----------------------------------------------------------------------------

; Verification Task 1. (Soundness).
;
; Goal: Verify that whenever function bfs returns an integer n, this is indeed
; the length of the shortest path from source to dest.


(defsection soundness

; The hard work of establishing the invariants is over.  The consequence of the
; invariant about Cs is it has exactly those vertices that are reachable at N
; steps but not before N steps.  So, a simple inductive proof using this
; invariant lets us see that BFS-MAIN-LOOP produces the right result, and the
; soundness of BFS follows immediately.

  (defthm cs-correct-consequence
    (implies (and (cs-correct Cs source d)
                  (sets::in dest Cs))
             (and (reachable-at-n source dest d)
                  (not (reachable-before-n source dest d))))
    :hints(("Goal"
            :use ((:instance cs-correct-necc
                             (to   dest)
                             (from source)
                             (d    d))))))

  (defthm soundness-crux
    (let ((n (bfs-main-loop-alt2 dest Vs Cs d clk)))
      (implies (and (vs-correct Vs source d)
                    (cs-correct Cs source d)
                    (natp d)
                    (integerp n))
               (and (reachable-at-n source dest n)
                    (not (reachable-before-n source dest n)))))
    :hints(("Goal" :induct (bfs-main-loop-alt2 dest Vs Cs d clk))))

  (defthm soundness
    (let ((n (bfs source dest)))
      (implies (integerp n)
               (and (reachable-at-n source dest n)
                    (not (reachable-before-n source dest n)))))))




; -----------------------------------------------------------------------------
;
;                                COMPLETENESS
;
; -----------------------------------------------------------------------------

; Verification Task 2.  (Completeness).
;
; Goal: Verify that whenever BFS reports failure, there is no path from source
; to dest.
;
; We're going to interpret "reporting failure" as either diverging or returning
; NO-PATH.

(defsection completeness-crux

; The main completeness argument is the following, inductive proof.  The basic
; idea is: suppose N is the minimum number of steps that are necessary to reach
; the destination.  Then, if we haven't yet reached depth N, and we have enough
; clock to get to N, then we can be sure that the algorithm is going to return
; N as the distance to dest.

  (local (defun induct-hint (source dest Vs Cs d clk)
           ;; This is like BFS-MAIN-LOOP-ALT2, but doesn't include a base case
           ;; for when CS is empty.  That case seems kind of tricky to cover,
           ;; and we don't really need to regard it separately for this proof.
           (declare (xargs :measure (nfix clk)))
           (b* (((when (zp clk)) :diverge)
                ((when (sets::in dest Cs)) d)
                (Succs (union-succs Cs)))
             (induct-hint source dest
                          (sets::union Vs Succs)
                          (sets::difference Succs Vs)
                          (+ 1 d)
                          (- clk 1)))))

  (local (defthm cs-correct-consequence-for-completeness
           (implies (and (cs-correct Cs source d)
                         (reachable-at-n source dest d)
                         (not (reachable-before-n source dest d)))
                    (sets::in dest Cs))
           :hints(("Goal"
                   :use ((:instance cs-correct-necc
                                    (to   dest)
                                    (from source)
                                    (d    d)))))))

  (local (defthm base-case-help
           (implies (and (cs-correct cs source d)
                         (sets::in dest cs)
                         (not (reachable-before-n source dest n))
                         (< d n)
                         (natp n)
                         (natp d))
                    (not (reachable-at-n source dest n)))
           :hints(("Goal"
                   :in-theory (disable cs-correct-consequence)
                   :use ((:instance cs-correct-consequence))))))

  (local (defthm completeness-crux
           (implies (and (vs-correct Vs source d)
                         (cs-correct Cs source d)
                         (reachable-at-n source dest n)
                         (not (reachable-before-n source dest n))
                         (<= d n)
                         (< n (+ d clk))
                         (natp n)
                         (natp clk)
                         (natp d))
                    (equal (bfs-main-loop-alt2 dest Vs Cs d clk)
                           n))
           :hints(("Goal"
                   :induct (induct-hint source dest Vs Cs d clk)
                   ;; bozo yuck!
                   :in-theory (disable natp-when-reachable-at-n)))))


; With the inductive proof completed, we can specialize it to the initial
; values of Vs and Cs.  This is now pretty nice: if we know the minimal N
; needed to reach DEST, we can just run for N+1 steps and be sure to get back
; N.

  (defthm refined-crux
    (implies (and (reachable-at-n source dest n)
                  (not (reachable-before-n source dest n))
                  (natp n))
             (equal (bfs-main-loop-alt2 dest
                                        (sets::insert source nil)
                                        (sets::insert source nil)
                                        0
                                        (+ n 1))
                    n))
    :hints(("Goal"
            :in-theory (disable completeness-crux)
            :use ((:instance completeness-crux
                             (Vs  (sets::insert source nil))
                             (Cs  (sets::insert source nil))
                             (d   0)
                             (clk (+ n 1))))))))



(defsection find-least-reachable

; To make use of the refined crux, we need to know the smallest N such that
; dest is reachable at N.  Well, it's easy enough to find such an N as long as
; we know some depth for which dest is reachable.

; That is, assuming we can reach dest from source in N steps,
; find-least-reachable tries to find a better bound, M, where M <= N and dest
; is reachable in M steps but not before M steps.

  (defun find-least-reachable (source dest n)
    (if (zp n)
        0
      (if (reachable-before-n source dest n)
          (find-least-reachable source dest (- n 1))
        n)))

  (local (defthm lemma
           (implies (and (reachable-at-n source dest n)
                         (natp n)
                         (natp d)
                         (<= d n)
                         (reachable-before-n source dest (+ 1 d)))
                    (let ((m (find-least-reachable source dest d)))
                      (and (reachable-at-n source dest m)
                           (not (reachable-before-n source dest m)))))
           :hints(("Goal" :induct (find-least-reachable source dest d)))))

  (defthm find-least-reachable-correct
    (implies (and (reachable-at-n source dest n)
                  (natp n))
             (let ((m (find-least-reachable source dest n)))
               (and (reachable-at-n source dest m)
                    (not (reachable-before-n source dest m)))))
    :hints(("Goal" :use ((:instance lemma (d n)))))))




(defsection enough-clock

; We now use find-least-reachable to figure out the right clock to use with our
; refined-crux.
;
; The essential argument is: if dest is reachable, then it must be reachable at
; some N.  To find a particular N that it is reachable at, we can just take the
; path length of (reachable-witness source dest).
;
; Of course, this N might not be good enough, because for the completeness crux
; we need the minimum N that dest can be reached at.  But to correct for that
; we'll just use find-least-reachable to go find the smallest N.

  (defun enough-clock (source dest)
    (let* ((path   (reachable-witness source dest))
           (len    (path-length path))
           (shrink (find-least-reachable source dest len)))
      (+ shrink 1)))

  (local (defthm lemma-1
           (implies (reachable source dest)
                    (reachable-at-n source dest
                                    (path-length (reachable-witness source dest))))
           :hints(("Goal" :in-theory (enable reachable)))))


  (local (defthm refined-crux-2
           (implies (and (reachable-at-n source dest n)
                         (natp n))
                    (integerp (bfs-main-loop-alt2 dest
                                                  (sets::insert source nil)
                                                  (sets::insert source nil)
                                                  0
                                                  (+ (find-least-reachable source dest n) 1))))
           :hints(("Goal"
                   :use ((:instance refined-crux
                                    (n (find-least-reachable source dest n))))))))

  (defthm very-refined-crux
    (implies (reachable source dest)
             (integerp (bfs-main-loop-alt2 dest
                                           (sets::insert source nil)
                                           (sets::insert source nil)
                                           0
                                           (enough-clock source dest))))
    :hints(("Goal"
            :use ((:instance refined-crux-2
                             (n (path-length (reachable-witness source dest))))
                  (:instance lemma-1)))))

  (in-theory (disable enough-clock)))




(defsection final-clock-nonsense

; We now know that given enough clock, the main loop will produce an integer
; for any reachable source/dest.  But, our top-level BFS algorithm doesn't use
; our notion of enough-clock, intead it just asks if the algorithm is going to
; terminate, and runs for however many steps are needed to accomplish that.

; Bridging this gap is a little technical, but basically straightforward.

  (defthm non-diverging-clocks-always-agree
    (implies (and (not (equal (bfs-main-loop-alt2 dest vs cs d clk) :diverge))
                  (not (equal (bfs-main-loop-alt2 dest vs cs d clk2) :diverge)))
             (equal (bfs-main-loop-alt2 dest vs cs d clk2)
                    (bfs-main-loop-alt2 dest vs cs d clk)))
    :rule-classes nil)

  (defthm witness-does-not-diverge-if-any-clock-suffices
    (implies (not (equal (bfs-main-loop-alt2 dest Vs Cs d clk) :diverge))
             (equal (bfs-main-loop-alt2 dest Vs Cs d clk)
                    (bfs-main-loop-alt2 dest Vs Cs d
                                        (bfs-finishes-witness dest Vs Cs nil d))))
    :rule-classes nil
    :hints(("Goal"
            :do-not-induct t
            :use ((:instance non-diverging-clocks-always-agree
                             (clk  clk)
                             (clk2 (bfs-finishes-witness dest Vs Cs nil d)))
                  (:instance bfs-finishes-suff (ns nil))))))

  (defthm always-diverge-if-diverge-for-witness
    (implies (equal (bfs-main-loop-alt2 dest Vs Cs d
                                        (bfs-finishes-witness dest Vs Cs nil d))
                    :diverge)
             (equal (bfs-main-loop-alt2 dest Vs Cs d clk)
                    :diverge))
    :hints(("Goal"
            :do-not-induct t
            :use ((:instance non-diverging-clocks-always-agree
                             (clk  clk)
                             (clk2 (bfs-finishes-witness dest Vs Cs nil d)))
                  (:instance bfs-finishes-suff (ns nil))))))

  (defthm any-reachable-path-is-found
    (implies (reachable source dest)
             (integerp (bfs source dest)))
    :hints(("Goal"
            :in-theory (disable very-refined-crux)
            :use ((:instance very-refined-crux)
                  (:instance witness-does-not-diverge-if-any-clock-suffices
                             (vs  (sets::insert source nil))
                             (Cs  (sets::insert source nil))
                             (d   0)
                             (clk (enough-clock source dest)))))))

  (defthm completeness
    (implies (not (integerp (bfs source dest)))
             (not (reachable source dest)))
    :hints(("Goal" :use ((:instance any-reachable-path-is-found))))))

; -----------------------------------------------------------------------------
;
;                            EXECUTABILITY
;
; -----------------------------------------------------------------------------

; As promised in a comment in bfs-main-loop-exec, we now demonstrate that our
; use of mbe in that function can indeed support executability.  The first step
; is to verify the guards of bfs-main-loop-exec, allowing us to execute bfs
; directly, even though its logical definition has a quantifier in it.

(defthm setp-bfs-inner-loop
  (and (sets::setp (mv-nth 0 (bfs-inner-loop succs Vs Ns)))
       (sets::setp (mv-nth 1 (bfs-inner-loop succs Vs Ns)))))

(defthm setp-bfs-main-step
  (implies (and (sets::setp Vs)
                (sets::setp Cs)
                (sets::setp Ns))
           (and (sets::setp (mv-nth 0 (bfs-main-step dest Vs Cs Ns d)))
                (sets::setp (mv-nth 1 (bfs-main-step dest Vs Cs Ns d)))
                (sets::setp (mv-nth 2 (bfs-main-step dest Vs Cs Ns d))))))

(defthm natp-bfs-main-step
  (implies (and (natp d)
                (mv-nth 4 (bfs-main-step dest vs cs ns d)))
           (natp (mv-nth 3 (bfs-main-step dest Vs Cs Ns d)))))

(defthm bfs-main-step-does-not-diverge
  (implies (integerp d)
           (not (equal (mv-nth 3 (bfs-main-step dest vs cs ns d))
                       :diverge))))

(verify-guards bfs-main-loop-exec
  :otf-flg t
  :hints ('(:use
            ;; Here we're making use of a generic theorem, pf-run-is-loop, by
            ;; doing a functional instantiation.  Thus, we associate new
            ;; definitions with the generic functions that the generic theorem
            ;; depends on.  ACL2 verifies that these new definitions satisfy
            ;; all the constraints of those generic functions, which allows
            ;; ACL2 to conclude that the theorem holds with this second-order
            ;; substitution.  Pf-run-is-loop essentially states the equivalence
            ;; of two functions: a tail-recursive function with a clock (like
            ;; bfs-main-loop), where the clock is instantiated with a witness
            ;; that is picked to be "big enough"; and a similar tail-recursive
            ;; function that does not have a clock, but might not terminate.
            ((:instance
              (:functional-instance
               pf-run-is-loop
               (pf-next
                (lambda (st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st)
                       ((mv vs cs ns d &)
                        (bfs-main-step dest vs cs ns d)))
                    (mv dest vs cs ns d))))
               (pf-done
                (lambda (st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (not (mv-nth 4 (bfs-main-step dest vs cs ns d))))))
               (pf-retval
                (lambda (st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (mv-nth 3 (bfs-main-step dest vs cs ns d)))))
               (pf-meas
                (lambda (clk st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (two-nats-measure clk
                                      (sets::cardinality cs)))))
               (pf-over
                (lambda (clk st) (zp clk)))
               (pf-decr
                (lambda (clk st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (if (equal d (mv-nth 3 (bfs-main-step dest vs cs ns d)))
                        clk
                      (- clk 1)))))
               (pf-incr
                (lambda (clk) (+ 1 (nfix clk))))
               (pf-diverge (lambda () :diverge))
               (pf-run-clk
                (lambda (clk st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (bfs-main-loop dest vs cs ns d clk))))
               (pf-terminates
                (lambda (st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (bfs-finishes dest vs cs ns d))))
               (pf-terminates-witness
                (lambda (st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (bfs-finishes-witness dest vs cs ns d))))
               (pf-run
                (lambda (st)
                  (b* (((mv ?dest ?vs ?cs ?ns ?d) st))
                    (bfs-main-loop-exec dest vs cs ns d)))))
              (st (mv dest vs cs ns d))))
            :in-theory (Disable bfs-main-loop-redefinition
                                bfs-main-loop-redefinition-2
                                sets::in
                                bfs-main-step)
            :do-not-induct t)
          (and stable-under-simplificationp
               '(:in-theory (e/d (bfs-main-step)
                                 (bfs-main-loop-redefinition
                                  bfs-main-loop-redefinition-2
                                  sets::in))))))
                  
(verify-guards bfs)

; Now we can evaluate calls of bfs, once we arrange to be able to evaluate
; calls of succ.  We illustrate how to do so by defining a very simple concrete
; version of succ, succ-exec.  The defattach form below generates proof
; obligations (see
; http://www.cs.utexas.edu/users/moore/acl2/v4-3/DEFATTACH.html for
; documentation) to guarantee that succ-exec satisfies the constraints on the
; abstract function, succ.  Upon successful execution of the defattach form, a
; call of succ is evaluated by making the corresponding call to succ-exec, as
; checked in one instance by the assert-event form below.  (However, this
; attachment of succ-exec to succ is disabled during proofs, as explained in
; the documentation mentioned above.)

; Consider the following graph, where, for example, there are edges from 
; node 0 to nodes 1 and 2.  Note that node 6 is isolated.

;      5             6
;      ^
;      |
;      2 ----
;      ^     |
;      |     v
;      0 --> 1 --> 3
;            |
;            v
;            4

; Concretely, we could represent this graph with the successor function
; defined below, as the executable function succ-exec.

(defun succ-exec (v)
  (declare (xargs :guard t))
  (case v
    (0 (sets::insert 1 (sets::insert 2 nil)))
    (1 (sets::insert 3 (sets::insert 4 nil)))
    (2 (sets::insert 5 (sets::insert 1 nil)))
    (otherwise nil)))

; We can then tell ACL2 to ``attach'' succ-exec to succ in the sense that
; when asked to evaluate succ, ACL2 will evaluate succ-exec instead.
; This attachment is only possible because we have proved that succ-exec
; is guard verified and satisfies the constraints on succ.  Furthermore,
; ACL2 will use succ-exec for succ only in the ``evaluation theory,''
; and, in particular, will NOT use it when it is proving theorems about
; succ.

(defattach succ succ-exec)

; Finally, we exhibit the answers computed by bfs for all the source and
; destination node combinations in {0,1,2,3,4,5,6}.  This demonstrates
; that our bfs is an executable program when succ is executable.

(assert-event
 (equal (list (list 0 (list (bfs 0 0) (bfs 0 1) (bfs 0 2)
                            (bfs 0 3) (bfs 0 4) (bfs 0 5) (bfs 0 6)))
              (list 1 (list (bfs 1 0) (bfs 1 1) (bfs 1 2) (bfs 1 3)
                            (bfs 1 4) (bfs 1 5) (bfs 1 6)))
              (list 2 (list (bfs 2 0) (bfs 2 1) (bfs 2 2)
                            (bfs 2 3) (bfs 2 4) (bfs 2 5) (bfs 2 6)))
              (list 3 (list (bfs 3 0) (bfs 3 1) (bfs 3 2)
                            (bfs 3 3) (bfs 3 4) (bfs 3 5) (bfs 3 6)))
              (list 4 (list (bfs 4 0) (bfs 4 1) (bfs 4 2)
                            (bfs 4 3) (bfs 4 4) (bfs 4 5) (bfs 4 6)))
              (list 5 (list (bfs 5 0) (bfs 5 1) (bfs 5 2)
                            (bfs 5 3) (bfs 5 4) (bfs 5 5) (bfs 5 6)))
              (list 6 (list (bfs 6 0) (bfs 6 1) (bfs 6 2)
                            (bfs 6 3) (bfs 6 4) (bfs 6 5) (bfs 6 6))))

;             0        1        2        3        4        5        6

        '((0 (0        1        1        2        2        2        :NO-PATH))
          (1 (:NO-PATH 0        :NO-PATH 1        1        :NO-PATH :NO-PATH))
          (2 (:NO-PATH 1        0        2        2        1        :NO-PATH))
          (3 (:NO-PATH :NO-PATH :NO-PATH 0        :NO-PATH :NO-PATH :NO-PATH))
          (4 (:NO-PATH :NO-PATH :NO-PATH :NO-PATH 0        :NO-PATH :NO-PATH))
          (5 (:NO-PATH :NO-PATH :NO-PATH :NO-PATH :NO-PATH 0        :NO-PATH))
          (6 (:NO-PATH :NO-PATH :NO-PATH :NO-PATH :NO-PATH :NO-PATH 0)))))


