; This book models a state machine using several functions, including:
; - a tail-recursive function pf-run-clk; and
; - a function pf-run, which calls pf-run-clk with a clock that is big enough
;   to guarantee termination, chosen nonconstructively, if such exists.
; The final theorem in this book, pf-run-is-loop, shows that pf-run satisfies
; the expected clock-free recursion equation, which can be viewed as a
; clock-free definition that runs the state machine in the obvious way but
; might not terminte.

(in-package "ACL2")

(encapsulate
  ((pf-next (st) st)
   (pf-done (st) flg)
   (pf-retval (st) res)
   (pf-meas (clk st) ord)
   (pf-over (clk st) flg)
   (pf-decr (clk st) clk)
   (pf-incr (clk) clk)
   (pf-diverge () fail))

  (set-ignore-ok t)
  (set-irrelevant-formals-ok t)

  (local
   (progn

     (defun pf-next (st)
       (list (+ 1 (car st)) (not (cadr st))))

     (defun pf-done (st)
       (equal (car st) 100))

     (defun pf-retval (st)
       (- st (if (cadr st) 100 0)))

     (defun pf-meas (clk st)
       (+ (* 2 (nfix clk))
          (if (cadr st) 1 0)))
          

     (defun pf-over (clk st)
       (zp clk))

     (defun pf-decr (clk st)
       (let ((clk (nfix clk)))
         (if (cadr st) clk (1- clk))))

     (defun pf-incr (clk)
       (1+ (nfix clk)))

     (defun pf-diverge () :pf-diverge)))

  (defthm run-clk-pf-measure
    (and (o-p (pf-meas clk st))
         (implies (and (not (pf-over clk st))
                       (not (pf-done st)))
                  (o< (pf-meas (pf-decr clk st) (pf-next st))
                      (pf-meas clk st)))))

  (defthm pf-decr-incr
    (and (o< (pf-meas clk st)
             (pf-meas (pf-incr clk) st))
         (o<= (pf-meas clk (pf-next st))
              (pf-meas (pf-decr (pf-incr clk) st) (pf-next st)))))

  (defthm pf-over-incr
    (not (pf-over (pf-incr clk) st)))

  ;;(defthm pf-retval-diverge
  ;;  (not (equal (pf-retval st) (pf-diverge))))

  (defthm pf-decr-preserves-o<
    (implies (not (pf-over clk1 st))
             (iff (o< (pf-meas (pf-decr clk1 st) (pf-next st))
                      (pf-meas (pf-decr clk2 st) (pf-next st)))
                  (o< (pf-meas clk1 st)
                      (pf-meas clk2 st)))))

  (defthm pf-over-respects-o<
    (implies (and (not (pf-over clk1 st))
                  (o<= (pf-meas clk1 st)
                       (pf-meas clk2 st)))
             (not (pf-over clk2 st)))))

(defun pf-run-clk (clk st)
  (declare (xargs :measure (pf-meas clk st)))
  (cond ((pf-over clk st) (pf-diverge))
        ((pf-done st)     (pf-retval st))
        (t (pf-run-clk (pf-decr clk st)
                       (pf-next st)))))

(defun-sk pf-terminates (st)
  (exists clk
          (not (equal (pf-run-clk clk st) (pf-diverge)))))

(defun pf-run (st)
  (if (pf-terminates st)
      (pf-run-clk (pf-terminates-witness st) st)
    (pf-diverge)))

(defthm not-pf-over-and-done
  (implies (and (pf-done st)
                (not (equal (pf-retval st) (pf-diverge))))
           (not (pf-over (pf-terminates-witness st) st)))
  :hints (("goal" :use ((:instance pf-terminates-suff
                         (clk (pf-incr clk)))))))

(local (defun induct-hint (clk1 clk2 st)
         (declare (xargs :measure (pf-meas clk1 st)))
         (cond ((pf-over clk1 st) (pf-diverge))
               ((pf-over clk2 st) (pf-diverge))
               ((pf-done st) (pf-retval st))
               (t (induct-hint (pf-decr clk1 st)
                               (pf-decr clk2 st)
                               (pf-next st))))))

(defthm pf-run-clk-with-sufficient-clks
  (implies (and (not (equal (pf-run-clk clk1 st) (pf-diverge)))
                (not (equal (pf-run-clk clk2 st) (pf-diverge))))
           (equal (pf-run-clk clk1 st)
                  (pf-run-clk clk2 st)))
  :hints (("goal" :induct (induct-hint clk1 clk2 st)))
  :rule-classes nil)

(defthm pf-run-clk-normalize
  (implies (and (not (equal (pf-run-clk clk st) (pf-diverge)))
                (syntaxp (not (equal clk `(pf-terminates-witness ,st)))))
           (equal (pf-run-clk clk st)
                  (pf-run-clk (pf-terminates-witness st) st)))
  :hints (("goal" :use ((:instance
                         pf-run-clk-with-sufficient-clks
                         (clk1 clk)
                         (clk2 (pf-terminates-witness st)))
                        pf-terminates-suff)
           :in-theory (disable pf-terminates-suff))))

(defthm pf-run-of-o<-does-not-diverge
  (implies (and (equal (pf-run-clk clk2 st) (pf-diverge))
                (o<= (pf-meas clk1 st)
                     (pf-meas clk2 st)))
           (equal (pf-run-clk clk1 st) (pf-diverge)))
  :hints (("Goal" :induct (induct-hint clk1 clk2 st)
           :in-theory (disable pf-run-clk-normalize))))
(local
 (progn
   (defthm o<=-implies-o<
     (implies (and (o-p x) (o-p y)
                   (o< x y))
              (o<= x y)))

   (defthm pf-run-of-incr-does-not-diverge
     (implies (not (equal (pf-run-clk clk st) (pf-diverge)))
              (not (equal (pf-run-clk (pf-incr clk) st) (pf-diverge)))))

   (defthm not-pf-done-impl-next-terminates
     (implies (and (pf-terminates st)
                   (not (pf-done st)))
              (not (equal (pf-run-clk (pf-terminates-witness (pf-next st))
                                   (pf-next st))
                          (pf-diverge))))
     :hints(("Goal" :in-theory (e/d () (pf-terminates-suff))
             :expand ((:free (clk) (pf-run-clk clk st)))
             :use ((:instance pf-terminates-suff
                    (st (pf-next st))
                    (clk (pf-decr (pf-terminates-witness st) st)))))))

   (defthm not-pf-done-impl-terminates-if-next-terminates
     (implies (and (pf-terminates (pf-next st))
                   (not (and (pf-done st)
                             (equal (pf-retval st) (pf-diverge)))))
              (not (equal (pf-run-clk (pf-terminates-witness st)
                                   st)
                          (pf-diverge))))
     :hints (("goal" :expand ((:free (clk) (pf-run-clk clk st)))
              :use ((:instance pf-terminates-suff
                     (clk (pf-incr (pf-terminates-witness (pf-next st)))))))))))

(defthm not-pf-done-impl-next-equal
  (implies (not (pf-done st))
           (equal (pf-run-clk (pf-terminates-witness (pf-next st))
                           (pf-next st))
                  (pf-run-clk (pf-terminates-witness st) st)))
  :hints (("goal" :cases ((pf-terminates st)))
          (and stable-under-simplificationp
               '(:expand ((:free (clk) (pf-run-clk clk st)))))))

(defthm pf-run-is-loop
  (equal (pf-run st)
         (if (pf-done st)
             (pf-retval st)
           (pf-run (pf-next st))))
  :rule-classes nil
  :otf-flg t)
