;;;; -*- Mode:Lisp; Package:WFB; Syntax:COMMON-LISP; Base:10 -*-

(in-package :WFB)
(in-package :WFB)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1999 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU GPL2 guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; ********************************************************************
;;;
;;; wfb.lisp   -  Reference implementation of the WFB.
;;;
;;; Mike Hewett    22 November 1999
;;;
;;; This is the original implementation of the Web Frame Browser.
;;; Definition is at  http://www.cs.utexas.edu/users/qr/algy/browser/protocol.html
;;;
;;; ********************************************************************


;;; ----------  Constants  --------------------------

(defconstant *default-services-port*     5714)
(defconstant *default-html-port*         6215)
(defconstant *default-passthrough-port*  6607)


;;; ----------  Global variables  -------------------

(defparameter *services-port*      *default-services-port*)
(defparameter *html-port*          *default-html-port*)
(defparameter *passthrough-port*   *default-passthrough-port*)

(defparameter *service-connections*      NIL)
(defparameter *html-connections*         NIL)
(defparameter *passthrough-connections*  NIL)


;;; ----------  Structures  -------------------------

;;; A connection from an external source.
(defstruct conn
  type                ;; :SERVICE, :HTML, :PASSTHROUGH, :KBMS
  socket              ;; 
  host                ;; "localhost" for servers.
  port
  input-stream
  output-stream
  parameters
  open-time
  close-time)



(defun start-wfb (&KEY
		  (services-port    *services-port*)
		  (html-port        *html-port*)
		  (passthrough-port *passthrough-port*)
		  )

  (start-services-server     services-port)
  (start-html-server         html-port)
  (start-passthrough-server  passthrough-port)
  )

(defun stop-wfb (&KEY
		 (services-port    *services-port*)
		 (html-port        *html-port*)
		 (passthrough-port *passthrough-port*)
		 )

  (stop-services-server     services-port)
  (stop-html-server         html-port)
  (stop-passthrough-server  passthrough-port)
  )



;;; ----------  Server setup  -------------------------

(defun start-services-server (&OPTIONAL (port *services-port*))
  "Starts the WFB listening on the services port."

  (unless (= port *services-port*)
    (setq port *services-port*))

  (format t "~%WFB: Initiating WFB Services service on port ~a at"
	  port (create-time-string))
  (create-server :SERVICE port "WFB Services service" #'services-interface)
  )


(defun start-html-server (&OPTIONAL (port *html-port*))
  "Starts the WFB listening on the html port."

  (unless (= port *html-port*)
    (setq port html-port*))

  (format t "~%WFB: Initiating WFB HTML service on port ~a at"
	  port (create-time-string))
  (create-server :HTML port "WFB HTML service" #'html-interface)
  )


(defun start-passthrough-server (&OPTIONAL (port *passthrough-port*))
  "Starts the WFB listening on the passthrough port."

  (unless (= port *passthrough-port*)
    (setq port *passthrough-port*))

  (format t "~%WFB: Initiating WFB Passthrough service on port ~a at"
	  port (create-time-string))
  (create-server :PASSTHROUGH port "WFB Passthrough service" #'passthrough-server-interface)
  )


(defun contact-kbms-server (&KEY
			    (host "localhost")
			    (port -1)
			    (parameters NIL))
  "Contacts a KBMS server and opens a connection."

  (format t "~%WFB: Contacting KBMS at ~A:~A..." host port)
  (let ((connection (make-conn
		     :type :KBMS
		     :host host
		     :port port
		     :input-stream  NIL
		     :output-stream NIL
		     :parameters    parameters
		     :open-time     (cons (get-universal-time)
					  (create-time-string))
		     :close-time    NIL)
		    )
	)

    (setf (conn-socket connection) (create-client host port))

    (cond ((null (conn-socket connection))
	   (format t "UNSUCCESSFUL - ~a" (cdr (conn-open-time connection)))
	   (setf (conn-close-time connection)
	     (conn-open-time connection))
	   )

	  (T
	   (format t "opened at ~a" (cdr (conn-open-time connection)))
	   (setf (conn-input-stream  connection)
	     (conn-socket connection))
	   (setf (conn-output-stream connection)
	     (conn-socket connection))
	   )
	  )
    )
  )


(defun create-client (host port)
  "Creates a new connection to a server."

  #+ACL5
  (socket:make-socket
   :remote-host     host
   :remote-port     port
   :connect         :active
   :address-family  (cond ((stringp port)         :FILE)
			  ((or (null port)
			       (integerp port))   :INTERNET)
			  (T
			   (error "WFB (create-client): illegal value for port: ~s" port))
			  )  
   :format format
   ) 

  #-ACL5
  (progn
    (format t "~%WFB (create-client): I don't know how to create a client in this LISP.")
    (format t "~%WFB:  This LISP is ~A ~A"
	    (lisp-implementation-type)
	    (lisp-implementation-version))
    )
  )


(defun server-process (connection name command-fn)
  (let (p)
    (setf p
      (mp:process-run-function
       name
       #'(lambda ()
	   (do (command)
	       ((eq command :EOF) t)
	     (mp::wait-for-input-available (conn-input-stream connection))
	     (setf command (read (conn-input-stream connection) NIL :EOF))
	     (unless (eq command :EOF)
	       (funcall command-fn command)
	       )
	     )
	   (format t "~%WFB: connection ~A ended at ~a" name (create-time-string))
	   (setf (conn-close-time connection) (create-time-string))
	   )
       )
      )
    (setf (mp::process-quantum p) 0.1)
    )
  )



;;; This is copied from the file "server.cl" that
;;; comes with Allegro CL 5.0.1 (SunOS).
;;;
(defun create-server (type port name server-fn
		      &KEY (wait NIL) (format :TEXT))
  "Creates a new server on the given port.
TYPE is :HTML, :PASSTHROUGH or :SERVICE.
NAME is a string giving the name of a process
INTERFACE-FN is a function of one argument that
is called on the opened socket stream."

  ;;
  ;; create a server process with the given name, listening on the
  ;; given port, running the given function on each connection that
  ;; comes in, and possibly waiting for that function's completion before
  ;; accepting a new connection.
  ;;
  ;; name - a string naming the server process -- if nil, then this 
  ;;	function will create a name.
  ;; port - if nil then an internet domain port number will be chosen
  ;;	    by the operating system.   If a number is given then that
  ;;	    port will be used (or an error will be signalled if it
  ;;	    is already in use).    If port is a string then a unix
  ;;	    domain port will be used.  (this will not work on Windows).
  ;; function - the function to run when a connection is made.  This
  ;;	    function must take one argument which is the stream used
  ;;	    used for reading from and writing to the process that connected
  ;;	    to this socket. 
  ;; wait - if true, then the function will be run in the server process
  ;;	    and thus the server won't accept a new connection until
  ;;	    the function finishes.
  ;; format  - :text (the default) or :binary.   This determes what kind
  ;;	    of data can sent to and read from the socket stream.
  ;;	    
  ;;
  ;;
  ;; The return value is the port number on which the server is
  ;;	listening.
  ;;

  (format t "~%WFB: Starting ~A server on port~A..." type port)

  (let ((connection (make-conn
		     :type          type
		     :host          "localhost"
		     :port          port
		     :input-stream  NIL
		     :output-stream NIL
		     :parameters    NIL
		     :open-time     (cons (get-universal-time)
					  (create-time-string))
		     :close-time    NIL))

	(passive-socket
	 (socket:make-socket
	  :connect         :passive
	  :local-port      port
	  :address-family  (cond ((stringp port)         :FILE)
				 ((or (null port)
				      (integerp port))   :INTERNET)
				 (T
				  (error "illegal value for port: ~s" port))
				 )
	  :format          format
	  ))
	)

    ;; This will loop, accepting new connections
    (start-socket-server name passive-socket :function server-fn :wait NIL)

    (format t "opened at ~a" (cdr (conn-open-time connection)))
    (setf (conn-input-stream  connection) stream)
    (setf (conn-output-stream connection) stream)
    (socket:local-port passive-socket)
    )
  )


(defun start-socket-server (connection name passive-socket
			    &KEY
			    (function #'identity)
			    (wait     NIL))
  ;; internal function run in the server lightweight process 
  ;; that continually processes the connection.
  ;; This code is careful to ensure that the sockets are 
  ;; properly closed something abnormal happens.
  (unwind-protect
      (loop (let ((connection (socket:accept-connection passive-socket)))
	      (if* wait
		 then (unwind-protect
			  (funcall function connection)
			(errorset (close connection) nil))
		 else (mp:process-run-function
		       name
		       #'(lambda (con)
			   (unwind-protect
			       (funcall function con)
			     (errorset (close con) nil)))
		       connection))))
    (errorset (close passive-socket) nil))
  )





(defun create-time-string (&OPTIONAL (short-version? nil))
  "Interprets (get-decoded-time) and returns a string with the date and time in some nice format."

  (multiple-value-bind (second minute hour date month year day-of-week ignore1 ignore2)
      (get-decoded-time)
    ignore1 ignore2                                                                    

    (if short-version?
	(format NIL "~D ~A ~D  ~D:~2,'0D" ;1 January 1987 16:14"
		date 
		(cdr (assoc month '((1 . "January")    (2 . "February") (3 . "March")     (4 . "April")
				    (5 . "May")        (6 . "June")     (7 . "July")      (8 . "August")
				    (9 . "September") (10 . "October") (11 . "November") (12 . "December"))))
		year
		hour
		minute)
	
      ;;Else give long verbose date and time
	
      (format NIL "~A, ~D ~A ~D  ~D:~2,'0D:~2,'0D" ;Sunday, 1 January 1986  16:14:12
		
	      (cdr (assoc day-of-week '((0 . "Monday") (1 . "Tuesday")  (2 . "Wednesday") (3 . "Thursday")
					(4 . "Friday") (5 . "Saturday") (6 . "Sunday"))))
	      date
	      (cdr (assoc month '((1 . "January")    (2 . "February") (3 . "March")     (4 . "April")
				  (5 . "May")        (6 . "June")     (7 . "July")      (8 . "August")
				  (9 . "September") (10 . "October") (11 . "November") (12 . "December"))))
	      year
	      hour
	      minute
	      second)))
  )
