;;;; -*- Mode:Lisp; Package:WFB; Syntax:COMMON-LISP; Base:10 -*-

(in-package :WFB)
(in-package :WFB)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1999 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU GPL2 guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; ********************************************************************
;;;
;;; wfb.lisp   -  Reference implementation of the WFB.
;;;
;;; Mike Hewett    22 November 1999
;;;
;;; This is the original implementation of the Web Frame Browser.
;;; Definition is at  http://www.cs.utexas.edu/users/qr/algy/browser/protocol.html
;;;
;;; ********************************************************************


;;; ----------  Constants  --------------------------

(defconstant *default-services-port*     5714)
(defconstant *default-html-port*         6215)
(defconstant *default-passthrough-port*  6607)


;;; ----------  Global variables  -------------------

(defparameter *services-port*      *default-services-port*)
(defparameter *html-port*          *default-html-port*)
(defparameter *passthrough-port*   *default-passthrough-port*)

(defparameter *service-connections*      NIL)
(defparameter *html-connections*         NIL)
(defparameter *passthrough-connections*  NIL)


;;; ----------  Structures  -------------------------

;;; A connection from an external source.
(defstruct conn
  type                ;; :SERVICE, :HTML, :PASSTHROUGH, :KBMS
  host                ;; "localhost" for servers.
  port
  input-stream
  output-stream
  parameters
  open-time
  close-time)



;;; ----------  Server setup  -------------------------

(defun start-services-server (&OPTIONAL (port *services-port*))
  "Starts the WFB listening on the services port."

  (unless (= port *services-port*)
    (setq port *services-port*))

  (format t "~%WFB: Initiating WFB Services service on port ~a at"
	  port (create-time-string))
  (create-server port "WFB Services service" #'services-interface)
  )


(defun start-html-server (&OPTIONAL (port *html-port*))
  "Starts the WFB listening on the html port."

  (unless (= port *html-port*)
    (setq port html-port*))

  (format t "~%WFB: Initiating WFB HTML service on port ~a at"
	  port (create-time-string))
  (create-server port "WFB HTML service" #'html-interface)
  )


(defun start-passthrough-server (&OPTIONAL (port *passthrough-port*))
  "Starts the WFB listening on the passthrough port."

  (unless (= port *passthrough-port*)
    (setq port *passthrough-port*))

  (format t "~%WFB: Initiating WFB Passthrough service on port ~a at"
	  port (create-time-string))
  (create-server port "WFB Passthrough service" #'passthrough-server-interface)
  )


(defun contact-kbms-server (&KEY
			    (host "localhost")
			    (port -1)
			    (parameters NIL))
  "Contacts a KBMS server and opens a connection."

  (format t "~%WFB: Contacting KBMS at ~A:~A..." host port)
  (let ((connection (make-conn
		     :type :KBMS
		     :host host
		     :port port
		     :input-stream  NIL
		     :output-stream NIL
		     :parameters    parameters)
		    :open-time     (cons (get-universal-time)
					 (create-time-string))
		    :close-time    NIL)
	(stream      NIL)
	)

    (setq stream (create-client host port))

    (cond ((null stream)
	   (format t "UNSUCCESSFUL - ~a" (cdr (conn-open-time connection)))
	   (setf (conn-close-time connection)
	     (conn-open-time connection))
	   )

	  (T
	   (format t "opened at ~a" (cdr (conn-open-time connection)))
	   (setf (conn-input-stream  connection) stream)
	   (setf (conn-output-stream connection) stream)
	   )
	  )
    )
  )


(defun create-server (port)
  "Creates a new server on the given port."

  #+ACL5
  (socket:make-socket :port port)

  #-ACL5
  (progn
    (format t "~%WFB (create-server): I don't know how to create a server in this LISP.")
    (format t "~%WFB:  This LISP is ~A ~A"
	    (lisp-implementation-type)
	    (lisp-implementation-version))
    )
  )


(defun create-client (host port)
  "Creates a new connection to a server."

  #+ACL5
  (socket:make-socket :remote-host host :remote-port port)

  #-ACL5
  (progn
    (format t "~%WFB (create-server): I don't know how to create a server in this LISP.")
    (format t "~%WFB:  This LISP is ~A ~A"
	    (lisp-implementation-type)
	    (lisp-implementation-version))
    )
  )


(defun server-loop (connection server name command-fn)
  (let (p)
    (setf p
      (mp:process-run-function
       name
       #'(lambda ()
	   (do (command)
	       ((eq command :EOF) t)
	     (mp::wait-for-input-available (conn-input-stream connection))
	     (setf command (read (conn-input-stream connection)))
	     (funcall command-fn command)
	     )
	   (format t "~%WFB: connection ~A ended at ~a" name (create-time-string))
	   (setf (conn-close-time connection) (create-time-string))
	   )
       )
      )
    (setf (mp::process-quantum p) 0.1)
    )
  )


(defun create-time-string (&OPTIONAL (short-version? nil))
  "Interprets (get-decoded-time) and returns a string with the date and time in some nice format."

  (multiple-value-bind (second minute hour date month year day-of-week ignore1 ignore2)
      (get-decoded-time)
    ignore1 ignore2                                                                    

    (if short-version?
	(format NIL "~D ~A ~D  ~D:~2,'0D" ;1 January 1987 16:14"
		date 
		(cdr (assoc month '((1 . "January")    (2 . "February") (3 . "March")     (4 . "April")
				    (5 . "May")        (6 . "June")     (7 . "July")      (8 . "August")
				    (9 . "September") (10 . "October") (11 . "November") (12 . "December"))))
		year
		hour
		minute)
	
      ;;Else give long verbose date and time
	
      (format NIL "~A, ~D ~A ~D  ~D:~2,'0D:~2,'0D" ;Sunday, 1 January 1986  16:14:12
		
	      (cdr (assoc day-of-week '((0 . "Monday") (1 . "Tuesday")  (2 . "Wednesday") (3 . "Thursday")
					(4 . "Friday") (5 . "Saturday") (6 . "Sunday"))))
	      date
	      (cdr (assoc month '((1 . "January")    (2 . "February") (3 . "March")     (4 . "April")
				  (5 . "May")        (6 . "June")     (7 . "July")      (8 . "August")
				  (9 . "September") (10 . "October") (11 . "November") (12 . "December"))))
	      year
	      hour
	      minute
	      second)))
  )
