(in-package :user)

;;; BELIEF.LISP
;;; Copyright (c) 1990 by Coy C. Day, Jr.
;;; Implementation of tree-structured subset of
;;; Judea Pearl's belief networks.

;;; To initialize:
;;; Nodes are structured, according to dependency, by parent
;;; and child relations.  Then, simply assert the conditional
;;; probability matrices between links, the initial pi value
;;; for the root node and initial lambda values for the leaf
;;; nodes.  The initial pi value for the root indicates the
;;; prior probabilities of the root values.  The initial
;;; lambda values for the leaves indicate that they are
;;; anticipatory nodes, i.e. they have no support yet from
;;; below.

;;; Architecture:
;;; Each variable is represented by an object called a "node".
;;; Its parent is given by (parent node ?p) and its children
;;; by (child node ?c).  Each node contains slots for its
;;; belief values and the messages it has received from other
;;; nodes.  These messages are "sent" by asserting the
;;; appropriate slot value in the receiving node.

;;; Note:
;;; This code is written for understandability
;;; rather than efficiency.  Specific improvements to the
;;; basic algorithm are mentioned on p. 260 of Pearl.

;;; Reference:
;;; Pearl, Judea.  Fusion, propagation, and structuring in
;;; belief networks.  Artificial Intelligence 29 (1986) 241-288.

;;; From: David Heckerman <heckerma@microsoft.com>
;;; To: ai-medicine@MED.Stanford.EDU
;;; Date: Wed, 12 May 93 23:52:11 PDT
;;; 
;;; You may want to consider using a belief network representation.
;;; There's a nice tutorial in:
;;;   Charniak, Bayesian Networks Without Tears, AI Mag Volume 12, Number 4.
;;;   p50.

(defun bel-frames ()
  (tell '((:taxonomy (things (objects (contexts current-context)
					 (nodes))))))

  (tell '((:slot net-root (contexts nodes) :cardinality 1)
	  (:slot leaf     (contexts nodes))
	  (:slot init-bnet-leaves (contexts booleans) :cardinality 1)
	  
	  (:slot description (nodes :string) :cardinality 1)
	  (:slot values      (nodes nil)     :cardinality 1)
	  (:slot child       (nodes nodes))
	  (:slot parent      (nodes nodes)   :cardinality 1)
	  (:slot dummy-child (nodes nodes)   :cardinality 1)
	  
	  (:slot matrix (nodes nil) :cardinality 1)
	  (:slot lambda (nodes nil) :cardinality 1)
	  (:slot pi     (nodes nil) :cardinality 1)
	  (:slot belief (nodes nil) :cardinality 1)
	  
	  (:slot new-evidence      (nodes nil)       :cardinality 1)
	  (:slot lambda-message    (nodes nodes nil) :cardinality 1)
	  (:slot pi-message        (nodes nil)       :cardinality 1)
	  (:slot delete-lambda-msg (nodes nodes)     :cardinality 1))))

(defun setup-belief ()
  (acom-reset)
  (bel-frames)
  (bel-rules)
;;  (kb-snapshot 'belief)
  )

(defun bel-rules ()

  ;; House-keeping rules

  (tell '((:rules contexts

	   ;; Initialize the belief network by setting the lambda values
	   ;; of all leaf nodes to be vectors of ones.  This causes propagation
	   ;; of belief values across the network and so may take a long time.

	          ((init-bnet-leaves ?context true)
		   ->
		   (leaf ?context ?leaf)
		   (:bind ?v (:values ?leaf values))
		   (:bind ?all-ones (fill (copy-list (car '?v)) 1.0))
		   (lambda ?leaf ?all-ones)))

	  (:rules nodes

		  ((child ?node1 ?node2) -> (parent ?node2 ?node1))

		  ((delete-lambda-msg ?a ?b)
		   ->
		   (:unp (lambda-message ?a ?b ?msg))
		   (:clear-slot ?a delete-lambda-msg))

		  ((delete-lambda-msg ?a ?b)
		   ->
		   (lambda-message ?a ?b ?msg)
		   (:clear-slot ?a delete-lambda-msg)
		   (:delete (lambda-message ?a ?b ?msg)))

	   )))

  ;; Update rules

  (tell '((:rules nodes

  ;; Assert new evidence to a leaf node by sending a lambda message
  ;; from its dummy child.  A dummy is used so that the lambda message
  ;; can be the same slot that is used between "real" nodes without
  ;; making the dummy part of the network, in which case it would
  ;; receive messages and compute its belief value.

	          ((new-evidence ?a ?lambda-msg)
		   ->
		   (:clear-slot ?a new-evidence)
		   (:forc ?node2 (dummy-child ?a ?node2))
		   (delete-lambda-msg ?a ?node2)
		   (lambda-message ?a ?node2 ?lambda-msg))

		  ;; update lambda

		  ((lambda-message ?a ?b ?lambda-msg)
		   ->
		   (:unp (lambda ?a ?old-lambda))
		   (:bind ?lambda-list (:values ?a lambda-message))
		   (:bind ?lambda (compute-lambda '?lambda-list))
		   (lambda ?a ?lambda))

		  ((lambda-message ?a ?b ?lambda-msg)
		   ->
		   (lambda ?a ?old-lambda)
		   (:bind ?lambda-list (:values ?a lambda-message))
		   (:bind ?lambda (compute-lambda '?lambda-list))
		   (:test (vector-/= '?lambda '?old-lambda))
		   (:delete (lambda ?a ?old-lambda))
		   (lambda ?a ?lambda))

		  ;; update pi

		  ((pi-message ?b ?pi-msg)
		   ->
		   (:unp (pi ?b ?old-pi))
		   (matrix ?b ?mat)
		   (:bind ?pi (compute-pi '?mat '?pi-msg))
		   (pi ?b ?pi))

		  ((pi-message ?b ?pi-msg)
		   ->
		   (pi ?b ?old-pi)
		   (matrix ?b ?mat)
		   (:bind ?pi (compute-pi '?mat '?pi-msg))
		   (:test (vector-/= '?pi '?old-pi))
		   (:delete (pi ?b ?old-pi))
		   (pi ?b ?pi))

		  ;; update lambda-message

		  ((lambda ?b ?lambda)
		   ->
		   (parent ?b ?a)
		   (matrix ?b ?mat)
		   (:bind ?lambda-msg (compute-lambda-msg '?mat '?lambda))
		   (delete-lambda-msg ?a ?b)
		   (lambda-message ?a ?b ?lambda-msg))


		  ;; update pi-message

		  ((lambda ?b ?lambda)
		   ->
		   (pi ?b ?pi)
		   (child ?b ?c)
		   (:bind ?lambda-list (:values ?b lambda-message))
		   (:bind ?pi-msg (compute-pi-msg '?pi '?c '?lambda-list))
		   (:clear-slot ?c pi-message)
		   (pi-message ?c ?pi-msg))

		  ((pi ?b ?pi)
		   ->
		   (child ?b ?c)
		   (:bind ?lambda-list (:values ?b lambda-message))
		   (:bind ?pi-msg (compute-pi-msg '?pi '?c '?lambda-list))
		   (:clear-slot ?c pi-message)
		   (pi-message ?c ?pi-msg))


		  ;; update belief
		  
		  ((lambda ?a ?lambda)
		   ->
		   (pi ?a ?pi)
		   (:bind ?bel (compute-belief '?lambda '?pi))
		   (:clear-slot ?a belief)
		   (belief ?a ?bel))

		  ((pi ?a ?pi)
		   ->
		   (lambda ?a ?lambda)
		   (:bind ?bel (compute-belief '?lambda '?pi))
		   (:clear-slot ?a belief)
		   (belief ?a ?bel))
	   ))))


;;; Supporting LISP functions.

(defparameter *trace-bayes* nil)

;;; Lambda is the product of the received lambda messages.

(defun compute-lambda (lambda-list)
  (reduce #'vector-* (mapcar #'cadr lambda-list)))

;;; Multiply the link matrix by the received pi message.

(defun compute-pi (matrix pi-vec)
  (matrix-trans-* matrix pi-vec))

;;; Belief is alpha * lambda * pi.

(defun compute-belief (lambda pi-vec)
  (let* ((prod (vector-* lambda pi-vec))
	 (alpha (/ 1 (apply #'+ prod)))
	 (bel (scale prod alpha)))
    (when *trace-bayes*
      (format t "~%BEL (~s) = alpha (~5,3f) * lambda (~s) * pi (~s)"
	      bel alpha lambda pi-vec))	 
    bel))

;;; Lambda message is the link matrix times lambda.

(defun compute-lambda-msg (matrix lambda)
  (matrix-* matrix lambda))

;;; The pi message is pi times the product of all lambda messages
;;; except for that of the destination node.
;;; NEED TO CHANGE!

(defun compute-pi-msg (pi-vec dest lambda-list)
  (let* ((lambda-prod (mapcar #'(lambda (pair)
				  (if (eq (car pair) dest)
				      (mapcar #'(lambda (value) 1.0) pi-vec)
				      (cadr pair)))
			      lambda-list))
	 (prod (vector-* pi-vec (cond ((null lambda-prod) '(1.0 1.0 1.0))
				      ((null (cdr lambda-prod)) (car lambda-prod))
				      (t (reduce #'vector-* lambda-prod))))))

    (scale prod (/ 1 (apply #'+ prod)))))

;;; Multiply a matrix by a vector.

(defun matrix-* (matrix vector)
  (mapcar #'(lambda (vec)
	      (sdot vec vector))
	  matrix))

;;; Multiply transpose of matrix by vector.

(defun matrix-trans-* (matrix vector)
  (let ((result (mapcar #'(lambda (x) 0.0) (car matrix))))
    (mapc #'(lambda (vec x)
	      (setf result (vector-+ result (scale vec x))))
	  matrix vector)
    result))

;;; Dot product of two vectors.
 
(defun sdot (vec1 vec2)
  (apply #'+ (vector-* vec1 vec2)))

;;; Multiply elements of one vector by a scalar.

(defun scale (vector scalar)
  (mapcar #'(lambda (x)
	      (* x scalar))
	  vector))

;;; Multiply two vectors, item by item.

(defun vector-* (a b)
  (cond ((null a) b)
	((null b) a)
	(t (mapcar #'* a b))))

;;; Add two vectors a and b.

(defun vector-+ (a b)
  (mapcar #'+ a b))

;;; Parameters for vector-/=

(defvar *threshold* 0.01 "indication of how much the elements may differ")
(defvar *epsilon* 1e-10 "smallest divisor allowed for normalizing elements")

;;; Predicate returning nil iff the elements of vector a are
;;; are all "close to" the corresponding element of vector b.

(defun vector-/= (a b)
  (or (not (listp a)) (not (listp b))
      (not (= (length a) (length b)))
      (progn (mapc #'(lambda (x y)
		       (let ((minimum (min x y)))
			 (when (> (abs (/ (- y x) (max minimum *epsilon*)))
				  *threshold*)
			   (return-from vector-/= t))))
		   a b)
	     nil)))

