;;; -*- Syntax: Common-Lisp; Package: USER -*-
;;; Copyright (c) 1991 Benjamin Kuipers.

;;;
;;;Date: Sun, 30 Jun 1996 22:01:48 -0500 (CDT)
;;;From: kuipers@cs.utexas.edu

#+(or GCL LUCID) (in-package :user)
#-(or GCL LUCID) (in-package :cl-user)

; The Establish-Refine approach to diagnosis [Gomez & Chandrasekaran, IEEE SMC, 1981;
; reprinted in [Clancey & Shortliffe, 1985]].
; Implemented as Necessary and Sufficient conditions for set membership.

(defun test-ER ()				; Load Establish-Refine KB.
  (acom-reset)
  (setup-classification-sets)
  (facts-about-animals)
  
  (tell '((isa Animals Classification-Roots))
	:comment "The set Animals inherits the Establish-Refine machinery")

  (ask-slots)
  (bwd-zoo-rules)
  (extra-zoo-rules)			; for concluding not-isa
  )

(defun ER-example (name)			; Classify a new animal.
  (let ((*trace-new-values* t))		
    (with-normal-output
	(ask `((:forc ?s (name ?s ,(string name)) (isa ?s Animals))
	       (isa ?s ?set))
	     :comment  (format nil "Classifying ~a." name)
	     :collect `(?s is a ?set))
      )))

(defun ERx (name)			; Classify a new animal.
  (let ((*trace-new-values* t))		
    (with-normal-output
	(ask `((:forc ?s (name ?s ,(string name))  (:assume (isa ?s Animals)))
	       (isa ?s ?set))
	     :comment  (format nil "Classifying ~a." name)
	     :collect `(?s is a ?set))
      )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun test-Mycin-style ()
  (setup-classification-sets)
  (facts-about-animals)  
  (ask-slots)
  (Backward-Chaining-Zoo-Rules))

(defun setup-classification-sets ()

  ;; Classification sets are sets in a hierarchy for establish-refine classification.
  ;; The root is a speecial classification set that holds the rules.

  (tell '((:taxonomy (Sets
		      (Classification-Sets
		       (Classification-Roots)))))
	:comment "Classification Sets get special inference rules")

  (tell '((:rules Classification-Roots
	   
	   ((isa ?S Classification-Sets)
	    (subset ?S ?Sb)
	    ->
	    (isa ?Sb Classification-Sets)))))

  ;; The structure of a set can be described by giving exclusive and exhaustive
  ;; decompositions of the set.

  (tell '((:slot Exhaustive-Set-of-Subsets (Sets Sets))
	  (:slot Set-of-Exclusive-Subsets  (Sets Sets)))
	:comment "Slots representing sets of subsets")

  (tell '((:srules Exhaustive-Set-of-Subsets
	   ((exhaustive-set-of-subsets ?U ?subsets)
	    (member ?subsets ?set)
	    ->
	    (subset ?U ?set)))

	  (:srules Set-of-Exclusive-Subsets
	   ((set-of-exclusive-subsets ?U ?subsets)
	    (member ?subsets ?set)
	    ->
	    (subset ?U ?set)))))

  ;; Establish-Refine:  upon establishing membership in a set, try to
  ;; extablish membership in one of its subsets.

  (tell '((:rules Classification-Roots
	   ((isa ?S Classification-Roots)
	    ->
	    (:rules ?S
		    ((isa ?x ?A)
		     (:retrieve (isa ?A Classification-Sets))
		     (:no-completion
		      (:cut (subset ?A ?B)
			    (:fail (:retrieve (not (isa ?x ?B))))
			    (isa ?x ?B)))
		     -> )
		    
		    ((isa ?x ?A)
		     (:retrieve (isa ?A Classification-Sets))
		     (:no-completion
		      (:cut (subset ?A ?B)
			    (:fail (:retrieve (isa ?x ?B)))
			    (not (isa ?x ?B))))
		     -> )
		    ))) )
	:comment "The basic Establish-Refine rules")

  (tell '((:rules Classification-Roots
	   ((isa ?S Classification-Roots)
	    ->
	    (:rules ?S
		    
		    ((isa ?x ?A)
		     (:retrieve (isa ?A Classification-Sets))
		     (exhaustive-set-of-subsets ?A ?ES)
		     (member ?ES ?Bi)
		     (:no-completion
		      (:all-paths ((member ?ES ?Bj) (:neq ?Bi ?Bj))
				  ((not (isa ?x ?Bj)))))
		     ->
		     (isa ?x ?Bi))

		    ((isa ?x ?Ci)
		     (:retrieve (isa ?Ci Classification-Sets))
		     (superset ?Ci ?D)
		     (:retrieve (isa ?D Classification-Sets))
		     (set-of-exclusive-subsets ?D ?XS)
		     (member ?XS ?Ci)
		     (member ?XS ?Cj)
		     (:neq ?Ci ?Cj)
		     ->
		     (not (isa ?x ?Cj)))
		    ))))
	:comment "Reasoning by exclusion or exhaustion")  )

; The following example is taken from Winston & Horn, LISP, 3ed, p.388-390.

(defun facts-about-animals ()

  (tell '((:taxonomy (Physical-Objects
		      (LivingThings
		       (Animals
			(Mammals
			 (Carnivores
			  (Cheetahs)
			  (Tigers))
			 (Ungulates
			  (Giraffes)
			  (Zebras)))
			(Birds
			 (Canaries)
			 (Robins)
			 (Penguins)
			 (Ostriches)
			 (Albatrosses) ))))))
	:comment "Taxonomy")

  (tell '((:forc ?a (set-of-exclusive-subsets Animals ?a)
	             (member ?a Mammals)
		     (member ?a Birds))

	  (:forc ?f (exhaustive-set-of-subsets Animals ?f)
                     (member ?f Mammals)
		     (member ?f Birds))
	             

	  (:forc ?b (set-of-exclusive-subsets Mammals ?b)
		     (member ?b Carnivores)
		     (member ?b Ungulates))

	  (:forc ?c (set-of-exclusive-subsets Carnivores ?c)
		     (member ?c Cheetahs)
		     (member ?c Tigers))

	  (:forc ?d (set-of-exclusive-subsets Ungulates ?d)
		     (member ?d Giraffes)
		     (member ?d Zebras))

	  (:forc ?e (set-of-exclusive-subsets Birds ?e)
		     (member ?e Canaries)
		     (member ?e Penguins)
		     (member ?e Ostriches)
		     (member ?e Albatrosses)))
	:comment "Exclusive sets")

  ;; Values of properties are also added to the taxonomy.
  
  (tell '((:taxonomy (Physical-Attributes
		      (Colors Yellow Black+White Tawny)
		      (Patterns Black-Stripes Dark-Spots Red-Breast)))
	  (:taxonomy (Things
		      (Stuff
		       (Materials
			(Biological-Materials Skin Fur Hair Feathers))))) )
	:comment "Taxonomy of properties")

  (tell '((:slot Covering      (Objects Materials))
	  (:slot Pattern       (Objects Patterns))
	  (:slot Flies         (Animals Booleans) :cardinality 1)
	  (:slot gives-milk    (Animals Booleans) :cardinality 1)
	  (:slot lays-eggs     (Animals Booleans) :cardinality 1)
	  (:slot eats-meat     (Animals Booleans) :cardinality 1)
	  (:slot pointed-teeth (Animals Booleans) :cardinality 1)
	  (:slot has-claws     (Animals Booleans) :cardinality 1)
	  (:slot forward-eyes  (Animals Booleans) :cardinality 1)
	  (:slot has-hoofs     (Animals Booleans) :cardinality 1)
	  (:slot chews-cud     (Animals Booleans) :cardinality 1)
	  (:slot long-neck     (Animals Booleans) :cardinality 1)
	  (:slot long-legs     (Animals Booleans) :cardinality 1)
	  (:slot swims         (Animals Booleans) :cardinality 1)
	  (:slot flies-well    (Animals Booleans) :cardinality 1)
	  (:slot parent        (Animals Animals) :cardinality 1)	      
	  )
	:comment "Slots"))

; Ask slots provide automatic user requests via :ask.

(defun ask-slots ()

  (tell '((:taxonomy (Slots
		      (Ask-slots)))))

  (tell '((:srules ask-slots
	   ((?p ?x true)  <- (:ask (?p ?x true)))
	   ;; This next rule makes sense ONLY under depth-first search:
	   ((?p ?x false) <- (:fail (:ask (?p ?x true))))
	   ;;
	   ((?p ?x false) -> (not (?p ?x true)))
	   ((?p ?x true)  -> (not (?p ?x false)))
	   ((not (?p ?x true))  -> (?p ?x false))
	   ((not (?p ?x false)) -> (?p ?x true))))
	:comment "Define Ask-slots and their rules")

  (tell '((:rules Objects
	   ((covering ?x ?c) <- (:ask (covering ?x ?c)))
	   ((color ?x ?c)    <- (:ask (color ?x ?c)))
	   ((pattern ?x ?c)  <- (:ask (pattern ?x ?c)))))
	:comment "Ask about covering")

  (tell '((isa (:slot flies)         ask-slots)
	  (isa (:slot gives-milk)    ask-slots)
	  (isa (:slot lays-eggs)     ask-slots)
	  (isa (:slot eats-meat)     ask-slots)
	  (isa (:slot pointed-teeth) ask-slots)
	  (isa (:slot has-claws)     ask-slots)
	  (isa (:slot forward-eyes)  ask-slots)
	  (isa (:slot has-hoofs)     ask-slots)
	  (isa (:slot chews-cud)     ask-slots)
	  (isa (:slot long-neck)     ask-slots)
	  (isa (:slot long-legs)     ask-slots)
	  (isa (:slot swims)         ask-slots)
	  (isa (:slot flies-well)    ask-slots)
	  )
	:comment "The :ask slots"))

; These rules provide sufficient conditions for set membership.
; We could make representative-based forms to provide necessary conditions.
;    That form generalizes more naturally to the case-based approach.

(defun Bwd-Zoo-Rules ()

  (tell '((:rules Animals
	   ((isa ?a Mammals) <- (covering ?a hair))
	   ((isa ?a Mammals) <- (gives-milk ?a true))
	   ((isa ?a Birds) <- (covering ?a feathers))
	   ((isa ?a Birds) <- (flies ?a true) (lays-eggs ?a true))
	   ;;  ((isa ?x ?species) <- (parent ?x ?y) (isa ?y ?species))
	   )

	  (:rules Mammals
	   ((isa ?a Carnivores) <- (eats-meat ?a true))
	   ((isa ?a Carnivores) <- (pointed-teeth ?a true)
	                           (has-claws ?a true)
	                           (forward-eyes ?a true))
	   ((isa ?a Ungulates) <- (has-hoofs ?a true))
	   ((isa ?a Ungulates) <- (chews-cud ?a true)))
	  
	  (:rules Carnivores
	   ((isa ?a Cheetahs) <- (color ?a tawny) (pattern ?a dark-spots))
	   ((isa ?a Tigers)   <- (color ?a tawny) (pattern ?a black-stripes)))

	  (:rules Ungulates
	   ((isa ?a giraffes) <- (long-neck ?a true)
		                     (long-legs ?a true)
				     (pattern ?a dark-spots))
	   ((isa ?a zebras) <- (pattern ?a black-stripes)))

	  (:rules Birds
	   ((isa ?b Ostriches) <- (flies ?b false)
		                      (long-neck ?b true)
				      (long-legs ?b true)
				      (color ?b black+white))
	   ((isa ?b Penguins) <- (flies ?b false)
		                     (swims ?b true)
				     (color ?b black+white))
	   ((isa ?b Albatrosses) <- (flies ?b true) (flies-well ?b true)))
	  )
	:comment "Bwd Zoo Rules modified from Winston+Horn 3e, pp.388-390."))

;;; The following rules are here to test whether we can refute the top-level
;;; hypothesis.

(defun extra-zoo-rules ()
  (tell '((:rules Animals
	   ((not (covering ?a feathers)) -> (not (isa ?a Birds)))
	   ((not (covering ?a hair)) -> (not (isa ?a Mammals)))))))


;;; These rules are written in the spirit of Mycin, backchaining from a
;;; suspected individual to the categories it is in.

(defun Backward-Chaining-Zoo-Rules ()

  (tell '((:rules Animals
	   ;;
	   ;; These rules are invoked by querying an individual species, and
	   ;; backchain to query the properties of a category.
	   ;;	  
	   ((isa ?a Cheetahs)  <- (isa ?a Carnivores) (color ?a tawny) (pattern ?a dark-spots))
	   ((isa ?a Tigers)    <- (isa ?a Carnivores) (color ?a tawny) (pattern ?a black-stripes))

	   ((isa ?a giraffes)  <- (isa ?a Ungulates)
	                             (long-neck ?a true)
		                     (long-legs ?a true)
				     (pattern ?a dark-spots))
	   ((isa ?a zebras)    <- (isa ?a Ungulates) (pattern ?a black-stripes))

	   ((isa ?b Ostriches) <- (isa ?b Birds)
	                              (flies ?b false)
		                      (long-neck ?b true)
				      (long-legs ?b true)
				      (color ?b black+white))
	   ((isa ?b Penguins)  <- (isa ?b Birds)
	                             (flies ?b false)
		                     (swims ?b true)
				     (color ?b black+white))
	   ((isa ?b Albatrosses) <- (isa ?b Birds) (flies ?b true) (flies-well ?b true))
	   ((isa ?b Robins)      <- (isa ?b Birds) (flies ?b true) (pattern ?b red-breast))
	   ;;
	   ;; Here are the rules for establishing the categories.  They are only
	   ;; reached by backchaining from the individual species.
	   ;;
	   ((isa ?a Mammals)    <- (covering ?a hair))
	   ((isa ?a Mammals)    <- (gives-milk ?a true))
	   ((isa ?a Birds)      <- (covering ?a feathers))
	   ((isa ?a Birds)      <- (flies ?a true) (lays-eggs ?a true))

	   ((isa ?a Carnivores) <- (eats-meat ?a true))
	   ((isa ?a Carnivores) <- (pointed-teeth ?a true)
	                           (has-claws ?a true)
	                           (forward-eyes ?a true))
	   ((isa ?a Ungulates)  <- (has-hoofs ?a true))
	   ((isa ?a Ungulates)  <- (chews-cud ?a true))
	   ;;
	   )))

  (tell '((:rules Animals
	   ;;
	   ;; These are the "triggering" rules that suggest a specific individual
	   ;; species to query, given an observation.
	   ;;
	   ((pattern ?a dark-spots) (isa ?a Cheetahs) ->)
	   ((pattern ?a black-stripes) (isa ?a Tigers) ->)
	   ((flies ?a true) (isa ?a Birds) ->)
	   ;;
	   ;; These are "typicality" rules that suggest an individual species,
	   ;; given a category.
	   ((isa ?a Birds) (isa ?a Robins) ->)
	   )))
  )

;;; Need an example where initial observation backchains to critical question;
;;; critical question suggests new possibility, which is then confirmed.

;;; These rules are just for the record, since they are the form in 
;;; Winston+Horn 3e, pp.388-390.

(defun Fwd-Zoo-Rules ()

  (tell '((:rules Animals
	   ((covering ?a hair) -> (isa ?a Mammals))
	   ((gives-milk ?a true) -> (isa ?a Mammals))
	   ((covering ?a feathers) -> (isa ?a Birds))
	   ((flies ?a true) (lays-eggs ?a true) -> (isa ?a Birds))
	   ((eats-meat ?a true) -> (isa ?a Carnivores))
	   ((pointed-teeth ?a true) (has-claws ?a true) (forward-eyes ?a true)
	    -> (isa ?a Carnivores))
	   ;;  ((isa ?x ?species) (parent ?x ?y) -> (isa ?y ?species))
	   )

	  (:rules Mammals
	   ((has-hoofs ?a true) -> (isa ?a Ungulates))
	   ((chews-cud ?a true) -> (isa ?a Ungulates)))

	  (:rules Carnivores
	   ((color ?a tawny) (pattern ?a dark-spots)    -> (isa ?a Cheetahs))
	   ((color ?a tawny) (pattern ?a black-stripes) -> (isa ?a Tigers)))
	  
	  (:rules Ungulates
	   ((long-neck ?a true) (long-legs ?a true) (pattern ?a dark-spots)
	    -> (isa ?a giraffes))
	   ((pattern ?a black-stripes) -> (isa ?a zebras)))
	  
	  (:rules Birds
	   ((flies ?b false) (long-neck ?b true) (long-legs ?b true) (color ?b black+white)
	    -> (isa ?b Ostriches))
	   ((flies ?b false) (swims ?b true) (color ?b black+white)
	    -> (isa ?b Penguins))
	   ((flies ?b true) (flies-well ?b true) -> (isa ?b Albatrosses)))
	  )
	:comment "Fwd Zoo Rules from Winston+Horn 3e, pp.388-390."))

