;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+(or GCL LUCID) (in-package :user)
#-(or GCL LUCID) (in-package :cl-user)


;;; Simple expert system.
;;;
;;; This is an example of a very simple expert system built on top
;;; of the finite-state machine.  There are only two modifications to
;;; the basic fsm (in fsm.lisp):
;;;
;;;  1. The current time of the fsm is non-monotonically stored in the current context
;;;  (this allows the rules to retrieve the "current state" of the fsm).
;;;
;;;  2. There are no "inputs" here and the transitions are triggered by
;;;  rules in the expert system asserting the next state explicitly.
;;;
;;; The states are as follows:
;;;
;;;   Diagnosing -> Testing -> Prescribing
;;;       ^            |
;;;       |            |
;;;       --------------
;;;
;;; In this simple example, each state fires rules to fill a given slot
;;; and then passes control to the next state.

(defun facts-about-expert ()

  (acom-reset)
  
  (tell 
   '((:taxonomy
      (objects
       (times fsm-time)
       (states (non-final-states diagnosing testing Prescribing)
	       (final-states final-state))
       (symptoms low-fever high-fever cough tiredness)
       (diseases flu mono pneumonia)
       (tests mono-test xray)
       (results positive negative)
       (prescriptions rest lots-of-rest penicillin))))
   :comment "Taxonomy")

  (tell '((:slot next-time (times times) :cardinality 1)
	      (:slot current-time (contexts times) :cardinality 1)
	      (:slot state (times states) :cardinality 1)
	      (:slot next-state (states times states))

              (:slot temperature (physical-objects :NUMBER)
                     :cardinality 1)
	      (:slot has-symptom (people symptoms))
	      (:slot diagnosis (people diseases))
	      (:slot probable-diagnosis (people diseases))
	      (:slot patient (contexts people) :cardinality 1)
	      (:slot test (diseases tests))
	      (:slot result (tests people results))
	      (:slot prescription (people prescriptions)))
	:comment "New slots")

  ;; Where do these go??  (mh) 16 Sep 1996
  (tell '((:slot current-context (CONTEXTS contexts))
	  (:taxonomy (contexts  global-context)))
	:comment "Contexts")


  (tell '((:rules contexts
	       ;; There's a first time for everything:
	       ;; (this saves us from having to assert a first time
	       ;; in each new context).
	       ((current-time ?cc fsm-time)
		<-
		(:unp (current-time ?cc ?t))))
	      (:rules times
	       ;; And there's always next time:
	       ((next-time ?t1 ?t2)
		<-
		(:forc (?t2 TIMES) (name ?t2 "fsm-time")
		       (next-time ?t1 ?t2)))))
	:comment "Times")

  ;; The transition rule is just like before except that:
  ;;  1. It changes the current-time in the current-context.
  ;;  2. It will not fire to create a new state for any state except for the
  ;;  state at the current time (this avoids some strange bugs
  ;;  involving old states executing transitions ...).
  ;;
  (tell '((:rules times
	       ;; Transitions:
	       ((state ?t1 ?s1)
		(isa ?s1 non-final-states)
		(next-time ?t1 ?t2)
		(next-state ?s1 ?t1 ?s2)
		(current-context global-context ?cc)
		(current-time ?cc ?t1)
		->
		(:clear-slot ?cc current-time)
		(current-time ?cc ?t2)
                ;; Output
                (:eval (format t "~% State is now ~(~a~).~%" '?s2))
		(state ?t2 ?s2))))
	:comment "Transitions")


  (tell '((:rules people
	       ((temperature ?x ?t)
		<-
		(:ask (temperature ?x ?t)))
	       ((has-symptom ?x low-fever)
		<-
		(temperature ?x ?t)
		(:test (and (> ?t 99) (< ?t 102))))
	       ((has-symptom ?x high-fever)
		<-
		(temperature ?x ?t)
		(:test (or (= ?t 102) (> ?t 102))))
	       ;;
	       ((has-symptom ?x tiredness)
		<-
		(:ask (has-symptom ?x tiredness)))
	       ((has-symptom ?x cough)
		<-
		(:ask (has-symptom ?x cough)))))
	:comment "Symptoms")

  (tell '((test mono mono-test)
	      (test pneumonia xray)
	      ;;
	      ;; Performing a test:
	      (:rules tests
	       ((result ?test ?patient ?result)
		<-
		;; Only instruct user to perform test if no value
		;; is known at all:
		(:unp (:retrieve (result ?test ?patient ?any-result)))
		(:eval (format t "~% Apply test ~(~a~).~%" '?test))
		(:ask (result ?test ?patient ?result)))
	       
	       ;; These last two rules should be handled in a more
	       ;; general way:
	       ((result ?test ?patient positive)
		->
		(not (result ?test ?patient negative)))
	       ((result ?test ?patient negative)
		->
		(not (result ?test ?patient positive)))))
	:comment "Tests")


  ;; For each state in the fsm we have three types of rules:
  ;;   <back-chaining rules to fill the slot for the state>
  ;;   <forward-chaining rules to print some tracing information>
  ;;   <transition rules to find the next state>
  ;;
  (tell '((:rules people
	       ;;
	       ;; Finding the probable-diagnosis:
	       ((probable-diagnosis ?p flu)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct diagnosing)
		(has-symptom ?p low-fever))
	       ((probable-diagnosis ?p flu)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct diagnosing)
		(not (diagnosis ?p pneumonia))
		(has-symptom ?p high-fever))
	       ((probable-diagnosis ?p mono)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct diagnosing)
		(has-symptom ?p low-fever)
		(has-symptom ?p tiredness))
	       ((probable-diagnosis ?p pneumonia)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct diagnosing)
		(has-symptom ?p high-fever)
		(has-symptom ?p cough))
	       ;;
	       ;; Output
	       ((probable-diagnosis ?p ?x)
		->
		(:eval (format t "~% ~(~a~) may have ~(~a~).~%" '?p '?x))))
	      ;;
	      ;; Transition rule:
	      (:rules times
	       ((state ?t diagnosing)
		(current-context global-context ?cc)
		(patient ?cc ?pat)
		(probable-diagnosis ?pat ?d)
		->
		(next-state diagnosing ?t testing))))
	:comment "Diagnosing")


  (tell '((:rules people
	       ;;
	       ;; Finding the diagnosis:
	       ((diagnosis ?p ?d)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct testing)
		(probable-diagnosis ?p ?d)
		(test ?d ?test)
		(result ?test ?p positive))

	       ((not (diagnosis ?p ?d))
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct testing)
		(probable-diagnosis ?p ?d)
		(test ?d ?test)
		(result ?test ?p negative))
	       ;; No test exists:
	       ((diagnosis ?p ?d)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct testing)
		(probable-diagnosis ?p ?d)
		(:unp (test ?d ?test))
		;; And everything else has been rules out:
		(:all-paths ((probable-diagnosis ?p ?d2) (:neq ?d ?d2))
		            ((not (diagnosis ?p ?d2)))))
	       ;;
	       ;; Output
	       ((diagnosis ?p ?x)
		->
		(:eval (format t "~% ~(~a~) has ~(~a~).~%" '?p '?x))))
	      ;;
	      ;; Transition rules:
	      (:rules times
	       ((state ?t testing)
		(current-context global-context ?cc)
		(patient ?cc ?pat)
		(diagnosis ?pat ?d)
		->
		(next-state testing ?t Prescribing))
	       ((state ?t testing)
		(current-context global-context ?cc)
		(patient ?cc ?patient)
		(not (diagnosis ?patient ?d))
		;; If all probable-diagnosis's fail then its back to
		;; the drawing board:
		(:all-paths ((probable-diagnosis ?patient ?diag))
		            ((not (diagnosis ?patient ?diag))))
		->
		(next-state testing ?t diagnosing))))
	:comment "Testing")


  (tell '((:rules people
	       ;;
	       ;; Finding the prescription:
	       ((prescription ?p rest)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct Prescribing)
		(diagnosis ?p flu))
	       ((prescription ?p lots-of-rest)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct Prescribing)
		(diagnosis ?p mono))
	       ((prescription ?p  penicillin)
		<-
		(current-context global-context ?cc)
		(current-time ?cc ?ct)
		(state ?ct Prescribing)
		(diagnosis ?p pneumonia))
	       ;;
	       ;; Output
	       ((prescription ?p ?x)
		->
		(:eval (format t "~% The prescription for ~(~a~) is ~(~a~).~%" '?p '?x))))
	      ;;
	      ;; Transition rule:
	      (:rules times
	       ((state ?t Prescribing)
		(current-context global-context ?cc)
		(patient ?cc ?pat)
		(prescription ?pat ?p)
		->
		(next-state Prescribing ?t final-state))))
	:comment "Prescribing"))


(defun queries-about-expert ()

  (tell '((:a (?con contexts) (name ?con "CONTEXT"))
	  (:a (?j People) (name ?j "JOHN"))
	  (:clear-slot global-context current-context)
	  (current-context global-context ?con)
	  (patient ?con ?j)
	  (current-time ?con ?ct)
	  (state ?ct diagnosing))
	:comment "Diagnosing John.")
  )

