;;; Copyright 1993, Benjamin Kuipers

#+(or GCL LUCID) (in-package :user)
#-(or GCL LUCID) (in-package :cl-user)

;;;                        ****** FLU ******
;;;
;;; This example illustrates the use of Algernon to build a very simple
;;; expert system.  In particular it illustrates the use of the :ask
;;; form to query the user.

(defun facts-about-flu ()

  (acom-reset)
  
  (with-normal-output
      
  (tell '((:taxonomy (Objects
		      (Symptoms Fever High-Fever Nausea Nodules)
	              (Physical-Objects
		       (People)
		       (Diseases Flu Plague)))))
	:comment "Taxonomy")

  ;; Four new slots:
  ;;  (has-disease x d) = x has disease d.
  ;;  (has-symptom x s) = x has symptom s.
  ;;  (temperature x t) = x has temperature t.
  ;;  (symptom d s)     = disease d causes symptom s.
  ;;
  (tell '((:slot has-disease (people diseases))
	  (:slot has-symptom (people symptoms))
	  (:slot symptom     (diseases symptoms))
	  (:slot temperature (physical-objects :number) :cardinality 1))
	:comment "Slot specifications")

  ;; In the diagnosis rule below we will need to be able to access all
  ;; diseases so store all diseases in the member slot of the set of
  ;; diseases: (this is not done for all sets since some sets (e.g.
  ;; things) could have thousands of members).
  ;;
  (tell '((:rules Diseases
	   ((isa ?d1 Diseases) -> (member Diseases ?d1))))
	:comment "Backlink diseases")

  (tell '((:rules People
	   ((has-symptom ?p fever)      <- (temperature ?p ?t) (:test (> ?t 99)))
	   ((has-symptom ?p high-fever) <- (temperature ?p ?t) (:test (> ?t 102)))
	   ((has-symptom ?p nausea)     <- (:ask (has-symptom ?p nausea)))
	   ((has-symptom ?p nodules)    <- (:ask (has-symptom ?p nodules)))
	   ((temperature ?p ?t)         <- (:ask (temperature ?p ?t)))))
	:comment "Recognizing symptoms")

  (tell '((:rules People
	   ;; A person has a disease if they have all symptoms of the disease ...
	   ((has-disease ?x ?d1)
	    <-
	    (member Diseases ?d1)
	    (:all-paths ((symptom ?d1 ?s)) ((has-symptom ?x ?s))))))
	:comment "General-purpose diagnostic rules")

  (tell '((:rules People
	   ((has-disease ?x Flu)     <-  (has-symptom ?x fever) (has-symptom ?x nausea))
	   ((has-disease ?x Plague)  <-  (has-symptom ?x high-fever) (has-symptom ?x nodules))
	   ))
	:comment "Special-purpose diagnostic rules")

  (tell '((symptom Flu Nausea)	  (symptom Plague Nodules)
	  (symptom Flu Fever)	  (symptom Plague High-Fever)
	  )
	:comment "Symptoms of known diseases")
  ))


(defun queries-about-flu ()
  (dx-patient 'John)
  )


(defun dx-patient (name)

  ;; Now we access the frame for the person John and find out
  ;; what disease he has:
  ;;
  (ask `((:the ?j (name ?j ,(string name)) (isa ?j People))
	 (has-disease ?j ?d))
       :comment "What does the patient have?"
       :collect '(?j has ?d)))
