;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+(or GCL LUCID) (in-package :user)
#-(or GCL LUCID) (in-package :cl-user)


;;; FINITE-STATE MACHINE
;;;
;;; Primarily intended for embedding in expert systems.
;;;
;;; When embedded the assertion "User interface" would be replaced by
;;; the expert system which must then provide rules to answer the query
;;; "(input ?t ?i)" (i.e. "What is the next input ?").  Such queries are
;;; generated by this "routine" until a final state is reached and are
;;; intended to "drive" the expert system.
;;;
;;; Algernon is logic based and is generally monotonic.  What this means
;;; in practice is that liberal use of :clear-slot or :delete can cause
;;; strange behavior.  The fsm gets around this by simulating an unbounded
;;; "linked list" of "times" (linked by the relation next-time) and "running"
;;; the simulation along this list.  Thus the simulation is actually monotonic.
;;;
(defun facts-about-fsm ()

  (acom-reset)
  
  (a-assert "Taxonomy"
	    '((:taxonomy (objects
			  (times fsm-time)
			  (states (non-final-states)
				  (final-states final-state))
			  (inputs)))))

  ;; New slots:
  ;;  (next-time t1 t2) = The time after t1 is t2.
  ;;  (state t s) = The state at time t is s.
  ;;  (input t i) = The input to the fsm at time t is i.
  ;;  (next-state s1 i s2) = The next state after s1 under input i is s2.
  
  (a-assert "New slots"
	    '((:slot next-time (times times) :cardinality 1)
	      (:slot state (times states) :cardinality 1)
	      (:slot input (times inputs) :cardinality 1)
	      (:slot next-state (states inputs states))))


  (a-assert "Time"
	    '((:rules times
	       (;; There's always next time:
		(next-time ?t1 ?t2) <- (:A (?t2 TIMES) (name ?t2 "fsm-time"))))))


  ;; Transitions are done as for fsm's in the literature ...
  ;;
  (a-assert "Transitions"
	    '((:rules times
	       ;; When we know everything about current time then we can go to next time:
	       ((state ?t1 ?s1) (isa ?s1 non-final-states)
		(next-time ?t1 ?t2)
		(input ?t1 ?i1)
		(next-state ?s1 ?i1 ?s2)
		->
		(state ?t2 ?s2)))))

  (a-assert "User interface"
	    '((:rules times
	       ((state ?t ?s) -> (:eval (format t "~% State of ~S is ~(~a~).~%" '?t '?s))))
	      (:rules final-states
	       ((state ?t ?s) -> (:eval (format t "~%~S reached final state.~%" '?t))))
	      (:rules times
	       ((input ?t ?i) <- (:ask (input ?t ?i)))))))

(defun queries-about-fsm ()
  ;; Simple fsm to recognize an "aba".

  (tell '((:taxonomy (non-final-states s1 s2 s3))
              (:taxonomy (inputs a b)))
            :comment "States")

  (tell '((next-state s1 a s2)
	      (next-state s2 b s3)
	      (next-state s3 a final-state)
	      (next-state s1 b s1)
	      (next-state s2 a s2)
	      (next-state s3 b s1))
	    :comment "Transitions")

  (tell  '((state fsm-time s1))
	    :comment "Initialize and run.")
  )



;;  (a-assert "States"
;;            '((:taxonomy (non-final-states s1 s2 s3))
;;              (:taxonomy (inputs a b))))
;;  (a-assert "Transitions"
;;	    '((next-state s1 a s2)
;;	      (next-state s2 b s3)
;;	      (next-state s3 a final-state)
;;	      (next-state s1 b s1)
;;	      (next-state s2 a s2)
;;	      (next-state s3 b s1)))
;;  (a-assert "Initialize and run."
;;	    '((state fsm-time s1))))

